(* RUN TIME OBJECTS *)

(* The type of a source reference: line number, filename *)
type sref = int * string;;
type 'a comparison_t' = 'a -> 'a -> bool;;

(* ABSTRACTIONS *)

class type ['expr_t] dictionary_t' =
  object 
    method del_item : 'expr_t -> bool
    method get_item : 'expr_t -> 'expr_t option
    method has_item : 'expr_t -> bool
    method set_item : 'expr_t -> 'expr_t -> bool

    method len : int
    method iter: ('expr_t -> 'expr_t -> unit) -> unit
    method clear : unit
    method keys : 'expr_t list
    method sorted_keys : 'expr_t comparison_t' -> 'expr_t list
    method cmp : 'expr_t comparison_t' -> 'expr_t dictionary_t' -> bool
  end

class type ['expr_t] environment_t'  =
  object
    (* python compatible name lookup *)
    method get_attr : 'expr_t -> 'expr_t option
    method set_attr : 'expr_t -> 'expr_t -> bool
    method del_attr : 'expr_t -> bool
    method has_attr : 'expr_t -> bool

    (* test what will work *)
    method can_get_attr : 'expr_t -> bool
    method can_set_attr : 'expr_t -> 'expr_t -> bool
    method can_set_any_attr : 'expr_t -> bool
    method can_del_attr : 'expr_t -> bool

    (* list of keys for dir() function *)
    method keys : 'expr_t list

    (* python compatible globals() and locals() support *)
    method get_locals: 'expr_t dictionary_t'
    method get_globals: 'expr_t dictionary_t'

    (* indexed lookup *)
    method get_indexed: int -> int -> 'expr_t
    method set_indexed: int -> int -> 'expr_t -> unit
    method del_indexed: int -> int -> unit

    (* get underlying module *)
    method get_module: 'expr_t module_t'
  end

and ['expr_t] module_t' = 
  object
    method get_name : string list
    method get_dotted_name : string
    method get_filename : string option
    method get_file_directory : string option
    method get_dictionary : 'expr_t dictionary_t'
    method get_attr : 'expr_t -> 'expr_t option
    method get_environment: 'expr_t environment_t' option
    
    method get_index_to_global_name : string array
    method get_global_name_to_index : int Py_mtypes.VarMap.t
    method get_vars : 'expr_t array

    method get_indexed: int -> int -> 'expr_t
    method set_indexed: int -> int -> 'expr_t -> unit
    method del_indexed: int -> int -> unit
  end

class type ['expr_t] class_t' = 
  object 
    method get_name : string
    method get_bases : 'expr_t class_t' list
    method get_environment: 'expr_t environment_t'
    method get_dictionary: 'expr_t dictionary_t'
    method get_methods: 'expr_t dictionary_t'
    method get_attr : 'expr_t -> 'expr_t option

    method get_index_to_global_name : string array
    method get_global_name_to_index : int Py_mtypes.VarMap.t
    method get_vars : 'expr_t array

    (* we should have a create instance method here *)
  end

class type ['expr_t] instance_t' =
  object 
    method can_get_attr : 'expr_t -> bool
    method get_attr : 'expr_t -> 'expr_t option

    method can_set_attr : 'expr_t -> 'expr_t -> bool
    method can_set_any_attr : 'expr_t -> bool
    method set_attr : 'expr_t -> 'expr_t -> bool

    method can_del_attr : 'expr_t -> bool
    method del_attr : 'expr_t -> bool

    method get_class : 'expr_t class_t'
    method get_dictionary : 'expr_t dictionary_t'
  end

type 'expr_t var_t' = { name: string; value: 'expr_t }

class type ['expr_t, 'parameters_t, 'statement_t] function_t' =
  object
    method can_get_attr : 'expr_t -> bool
    method get_attr : 'expr_t -> 'expr_t option

    method can_set_attr : 'expr_t -> 'expr_t -> bool
    method can_set_any_attr : 'expr_t -> bool
    method set_attr : 'expr_t -> 'expr_t -> bool

    method get_name : string
    method get_parameters: 'parameters_t
    method get_code : 'statement_t
    method get_environment: 'expr_t environment_t'

    method get_global_names : string list
    method get_index_to_local_name : string array
    method get_local_name_to_index : int Py_mtypes.VarMap.t

  end

class type ['expr_t] interpreter_t' =
  object
    (* source tracking *)
    method set_line: sref -> unit
    method push_line: unit
    method pop_line: unit

    (* trace control *)
    method incr_tracecount : unit
    method decr_tracecount : unit
    method get_tracecount : int

    (* system access *)
    method get_builtins_dictionary: 'expr_t dictionary_t'
    method get_builtins_environment : 'expr_t environment_t'
    method get_sys : 'expr_t module_t'

    (* exception handling *)
    method py_raise : exn -> unit 
    method set_exc_info : 'expr_t -> 'expr_t -> 'expr_t -> unit
    method get_exc_info : 'expr_t
    method clear_exc_info : unit
    method get_traceback: sref list 
    method print_tb : unit
    method print_exc_tb : unit

    (* type control *)
    method type_of_object : 'expr_t -> 'expr_t
    method type_with_name : string -> 'expr_t

    (* execution *)
    method import: 'expr_t module_t' -> string list -> 'expr_t module_t' * 'expr_t module_t'
    method run_file : string -> int 
    method run_string : string -> int
  end
;;

(* LANGUAGE TERMS *)

type dict_elt_t = expr_t * expr_t
and star_param_t =  StarParam of string | NoStarParam
and starstar_param_t = StarStarParam of string | NoStarStarParam
and param_name_t =
  Param of string
  | Paramtuple of param_name_t list
and argument_t = 
  Argument2 of string * expr_t
  | Argument1 of expr_t 
and parameter_t = 
  Parameter2 of param_name_t * expr_t
  | Parameter1 of param_name_t 
and parameters_t = parameter_t list * star_param_t * starstar_param_t
and subscript_entry =
  Defsub
  | Pos of expr_t 
and subscript_t = 
  Ellipsis
  | Subscript2 of subscript_entry * subscript_entry * subscript_entry
  | Subscript1 of subscript_entry * subscript_entry 
  | Subscript0 of subscript_entry 
and except_t= 
  Except0
  | Except1 of sref * expr_t
  | Except2 of sref * expr_t * expr_t
and handler_t = except_t * statement_t
and trailer_t = 
  Arglist of argument_t list
  | Dotname of string
  | Sublist of subscript_t list
and statement_t = 
  | Empty
  | While of sref * expr_t * statement_t * statement_t
  | For of sref * expr_t * expr_t * statement_t * statement_t
  | Def of sref * string * parameters_t * statement_t
  | Class of sref * string * expr_t * statement_t
  | TryFinally of statement_t * statement_t
  | TryElse of statement_t * handler_t list * statement_t
  | Global of sref * string list
  | Break of sref
  | Continue of sref
  | Return of sref * expr_t 
  | Raise0 of sref
  | Raise1 of sref * expr_t
  | Raise2 of sref * expr_t * expr_t
  | Raise3 of sref * expr_t * expr_t * expr_t
  | Print of  sref * expr_t list
  | PrintComma of  sref * expr_t list
  | Pass
  | Exec1 of sref * expr_t
  | Exec2 of sref * expr_t * expr_t
  | Exec3 of sref * expr_t * expr_t * expr_t
  | Assert2 of sref * expr_t * expr_t
  | Assert1 of sref * expr_t 
  | Suite of statement_t list
  | Import of sref * string list list
  | ImportFrom of sref * string list * string list
  | ImportAll of sref * string list 
  | If of (sref * expr_t * statement_t) list * statement_t 
  | Assign of sref * expr_t list
  | Del of sref * expr_t
  | Expr of sref * expr_t


  (* Viper extended assignments *)
  | PlusEqual of sref * expr_t * expr_t 
  | MinusEqual of sref * expr_t * expr_t 
  | StarEqual of sref * expr_t * expr_t 
  | SlashEqual of sref * expr_t * expr_t 
  | PercentEqual of sref * expr_t * expr_t 
  | AmperEqual of sref * expr_t * expr_t 
  | VbarEqual of sref * expr_t * expr_t 
  | CaretEqual of sref * expr_t * expr_t 
  | ColonEqual of sref * expr_t * expr_t 
  | LeftShiftEqual of sref * expr_t * expr_t 
  | RightShiftEqual of sref * expr_t * expr_t 
  | PlusPlus of sref * expr_t 
  | MinusMinus of sref * expr_t 

and comparator_t = 
  Less of expr_t
  | Greater of expr_t
  | LessEqual of expr_t
  | GreaterEqual of expr_t
  | Equal of expr_t
  | NotEqual of expr_t
  | Is of expr_t
  | IsNot of expr_t
  | In of expr_t
  | NotIn of expr_t
and binop_t =
  Add of expr_t 
  | Sub of expr_t 
  | Mul of expr_t 
  | Div of expr_t 
  | Mod of expr_t 
  | Asl of expr_t 
  | Lsr of expr_t 
  | Pow of expr_t 
and dict_t = (string, expr_t) Hashtbl.t
and expr_t = 
  | PyNone
  | PyName of string
  | PyVarIndex of int * int
  | PyString of string
  | PyInt of int
  | PyLong of Big_int.big_int
  | PyRational of Num.num
  | PyFloat of float
  | PyComplex of float * float
  | PyRegexp of Pcre.regexp
  | PyTuple of expr_t list
  | PyList of expr_t list 
  | PyMutableList of pylist_t 
  | PyDict of dictent_t list
  | PyFile of Unix.file_descr
  | PyRepr of expr_t 
  | Or of expr_t list
  | And of expr_t list
  | Not of expr_t 
  | Neg of expr_t 
  | Compare of expr_t * comparator_t list
  | BitOr of expr_t list
  | BitAnd of expr_t list
  | BitXor of expr_t list
  | Complement of expr_t 
  | Eval of expr_t * binop_t list
  | AtomWithTrailers of expr_t * trailer_t list
  | Lambda of parameters_t * expr_t 
  | IntRange of int * int * int
  | PyDictionary of expr_t dictionary_t'
  | PyModule of expr_t module_t'
  | PyClass of expr_t class_t'
  | PyFunction of (expr_t, parameters_t, statement_t) function_t'
  | PyNativeMacro of string * (
    expr_t environment_t' -> 
    expr_t interpreter_t' -> 
    expr_t list -> 
    expr_t dictionary_t' 
    -> expr_t
  )
  | PyNativeFunction of string * (
    expr_t interpreter_t' -> 
    expr_t list -> 
    expr_t dictionary_t' 
    -> expr_t
  )
  | PyInstance of expr_t instance_t'
  | PyBoundMethod of expr_t * expr_t 
  | PyTraceback of (int * string) list
  | PyStatement of statement_t

  | PyInitial (* less than all other objects *)
  | PyTerminal (* greater than all other objects *)

  | PyInterpreter of expr_t interpreter_t'
  | PyEnv of expr_t environment_t'
  | PyClosure of expr_t environment_t' * expr_t

  (* GUI and graphics *)
  | PyWidget of Gtk.Unsafe.gtkobject
  | PyFont of Gdk.font
  | PyColor of Gdk.color 
  | PyImage of Gdk.image
  | PyGraphicsContext of Gdk.gc
  | PyCanvas of Gdk.drawable

  (* GDK defines some other types: visual, pixmap, colormap, bitmap, point ..
    we don't need these hopefully :-*)

  (* who knows what to do with this? *)
  | Unknown

and dictent_t = 
  DictEnt of expr_t * expr_t
and pylist_t = expr_t Varray.varray_t

type var_t = expr_t var_t'

class type virtual dictionary_t = [expr_t] dictionary_t'
class type virtual environment_t = [expr_t] environment_t'
class type virtual function_t = [expr_t, parameters_t, statement_t] function_t'
class type virtual module_t = [expr_t] module_t'
class type virtual instance_t = [expr_t] instance_t'
class type virtual class_t = [expr_t] class_t'
class type virtual interpreter_t = [expr_t] interpreter_t'
type comparison_t = expr_t comparison_t'

(* exceptions *)
type py_exception = 
| PyStringException of string * expr_t
| PyInstanceException of instance_t

type py_native_function =
  interpreter_t ->
  expr_t list -> 
  dictionary_t ->
  expr_t

type py_native_macro =
  environment_t ->
  interpreter_t ->
  expr_t list -> 
  dictionary_t ->
  expr_t


