open Py_types
open Py_exceptions
open Py_builtins_util
open Util

let py_time_time
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 0;
  PyFloat (Unix.time())
;;

let py_time_clock
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 0;
  PyFloat (Unix.time())
;;

module Time = struct
open Unix
let py_time tm = PyTuple [
    PyInt tm.tm_year;      (* 0 year *)
    PyInt (tm.tm_mon+1);   (* 1 month 1-12 *)
    PyInt tm.tm_mday;      (* 2 day of month 1-31 *)
    PyInt tm.tm_hour;      (* 3 hour 0-23 *)
    PyInt tm.tm_min;       (* 4 minute 0-59 *)
    PyInt tm.tm_sec;       (* 5 second 0-59 *)
    PyInt tm.tm_wday;      (* 6 day of week; sun=0 *)
    PyInt tm.tm_yday;      (* 7 day of yearr 1-366 *)
    PyInt (int_of_bool tm.tm_isdst) (* 8 daylight saving is in effect=1 *)  
  ]

let mktime year mon mday hour min sec wday yday isdst =
      PyFloat (fst (Unix.mktime {
        tm_year=year;
        tm_mon=mon;
        tm_mday=mday;
        tm_hour=hour;
        tm_min=min;
        tm_sec=sec;
        tm_wday=wday;
        tm_yday=yday;
        tm_isdst=isdst
      }))

end

let py_time_gmtime
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  let tm = 
    match arg e 0 with
    | PyFloat f -> Unix.gmtime f
    | _ -> raise (TypeError "gmtime requires float for time")
  in Time.py_time tm


let py_time_localtime
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  let tm = 
    match arg e 0 with
    | PyFloat f -> Unix.localtime f
    | _ -> raise (TypeError "localtime requires float for time")
  in Time.py_time tm


let py_time_mktime 
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  match arg e 0 with
  | PyTuple [
      PyInt year;
      PyInt mon;
      PyInt mday;
      PyInt hour;
      PyInt min;
      PyInt sec;
      PyInt wday;
      PyInt yday;
      PyInt isdst
    ] -> Time.mktime year (mon-1) mday hour min sec wday yday (isdst<>0)
  | _ -> raise (TypeError "mktime requires tuple of 9 integers for time")


let py_time_sleep
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  match arg e 0 with
  | PyInt secs -> Unix.sleep secs; PyNone
  | _ -> raise (TypeError "sleep requires integer seconds to sleep")

let dayname day = match day with
  | 0 -> "Sun"
  | 1 -> "Mon"
  | 2 -> "Tue"
  | 3 -> "Wed"
  | 4 -> "Thu"
  | 5 -> "Fri"
  | 6 -> "Sat"
  | _ -> raise (ValueError "day must be 0-6")

let monthname month = match month with
  | 1 -> "Jan"
  | 2 -> "Feb"
  | 3 -> "Mar"
  | 4 -> "Apr"
  | 5 -> "May"
  | 6 -> "Jun"
  | 7 -> "Jul"
  | 8 -> "Aug"
  | 9 -> "Sep"
  | 10 -> "Oct"
  | 11 -> "Nov"
  | 12 -> "Dec"
  | _ -> raise (ValueError "month must be 0-6")

let digits i =
  let s = "00" ^ (string_of_int i)
  in let n = String.length s 
  in String.sub s (n-2) 2

let py_time_asctime
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  match arg e 0 with
  | PyTuple [
      PyInt year;
      PyInt mon;
      PyInt mday;
      PyInt hour;
      PyInt min;
      PyInt sec;
      PyInt wday;
      PyInt yday;
      PyInt isdst
    ] -> PyString (
      (dayname wday) ^ " " ^
      (monthname mon) ^ " " ^
      (digits mday) ^ " " ^
      (digits hour) ^ ":" ^
      (digits min) ^ ":" ^
      (digits sec) ^ " " ^
      (string_of_int year)
    )
  | _ -> raise (TypeError "asctime requires tuple of 9 integers for time")


