(* builtin functions for struct module *)
open Py_types
open Py_exceptions
open Py_builtins_util

let standard_sizeof code = 
  match code with
  | 'x' -> 1 (* pad *)
  | 'c' -> 1 (* char *)
  | 'b' -> 1 (* schar *)
  | 'B' -> 1 (* char *)
  | 'h' -> 2 (* short *)
  | 'H' -> 2 (* ushort *)
  | 'i' -> 4 (* int *)
  | 'I' -> 4 (* uint *)
  | 'l' -> 4 (* long *)
  | 'L' -> 4 (* ulong *)
  | 'f' -> 4 (* float *)
  | 'd' -> 8 (* double *)
  | 's' -> 1 (* string *)
  | 'p' -> 4 (* pascal string *)
  | 'P' -> 4 (* void* *)
  | _ -> raise (ValueError ("Unknown code '" ^ String.make 1 code ^ "' in struct fmt string"))
;;

(* WARNING: this table is set up for x86 machines *)
let native_sizeof code = 
  match code with
  | 'x' -> 1 (* pad *)
  | 'c' -> 1 (* char *)
  | 'b' -> 1 (* schar *)
  | 'B' -> 1 (* char *)
  | 'h' -> 2 (* short *)
  | 'H' -> 2 (* ushort *)
  | 'i' -> 4 (* int *)
  | 'I' -> 4 (* uint *)
  | 'l' -> 4 (* long *)
  | 'L' -> 4 (* ulong *)
  | 'f' -> 4 (* float *)
  | 'd' -> 8 (* double *)
  | 's' -> 1 (* string *)
  | 'p' -> 4 (* pascal string * *)
  | 'P' -> 4 (* void* *)
  | _ -> raise (ValueError ("Unknown code '" ^ String.make 1 code ^ "' in struct fmt string"))
;;  

let standard_alignment code = 1;;

(* WARNING: this is compiler and machine dependent *)
let native_alignment code = 
  match code with
  | 'x' -> 1 (* pad *)
  | 'c' -> 1 (* char *)
  | 'b' -> 1 (* schar *)
  | 'B' -> 1 (* char *)
  | 'h' -> 2 (* short *)
  | 'H' -> 2 (* ushort *)
  | 'i' -> 4 (* int *)
  | 'I' -> 4 (* uint *)
  | 'l' -> 4 (* long *)
  | 'L' -> 4 (* ulong *)
  | 'f' -> 4 (* float *)
  | 'd' -> 8 (* double *)
  | 's' -> 1 (* string *)
  | 'p' -> 4 (* pascal string * *)
  | 'P' -> 4 (* void* *)
  | _ -> raise (ValueError ("Unknown code '" ^ String.make 1 code ^ "' in struct fmt string"))
;;


let isdigit x = match x with '0'..'9' -> true | _ -> false;;

let parse_int s i' =
  let acc = ref 0 and i = ref i' and n = String.length s in
  while !i < n && isdigit s.[!i] do
    acc := !acc * 10 + int_of_char s.[!i];
    incr i
  done;
  !acc, !i
;;

let calcsize fmt = 
  let n = String.length fmt in
  if n < 1 then raise (ValueError "Struct format string is empty");
  let code = fmt.[0] in
  let alignment, sizeof = match code with
  | '@' -> native_alignment, native_sizeof
  | '=' | '<' | '>' | '!' -> standard_alignment, standard_sizeof
  | _ -> raise 
    (ValueError 
      ("Calcsize format string must start with @=<>!, found '" ^
        (String.make 1 code)^ 
        "'"
        )
      )
  in let i = ref 1 and size = ref 0 in
  let align j = 
    if !size mod j <> 0 
    then size := !size + j - (!size mod j)
  in while !i < n do
    let count = ref 1 in
    let code = ref fmt.[!i] in
    if isdigit !code 
    then begin 
      let v,j = parse_int fmt !i in i:=j; count := v;
      if j >= n 
      then raise (ValueError "struct fmt string ended after integer, code required");
      code := fmt.[j]
    end;
    align (alignment !code);
    size := !size + !count * sizeof !code;
    incr i
  done;
  !size
;;

let py_struct_calcsize
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  match arg e 0 with
  | PyString s -> PyInt (calcsize s)
  | _ -> raise (ValueError "struct_calcsize requires string argument for format")
;;

let pack_s data offset count arg = 
  match arg with
  | PyString s -> 
    let n = String.length s in
    let s' = s ^ (String.make n (char_of_int 0)) in
    for i = 0 to n-1 do
      data.[offset + i] <- s'.[i]
    done
  | _ -> raise (TypeError "Struct pack 'c' format code requires string")
;;

let pack_p data offset count arg = 
  match arg with
  | PyString s -> 
    let n = String.length s in
    let s' = s ^ (String.make n (char_of_int 0)) in
    for i = 0 to n-1 do
      data.[offset + i+1] <- s'.[i];
      data.[offset] <- char_of_int (n+1)
    done
  | _ -> raise (TypeError "Struct pack 'c' format code requires string")
;;

type endien = Big | Little;;

let py_struct_pack
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  atleast e 1;
  let n = argcount e in
  let fmt = 
    match arg e 0 with
    | PyString s -> s
    | _ -> raise (ValueError "struct_pack requires string argument for format")
  in
    let args = ref (List.tl e) in
    let ssize = calcsize fmt in
    let data = String.make ssize (char_of_int 0) in
    let n = String.length fmt in
    if n < 1 then raise (ValueError "Struct format string is empty");
    let code = fmt.[0] in
    let alignment, sizeof, endienness = match code with
    | '@' -> native_alignment, native_sizeof, Little
    | '=' -> standard_alignment, standard_sizeof, Big
    | '<' -> standard_alignment, standard_sizeof, Little 
    | '>' -> standard_alignment, standard_sizeof, Big
    | '!' -> standard_alignment, standard_sizeof, Big
    | _ -> raise 
      (ValueError 
        ("struct pack format string must start with @=<>!, found '" ^
          (String.make 1 code)^ 
          "'"
          )
        )
    in let i = ref 1 and used = ref 0 in
    let align j = 
      if !used mod j <> 0 
      then used := !used + j - (!used mod j)
    in while !i < n do
      let count = ref 1 in
      let code = ref fmt.[!i] in
      if isdigit !code 
      then begin 
        let v,j = parse_int fmt !i in i:=j; count := v;
        if j >= n 
        then raise (ValueError "struct fmt string ended after integer, code required");
        code := fmt.[j]
      end;

      (* align write position *)
      align (alignment !code);

      (* handle strings *)
      if !code = 's' 
      then begin 
        pack_s data !used !count (List.hd !args);
        used := !used + !count;
        args := List.tl !args
      end else if !code = 'p'
      then begin
        pack_p data !used !count (List.hd !args);
        used := !used + !count;
        args := List.tl !args
      end 
      else 
      
      (* handle other cases *)
      for k = 0 to !count - 1 do
        match !code with
        | 'x' -> (* pad *) ()
        | 'c' -> (* char *) 
          data.[!used] <-  
          begin match List.hd !args with 
          | PyInt c -> char_of_int c 
          | _ -> raise (TypeError "struct_pack 'c' format requires integer")
          end
        | 'b' -> (* schar *)
          data.[!used] <-  
          begin match List.hd !args with 
          | PyInt c -> char_of_int c 
          | _ -> raise (TypeError "struct_pack 'b' format requires integer")
          end
        | 'B' -> (* uchar *)
          data.[!used] <-  
          begin match List.hd !args with 
          | PyInt c -> char_of_int c 
          | _ -> raise (TypeError "struct_pack 'B' format requires integer")
          end
        | 'h' -> (* short *)
          begin match List.hd !args with 
          | PyInt i -> 
            begin match endienness with
            | Little ->
              data.[!used] <- char_of_int (i land 0xFF);
              data.[!used+1] <- char_of_int ((i lsr 8) land 0xFF);
            | Big ->
              data.[!used+1] <- char_of_int (i land 0xFF);
              data.[!used] <- char_of_int ((i lsr 8) land 0xFF);
             end
          | _ -> raise (TypeError "struct_pack 'h' format requires integer")
          end
        | 'H' -> (* ushort *)
          begin match List.hd !args with 
          | PyInt i -> 
            begin match endienness with
            | Little ->
              data.[!used] <- char_of_int (i land 0xFF);
              data.[!used+1] <- char_of_int ((i lsr 8) land 0xFF);
            | Big ->
              data.[!used+1] <- char_of_int (i land 0xFF);
              data.[!used] <- char_of_int ((i lsr 8) land 0xFF);
            end
          | _ -> raise (TypeError "struct_pack 'H' format requires integer")
          end
        | 'i' -> (* int *)
          begin match List.hd !args with 
          | PyInt i -> 
            begin match endienness with
            | Little ->
              data.[!used+0] <- char_of_int (i land 0xFF);
              data.[!used+1] <- char_of_int ((i lsr 8) land 0xFF);
              data.[!used+2] <- char_of_int ((i lsr 16) land 0xFF);
              data.[!used+3] <- char_of_int ((i lsr 24) land 0xFF);
            | Big ->
              data.[!used+3] <- char_of_int (i land 0xFF);
              data.[!used+2] <- char_of_int ((i lsr 8) land 0xFF);
              data.[!used+1] <- char_of_int ((i lsr 16) land 0xFF);
              data.[!used+0] <- char_of_int ((i lsr 24) land 0xFF);
            end
          | _ -> raise (TypeError "struct_pack 'i' format requires integer")
          end 
        | 'I' -> (* uint *)
          begin match List.hd !args with 
          | PyInt i -> 
            begin match endienness with
            | Little ->
              data.[!used+0] <- char_of_int (i land 0xFF);
              data.[!used+1] <- char_of_int ((i lsr 8) land 0xFF);
              data.[!used+2] <- char_of_int ((i lsr 16) land 0xFF);
              data.[!used+3] <- char_of_int ((i lsr 24) land 0xFF);
            | Big ->
              data.[!used+3] <- char_of_int (i land 0xFF);
              data.[!used+2] <- char_of_int ((i lsr 8) land 0xFF);
              data.[!used+1] <- char_of_int ((i lsr 16) land 0xFF);
              data.[!used+0] <- char_of_int ((i lsr 24) land 0xFF);
            end
          | _ -> raise (TypeError "struct_pack 'I' format requires integer")
          end 

        | 'l' -> (* long *)
          begin match List.hd !args with 
          | PyInt i -> 
            begin match endienness with
            | Little ->
              data.[!used+0] <- char_of_int (i land 0xFF);
              data.[!used+1] <- char_of_int ((i lsr 8) land 0xFF);
              data.[!used+2] <- char_of_int ((i lsr 16) land 0xFF);
              data.[!used+3] <- char_of_int ((i lsr 24) land 0xFF);
            | Big ->
              data.[!used+3] <- char_of_int (i land 0xFF);
              data.[!used+2] <- char_of_int ((i lsr 8) land 0xFF);
              data.[!used+1] <- char_of_int ((i lsr 16) land 0xFF);
              data.[!used+0] <- char_of_int ((i lsr 24) land 0xFF);
            end
          | _ -> raise (TypeError "struct_pack 'l' format requires integer")
          end 

        | 'L' -> (* ulong *)
          begin match List.hd !args with 
          | PyInt i -> 
            begin match endienness with
            | Little ->
              data.[!used+0] <- char_of_int (i land 0xFF);
              data.[!used+1] <- char_of_int ((i lsr 8) land 0xFF);
              data.[!used+2] <- char_of_int ((i lsr 16) land 0xFF);
              data.[!used+3] <- char_of_int ((i lsr 24) land 0xFF);
            | Big ->
              data.[!used+3] <- char_of_int (i land 0xFF);
              data.[!used+2] <- char_of_int ((i lsr 8) land 0xFF);
              data.[!used+1] <- char_of_int ((i lsr 16) land 0xFF);
              data.[!used+0] <- char_of_int ((i lsr 24) land 0xFF);
            end
          | _ -> raise (TypeError "struct_pack 'L' format requires integer")
          end 

        | 'f' -> raise (NotImplemented "struct_pack float") (* float *)
        | 'd' -> raise (NotImplemented "struct_pack double") (* double *)
        | 'P' -> raise (NotImplemented "struct_pack void* ") (* void* *)
        | _ -> raise (ValueError ("Unknown code '" ^ String.make 1 !code ^ "' in struct fmt string"))
        ;
        used := !used + sizeof !code;
        args := List.tl !args
      done;
      incr i
    done  
    ;
    PyString data
;;
  
let py_struct_unpack
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 2;
  let fmt = 
    match arg e 0 with
    | PyString s -> s
    | _ -> raise (ValueError "struct_pack requires string argument for format")
  and data =
    match arg e 1 with
    | PyString s -> s
    | _ -> raise (ValueError "struct_pack requires string argument for data")
  in
    let ssize = calcsize fmt in
    if ssize <> String.length data
    then raise (ValueError "struct_unpack: length of data does not agree with size computed from format")
    ;
    let n = String.length fmt in
    if n < 1 
    then raise (ValueError "Struct format string is empty")
    ;
    let code = fmt.[0] in
    let alignment, sizeof, endienness = match code with
    | '@' -> native_alignment, native_sizeof, Little
    | '=' -> standard_alignment, standard_sizeof, Big
    | '<' -> standard_alignment, standard_sizeof, Little 
    | '>' -> standard_alignment, standard_sizeof, Big
    | '!' -> standard_alignment, standard_sizeof, Big
    | _ -> raise 
      (ValueError 
        ("struct pack format string must start with @=<>!, found '" ^
          (String.make 1 code)^ 
          "'"
          )
        )
    in let i = ref 1 and used = ref 0 and result = ref [] in
    let align j = 
      if !used mod j <> 0 
      then used := !used + j - (!used mod j)
    in while !i < n do
      let count = ref 1 in
      let code = ref fmt.[!i] in
      if isdigit !code 
      then begin 
        let v,j = parse_int fmt !i in i:=j; count := v;
        if j >= n 
        then raise (ValueError "struct fmt string ended after integer, code required");
        code := fmt.[j]
      end;

      (* align read position *)
      align (alignment !code);

      (* handle strings *)
      if !code = 's' 
      then begin 
        result := PyString (String.sub data !used !count) :: !result;
        used := !used + !count;
      end else if !code = 'p'
      then begin
        result := PyString (String.sub data (!used+1) !count) :: !result;
        used := !used + !count;
      end 
      else 
      
      (* handle other cases *)
      for k = 0 to !count - 1 do
        match !code with
        | 'x' -> (* pad *) ()
        | 'c' -> (* char *) 
          result := PyInt (int_of_char data.[!used]) :: !result
        | 'b' -> (* schar *)
          result := PyInt (int_of_char data.[!used]) :: !result
        | 'B' -> (* uchar *)
          result := PyInt (int_of_char data.[!used]) :: !result
        | 'h' -> (* short *)
          begin match endienness with
          | Little ->
            result := PyInt (
              int_of_char data.[!used] +
              int_of_char data.[!used+1] * 256
            ) :: !result
          | Big ->
            result := PyInt (
              int_of_char data.[!used+1] +
              int_of_char data.[!used] * 256
            ) :: !result
          end
        | 'H' -> (* ushort *)
          begin match endienness with
          | Little ->
            result := PyInt (
              int_of_char data.[!used] +
              int_of_char data.[!used+1] * 256
            ) :: !result
          | Big ->
            result := PyInt (
              int_of_char data.[!used+1] +
              int_of_char data.[!used] * 256
            ) :: !result
          end
        | 'i' -> (* int *)
          begin match endienness with
          | Little ->
            result := PyInt (
              int_of_char data.[!used] +
              256 * (int_of_char data.[!used+1] +
              256 * (int_of_char data.[!used+1] +
              256 * int_of_char data.[!used+1]))
            ) :: !result
          | Big ->
            result := PyInt (
              int_of_char data.[!used+1] +
              int_of_char data.[!used] * 256
            ) :: !result
          end

        | 'I' -> (* uint *)
          begin match endienness with
          | Little ->
            result := PyInt (
              int_of_char data.[!used] +
              256 * (int_of_char data.[!used+1] +
              256 * (int_of_char data.[!used+1] +
              256 * int_of_char data.[!used+1]))
            ) :: !result
          | Big ->
            result := PyInt (
              int_of_char data.[!used+1] +
              int_of_char data.[!used] * 256
            ) :: !result
          end

        | 'l' -> (* long *)
          begin match endienness with
          | Little ->
            result := PyInt (
              int_of_char data.[!used] +
              256 * (int_of_char data.[!used+1] +
              256 * (int_of_char data.[!used+1] +
              256 * int_of_char data.[!used+1]))
            ) :: !result
          | Big ->
            result := PyInt (
              int_of_char data.[!used+1] +
              int_of_char data.[!used] * 256
            ) :: !result
          end

        | 'L' -> (* ulong *)
          begin match endienness with
          | Little ->
            result := PyInt (
              int_of_char data.[!used] +
              256 * (int_of_char data.[!used+1] +
              256 * (int_of_char data.[!used+1] +
              256 * int_of_char data.[!used+1]))
            ) :: !result
          | Big ->
            result := PyInt (
              int_of_char data.[!used+1] +
              int_of_char data.[!used] * 256
            ) :: !result
          end

        | 'f' -> raise (NotImplemented "struct_unpack float") (* float *)
        | 'd' -> raise (NotImplemented "struct_unpack double") (* double *)
        | 'P' -> raise (NotImplemented "struct_unpack void* ") (* void* *)
        | _ -> raise (ValueError ("Unknown code '" ^ String.make 1 !code ^ "' in struct fmt string"))
        ;
        used := !used + sizeof !code;
      done;
      incr i
    done  
    ;
    PyTuple (List.rev !result)
;;
  
