open Py_types
open Py_exceptions
open Py_builtins_util

exception Found of int

let py_string_strip
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  match arg e 0 with 
  | PyString s -> 
    begin let n = String.length s in
    try 
      for i = 0 to n - 1 do 
        if s.[i] > ' ' then raise (Found i)
      done;
      PyString ""
    with Found first ->
    try 
      for i = n-1 downto first do
        if s.[i] > ' ' then raise (Found i)
      done;
      PyString ""
    with Found last ->
    PyString (String.sub s first (last - first + 1))
    end
  | _ -> raise (TypeError "string.strip requires string argument")

let py_string_lstrip
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  match arg e 0 with 
  | PyString s -> 
    begin let n = String.length s in
    try 
      for i = 0 to n - 1 do 
        if s.[i] > ' ' then raise (Found i)
      done;
      PyString ""
    with Found first ->
    PyString (String.sub s first (n - first))
    end
  | _ -> raise (TypeError "string.lstrip requires string argument")

let py_string_rstrip
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  match arg e 0 with 
  | PyString s -> 
    begin let n = String.length s in
    try 
      for i = n-1 downto 0 do
        if s.[i] > ' ' then raise (Found i)
      done;
      PyString ""
    with Found last ->
    PyString (String.sub s 0 (last+1))
    end
  | _ -> raise (TypeError "string.rstrip requires string argument")

(* find a string s' in s, starting at position pos, return the index in s
  of the string, and the position after the string; raise Not_found if the
  string is not found.

  Note this is a naive implementation! (Boyer-Moore is faster)
  WARNING: endpos must be less than or equal to the string length
*)

let find_string s1 pos endpos s2 =
  let n2 = String.length s2 
  in
    try
      for i = pos to endpos - n2 do
        try
          for j = 0 to n2 - 1 do
            if s1.[i+j] <> s2.[j] then raise Not_found
          done;
          raise (Found i)
        with Not_found -> ()
      done;
      raise Not_found
    with Found i -> i, i+ n2 

(* unbounded split on string *)
let string_split s s' =
  let n = String.length s in
  let rec split pos = 
    try 
      let last,next = find_string s pos n s' 
      in (String.sub s pos (last-pos)) :: split next
    with Not_found -> [String.sub s pos (n-pos)]
  in split 0

(* bounded split on string *)
let bounded_string_split s s' limit =
  let n = String.length s in
  let count = ref 0 in
  let rec split pos = 
    if !count = limit
    then [String.sub s pos (n - pos)]
    else
      try 
        let last,next = find_string s pos n s' 
        in 
          incr count; 
          (String.sub s pos (last-pos)) :: split next
      with Not_found -> [String.sub s pos (n-pos)]
  in split 0

exception Found2 of int * int

(* find whitespace in string s, starting at pos; raise Not_found
  if there is no whitespace found 
*)
let find_white s pos =
  let n = String.length s
  in 
   try
     for i = pos to n - 1 do
       if s.[i] <= ' '
       then begin 
         for j = i+1 to n - 1 do
           if s.[j] > ' ' then raise (Found2 (i,j))
         done;
         raise (Found2 (i,n))
       end
     done;
     raise Not_found
   with Found2 (i,j) -> i,j

(* unbounded split on whitespace *)
let white_split s =
  let n = String.length s in
  let rec split pos = 
    try 
      let last,next = find_white s pos  
      in (String.sub s pos (last-pos)) :: split next
    with Not_found -> [String.sub s pos (n-pos)]
  in split 0

(* bounded split on whitespace *)
let bounded_white_split s limit =
  let n = String.length s in
  let count = ref 0 in
  let rec split pos = 
    if !count = limit
    then [String.sub s pos (n - pos)]
    else
      try 
        let last,next = find_white s pos  
        in 
          incr count; 
          (String.sub s pos last) :: split next 
      with Not_found -> [String.sub s pos (n-pos)]
  in split 0

let py_string_split
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 3;
  let src, split, limit =
    begin match arg e 0 with 
    | PyString s -> s
    | _ -> raise (TypeError "string.split requires string argument")
    end
  ,
    begin match arg e 1 with
    | PyNone -> None 
    | PyString s -> Some s
    | _ -> raise (TypeError "string.split requires string or None for separator")
    end
  ,
    begin match arg e 2 with
    | PyInt 0 -> None
    | PyInt i -> Some i
    | _ -> raise (TypeError "string.split requires integer limit")
    end
  in 
    PyMutableList 
      begin Varray.of_list
        begin 
          List.map
            begin fun x -> PyString x end

            begin match split, limit with
            | Some s, Some k -> bounded_string_split src s k
            | Some s, None -> string_split src s
            | None, Some k -> bounded_white_split src k
            | None, None -> white_split src 
            end
        end 
      end

let py_string_join
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 2;
  let words = 
    List.map
    begin fun x -> 
      match x with
      | PyString s -> s
      | _ -> raise (TypeError "string.join requires sequence elements be strings")
    end  
    begin
      try Py_util.list_of_sequence (arg e 0) 
      with NonSequence -> raise (TypeError "string.join requires sequence argument")
    end
  and sep = 
    match arg e 1 with
    | PyString s -> s
    | _ -> raise (TypeError "string.join requires string separator")
  in PyString (String.concat sep words)

(* this routine never fails, it returns -1 if not found *)
let py_string_find
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 4;
  let str, sub, pos, endpos = 
    begin match arg e 0 with
    | PyString s -> s
    | _ -> raise (TypeError "string.index requires string argument")
    end  
  ,
    begin match arg e 1 with
    | PyString s -> s
    | _ -> raise (TypeError "string.index requires string second (substr) argument")
    end  
  ,
    begin match arg e 2 with
    | PyInt i -> i
    | _ -> raise (TypeError "string.index requires integer startpos argument")
    end  
  ,
    begin match arg e 3 with
    | PyInt i -> i
    | _ -> raise (TypeError "string.index requires integer endpos argument")
    end  
  in
    let n = String.length str 
    and m = String.length sub 
    in
    let pos' = max 0 (if pos < 0 then n - pos else pos)
    and endpos' = min n (if endpos < 0 then n - endpos else endpos)
    in
      PyInt
        begin 
          if pos' + m > endpos' then -1
          else 
            try fst (find_string str pos' endpos' sub)
            with Not_found -> -1
        end
