%{
exception EndOfInput
open Py_types

type testlist_t = 
  ListWithComma of expr_t list 
  | ListWithoutComma of expr_t list

let parse_error (s:string) = 
  raise (Py_exceptions.ParseError "Error parsing input")
%}
/* declarations */
%token NONE 
%token ENDMARKER
%token <string> NAME
%token <int> INTEGER
%token <Big_int.big_int> LONG
%token <float> FLOAT
%token <float * float> COMPLEX
%token <string> STRING
%token <int * string> NEWLINE 
%token <int * string> SETLINE 
%token INDENT DEDENT
%token LPAR RPAR LSQB RSQB COLON COMMA SEMI PLUS MINUS STAR SLASH
%token VBAR AMPER LESS GREATER EQUAL DOT PERCENT BACKQUOTE
%token LBRACE RBRACE EQEQUAL NOTEQUAL LESSEQUAL GREATEREQUAL
%token TILDE CIRCUMFLEX LEFTSHIFT RIGHTSHIFT DOUBLESTAR
%token TRAILING_COMMA 
%token POWER
%token PLUSPLUS MINUSMINUS PLUSEQUAL MINUSEQUAL STAREQUAL SLASHEQUAL PERCENTEQUAL
%token CARETEQUAL VBAREQUAL AMPEREQUAL TILDEEQUAL COLONEQUAL
%token LEFTSHIFTEQUAL RIGHTSHIFTEQUAL LEFTARROW RIGHTARROW

/* KEYVALSEP and SLICESEP are colons in dictionaries and slices, resp */
%token KEYVALSEP SLICESEP

/* CTRL is a colon at the outer level which announces a suite */
%token CTRL
%token <Py_types.sref> LOC
%token RARR
/* keywords */
%token AND ASSERT BREAK CLASS CONTINUE DEF DEL ELIF ELSE
%token EXCEPT EXEC FINALLY FOR FROM GLOBAL IF IMPORT IN IS LAMBDA NOT OR PASS
%token PRINT RAISE RETURN TRY WHILE

%token <Py_types.sref> LOC_ELIF LOC_ELSE LOC_FINALLY LOC_EXCEPT LOC_IF
/* none of these are actually used, they're pre-processing tokens only */
%token SLOSH
%token BLANK_LINE
%token <string> COMMENT 
%token <(int * string) * string> COMMENT_NEWLINE
%token <int> WHITE 
%token <string> ERRORTOKEN

%type <statement_t> small_stmt
%type <Py_types.statement_t> file_input 
%type <Py_types.expr_t> eval_input

%start file_input 
%start eval_input 

%type <Py_types.expr_t> testlist 
%type <testlist_t> special_testlist 

%type <Py_types.expr_t> expr, xor_expr, and_expr, shift_expr
%type <Py_types.expr_t> arith_expr, term, factor, power 

%type <Py_types.statement_t> if_stmt  
%type <Py_types.statement_t> while_stmt
%type <Py_types.statement_t> for_stmt
%type <Py_types.statement_t> try_stmt
%type <Py_types.statement_t> funcdef
%type <Py_types.statement_t> classdef
%type <Py_types.statement_t> expr_stmt
%type <Py_types.statement_t> print_stmt
%type <Py_types.statement_t> del_stmt
%type <Py_types.statement_t> pass_stmt
%type <Py_types.statement_t> flow_stmt
%type <Py_types.statement_t> import_stmt
%type <Py_types.statement_t> global_stmt
%type <Py_types.statement_t> exec_stmt
%type <Py_types.statement_t> assert_stmt

%%

newline_or_stmt:
  NEWLINE { Empty }
  | stmt { $1 }

newline_or_stmt_seq:
  newline_or_stmt newline_or_stmt_seq { $1 :: $2 }
  | newline_or_stmt { [$1] }
  
file_input: 
  newline_or_stmt_seq ENDMARKER {  Suite $1 }
  | ENDMARKER { Empty }

eval_input:
   testlist ENDMARKER { $1 }

funcdef: 
  LOC DEF NAME parameters colon_suite { Def ($1, $3, $4, $5) }

parameters: 
  LPAR  varargslist RPAR { $2 }
  | LPAR  varargslist TRAILING_COMMA RPAR { $2 }
  | LPAR RPAR { [], NoStarParam, NoStarStarParam }
 
varargslist:
  | fpdef_dflts COMMA starargs { $1, fst $3, snd $3 }
  | fpdef_dflts { $1, NoStarParam, NoStarStarParam }
  | starargs {  [], fst $1, snd $1 }

/* *x or **x or *x, **x */
starargs:
  STAR NAME COMMA STAR STAR NAME { StarParam $2, StarStarParam $6 }
  | STAR NAME COMMA DOUBLESTAR NAME { StarParam $2, StarStarParam $5 }
  | STAR NAME { StarParam $2, NoStarStarParam }
  | STAR STAR NAME { NoStarParam, StarStarParam $3}
  | DOUBLESTAR NAME { NoStarParam, StarStarParam $2}


fpdef: 
  NAME  {Param $1 }
  | LPAR fplist RPAR { Paramtuple $2 }
  | LPAR fplist TRAILING_COMMA RPAR { Paramtuple $2 }
fplist:
  fpdef COMMA fplist { $1::$3 }
  | fpdef { [$1] }

/* argument with optional default */
fpdef_dflt:
  fpdef EQUAL test { Parameter2 ( $1, $3 ) }
  | fpdef { Parameter1 $1 }

/* list of arguments with optional defaults */ 
fpdef_dflts:
  fpdef_dflts COMMA fpdef_dflt { List.concat [$1; [$3]] }
  | fpdef_dflt { [$1] }

stmt: 
  simple_stmt  { Suite [$1] }
  | compound_stmt { $1 }

small_list:
  small_stmt SEMI small_list { $1 :: $3 }
  | small_stmt { [$1] }

simple_stmt: 
  small_list NEWLINE { Suite $1 }

small_stmt: 
  | assign_stmt  {$1}
  | expr_stmt    {$1}
  | print_stmt   {$1}
  | del_stmt     {$1}
  | pass_stmt    {$1}
  | flow_stmt    {$1}
  | import_stmt  {$1}
  | global_stmt  {$1}
  | exec_stmt    {$1}
  | assert_stmt  {$1}

eqlist: 
  | testlist EQUAL eqlist { $1 :: $3 }
  | testlist EQUAL testlist { [$1; $3] }

  | testlist EQUAL eqlist { $1 :: $3 }
  | testlist EQUAL testlist { [$1; $3] }


assign_stmt:
  | LOC eqlist { Assign ($1, $2) }
  | LOC testlist PLUSEQUAL testlist { PlusEqual ($1, $2, $4) }
  | LOC testlist MINUSEQUAL testlist { MinusEqual ($1, $2, $4) }
  | LOC testlist STAREQUAL testlist { StarEqual ($1, $2, $4) }
  | LOC testlist SLASHEQUAL testlist { SlashEqual ($1, $2, $4) }
  | LOC testlist AMPEREQUAL testlist { AmperEqual ($1, $2, $4) }
  | LOC testlist VBAREQUAL testlist { VbarEqual ($1, $2, $4) }
  | LOC testlist PERCENTEQUAL testlist { PercentEqual ($1, $2, $4) }
  | LOC testlist CARETEQUAL testlist { CaretEqual ($1, $2, $4) }
  | LOC testlist COLONEQUAL testlist { ColonEqual ($1, $2, $4) }
  | LOC testlist LEFTSHIFTEQUAL testlist { LeftShiftEqual ($1, $2, $4) }
  | LOC testlist RIGHTSHIFTEQUAL testlist { RightShiftEqual ($1, $2, $4) }
  | LOC testlist PLUSPLUS { PlusPlus ($1, $2) }
  | LOC PLUSPLUS testlist { PlusPlus ($1, $3) }
  | LOC testlist MINUSMINUS { MinusMinus ($1, $2) }
  | LOC MINUSMINUS testlist { MinusMinus ($1, $3) }
  
expr_stmt:
  | LOC testlist { Expr ($1,$2) } 
  
elist_ntc:
  | elist_ntc COMMA expr { List.concat [$1; [$3]] }
  | expr { [$1] }
 
special_elist:
  | elist_ntc TRAILING_COMMA { ListWithComma $1 }
  | elist_ntc { ListWithoutComma $1 }

elist:
  special_elist {
    match $1 with
    | ListWithComma xx -> PyTuple xx
    | ListWithoutComma [x] -> x
    | ListWithoutComma xx -> PyTuple xx
  }

testlist_ntc:
  | testlist_ntc COMMA test { List.concat [$1; [$3]] }
  | test { [$1] }
 
special_testlist:
  | testlist_ntc TRAILING_COMMA { ListWithComma $1 }
  | testlist_ntc { ListWithoutComma $1 }

testlist:
  special_testlist {
    match $1 with
    | ListWithComma xx -> PyTuple xx
    | ListWithoutComma [x] -> x
    | ListWithoutComma xx -> PyTuple xx
  }

print_stmt: 
  | LOC PRINT special_testlist { 
    match $3 with
    | ListWithComma x -> PrintComma ($1, x)
    | ListWithoutComma x -> Print ($1, x)
    }
  | LOC PRINT { Print ($1, []) }

del_stmt: 
  LOC DEL testlist { Del ($1, $3) }

pass_stmt: 
  LOC PASS { Pass }

flow_stmt: 
  break_stmt  { $1 }
  | continue_stmt { $1 }
  | return_stmt {  $1 }
  | raise_stmt  { $1 }

break_stmt: 
  LOC BREAK { Break $1 }

continue_stmt: 
  LOC CONTINUE { Continue $1 }

return_stmt: 
  LOC RETURN testlist { Return ($1, $3) }
  | LOC RETURN  { Return ($1, PyNone) }

raise_stmt:
  LOC RAISE test COMMA test COMMA test { Raise3 ($1, $3, $5, $7) }
  | LOC RAISE test COMMA test { Raise2 ($1, $3, $5) }
  | LOC RAISE test { Raise1 ($1, $3) }
  | LOC RAISE {Raise0 $1 }

import_stmt: 
  LOC IMPORT dotted_name_list  {Import ($1, $3) }
  | LOC FROM dotted_name IMPORT name_list {ImportFrom ($1, $3, $5) }
  | LOC FROM dotted_name IMPORT STAR {ImportAll ($1, $3) }
  
dotted_name: 
  NAME DOT dotted_name {$1 :: $3 }
  | NAME              { [$1] }

dotted_name_list:
  dotted_name COMMA dotted_name_list { $1 :: $3 }
  | dotted_name { [$1]}

name_list:
  NAME COMMA name_list { $1 :: $3 }
  | NAME { [$1] }

global_stmt: 
  LOC GLOBAL name_list  {Global ($1, $3) }

exec_stmt: 
  LOC EXEC expr IN test COMMA test { Exec3 ($1, $3, $5, $7) }
  | LOC EXEC expr IN test { Exec2 ($1, $3, $5) } 
  | LOC EXEC expr  { Exec1 ($1, $3) }

assert_stmt: 
  LOC ASSERT test COMMA test { Assert2 ($1, $3, $5) }
  | LOC ASSERT test {Assert1 ($1, $3) }

compound_stmt: 
  if_stmt  { $1 }
  | while_stmt  { $1 }
  | for_stmt  { $1 }
  | try_stmt  { $1 }
  | funcdef  { $1 }
  | classdef { $1 }

elif_clause:
  LOC_ELIF test colon_suite { $1, $2, $3 }

else_clause:
  LOC_ELSE colon_suite {$2}

if_clause:
  LOC_IF test colon_suite  { $1, $2, $3 } 

elif_clauses:
  elif_clause elif_clauses { $1 :: $2 }
  | elif_clause { [$1] }

ifelif_clause:
  if_clause elif_clauses { $1 :: $2 }
  | if_clause { [$1]}

if_stmt: 
  | ifelif_clause else_clause { If ($1, $2) } 
  | ifelif_clause { If ($1, Empty) } 

while_stmt: 
  LOC WHILE test colon_suite LOC_ELSE colon_suite { 
    While ($1, $3, $4, $6) }
  | LOC WHILE test colon_suite  { 
    While ($1, $3, $4, Empty) }


for_stmt: 
  LOC FOR elist IN testlist colon_suite LOC_ELSE colon_suite { 
    For ($1, $3, $5, $6, $8) }
  | LOC FOR elist IN testlist colon_suite { 
    For ($1, $3, $5, $6, Empty) }


try_stmt: 
  LOC TRY colon_suite handlers LOC_ELSE colon_suite { TryElse ($3, $4, $6) }
  | LOC TRY colon_suite handlers { TryElse ($3, $4, Empty) }
  | LOC TRY colon_suite LOC_ELSE colon_suite { TryElse ($3, [], $5) }
  | LOC TRY colon_suite LOC_FINALLY colon_suite {TryFinally ($3, $5) }

except_clause: 
  LOC_EXCEPT test COMMA test { Except2 ($1, $2, $4) }
  | LOC_EXCEPT test { Except1 ($1, $2) }
  | LOC_EXCEPT { Except0 }

handler:
  except_clause colon_suite { $1, $2 }

handlers:
  handler handlers { $1 :: $2 }
  | handler        { [$1] }

colon_suite: 
  CTRL simple_stmt  { $2 }
  | CTRL NEWLINE INDENT stmts DEDENT { Suite $4 }

stmts:
  stmt stmts { $1 :: $2 }
  | stmt { [$1] }

test: 
  | and_test or_list { Or ($1 :: $2) }
  | and_test { $1 }
  | lambdef { $1 }

or_list:
  OR and_test or_list { $2 :: $3 }
  | OR and_test { [$2] }
  
and_test: 
  not_test and_list { And ($1 :: $2) }
  | not_test { $1 }
  
and_list:
  AND not_test and_list { $2 :: $3 }
  | AND not_test  { [$2] }

not_test: 
  NOT not_test  { Not $2 } 
  | comparison { $1 }

comparison: 
  expr comp_list { Compare ($1, $2) }
  | expr { $1 }

comp_list:
  comparator comp_list { $1 :: $2 }
  | comparator  { [$1] }

comparator: 
  LESS  expr { Less $2 }
  | GREATER expr { Greater $2 }
  | EQEQUAL expr {Equal $2 }
  | GREATEREQUAL expr {GreaterEqual $2 }
  | LESSEQUAL expr {LessEqual $2 }
  | NOTEQUAL expr {NotEqual $2 }
  | IN expr { In $2 }
  | NOT IN  expr { NotIn $3 }
  | IS expr { Is $2 }
  | IS NOT expr {IsNot $3 }

expr: 
  xor_expr expr_list { BitOr ($1 :: $2) }
  | xor_expr { $1 }
expr_list:
 VBAR xor_expr expr_list { $2 :: $3 }
 | VBAR xor_expr  { [$2] }
 
 
xor_expr_list:
  CIRCUMFLEX and_expr xor_expr_list { $2 :: $3 }
  | CIRCUMFLEX and_expr  { [$2] }
xor_expr: 
  and_expr xor_expr_list  { BitXor ($1 :: $2) }
  | and_expr { $1 }

and_expr_list:
  AMPER shift_expr and_expr_list { $2 :: $3 }
  | AMPER shift_expr { [$2] }
and_expr: 
  shift_expr and_expr_list {BitAnd ($1 :: $2 )}
  | shift_expr  { $1 }

shift_op:
  LEFTSHIFT arith_expr { Asl $2 }
  | RIGHTSHIFT arith_expr { Lsr $2 }
shift_expr_list:
  shift_op shift_expr_list { $1 :: $2 }
  | shift_op { [$1] } 
shift_expr: 
  arith_expr shift_expr_list { Eval ($1, $2) }
  | arith_expr { $1 }

add_op:
  PLUS term { Add $2 }
  | MINUS term {Sub $2 }
arith_expr_list:
  add_op arith_expr_list { $1 :: $2 }
  | add_op  { [$1] }
arith_expr: 
  term arith_expr_list {Eval ($1, $2) }
  | term {$1} 

mul_op: 
  STAR factor { Mul $2 }
  | SLASH factor {Div $2}
  | PERCENT factor {Mod $2}
term_list:
  mul_op term_list { $1 :: $2 }
  | mul_op { [$1] } 
term: 
  factor term_list {Eval ($1, $2) }
  | factor   { $1 }
  
factor:
  PLUS factor { $2 }
  | MINUS factor { Neg $2 }
  | TILDE factor { Complement $2}
  | power { $1 }

pow_op:
  POWER atomic { Pow $2 }
pow_expr_list:
  pow_op pow_expr_list { $1 :: $2 }
  | pow_op { [$1] }
power:
  atomic pow_expr_list { Eval ($1, $2) }
  | atomic {$1}

trailers:
  trailer trailers { $1 :: $2 }
  | trailer { [$1] }
 
atomic:
  atom trailers { AtomWithTrailers ($1, $2) }
  | atom { $1 }

strings:
  STRING strings { $1 ^ $2 }
  | STRING { $1 }

atom: 
  LPAR RPAR { PyTuple [] }
  | LPAR testlist_ntc TRAILING_COMMA RPAR { PyTuple $2 }
  | LPAR testlist_ntc RPAR { 
    match $2 with
    | [x] -> x
    | x  -> PyTuple x
    }
  | LSQB RSQB { PyList [] }
  | LSQB testlist_ntc TRAILING_COMMA RSQB { PyList $2 }
  | LSQB testlist_ntc RSQB { PyList $2 }
  | LBRACE RBRACE { PyDict [] }
  | LBRACE dictmaker RBRACE {PyDict $2 }
  | LBRACE dictmaker TRAILING_COMMA RBRACE {PyDict $2 }
  | BACKQUOTE testlist BACKQUOTE  { PyRepr $2 }
  | NAME {PyName $1}
  | NONE { PyNone }
  | INTEGER {PyInt $1}
  | FLOAT {PyFloat $1}
  | LONG {PyLong $1}
  | COMPLEX {PyComplex (fst $1, snd $1) }
  | strings {PyString $1}

lambdef: 
  LAMBDA varargslist CTRL test { Lambda ($2, $4) }
  | LAMBDA varargslist CTRL LOC test { Lambda ($2, $5) }
  | LAMBDA CTRL test { Lambda (([], NoStarParam, NoStarStarParam), $3) }
  | LAMBDA CTRL LOC test { Lambda (([], NoStarParam, NoStarStarParam), $4) }

trailer: 
   LPAR RPAR { Arglist [] }
  | LPAR arglist RPAR  { Arglist $2 }
  | LPAR arglist TRAILING_COMMA RPAR  { Arglist $2 }
  | LSQB subscriptlist RSQB {Sublist $2 }
  | LSQB subscriptlist TRAILING_COMMA RSQB {Sublist $2 }
  | DOT NAME {Dotname $2}

comma_sublist:
  COMMA subscript comma_sublist { $2 :: $3 }
  | COMMA subscript { [$2] }

subscriptlist: 
  | subscript comma_sublist  { $1 :: $2 }
  | subscript  { [$1] }

subscript: 
  DOT DOT DOT { Ellipsis }
  | test SLICESEP test SLICESEP test { Subscript2 (Pos $1, Pos $3, Pos $5) }
  
  | test SLICESEP test SLICESEP { Subscript2 (Pos $1, Pos $3, Defsub) }
  | test SLICESEP SLICESEP test { Subscript2 (Pos $1, Defsub, Pos $4) }
  | SLICESEP test SLICESEP test { Subscript2 (Defsub, Pos $2, Pos $4) }
  | test SLICESEP SLICESEP { Subscript2 (Pos $1, Defsub, Defsub) }
  | SLICESEP test SLICESEP { Subscript2 (Defsub, Pos $2, Defsub) }
  | SLICESEP SLICESEP test { Subscript2 (Defsub, Defsub, Pos $3) }
  | SLICESEP SLICESEP { Subscript2 (Defsub, Defsub, Defsub) }

  | test SLICESEP test {Subscript1 (Pos $1, Pos $3) }
  | test SLICESEP {Subscript1 (Pos $1, Defsub) }
  | SLICESEP test {Subscript1 (Defsub, Pos $2) }
  | SLICESEP {Subscript1 (Defsub, Defsub) }
  | test {Subscript0 (Pos $1) }

dictent:
  test KEYVALSEP test { DictEnt ($1, $3) }
dictmaker:
  dictent COMMA dictmaker { $1 :: $3 }
  | dictent { [$1] }

classdef: 
  LOC CLASS NAME LPAR testlist RPAR colon_suite { Class ($1, $3, $5, $7) }
  | LOC CLASS NAME colon_suite { Class ($1, $3, PyNone, $4) }

arglist:
  argument COMMA arglist { $1 :: $3 }
  | argument { [$1] }

argument: 
  NAME EQUAL test { Argument2 ($1, $3) }
  | test { Argument1 $1 }

%%
(* trailer *)


