open Py_types
open Py_exceptions
open Py_builtins_util

(* This has to be defined before opening Gtk.Unsafe, because it
  unfortunately defines names like String which are also top level
  module names from the standard library :-((
*)

type return_t = Unit_t | Bool_t

let callback_wrapper
  (return_type:return_t)

  (* These args must be curried by the caller to create a gtkCallBack *)
  (interp:interpreter_t) 
  (python_function:expr_t) 
  (* These args are supplied by (ml)gtk *) 
  (widget:Gtk.Unsafe.gtkobject)
  (gtkargs:Gtk.Unsafe.gtkArg list): Gtk.Unsafe.gtkArg =
  begin try
    let arguments = 
      Argument1 (PyWidget widget) 
      ::
      begin
        List.map
        begin fun arg ->
        Argument1 begin
          match arg with
          | Gtk.Unsafe.Unit -> PyNone
          | Gtk.Unsafe.Other -> PyNone
          | Gtk.Unsafe.Invalid -> PyNone
          | Gtk.Unsafe.Bool b -> PyInt (Util.int_of_bool b)
          | Gtk.Unsafe.Char ch -> PyString (String.make 1 ch)
          | Gtk.Unsafe.Int i -> PyInt i
          | Gtk.Unsafe.Float f -> PyFloat f
          | Gtk.Unsafe.String s -> PyString s
          | Gtk.Unsafe.Pointer p -> PyNone
          end (* argument *)
        end (* function *)
        gtkargs
      end (* argument list tail *)
    in let result = Py_exec.py_call interp python_function arguments
    in match return_type with
    | Bool_t -> Gtk.Unsafe.Bool (Py_datum.py_istrue result)
    | Unit_t -> Gtk.Unsafe.Unit
  with x ->
    print_endline "ERROR IN CALLBACK!!";
    print_endline (Printexc.to_string x);
    flush stdout;
    Gtk.Unsafe.Bool false
  end (* callback wrapper *)
  
open Gtk.Unsafe

let py_widget_create
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  exactly e 1;
  let widget_name = 
    match arg e 0 with
    | PyString s -> s
    | _ -> raise (TypeError "create_widget requires string naming widget kind")
  in let widget =
    match widget_name with
    | "adjustment"  ->
       let pos = 
         match d#get_item (PyString "position") with
         | Some (PyFloat v) -> v
         | None -> 0.0
         | _ -> raise (TypeError "adjustment position must be float")
       and lower = 
         match d#get_item (PyString "lower") with
         | Some (PyFloat v) -> v
         | None -> 0.0
         | _ -> raise (TypeError "adjustment lower must be float")
       and upper = 
         match d#get_item (PyString "upper") with
         | Some (PyFloat v) -> v
         | None -> 1.0
         | _ -> raise (TypeError "adjustment upper must be float")
       and step_increment = 
         match d#get_item (PyString "step_increment") with
         | Some (PyFloat v) -> v
         | None -> 0.02
         | _ -> raise (TypeError "adjustment step_increment must be float")
       and page_increment = 
         match d#get_item (PyString "page_increment") with
         | Some (PyFloat v) -> v
         | None -> 0.1
         | _ -> raise (TypeError "adjustment page_incrementmust be float")
       and page_size = 
         match d#get_item (PyString "page_size") with
         | Some (PyFloat v) -> v
         | None -> 0.2
         | _ -> raise (TypeError "adjustment page_size must be float")
       in
         adjustment_new pos lower upper step_increment page_increment page_size
         
    | "alignment"  -> 
       let xalign = 
         match d#get_item (PyString "xalign") with
         | Some (PyFloat v) -> v
         | None -> 0.0
         | _ -> raise (TypeError "alignment xalign must be float")
       and yalign = 
         match d#get_item (PyString "yalign") with
         | Some (PyFloat v) -> v
         | None -> 0.0
         | _ -> raise (TypeError "alignment yalign must be float")
       and xscale  = 
         match d#get_item (PyString "xscale") with
         | Some (PyFloat v) -> v
         | None -> 1.0
         | _ -> raise (TypeError "alignment xscale must be float")
       and yscale  = 
         match d#get_item (PyString "yscale") with
         | Some (PyFloat v) -> v
         | None -> 0.02
         | _ -> raise (TypeError "alignment yscale must be float")
       in
         alignment_new xalign yalign xscale yscale
     
    | "arrow"  -> raise (NotImplemented "arrow widget")
    | "aspect_frame"  -> raise (NotImplemented "aspect frame widget")
    | "button"  ->  
      begin match d#get_item (PyString "text") with
      | Some (PyString text) -> button_new_with_label text
      | Some _ -> raise (ValueError "button 'text=' must be string")
      | None -> button_new ()
      end
    | "check_button"  ->
      begin match d#get_item (PyString "text") with
      | Some (PyString text) -> check_button_new_with_label text
      | Some _ -> raise (ValueError "check_button 'text=' must be string")
      | None -> check_button_new ()
      end

    | "check_menu_item"  ->
      begin match d#get_item (PyString "text") with
      | Some (PyString text) -> check_menu_item_new_with_label text
      | Some _ -> raise (ValueError "check_menu_item 'text=' must be string")
      | None -> check_menu_item_new ()
      end

    | "clist"  -> 
      begin match d#get_item (PyString "columns") with
      | Some (PyInt n) -> clist_new n
      | Some (PyMutableList headings) -> 
        let n = (Varray.length headings) in
        let widget = clist_new n
        in for i=0 to n-1 do
          let title = Py_functions.str (Varray.get headings i) 
          in clist_set_column_title widget i title
        done
        ;
        clist_column_titles_show widget
        ;
        widget
        
      | _ -> raise (ValueError "clist requires 'columns' [either number or list of strings]")
      end

    | "color_selection"  -> color_selection_new ()
    | "combo"  -> combo_new ()
    | "curve"  -> curve_new ()
    | "gamnma_curve"  -> gamma_curve_new ()
    | "dialog"  -> dialog_new ()
    | "drawing_area"  -> drawing_area_new ()
    | "entry"  -> 
      begin match d#get_item (PyString "max_length") with
      | Some (PyInt max_length) -> entry_new_with_max_length max_length 
      | Some _ -> raise (ValueError "entry 'max_length=' must be integer")
      | None -> entry_new ()
      end

    | "file_selection"  ->
      begin match d#get_item (PyString "text") with
      | Some (PyString text) -> file_selection_new text
      | _ -> raise (ValueError "file_selection requires string 'text=' for title")
      end

    | "frame"  -> 
      begin match d#get_item (PyString "text") with
      | Some (PyString text) -> frame_new text
      | _ -> raise (ValueError "frame requires string 'text=' for title")
      end

    | "hbox"  -> 
      let homogenous =
        begin match d#get_item (PyString "homogenous") with
        | Some x -> Py_datum.py_istrue x
        | None -> false
        end
      and padding =
        begin match d#get_item (PyString "spacing") with
        | Some (PyInt x) -> x
        | Some _ ->  raise (ValueError "hbox 'spacing=' must be integer")
        | None -> 0
        end
      in
        hbox_new homogenous padding

    | "hbutton_box"  -> hbutton_box_new ()
    | "hpaned"  -> hpaned_new ()
    | "hscrollbar"  -> 
      begin match d#get_item (PyString "adjustment") with
      | Some (PyWidget x) -> hscrollbar_new (Some x) 
      | Some _ -> raise (ValueError "hscrollbar 'adjustment=' must be widget")
      | None -> hscrollbar_new None
      end

    | "hseparator"  -> hseparator_new ()
    | "label"  -> 
      begin match d#get_item (PyString "text") with
      | Some (PyString text) -> label_new text
      | _ -> raise (ValueError "label requires 'text='")
      end

    | "list"  -> list_new ()
    | "list_item"  -> 
      begin match d#get_item (PyString "text") with
      | Some (PyString text) -> list_item_new_with_label text
      | _ -> raise (ValueError "list item (currently) requires 'text='")
      end

    | "menu"  -> menu_new ()
    | "menu_bar"  -> menu_bar_new ()
    | "menu_item"  -> 
      begin match d#get_item (PyString "text") with
      | Some (PyString text) -> menu_item_new_with_label text
      | Some _ -> raise (ValueError "menu_item 'text=' must be string")
      | None -> menu_item_new ()
      end

    | "notebook"  -> notebook_new ()
    | "pixmap"  ->  raise (NotImplemented "pixmap")
    | "progress_bar"  -> progress_bar_new ()
    | "radio_button"  -> (* radio button groups not yet implemented *)
      begin match d#get_item (PyString "text") with
      | Some (PyString text) -> radio_button_new_with_label None text
      | _ -> raise (ValueError "radio button (currently) requires 'text='")
      end

    | "radio_menu_item"  ->
      begin match d#get_item (PyString "text") with
      | Some (PyString text) -> radio_menu_item_new_with_label None text
      | _ -> raise (ValueError "radio menu item (currently) requires 'text='")
      end

    | "scrolled_window"  -> scrolled_window_new ()
    | "statusbar"  -> statusbar_new ()
    | "table"  -> 
      let homogenous =
        begin match d#get_item (PyString "homogenous") with
        | Some x -> Py_datum.py_istrue x
        | None -> false
        end
      and rows =
        begin match d#get_item (PyString "rows") with
        | Some (PyInt x) -> x
        | _ ->  raise (ValueError "integer table 'rows=' required")
        end
      and columns =
        begin match d#get_item (PyString "columns") with
        | Some (PyInt x) -> x
        | _ ->  raise (ValueError "integer table 'columns=' required")
        end
      in
        table_new rows columns homogenous

    | "text"  -> 
      begin
        let widget = text_new () in
        begin match d#get_item (PyString "editable") with
        | Some (PyInt 1) -> text_set_editable widget true
        | _ -> ()
        end
        ;
        begin match d#get_item (PyString "text") with
        | Some (PyString s) -> text_insert widget None None None s 0 (String.length s)
        | _ -> ()
        end
        ;
        widget
      end
    | "toggle_button"  -> 
      begin match d#get_item (PyString "text") with
      | Some (PyString text) -> toggle_button_new_with_label text
      | Some _ -> raise (ValueError "toggle_button 'text=' must be string")
      | None -> toggle_button_new ()
      end

    | "tooltips"  -> tooltips_new ()
    | "vbox"  ->
      let homogenous =
        begin match d#get_item (PyString "homogenous") with
        | Some x -> Py_datum.py_istrue x
        | None -> false
        end
      and padding =
        begin match d#get_item (PyString "spacing") with
        | Some (PyInt x) -> x
        | Some _ ->  raise (ValueError "vbox 'spacing=' must be integer")
        | None -> 0
        end
      in
        vbox_new homogenous padding

    | "vbutton_box"  -> vbutton_box_new ()
    | "vpaned"  -> vpaned_new ()
    | "vscrollbar"  -> 
      begin match d#get_item (PyString "adjustment") with
      | Some (PyWidget x) -> vscrollbar_new (Some x) 
      | Some _ -> raise (ValueError "vscrollbar 'adjustment=' must be widget")
      | None -> vscrollbar_new None
      end

    | "vseparator"  -> vseparator_new ()
    | "window"  ->  
      begin
        let widget =
          begin match d#get_item (PyString "type") with
          | Some (PyString "toplevel") -> window_new Gtk.WINDOW_TOPLEVEL
          | Some (PyString "dialog") ->  window_new Gtk.WINDOW_DIALOG
          | Some (PyString "popup") -> window_new Gtk.WINDOW_POPUP
          | Some _ -> raise (ValueError "window 'type=' must be 'toplevel', 'dialog' or 'popup'")
          | None -> window_new Gtk.WINDOW_TOPLEVEL (* default *) 
          end
        in 
          begin match d#get_item (PyString "title") with
          | Some (PyString title) -> window_set_title widget title
          | _ -> ()
          end
          ;
          widget
      end
      | _ -> raise (ValueError ("Unknown widget " ^ widget_name ^ " in widget_create"))

  in PyWidget widget


let py_gui_mainloop
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  exactly e 0;
  empty_dict d;
  Gtk.gtk_mainloop ();
  PyNone

(* NOTE: this function is a common callback, it takes any
  arguments and ignores them *)
let py_gui_quit
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  Gtk.main_quit();
  PyNone

let py_widget_destroy 
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  exactly e 1;
  empty_dict d;
  begin match arg e 0 with
  | PyWidget w -> object_destroy w
  | _ -> raise (TypeError "widget destroy requires widget")
  end
  ;
  PyNone

let py_widget_show
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  exactly e 1;
  empty_dict d;
  begin match arg e 0 with
  | PyWidget w -> widget_show w
  | _ -> raise (TypeError "widget show requires widget")
  end
  ;
  PyNone

let py_widget_hide
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  exactly e 1;
  empty_dict d;
  begin match arg e 0 with
  | PyWidget w -> widget_hide w
  | _ -> raise (TypeError "widget hide requires widget")
  end
  ;
  PyNone

let py_widget_realize
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  exactly e 1;
  empty_dict d;
  begin match arg e 0 with
  | PyWidget w -> widget_realize w
  | _ -> raise (TypeError "widget realize requires widget")
  end
  ;
  PyNone

let py_widget_unrealize
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  exactly e 1;
  empty_dict d;
  begin match arg e 0 with
  | PyWidget w -> widget_unrealize w
  | _ -> raise (TypeError "widget unrealize requires widget")
  end
  ;
  PyNone


let py_widget_set_usize
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  exactly e 3;
  empty_dict d;
  let widget =
    begin match arg e 0 with
    | PyWidget w -> w
    | _ -> raise (TypeError "widget set_usize requires widget")
    end
  and xsize = 
    begin match arg e 1 with
    | PyInt i -> i
    | _ -> raise (TypeError "widget set_usize requires x is int (arg 2)")
    end
  and ysize = 
    begin match arg e 2 with
    | PyInt i -> i
    | _ -> raise (TypeError "widget set_usize requires y is int (arg 3)")
    end
  in widget_set_usize widget xsize ysize
  ;
  PyNone

let py_container_add
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  exactly e 2;
  empty_dict d;
  let container = 
    begin match arg e 0 with
    | PyWidget w -> w
    | _ -> raise (TypeError "container_add requires container (arg 1)")
    end
  and widget =
    begin match arg e 1 with
    | PyWidget w -> w
    | _ -> raise (TypeError "container_add requires widget (arg 2)")
    end
  in 
    container_add container widget
    ;
    PyNone


let py_widget_reparent
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  exactly e 2;
  empty_dict d;
  let widget = 
    begin match arg e 0 with
    | PyWidget w -> w
    | _ -> raise (TypeError "widget reparent requires widget (arg 1)")
    end
  and newparent =
    begin match arg e 1 with
    | PyWidget w -> w
    | _ -> raise (TypeError "widget reparent requires widget (arg 2)")
    end
  in 
    widget_reparent widget newparent
    ;
    PyNone

let py_container_remove
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  exactly e 2;
  empty_dict d;
  let container = 
    begin match arg e 0 with
    | PyWidget w -> w
    | _ -> raise (TypeError "container_remove requires container (arg 1)")
    end
  and widget =
    begin match arg e 1 with
    | PyWidget w -> w
    | _ -> raise (TypeError "container_remove requires widget (arg 2)")
    end
  in 
    container_remove container widget
    ;
    PyNone

let py_signal_connect
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  exactly e 3;
  empty_dict d;
  let widget = 
    begin match arg e 0 with
    | PyWidget w -> w
    | _ -> raise (TypeError "signal connect requires widget (arg 1)")
    end
  and signal =
    begin match arg e 1 with
    | PyString s -> s
    | _ -> raise (TypeError "signal connect requires signal name (string) (arg 2)")
    end
  and py_callback = arg e 2 
  in 
    if not (Py_functions.callable py_callback)
    then raise (TypeError "signal connect requires callable object (arg 3)")
    else begin
      let return_type = 
        (* calculate required return type, to allow callback wrapper
          to coercre python return value
        *)
        match signal with 
        | "delete_event" -> Bool_t
        | _ -> Unit_t
      in let mlgtk_callback = callback_wrapper return_type interp py_callback 
      in let result = signal_connect widget signal mlgtk_callback 
      in PyInt result
    end

(* range *)
let py_range_get_adjustment
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  let widget = 
    begin match arg e 0 with
    | PyWidget w -> w
    | _ -> raise (TypeError "range get adjustment requires range widget (arg 1)")
    end
  in PyWidget (range_get_adjustment widget)

let py_range_set_adjustment
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 2;
  let widget = 
    match arg e 0 with
    | PyWidget w -> w
    | _ -> raise (TypeError "range get adjustment requires range widget (arg 1)")
  and adjustment = 
    match arg e 1 with
    | PyWidget w -> w
    | _ -> raise (TypeError "range set adjustment requires adjustment (arg 2)")
  in 
    range_set_adjustment widget adjustment
    ;
    PyNone

(* text *)
let py_text_length
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  let widget = 
    begin match arg e 0 with
    | PyWidget w -> w
    | _ -> raise (TypeError "text replace requires widget (arg 1)")
    end
  in PyInt (text_get_length widget)

let py_text_get_text
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  exactly e 3;
  let widget = 
    begin match arg e 0 with
    | PyWidget w -> w
    | _ -> raise (TypeError "text get text requires widget (arg 1)")
    end
  and first = 
    begin match arg e 1 with 
    | PyInt i -> i
    | _ -> raise (TypeError "text get_text requires integer (arg 2)")
    end
  and last = 
    begin match arg e 2 with 
    | PyInt i -> i
    | _ -> raise (TypeError "text get_text requires integer (arg 3)")
    end
  in 
    let n = text_get_length widget in
    if first < 0 
    then raise (ValueError "first must be >=0")
    else if last > n
    then raise (ValueError "last must be <= length")
    else if first > last 
    then raise (ValueError "first must be <= last")
    else PyString (editable_get_chars widget first last)

(* this function is designed to implement:

   text[a:b]=string
   del text[a:b]
   text.append(string)

   .. etc
*)

let py_text_replace
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  exactly e 4;
  let widget = 
    begin match arg e 0 with
    | PyWidget w -> w
    | _ -> raise (TypeError "text replace requires widget (arg 1)")
    end
  and first = 
    begin match arg e 1 with 
    | PyInt i -> i
    | _ -> raise (TypeError "text replace requires integer (arg 2)")
    end
  and last = 
    begin match arg e 2 with 
    | PyInt i -> i
    | _ -> raise (TypeError "text replace requires integer (arg 3)")
    end
  and text =
    begin match arg e 3 with
    | PyString s -> s
    | _ -> raise (TypeError "text replace requires string (arg 4)")
    end
  and font = 
    match d#get_item (PyString "font") with 
    | Some (PyFont f) -> Some f
    | None -> None
    | _ -> raise (TypeError "font argument must be a font")
  and fg_color = 
    match d#get_item (PyString "foreground") with
    | Some (PyColor f) -> Some f
    | None -> None
    | _ -> raise (TypeError "foreground argument must be a color")
  and bg_color = match d#get_item (PyString "background") with
    | Some (PyColor f) -> Some f
    | None -> None
    | _ -> raise (TypeError "foreground argument must be a color")
  in
    let n = text_get_length widget in
    if first < 0 then raise (ValueError "first must be >=0")
    else if last > n then raise (ValueError "last must be <= length") 
    else if first > last then raise (ValueError "last must be > first") 
    else begin
      text_freeze widget;
      let pos = text_get_point widget in
      text_set_point widget first;
      ignore (text_forward_delete widget (last-first));
      text_insert widget font fg_color bg_color text 0 (String.length text);
      if pos < first 
      then text_set_point widget pos
      else if pos < last
      then text_set_point widget last
      else text_set_point widget (pos - (last-first) + (String.length text));
      text_thaw widget
    end
    ;
    PyNone

let py_text_getattr
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  exactly e 2;
  empty_dict d;
  let widget = 
    begin match arg e 0 with
    | PyWidget w -> w
    | _ -> raise (TypeError "text getattr requires widget (arg 1)")
    end
  and attr =
    begin match arg e 1 with
    | PyString s -> s
    | _ -> raise (TypeError "text getattr requires string (arg 2)")
    end
  in match attr with
  | "vadj" -> PyWidget (text_get_vadj widget)
  | "hadj" -> PyWidget (text_get_hadj widget)
  | "point" -> PyInt (text_get_point widget)
  | "length" -> PyInt (text_get_length widget)
  | _ -> raise (ValueError "text getattr attr must be vadj,hadj,point or length")

let py_text_setattr
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  exactly e 3;
  empty_dict d;
  let widget = 
    begin match arg e 0 with
    | PyWidget w -> w
    | _ -> raise (TypeError "text setattr requires widget (arg 1)")
    end
  and attr =
    begin match arg e 1 with
    | PyString s -> s
    | _ -> raise (TypeError "text setattr requires string (arg 2)")
    end
  in match attr with
  | "frozen" -> 
    if Py_datum.py_istrue (arg e 2) 
    then text_freeze widget
    else text_thaw widget
    ;
    PyNone
  | "word_wrap" -> 
    text_set_word_wrap widget (Py_datum.py_istrue (arg e 2))
    ;
    PyNone
  | "editable" -> 
    text_set_editable widget (Py_datum.py_istrue (arg e 2))
    ;
    PyNone
  | _ -> raise (ValueError "text setattr attr must be frozen, editable or word_wrap")

(* paned *)
let py_pane_add
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  exactly e 3;
  empty_dict d;
  let widget1 = 
    begin match arg e 0 with
    | PyWidget w -> w
    | _ -> raise (TypeError "pane add requires paned widget (arg 1)")
    end
  and widget2 = 
    begin match arg e 1 with
    | PyWidget w -> w
    | _ -> raise (TypeError "pane add requires widget (arg 2)")
    end
  and pane_no  = 
    begin match arg e 2 with
    | PyInt i -> i  
    | _ -> raise (TypeError "pane add requires integer pane number (arg 2)")
    end
  in
    begin match pane_no with
    | 1 -> paned_add1 widget1 widget2
    | 2 -> paned_add2 widget1 widget2
    | _ -> raise (ValueError "Gtk only supports pane1 and pane2")
    end
    ;
    PyNone

(* file selections *)
let py_file_selection_getattr
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  exactly e 2;
  empty_dict d;
  let widget = 
    begin match arg e 0 with
    | PyWidget w -> w
    | _ -> raise (TypeError "file_selection_get requires file selection widget (arg 1)")
    end
  in match arg e 1 with
  | PyString "filename" -> PyString (file_selection_get_filename widget)
  | PyString "ok_button" -> PyWidget (file_selection_get_ok_button widget)
  | PyString "cancel_button" -> PyWidget (file_selection_get_cancel_button widget)
  | PyString "help_button" -> PyWidget (file_selection_get_help_button widget)
  | PyString "selection_entry" -> PyWidget (file_selection_get_selection_entry widget)
  | PyString "selection_text" -> PyWidget (file_selection_get_selection_text widget)
  | PyString "main_vbox" -> PyWidget (file_selection_get_main_vbox widget)
  | PyString "dir_list" -> PyWidget (file_selection_get_dir_list widget)
  | PyString "file_list" -> PyWidget (file_selection_get_file_list widget)
  | _ -> raise (ValueError ( "file_selection getattr requires one of " ^ 
    "filename, ok_button, cancel_button, help_button, selection_entry, selection_text," ^
    " main_vbox, dir_list, file_list as arg2"))

let py_file_selection_setattr
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  exactly e 3;
  empty_dict d;
  let widget = 
    begin match arg e 0 with
    | PyWidget w -> w
    | _ -> raise (TypeError "file_selection_get requires file selection widget (arg 1)")
    end
  and attr =
    begin match arg e 1 with
    | PyString s -> s
    | _ -> raise (ValueError  "file_selection setattr requires string as arg2")
    end
  in match attr with
  | "filename" ->
    begin match arg e 2 with
    | PyString filename -> 
      file_selection_set_filename widget filename 
      ;
      PyNone
    | _ -> raise (ValueError  "file_selection setattr filename requires the filename as arg3")
    end
  | _ -> raise (ValueError "file_selection setattr, attr must be 'filename'")

(* Note books *)
let py_notebook_insert_page
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  exactly e 4;
  empty_dict d;
  let parent = 
    begin match arg e 0 with
    | PyWidget w -> w
    | _ -> raise (TypeError "notebook insert requires notebook widget (arg 1)")
    end
  and child = 
    begin match arg e 1 with
    | PyWidget w -> w
    | _ -> raise (TypeError "notebook insert requires widget (arg 2)")
    end
  and label =
    begin match arg e 2 with
    | PyWidget w -> w
    | _ -> raise (TypeError "notebook insert requires widget for label (arg 3)")
    end
  and pane_no  = 
    begin match arg e 3 with
    | PyInt i -> i  
    | _ -> raise (TypeError "notebook insert requires integer page number (arg 4)")
    end
  in
    notebook_insert_page parent child label pane_no
    ;
    PyNone

let py_notebook_remove_page
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  exactly e 2;
  empty_dict d;
  let parent = 
    begin match arg e 0 with
    | PyWidget w -> w
    | _ -> raise (TypeError "notebook remove requires notebook widget (arg 1)")
    end
  and pane_no  = 
    begin match arg e 1 with
    | PyInt i -> i  
    | _ -> raise (TypeError "notebook remove requires integer page number (arg 2)")
    end
  in
    notebook_remove_page parent pane_no
    ;
    PyNone

let py_notebook_set_page
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  exactly e 2;
  empty_dict d;
  let parent = 
    begin match arg e 0 with
    | PyWidget w -> w
    | _ -> raise (TypeError "notebook set page requires notebook widget (arg 1)")
    end
  and pane_no  = 
    begin match arg e 1 with
    | PyInt i -> i  
    | _ -> raise (TypeError "notebook set page requires integer page number (arg 2)")
    end
  in
    notebook_set_page parent pane_no
    ;
    PyNone

let py_notebook_get_page
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  exactly e 1;
  empty_dict d;
  let parent = 
    begin match arg e 0 with
    | PyWidget w -> w
    | _ -> raise (TypeError "notebook remove requires notebook widget (arg 1)")
    end
  in
    PyInt (notebook_get_current_page parent)

let py_notebook_setattr
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  exactly e 3;
  empty_dict d;
  let parent = 
    begin match arg e 0 with
    | PyWidget w -> w
    | _ -> raise (TypeError "notebook setattr requires notebook widget (arg 1)")
    end
  and attr =
    begin match arg e 1 with
    | PyString s -> s
    | _ -> raise (ValueError  "notebook setattr requires string as arg2")
    end
  in match attr with
  | "tab_pos" ->
    let tab_pos = 
      match arg e 2 with
      | PyString "left" -> Gtk.POS_LEFT
      | PyString "right" -> Gtk.POS_LEFT
      | PyString "top" 
      | PyString "up" -> Gtk.POS_TOP
      | PyString "bottom" 
      | PyString "down" -> Gtk.POS_BOTTOM
      | _ -> raise (ValueError "notebook set tab_pos requires string 'left','right','top','bottom'")
    in
     notebook_set_tab_pos parent tab_pos
     ;
     PyNone

  | "show_tabs" ->
     notebook_set_show_tabs parent (Py_datum.py_istrue (arg e 2))
     ;
     PyNone
  | "show_border" ->
     notebook_set_show_border parent (Py_datum.py_istrue (arg e 2))
     ;
     PyNone
  | "scroll_tabs" ->
     notebook_set_scrollable parent (Py_datum.py_istrue (arg e 2))
     ;
     PyNone
  | _ -> raise (ValueError "notebook setattr requires 'tab_pos', 'show_tabs' or 'show_border'")


(* clist *)
let py_clist_insert
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  exactly e 3;
  empty_dict d;
  let widget = 
    begin match arg e 0 with
    | PyWidget w -> w
    | _ -> raise (TypeError "clist insert requires clist widget (arg 1)")
    end
  and position =
    begin match arg e 1 with
    | PyInt i -> i
    | _ -> raise (TypeError "clist insert requires row index (arg 2)")
    end
  and data =
    Array.map
    Py_functions.str
    begin 
      try Py_util.array_of_sequence (arg e 2)
      with NonSequence -> raise (TypeError "clist insert requires sequence arg 3")
    end
  in 
    clist_insert widget position data
    ;
    PyNone

(* scrolled window *)
let py_scrolled_window_add_with_viewport
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 2;
  let widget = 
    match arg e 0 with
    | PyWidget w -> w
    | _ -> raise (TypeError "scrolled window add with viewport requires scrolled window widget (arg 1)")
  and child = 
    match arg e 1 with
    | PyWidget w -> w
    | _ -> raise (TypeError "scrolled window add with viewport requires widget (arg 2)")
  in 
    scrolled_window_add_with_viewport widget child
    ;
    PyNone

