open Py_types
open Py_exceptions
open Py_builtins_util
open Util

let py_dictionary_clear
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  begin match arg e 0 with 
  | PyDictionary d -> d#clear 
  | _ -> raise (TypeError "dictionary_clear method requires dictionary object")
  end;
  PyNone
;;

let py_dictionary_copy
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  begin match arg e 0 with 
  | PyDictionary d1 ->
    let d2 = new Py_dict.py_dictionary in
    d1#iter
    begin fun k v -> ignore (d2#set_item k v) end;
    PyDictionary d2
  | _ -> raise (TypeError "dictionary_copy method requires dictionary object")
  end
;;

let py_dictionary_has_key
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 2;
  match arg e 0 with 
  | PyDictionary d -> PyInt (int_of_bool (d#has_item (arg e 1)))
  | _ -> raise (TypeError "dictionary_has_key method requires dictionary object")
;;

let py_dictionary_items
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  match arg e 0 with 
  | PyDictionary d ->
    let items = Varray.empty () in
    d#iter 
    begin fun k v -> Varray.append_element items (PyTuple [k;v]) end;
    PyMutableList items
  | _ -> raise (TypeError "dictionary_items method requires dictionary object")
;;

let py_dictionary_keys
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  match arg e 0 with 
  | PyDictionary d -> PyMutableList (Varray.of_list d#keys)
  | _ -> raise (TypeError "dictionary_keys method requires dictionary object")
;;

let py_dictionary_update
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 2;
  match arg e 0 with 
  | PyDictionary d1 ->
    begin match arg e 1 with
    | PyDictionary d2 ->
      d2#iter
      begin fun k v -> ignore (d1#set_item k v) end;
      PyNone
    | _ -> raise (TypeError "dictionary_update method requires dictionary argument")
    end
  | _ -> raise (TypeError "dictionary_update method requires dictionary object")
;;

module OrderedExpr = struct type t = expr_t let compare = Py_datum.py_compare end;;
module ExprSet = Set.Make(OrderedExpr);;


let py_dictionary_values
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  match arg e 0 with 
  | PyDictionary d -> 
    let values = ref ExprSet.empty in    
    d#iter
    begin fun k v -> values := ExprSet.add v !values end;
    let vlist = Varray.empty () in
    ExprSet.iter
    begin fun x -> Varray.append_element vlist x end
    !values;
    PyMutableList vlist
  | _ -> raise (TypeError "dictionary_values method requires dictionary object")
;;

let py_dictionary_get
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  atleast e 2;
  atmost e 3;
  let dflt = 
    match optional e 2 with 
    | Some x -> x
    | None -> PyNone
  in
  match arg e 0 with 
  | PyDictionary d ->
    begin match d#get_item (arg e 1) with
    | Some x -> x
    | None -> dflt
    end
  | _ -> raise (TypeError "dictionary_get method requires dictionary object")
;;

