open Py_types
open Py_exceptions
open Big_int
open Num
open Util

(* Argument parsing utilities *)
let empty_dict d = 
  if d#len <> 0 
  then raise (TypeError "Invalid keyword arguments in call to builtin function");
;;

let arg e i = List.nth e i;;

let atleast e n = 
  if (List.length e) < n 
  then raise (TypeError "Insufficient arguments to builtin function")
;;

let argcount e = List.length e
;;

let int_of_expr e i = 
  match List.nth e i with 
  | PyInt n -> n 
  | _ -> raise (TypeError "Integer expected")
;;

let atmost e n = 
  if (List.length e) > n 
  then raise (TypeError "Too many arguments to builtin function")
;;

let exactly e n = 
  if (List.length e) <> n
  then raise (TypeError "Wrong number of arguments to builtin function")
;;

let optional e n = 
  if List.length e > n 
  then Some (List.nth e n)
  else None
;;

(* other utilities *)
let empty_dictionary = new Py_dict.py_dictionary;;

let make_argument_list tuple dict =
  let args = ref 
  begin List.map (fun x -> Argument1 x) tuple end 
  in dict#iter
  begin fun k v -> 
    match k with
    | PyString k' -> args := !args @ [Argument2 (k', v)] 
    | _ -> 
      raise (TypeError "Third argument to apply must be dictionary with string keys")
  end;
  !args
;;

let py_tuple_of_sequence arg = 
  try PyTuple (Py_util.list_of_sequence arg)
  with NonSequence -> 
    raise (TypeError "Argument to 'tuple' must be a sequence")
;;

let py_list_of_sequence arg = 
  try PyMutableList (Varray.of_list (Py_util.list_of_sequence arg))
  with NonSequence -> 
    raise (TypeError "Argument to 'list' must be a sequence")
;;

(* Coercion utilities *)

(* COERCE: extended semantics!
   1. Coerce can accept any number n of numeric arguments, 
      specified positionally,
     it returns a tuple of n elements of the same type
     which is the least upper bound in the lattice whose
     arrows are conversions.

   2. The accepted non-identity conversions are:
      Int -> Float -> Complex
*)

(* actual conversions *)
let py_float_of_int i = 
  match i with PyInt i' -> PyFloat (float_of_int i')
  | _ -> raise (Failure "py_float_of_int")
;;

let py_complex_of_float f = 
  match f with PyFloat f' -> PyComplex (f',0.0)
  | _ -> raise (Failure "py_complex_of_float")
;;

let py_complex_of_int z = 
  match z with PyInt i' -> PyComplex (float_of_int i',0.0)
  | _ -> raise (Failure "py_complex_of_int")
;;

let py_long_of_int i = 
  match i with PyInt i' -> PyLong (big_int_of_int i')
  | _ -> raise (Failure "py_long_of_int")
;;

let py_rational_of_int i = 
  match i with PyInt i' -> PyRational (num_of_int i')
  | _ -> raise (Failure "py_rational_of_int")
;;

let py_rational_of_long i = 
  match i with PyLong i' -> PyRational (num_of_big_int i')
  | _ -> raise (Failure "py_rational_of_big_intt")
;;

let ident x:expr_t = x;;

(* type symbols: Z is an initial symbol *)
type tRep = I | F | C | S | Z | R | L;;

let string_of_tRep t = match t with 
| I -> "Integer"
| L -> "Long"
| R -> "Rational"
| F -> "Float"
| C -> "Complex"
| S -> "String"
| Z -> "Initial"
;;

(* get the conversion for a given symbolic conversion *)
let cvt_of_Rep (i:tRep) (j:tRep) : (expr_t -> expr_t) =
  match (i,j) with
  | (I,I) -> ident
  | (I,L) -> py_long_of_int
  | (I,R) -> py_rational_of_int
  | (I,F) -> py_float_of_int
  | (I,C) -> py_complex_of_int

  | (L,L) -> ident
  | (L,R) -> py_rational_of_long

  | (R,R) -> ident

  | (F,F) -> ident
  | (F,C) -> py_complex_of_float

  | (C,C) -> ident
  | _ -> raise (TypeError (
    "Cannot convert types " ^
    (string_of_tRep i) ^ " -> " ^
    (string_of_tRep j)
    ))
;;

(* reduced table of least upper bounds on type symbols *)
let lubRep = [
  ((I,L), L); 
  ((I,R), R); 
  ((I,F), F); 
  ((I,C),C);
  ((L,R), R); 
  ((F,C),C) 
];;

(* get the least upper bound of two symbolic types *)
let repCoerce (i:tRep) (j:tRep) : tRep =
  if i = j then i
  else if i = Z then j
  else try List.assoc (i,j) lubRep
  with Not_found -> 
    try List.assoc (j,i) lubRep
    with Not_found -> raise (TypeError "Cannot coerce types")
;;

(* get the symbolic type of an expression *)
let rep_of_expr (x:expr_t) : tRep = 
  match x with
  | PyInt _ -> I
  | PyLong _ -> L
  | PyRational _ -> R
  | PyFloat _ -> F
  | PyComplex _ -> C
  | PyString _ -> S 
  | _ -> raise (TypeError "Arguments to coerce must be int, long, rational, float, complex or string")
;;

(* get the lub of a list of expressions *)
let lub (l:expr_t list) =
  List.fold_left 
  begin fun x y -> repCoerce x (rep_of_expr y) end
  Z 
  l
;;

let coerce (l:expr_t list) =
  let cvt = fun x -> cvt_of_Rep x (lub l) in
    List.map
    (fun x -> (cvt (rep_of_expr x)) x)
    l
;;

