(* Abstact Data Linear data structures *)


(* A module is strict if the complexity of all operations are constant time

  Unlike STL, we have no notion of invalidating an iterator:
  iterators are always required to provide correct semantics
*)

(* Unbounded iterators: operate on conceptually infinite
  data structures
*)

module type OutputIterator =
  sig
    type 'a iterator
    val write: 'a iterator -> 'a -> 'a iterator
  end

(* Input iterators are characterized by advancing destructively,
  which is represented by the inability to compare them.

*)
  
module type InputIterator = 
  sig
    type 'a iterator
    val next : 'a iterator
    val deref : 'a iterator -> 'a option
  end

module type ForwardIterator =
  sig
    type 'a iterator 

    val next : 'a iterator -> 'a iterator
    val deref : 'a iterator -> 'a option 
    val equal : 'a iterator -> 'a iterator -> bool
  end
  
module type BiDirectionalIterator =
  sig
    type 'a iterator 

    val next : 'a iterator -> 'a iterator
    val prev : 'a iterator -> 'a iterator
    val deref : 'a iterator -> 'a option 
    val equal : 'a iterator -> 'a iterator -> bool
  end

module type RandomIterator =
  sig
    type 'a iterator 

    val advance : 'a iterator -> int -> 'a iterator
    val retreat : 'a iterator -> int -> 'a iterator
    val next : 'a iterator -> 'a iterator
    val prev : 'a iterator -> 'a iterator
    val deref : 'a iterator -> 'a option 
    val equal : 'a iterator -> 'a iterator -> bool
    val less : 'a iterator -> 'a iterator -> bool
  end

module type BiDirectional =
  sig
    type 'a iterator 

    val next : 'a iterator -> 'a iterator
    val prev : 'a iterator -> 'a iterator
    val deref : 'a iterator -> 'a option 
    val terminal : 'a iterator

    class ['a] dlist :
      object
        method as_list : 'a list
        method first : 'a iterator 
        method iter : ('a -> unit) -> unit
        method last : 'a iterator 
        method length : int
        method nth : int -> 'a option (* O(n) *)
        method pop_back : 'a option
        method pop_front : 'a option
        method push_back : 'a -> unit
        method push_front : 'a -> unit
        method terminal : 'a iterator
        method indexed_insert : int -> 'a -> unit
        method indexed_replace : int -> 'a -> unit
        method indexed_delete : int -> unit
      end

    val of_list : 'a list -> 'a dlist 
    val map : 'a dlist -> ('a -> 'b) -> 'b dlist
    val create : unit -> 'a dlist
    val concat : 'a dlist -> 'a dlist -> 'a dlist
  end

module DoublyLinkedList : BiDirectional =
  struct (* doubly linked list type *)
    type 't d_node = 
      {
        mutable nxt: 't iterator; 
        mutable prv: 't iterator;
        mutable data: 't
      }

    and 't iterator = 
      Empty 
      | Node of 't d_node

    let next x = match x with Empty -> Empty | Node n -> n.nxt
    let prev x = match x with Empty -> Empty | Node n -> n.prv
    let deref x = match x with Empty -> None | Node n -> Some n.data
    let terminal = Empty

    class ['t] dlist = 
      object(self)
        val mutable first': 't iterator = Empty
        val mutable last':  't iterator = Empty
        val mutable size: int = 0

        method private init node = 
          last' <- node; 
          first' <- node;
          size <- 1

        method length = size

        (* STL style mutators *)
        method push_back (data':'t): unit = 
          match last' with
          | Empty -> self#init (Node {nxt=Empty; prv=Empty; data=data'})
          | Node fin -> 
            let tmp = Node {nxt=Empty; prv=last'; data=data'}  in 
              fin.nxt <- tmp;
              last' <- tmp;
              size <- size + 1

        method push_front (data':'t): unit =
          match first' with
          | Empty -> self#init (Node {nxt=Empty; prv=Empty; data=data'})
          | Node start ->
            let tmp = Node {nxt=first'; prv=Empty; data=data'} in 
              start.prv <- tmp;
              first' <- tmp;
              size <- size + 1

        method pop_front : 't option = 
          match first' with
          | Empty -> None 
          | Node start ->
              first' <- start.nxt;
              begin match first' with
                | Empty -> last' <- Empty
                | Node n -> n.prv <- Empty
              end;
              size <- size - 1;
              Some start.data

        method pop_back : 't option = 
          match last' with
          | Empty -> None 
          | Node fin ->
              last' <- fin.prv;
              begin match last' with
                | Empty -> first' <- Empty 
                | Node n -> n.nxt <- Empty
              end;
              size <- size - 1;
              Some fin.data

       (* n'th element, from closest end *)
       method nth i : 't option = 
          if i < 0 or i >= size 
          then None
          else if i <= (size / 2)
          then let rec p d i = match d with
            | Empty -> None
            | Node n -> 
              if i = 0 
              then Some n.data 
              else p n.nxt (i-1)
            in p first' i
          else let rec p d i = match d with
            | Empty -> None
            | Node n ->
              if i = 0 
              then Some n.data
              else p n.prv (i-1)
            in p last' (size - i - 1)

        (* indexed mutators *)
        method indexed_insert i x =
          let insert n =
            let newnode =  Node {nxt= Node n; prv= n.prv; data=x} in
               begin match n.prv with 
               | Node nm1 -> nm1.nxt <- newnode 
               | Empty -> assert false (* impossible *)
               end;
               n.prv <- newnode;
               size <- size + 1
          in if i < 0 or i > size 
          then raise (Invalid_argument "Doubly linked list index out of range")
          else if i = 0 then self#push_front x
          else if i = size then self#push_back x
          else if i <= (size / 2)
          then let rec p d i = match d with
            | Empty -> assert false (* impossible *)
            | Node n -> 
              if i = 0 
              then insert n
              else p n.nxt (i-1)
            in p first' i
          else let rec p d i = match d with
            | Empty -> assert false (* impossible *)
            | Node n ->
              if i = 0 
              then insert n
              else p n.prv (i-1)
            in p last' (size - i - 1)

        method indexed_delete i =
          let delete n =
            begin match n.nxt with Node m -> m.prv <- n.prv | Empty -> assert false end;
            begin match n.prv with Node m -> m.nxt <- n.nxt | Empty -> assert false end;
            size <- size - 1
          in if i < 0 or i >= size 
          then raise (Invalid_argument "Doubly linked list index out of range")
          else if i = 0 then ignore (self#pop_front)
          else if i = (size - 1) then ignore (self#pop_back)
          else if i <= (size / 2)
          then let rec p d i = match d with
            | Empty -> assert false (* impossible *)
            | Node n -> 
              if i = 0 
              then delete n
              else p n.nxt (i-1)
            in p first' i
          else let rec p d i = match d with
            | Empty -> assert false (* impossible *)
            | Node n ->
              if i = 0 
              then delete n
              else p n.prv (i-1)
            in p last' (size - i - 1)

        method indexed_replace i x =
          if i < 0 or i >= size 
          then raise (Invalid_argument "Doubly linked list index out of range")
          else if i <= (size / 2)
          then let rec p d i = match d with
            | Empty -> assert false (* impossible *)
            | Node n -> 
              if i = 0 
              then n.data <- x
              else p n.nxt (i-1)
            in p first' i
          else let rec p d i = match d with
            | Empty -> assert false (* impossible *) 
            | Node n ->
              if i = 0 
              then n.data <- x
              else p n.prv (i-1)
            in p last' (size - i - 1)

        (* bidirectional iterators *)
        method first = first'
        method last = last'
        method terminal : 't iterator = Empty

        (* conveniences *)
        method iter (f:'t->unit) : unit =
          let rec p i = match i with
          | Empty -> ()
          | Node n -> f n.data; p n.nxt
          in p first'

        method as_list : 't list =
          let rec p i l = match i with
          | Empty -> l 
          | Node n -> p (n.prv) (n.data :: l)
          in p last' []

      end

    let create () = new dlist

    let of_list l = let m = new dlist in
      List.iter m#push_back l;
      m

    let map (h:'t dlist) (f:'t->'a) : 'a dlist =
      let m = new dlist in
        h#iter 
          begin fun x -> m#push_back (f x) end;
        m

    let concat (x:'a dlist) (y:'a dlist) :'a dlist = 
      let z = new dlist in
      let i = ref x#first in
      while deref !i <> None do 
        match deref !i with 
        | Some v -> z#push_back v; i := next !i 
        | None -> assert false
      done;
      let j = ref y#first in
      while deref !j <> None do 
        match deref !j with
        | Some v -> z#push_back v; j := next !j 
        | None -> assert false
      done;
      z

  end (* struct *)

