/**********************************************************************
Copyright 1999 by ITG Australia.

                        All Rights Reserved

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose and without fee is hereby granted,
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in
supporting documentation, and that the names of ITG Australia or ITGA
not be used in advertising or publicity pertaining to distribution of
the software without specific, written prior permission.

ITG AUSTRALIA DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
EVENT SHALL ITG AUSTRALIA BE LIABLE FOR ANY SPECIAL, INDIRECT OR
CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
PERFORMANCE OF THIS SOFTWARE.
**********************************************************************/
#include "Sybase.h"

static int next_cursor_num;	/* unique identifier for cursors */

/* ----------------------------------------------------- */

staticforward PyTypeObject CursorType;	/* shared type descriptor */

#define is_Cursor_object(v) ((v)->ob_type == &CursorObj)

/* ---------------------------------------------------------------- */

/* Wait for the confirmation status (CS_END_RESULTS) of a simple command.
 */
static int cursor_wait_results(CmdInfo *cmd_info)
{
    for (;;) {
	CS_INT result;

	switch (wrap_ct_results(cmd_info, &result)) {
	case CS_END_RESULTS:
	    return 1;
	case CS_SUCCEED:
	    if (!handle_result_type(cmd_info, result))
		return 0;
	    switch (result) {
	    case CS_MSG_RESULT:
	    case CS_CMD_FAIL:
	    case CS_DESCRIBE_RESULT:
	    case CS_COMPUTEFMT_RESULT:
	    case CS_ROWFMT_RESULT:
		/* These are handled above
		 */
		return 0;

	    case CS_CMD_DONE:
	    case CS_CMD_SUCCEED:
		/* So far, so good.
		 */
		break;

	    case CS_COMPUTE_RESULT:
	    case CS_CURSOR_RESULT:
	    case CS_PARAM_RESULT:
	    case CS_ROW_RESULT:
	    case CS_STATUS_RESULT:
		/* We definitely did not expect to see these
		 */
		cmd_abort_quietly(cmd_info);
		raise_exception_string(InterfaceError, "extra cursor results received");
		return 0;
	    }
	    break;
	default:
	    return 0;
	}
    }
}

/* Fetch one row from a cursor
 */
static PyObject *cursor_fetch_one(CmdInfo *cmd_info)
{
    PyObject *row_tuple;	/* the row fetched */
    int fetch_failed;		/* return status of cmd_fetch_row() */

    debug_msg(cmd_info->debug, "  cursor_fetch_one\n");

    /* Cursor access enters and exits the command sequence and fetch
     * loop.  A state machine is used to remember the state of the
     * command sequence and fetch loop.
     */
    for (;;) {
	switch (cmd_info->state) {
	case STATE_INIT:
	    /* Command has been allocated, but cursor has not been opened.
	     */
	    raise_exception_string(ProgrammingError, "you must call execute first");
	    return NULL;
	case STATE_FETCH:
	    /* Cursor has been opened and rows can be retrieved.
	     */
	    row_tuple = cmd_fetch_row(cmd_info, &fetch_failed);
	    if (row_tuple != NULL)
		return row_tuple;
	    if (fetch_failed)
		return NULL;

	    /* End of result set.  Now clean up the rest of the command
	     */
	    if (!cursor_wait_results(cmd_info))
		return NULL;
	    cmd_info->state = STATE_FINISHED;
	    break;
	case STATE_FINISHED:
	    /* All results have been fetched.
	     */
	    Py_INCREF(Py_None);
	    return Py_None;
	case STATE_CLOSED:
	    /* Cursor has been closed either manually via close()
	     * method, or as a side effect of a Sybase or Python error.
	     */
	    raise_exception_string(ProgrammingError, "cursor closed");
	    return NULL;
	}
    }
}

/* Fetch one or more rows from the cursor and return them in a Python
 * list.
 */
static PyObject *cursor_fetch(CmdInfo *cmd_info, int num_rows)
{
    PyObject *result_list;

    debug_msg(cmd_info->debug, "  cursor_fetch(%d)\n", num_rows);

    result_list = PyList_New(0);
    if (result_list == NULL) {
	/* Very bad things happened - we have to cancel the Sybase
	 * command now that we cannot store results.
	 */
	cmd_abort_quietly(cmd_info);
	return NULL;
    }

    /* Note that the fetchall() method calls us with @num_rows == -1,
     * so we cannot just check for @num_rows > 0 to terminate the
     * loop.
     */
    while (num_rows-- != 0) {
	PyObject *row_tuple;

	row_tuple = cursor_fetch_one(cmd_info);
	if (row_tuple == NULL) {
	    Py_DECREF(result_list);
	    return NULL;
	}
	if (row_tuple == Py_None) {
	    Py_DECREF(Py_None);
	    return result_list;
	}

	/* Append row to results list
	 */
	if (PyList_Append(result_list, row_tuple) < 0) {
	    cmd_abort_quietly(cmd_info);
	    Py_DECREF(row_tuple);
	    Py_DECREF(result_list);
	    return NULL;
	}
	Py_DECREF(row_tuple);
    }

    return result_list;
}

/* Open the cursor, send the command, then read results until the
 * first row is ready to be fetched.
 */
static int cursor_open(CursorObj *self,
		       char *sql_command, PyObject *param_seq)
{
    char cursor_name[64];
    CmdInfo *cmd_info;
    CS_INT option;

    cmd_info = &self->cmd_info;
    sprintf(cursor_name, "cursor_%d", next_cursor_num++);

    /* Construct the cursor and send the command
     */
    option = self->is_update ? CS_FOR_UPDATE : CS_READ_ONLY;
    debug_msg(cmd_info->debug, "    ct_cursor(declare)\n");
    if (ct_cursor(cmd_info->cmd, CS_CURSOR_DECLARE, cursor_name, CS_NULLTERM,
		  sql_command, CS_NULLTERM, option) != CS_SUCCEED) {
	raise_exception(cmd_info->conn_info, "ct_cursor(declare) failed");
	return 0;
    }
    debug_msg(cmd_info->debug, "    ct_cursor(open)\n");
    if (ct_cursor(cmd_info->cmd, CS_CURSOR_OPEN, NULL, CS_UNUSED, NULL,
		  CS_UNUSED, CS_UNUSED) != CS_SUCCEED) {
	raise_exception(cmd_info->conn_info, "ct_cursor(open) failed");
	return 0;
    }

    if (param_seq != NULL)
	cmd_send_params(cmd_info, param_seq);

    debug_msg(cmd_info->debug, "    ct_send()\n");
    if (ct_send(cmd_info->cmd) != CS_SUCCEED) {
	raise_exception(cmd_info->conn_info, "cursor ct_send() failed");
	return 0;
    }

    cmd_info->is_cursor_open = 1;

    for (;;) {
	CS_INT result;

	switch (wrap_ct_results(cmd_info, &result)) {
	case CS_END_RESULTS:
	    /* What?!?  We did not get any results from the cursor.
	     * We should only get this return value once all of the
	     * cursor results have been retrieved.
	     */
	    cmd_info->state = STATE_FINISHED;
	    return 1;
	case CS_SUCCEED:
	    if (!handle_result_type(cmd_info, result))
		return 0;
	    switch (result) {
	    case CS_MSG_RESULT:
	    case CS_CMD_FAIL:
	    case CS_DESCRIBE_RESULT:
	    case CS_COMPUTEFMT_RESULT:
	    case CS_ROWFMT_RESULT:
		/* These are handled above
		 */
		return 0;

	    case CS_CMD_DONE:
	    case CS_CMD_SUCCEED:
		/* So far, so good.  Keep going until we get one of
		 * the results below.
		 */
		break;

	    case CS_COMPUTE_RESULT:
	    case CS_CURSOR_RESULT:
	    case CS_PARAM_RESULT:
	    case CS_ROW_RESULT:
	    case CS_STATUS_RESULT:
		/* These result codes indicate data results to be
		 * fetched.
		 */
		if (!cmd_row_bind(cmd_info, &self->desc))
		    return 0;

		cmd_info->state = STATE_FETCH;
		return 1;
	    }
	    break;
	default:
	    return 0;
	}
    }
}

/* Submit an SQL command to the Sybase server for this cursor
 */
static PyObject *cursor_execute(CursorObj *self,
				char *sql_command, PyObject *param_seq)
{
    debug_msg(self->cmd_info.debug, "  cursor_execute\n");

    /* The DB API 2.0 spec is not clear on what should be done when
     * the execute() method is invoked in all cursor states.
     */
    for (;;) {
	switch (self->cmd_info.state) {
	case STATE_INIT:
	    /* Command has been allocated for the cursor, now open the
	     * cursor on the SQL command and get it ready for results to
	     * be fetched.
	     */
	    if (cursor_open(self, sql_command, param_seq)) {
		Py_INCREF(Py_None);
		return Py_None;
	    }
	    return NULL;
	case STATE_FETCH:
	case STATE_FINISHED:
	    /* The cursor was retrieving results, or had just
	     * finished.  Instead of raising an exception, close the
	     * cursor which will trigger the STATE_CLOSED state.
	     */
	    cmd_free(&self->cmd_info);
	    if (self->desc != NULL) {
		Py_DECREF(self->desc);
	    }
	    Py_INCREF(Py_None);
	    self->desc = Py_None;
	    break;
	case STATE_CLOSED:
	    /* Cursor has been closed.  Allocate a new command to
	     * trigger the STATE_INIT state.
	     */
	    if (!cmd_init(&self->cmd_info,
			  self->cmd_info.conn_info,
			  self->cmd_info.conn_info->debug, 1))
		return NULL;
	    if (self->desc != NULL) {
		Py_DECREF(self->desc);
	    }
	    Py_INCREF(Py_None);
	    self->desc = Py_None;
	    break;
	}
    }
}

/* ---------------------------------------------------------------- */

/* Implement the cursor.close() method as per DB API spec 2.0
 */
static PyObject *Cursor_close(CursorObj *self, PyObject *args)
{
    if (!PyArg_ParseTuple(args, ""))
	return NULL;
    cmd_free(&self->cmd_info);

    Py_INCREF(Py_None);
    return Py_None;
}

/* Implement the cursor.execute() method as per DB API spec 2.0
 */
static PyObject *Cursor_execute(CursorObj *self, PyObject *args)
{
    char *sql_command;
    PyObject *param_seq;

    sql_command = NULL;
    param_seq = NULL;
    if (!PyArg_ParseTuple(args, "s|O", &sql_command, &param_seq))
	return NULL;
    if (param_seq != NULL && !PySequence_Check(param_seq)) {
	raise_exception_string(ProgrammingError, "args must be a sequence");
	return NULL;
    }
    if (!conn_check(self->cmd_info.conn_info) || !cmd_check(&self->cmd_info))
	return NULL;

    return cursor_execute(self, sql_command, param_seq);
}

/* Implement the cursor.close() method as per DB API spec 2.0.
 */
static PyObject *Cursor_executemany(CursorObj *self, PyObject *args)
{
    char *sql_command;
    PyObject *param_seq;
    PyObject *result_list;
    int num_params;
    int idx;

    sql_command = NULL;
    param_seq = NULL;
    if (!PyArg_ParseTuple(args, "sO", &sql_command, &param_seq))
	return NULL;
    if (!PySequence_Check(param_seq)) {
	raise_exception_string(ProgrammingError, "args must be a sequence");
	return NULL;
    }
    if (!conn_check(self->cmd_info.conn_info) || !cmd_check(&self->cmd_info))
	return NULL;

    result_list = PyList_New(0);
    num_params = PySequence_Length(param_seq);

    for (idx = 0; idx < num_params; idx++) {
	PyObject *param;
	PyObject *result;

	param = PySequence_GetItem(param_seq, idx);
	if (!PySequence_Check(param)) {
	    raise_exception_string(ProgrammingError, "args must be a sequence");
	    Py_DECREF(result_list);
	    Py_DECREF(param);
	    return NULL;
	}
	result = cursor_execute(self, sql_command, param);
	if (result == NULL) {
	    Py_DECREF(result_list);
	    Py_DECREF(param);
	    return NULL;
	}
	if (PyList_Append(result_list, result) < 0) {
	    Py_DECREF(result);
	    Py_DECREF(result_list);
	}
	Py_DECREF(result);
	Py_DECREF(param);
    }

    return result_list;
}

/* Implement the cursor.fetchone() method as per DB API spec 2.0.
 */
static PyObject *Cursor_fetchone(CursorObj *self, PyObject *args)
{
    if (!PyArg_ParseTuple(args, ""))
	return NULL;
    if (!conn_check(self->cmd_info.conn_info) || !cmd_check(&self->cmd_info))
	return NULL;

    return cursor_fetch_one(&self->cmd_info);
}

/* Implement the cursor.fetchmany() method as per DB API spec 2.0.
 */
static PyObject *Cursor_fetchmany(CursorObj *self, PyObject *args)
{
    int num_rows;

    num_rows = 0;
    if (!PyArg_ParseTuple(args, "|i", &num_rows))
	return NULL;
    if (num_rows == 0)
	num_rows = self->array_size;
    if (!conn_check(self->cmd_info.conn_info) || !cmd_check(&self->cmd_info))
	return NULL;

    return cursor_fetch(&self->cmd_info, num_rows);
}

/* Implement the cursor.fetchall() method as per DB API spec 2.0.
 */
static PyObject *Cursor_fetchall(CursorObj *self, PyObject *args)
{
    if (!PyArg_ParseTuple(args, ""))
	return NULL;
    if (!conn_check(self->cmd_info.conn_info) || !cmd_check(&self->cmd_info)) 
	return NULL;

    return cursor_fetch(&self->cmd_info, -1);
}

/* Implement the cursor.setinputsizes() method as per DB API spec 2.0.
 *
 * Our implementation does nothing.  This is legal according to the
 * spec.
 */
static PyObject *Cursor_setinputsizes(CursorObj *self, PyObject *args)
{
    PyObject *obj;

    if (!PyArg_ParseTuple(args, "O", &obj))
	return NULL;

    Py_INCREF(Py_None);
    return Py_None;
}

/* Implement the cursor.setoutputsize() method as per DB API spec 2.0.
 *
 * Our implementation does nothing.  This is legal according to the
 * spec.
 */
static PyObject *Cursor_setoutputsize(CursorObj *self, PyObject *args)
{
    int size;
    int column;

    if (!PyArg_ParseTuple(args, "i|i", &size, &column))
	return NULL;

    Py_INCREF(Py_None);
    return Py_None;
}

static struct PyMethodDef Cursor_methods[] = {
    { "close", (PyCFunction)Cursor_close, METH_VARARGS },
    { "execute", (PyCFunction)Cursor_execute, METH_VARARGS },
    { "executemany", (PyCFunction)Cursor_executemany, METH_VARARGS },
    { "fetchone", (PyCFunction)Cursor_fetchone, METH_VARARGS },
    { "fetchmany", (PyCFunction)Cursor_fetchmany, METH_VARARGS },
    { "fetchall", (PyCFunction)Cursor_fetchall, METH_VARARGS },
    { "setinputsizes", (PyCFunction)Cursor_setinputsizes, METH_VARARGS },
    { "setoutputsize", (PyCFunction)Cursor_setoutputsize, METH_VARARGS },
    { NULL, NULL }  /* sentinel */
};

static void Cursor_dealloc(CursorObj *self)
{
    cmd_free(&self->cmd_info);

    if (self->desc != NULL) {
	Py_DECREF(self->desc);
    }
    PyMem_DEL(self);
}

#define OFF(x) offsetof(CursorObj, x)

static struct memberlist cursor_members[] = {
    { "description",	T_OBJECT,	OFF(desc),	RO },
    { "rowcount",	T_INT,		OFF(row_count),	RO },
    { "arraysize",	T_INT,		OFF(array_size) },
    { NULL }  /* sentinel */
};

static PyObject *Cursor_getattr(CursorObj *self, char *name)
{
    PyObject *res;

    res = Py_FindMethod(Cursor_methods, (PyObject*)self, name);
    if (res != NULL)
	return res;
    PyErr_Clear();
    return PyMember_Get((char *)self, cursor_members, name);
}

static int Cursor_setattr(CursorObj *self, char *name, PyObject *value)
{
    if (value == NULL) {
	PyErr_SetString(PyExc_AttributeError,
			"can't delete file attributes");
	return -1;
    }
    return PyMember_Set((char *)self, cursor_members, name, value);
}

/* ---------------------------------------------------------------- */

static PyTypeObject CursorType = { /* main python type-descriptor */
    /* type header */
    PyObject_HEAD_INIT(&PyType_Type)
    0,				/* ob_size */
    "Connect",			/* tp_name */
    sizeof(CursorObj),		/* tp_basicsize */
    0,				/* tp_itemsize */

    /* standard methods */
    (destructor)Cursor_dealloc,/* tp_dealloc */
    (printfunc)0,
    (getattrfunc)Cursor_getattr, /* tp_getattr */
    (setattrfunc)Cursor_setattr, /* tp_setattr */
    (cmpfunc)0,
    (reprfunc)0
};

/* ---------------------------------------------------------------- */

PyObject *cursor_new(ConnInfo *conn_info, int is_update, int debug)
{
    CursorObj *cursor;

    cursor = PyObject_NEW(CursorObj, &CursorType);
    if (cursor == NULL)
	return NULL;

    cursor->desc = NULL;
    cursor->is_update = is_update;
    cursor->row_count = -1;
    cursor->array_size = 1;
    if (cmd_init(&cursor->cmd_info, conn_info, debug, 1)) {
	Py_INCREF(Py_None);
	cursor->desc = Py_None;
	return (PyObject*)cursor;
    }

    Py_DECREF(cursor);
    return NULL;
}
