/**********************************************************************
Copyright 1999 by ITG Australia.

                        All Rights Reserved

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose and without fee is hereby granted,
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in
supporting documentation, and that the names of ITG Australia or ITGA
not be used in advertising or publicity pertaining to distribution of
the software without specific, written prior permission.

ITG AUSTRALIA DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
EVENT SHALL ITG AUSTRALIA BE LIABLE FOR ANY SPECIAL, INDIRECT OR
CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
PERFORMANCE OF THIS SOFTWARE.
**********************************************************************/
#include "Sybase.h"

staticforward PyTypeObject ConnectType;	/* shared type descriptor */

#define is_Connect_object(v) ((v)->ob_type == &ConnectObj)

/* ---------------------------------------------------------------- */

/* Return a list of logical results for a command.  On failure, raises
 * a Python exception and returns NULL.
 */
static PyObject *cmd_fetch_results(CmdInfo *cmd_info)
{
    PyObject *result_list;	/* list of logical results */

    debug_msg(cmd_info->debug, "  cmd_fetch_results\n");

    /* Parse results of command returned by server
     */
    result_list = PyList_New(0);
    if (result_list == NULL) {
	/* Very bad things happened - we have to cancel the Sybase
	 * command now that we cannot store results.
	 */
	cmd_abort_quietly(cmd_info);
	return NULL;
    }

    for (;;) {
	PyObject *logical_result = NULL;
	CS_INT result;

	switch (wrap_ct_results(cmd_info, &result)) {
	case CS_END_RESULTS:
	    return result_list;
	case CS_SUCCEED:
	    break;
	default:
	    Py_DECREF(result_list);
	    return NULL;
	}

	/* We only get down here when ct_results() returns CS_SUCCEED.
	 * Now we need to determine the type of result returned.
	 */
	handle_result_type(cmd_info, result);
	switch (result) {
	case CS_MSG_RESULT:
	case CS_CMD_FAIL:
	case CS_DESCRIBE_RESULT:
	case CS_COMPUTEFMT_RESULT:
	case CS_ROWFMT_RESULT:
	    Py_DECREF(result_list);
	    return NULL;

	case CS_CMD_DONE:
	case CS_CMD_SUCCEED:
	    break;

	case CS_COMPUTE_RESULT:
	case CS_CURSOR_RESULT:
	case CS_PARAM_RESULT:
	case CS_ROW_RESULT:
	case CS_STATUS_RESULT:
	    /* Fetch the results of a logical command
	     */
	    if (cmd_row_bind(cmd_info, NULL)) {
		logical_result = cmd_fetch_logical_result(cmd_info);
		if (logical_result != NULL) {
		    /* Append logical result to results list
		     */
		    if (PyList_Append(result_list, logical_result) < 0) {
			cmd_abort_quietly(cmd_info);
			Py_DECREF(logical_result);
			Py_DECREF(result_list);
			return NULL;
		    }
		    Py_DECREF(logical_result);
		    break;
		}
	    }
	    Py_DECREF(result_list);
	    return NULL;
	}
    }
}

/* Execute an SQL command and return a list of logical results.
 */
PyObject *cmd_execute(CmdInfo *cmd_info,
		      char *sql_command, PyObject *param_seq)
{
    PyObject *result_list;

    debug_msg(cmd_info->debug, "  cmd_execute\n");
    /* Build the command and send it to the server
     */
    debug_msg(cmd_info->debug, "    ct_command()\n");
    if (ct_command(cmd_info->cmd, CS_LANG_CMD, sql_command,
		   CS_NULLTERM, CS_UNUSED) != CS_SUCCEED) {
	raise_exception(cmd_info->conn_info, "ct_command failure");
	return NULL;
    }

    if (param_seq != NULL)
	cmd_send_params(cmd_info, param_seq);

    debug_msg(cmd_info->debug, "    ct_send()\n");
    if (ct_send(cmd_info->cmd) != CS_SUCCEED) {
	raise_exception(cmd_info->conn_info, "ct_send failure");
	return NULL;
    }

    result_list = cmd_fetch_results(cmd_info);
    conn_clear_messages(cmd_info->conn_info);

    return result_list;
}

/* ---------------------------------------------------------------- */

/* Execute an SQL command string on the server.  This is not part of
 * the DB API specification.  At the moment, this is the only way to
 * access a database other than the default.
 *
 * e.g.
 *     s = Sybase.connect('SYBASE', 'sa', '')
 *     s.execute('use other')
 *     c = s.cursor()
 *     c.execute('select * from something')
 */
static PyObject *Connect_execute(ConnectObj *self, PyObject *args)
{
    char *sql_command;
    PyObject *param_seq;

    param_seq = NULL;
    if (!PyArg_ParseTuple(args, "s|O", &sql_command, &param_seq))
	return NULL;
    if (param_seq != NULL && !PySequence_Check(param_seq)) {
	raise_exception_string(ProgrammingError, "args must be a sequence");
	return NULL;
    }
    if (!conn_check(&self->conn_info) || !cmd_check(&self->cmd_info))
	return NULL;

    return cmd_execute(&self->cmd_info, sql_command, param_seq);
}

/* Implement the connect.close() method as per DB API spec 2.0.
 */
static PyObject *Connect_close(ConnectObj *self, PyObject *args)
{
    if (!PyArg_ParseTuple(args, ""))
	return NULL;

    cmd_free(&self->cmd_info);
    conn_free(&self->conn_info);

    Py_INCREF(Py_None);
    return Py_None;
}

/* Implement the connect.cursor() method as per DB API spec 2.0.
 */
static PyObject *Connect_cursor(ConnectObj *self,
				PyObject *args, PyObject *keyword_args)
{
    static char *keywords[] = {
	"update", NULL
    };
    int is_update;

    is_update = 0;
    if (!PyArg_ParseTupleAndKeywords(args, keyword_args, "|i", keywords,
				     &is_update))
      return NULL;

    return cursor_new(&self->conn_info, is_update, self->conn_info.debug);
}

/* Implement the connect.bulkcopy() method
 */
static PyObject *Connect_bulkcopy(ConnectObj *self, PyObject *args)
{
    char *table;

    table = NULL;
    if (!PyArg_ParseTuple(args, "s", &table))
	return NULL;

    return bulkcopy_new(&self->conn_info, table, self->conn_info.debug);
}

static struct PyMethodDef Connect_methods[] = {
    { "close", (PyCFunction)Connect_close, METH_VARARGS },
    { "cursor", (PyCFunction)Connect_cursor, METH_VARARGS | METH_KEYWORDS },
    { "bulkcopy", (PyCFunction)Connect_bulkcopy, METH_VARARGS },
    { "execute", (PyCFunction)Connect_execute, METH_VARARGS },
    { NULL, NULL }
};

static void Connect_dealloc(ConnectObj *self)
{
    cmd_free(&self->cmd_info);
    conn_free(&self->conn_info);

    PyMem_DEL(self);
}

static PyObject *Connect_getattr(ConnectObj *self, char *name)
{
    return Py_FindMethod(Connect_methods, (PyObject*)self, name);
}

/* ---------------------------------------------------------------- */

static PyTypeObject ConnectType = { /* main python type-descriptor */
    /* type header */
    PyObject_HEAD_INIT(&PyType_Type)
    0,				/* ob_size */
    "Connect",			/* tp_name */
    sizeof(ConnectObj),		/* tp_basicsize */
    0,				/* tp_itemsize */

    /* standard methods */
    (destructor)Connect_dealloc,/* tp_dealloc */
    (printfunc)0,
    (getattrfunc)Connect_getattr, /* tp_getattr */
    (setattrfunc)0,
    (cmpfunc)0,
    (reprfunc)0
};

/* ---------------------------------------------------------------- */

/* Implement the Sybase.connect() method as per DB API spec 2.0.
 */
PyObject *ConnectType_new(PyObject *module,
			  PyObject *args, PyObject *keyword_args)
{
    ConnectObj *self;
    char *dsn, *user, *passwd;
    int debug, bulkcopy;
    static char *keywords[] = {
	"dsn", "user", "passwd", "bulkcopy", "debug", NULL
    };

    self = PyObject_NEW(ConnectObj, &ConnectType);
    if (self == NULL)
	return NULL;

    memset(&self->cmd_info, 0, sizeof(self->cmd_info));
    dsn = user = passwd = NULL;
    debug = bulkcopy = 0;

    if (PyArg_ParseTupleAndKeywords(args, keyword_args, "sss|ii", keywords,
				    &dsn, &user, &passwd, &bulkcopy, &debug)
	&& conn_init(&self->conn_info, dsn, user, passwd, bulkcopy, debug)) {
	conn_clear_messages(&self->conn_info);
	cmd_init(&self->cmd_info, &self->conn_info, debug, 0);
	return (PyObject*)self;
    }

    Py_DECREF(self);
    return NULL;
}
