/**********************************************************************
Copyright 1999 by ITG Australia.

                        All Rights Reserved

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose and without fee is hereby granted,
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in
supporting documentation, and that the names of ITG Australia or ITGA
not be used in advertising or publicity pertaining to distribution of
the software without specific, written prior permission.

ITG AUSTRALIA DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
EVENT SHALL ITG AUSTRALIA BE LIABLE FOR ANY SPECIAL, INDIRECT OR
CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
PERFORMANCE OF THIS SOFTWARE.
**********************************************************************/
#include "Sybase.h"

/* The following #defines and CS_DATETIME_to_time() conversion
 * function were shamelessly lifted directly out of the ctsybase
 * module (the other Sybase interface for Python).
 */
#define EPOCH_DAYS_SINCE_1900 25567
#define SECONDS_PER_DAY 86400
#define SYBASE_TICKS_PER_SECOND 300

/* Code to interface with Sybase.
 */
static time_t CS_DATETIME_to_time(CS_DATETIME *date_time)
{
    return ((date_time->dtdays - EPOCH_DAYS_SINCE_1900) * SECONDS_PER_DAY) + 
        (date_time->dttime / SYBASE_TICKS_PER_SECOND);
}

/* Set an item in a tuple.
 * On success, return non-zero.
 * On failure, decref the tuple and object, then return 0.
 */
static int set_tuple_idx(PyObject *tuple, int idx, PyObject *obj)
{
    if (obj != NULL && PyTuple_SetItem(tuple, idx, obj) == 0) {
	return 1;
    }
    Py_DECREF(tuple);
    if (obj != NULL) {
	Py_DECREF(obj);
    }
    return 0;
}

/* Free all memory buffers allocated to retrieve Sybase column data
 */
void free_column_buffers(CmdInfo *cmd_info)
{
    int idx;			/* iterate over columns */

    if (cmd_info->col_info == NULL)
	return;

    for (idx = 0; idx < cmd_info->num_cols; idx++) {
	ColInfo *col = cmd_info->col_info + idx;
	if (col->buff != NULL)
	    Py_Free(col->buff);
    }
    Py_Free(cmd_info->col_info);
    cmd_info->col_info = NULL;
}

/* Verify that a Sybase command structure exists.  Return non-zero if
 * OK.  Raise a Python exception and return 0 if it does not exist.
 */
int cmd_check(CmdInfo *cmd_info)
{
    if (cmd_info != NULL && cmd_info->cmd != NULL)
	return 1;
    raise_exception_string(InternalError, "command has not been allocated");
    return 0;
}

/* Initialise a CmdInfo structure and allocate a Sybase command.
 * Return non-zero on success.  Raise a Python exception and return 0 on
 * failure.
 */
int cmd_init(CmdInfo *cmd_info, ConnInfo *conn_info, int debug, int is_cursor)
{
    memset(cmd_info, 0, sizeof(*cmd_info));

    cmd_info->conn_info = conn_info;
    cmd_info->debug = debug;
    cmd_info->is_cursor = is_cursor;

    if (ct_cmd_alloc(conn_info->conn, &cmd_info->cmd) != CS_SUCCEED) {
	raise_exception(conn_info, "c_cmd_alloc() failure");
	return 0;
    }
    debug_msg(cmd_info->debug, "    ct_cmd_alloc() success\n");
    return 1;
}

/* Abort a command without raising exceptions.
 *
 * FIXME: Currently there is no code which recovers from the "Double
 * Ouch" (see code).  I have no idea what to do for the "Triple Ouch".
 */
void cmd_abort_quietly(CmdInfo *cmd_info)
{
    CS_RETCODE ret_code;

    if (cmd_info->is_eed) {
	/* Do not abort commands being used to gather extended error
	 * data.
	 */
	return;
    }

    debug_msg(cmd_info->debug, "  cmd_abort_quietly\n");

    /* To abort a command, an application must call ct_cancel() with
     * type as CS_CANCEL_ALL before using the affected command
     * structure to send another command.
     *
     * If ct_cancel() returns CS_FAIL, the application must call
     * ct_close(CS_FORCE_CLOSE) to force the connection closed.
     */
    ret_code = ct_cancel(NULL, cmd_info->cmd, CS_CANCEL_ALL);
    if (ret_code == CS_SUCCEED) {
	cmd_info->state = STATE_INIT;
	debug_msg(cmd_info->debug, "    ct_cancel(all): success\n");
	return;
    }
    debug_msg(cmd_info->debug, "    ct_cancel(all): failed\n");

    /* Double Ouch!  Now we have to close and drop the connection
     */
    if (ct_close(cmd_info->conn_info->conn, CS_FORCE_CLOSE) == CS_SUCCEED) {
	cmd_info->state = STATE_INIT;
	debug_msg(cmd_info->debug, "    ct_close(): success\n");
	ct_con_drop(cmd_info->conn_info->conn);
	debug_msg(cmd_info->debug, "    ct_con_drop()\n");
	cmd_info->conn_info->conn = NULL;
	cmd_info->conn_info->is_conn_ok = 0;
	return;
    }
    /* Triple Ouch!  No idea what happens now
     */
    debug_msg(cmd_info->debug, "    ct_close(): failed!!!\n");
}

/* Free a Sybase command structure in an orderly fashion.
 */
void cmd_free(CmdInfo *cmd_info)
{
    free_column_buffers(cmd_info);

    if (cmd_info->cmd != NULL) {
	if (cmd_info->is_cursor_open) {
	    debug_msg(cmd_info->debug, "    ct_cursor(close)\n");
	    ct_cursor(cmd_info->cmd, CS_CURSOR_CLOSE, NULL, CS_UNUSED, NULL,
		      CS_UNUSED, CS_DEALLOC);
	    debug_msg(cmd_info->debug, "    ct_send()\n");
	    if (ct_send(cmd_info->cmd) == CS_SUCCEED) {
		int finished = 0;
		while (!finished) {
		    CS_INT result;

		    switch (wrap_ct_results(cmd_info, &result)) {
		    case CS_END_RESULTS:
			finished = 1;
			break;
		    case CS_SUCCEED:
			if (result == CS_CMD_SUCCEED
			    || result == CS_CMD_DONE)
			    break;
			/* fall through and abort
			 */
 		    default:
			cmd_abort_quietly(cmd_info);
			cmd_info->cmd = NULL;
			cmd_info->state = STATE_CLOSED;
			finished = 1;
			return;
		    }
		}
		cmd_info->is_cursor_open = 0;
	    }
	}
	debug_msg(cmd_info->debug, "    ct_cmd_drop()\n");
	ct_cmd_drop(cmd_info->cmd);
	cmd_info->cmd = NULL;
    }

    cmd_info->state = STATE_CLOSED;
}

/* Obtain the format of the data rows returned by the current command
 * and bind the columns to buffers.  In an effort to cut down the amount
 * of allocation and free operations during cursor access, do not free /
 * alloc buffers that are the right size.
 *
 * If @desc is non-null, describe the format of the data as per DB API
 * 2.0 specification and pass that description back via @desc.
 *
 * Return non-zero on success.  Raise a Python exception and return 0
 * on failure.
 */
int cmd_row_bind(CmdInfo *cmd_info, PyObject **desc)
{
    int idx;			/* iterate over columns */
    int num_cols;		/* number of columns in results */
    PyObject *new_desc = NULL;	/* build new column description here */

    debug_msg(cmd_info->debug, "  cmd_row_bind\n");

    /* Find out how many columns are in the result set
     */
    if (ct_res_info(cmd_info->cmd, CS_NUMDATA, &num_cols,
		    CS_UNUSED, NULL) != CS_SUCCEED) {
	raise_exception_string(InterfaceError, "ct_res_info(CS_NUMDATA) failed");
	goto error;
    }
    debug_msg(cmd_info->debug, "    ct_res_info(): num_cols = %d\n", cmd_info->num_cols);

    if (num_cols != cmd_info->num_cols) {
	free_column_buffers(cmd_info);
	cmd_info->col_info = Py_Malloc(num_cols * sizeof(*cmd_info->col_info));
	if (cmd_info->col_info == NULL)
	    goto error;
	memset(cmd_info->col_info, 0, num_cols * sizeof(*cmd_info->col_info));
	cmd_info->num_cols = num_cols;

	if (desc != NULL) {
	    new_desc = PyTuple_New(num_cols);
	    if (new_desc == NULL)
		goto error;
	}
    }

    /* Now find out the format of each column
     */
    for (idx = 0; idx < cmd_info->num_cols; idx++) {
	ColInfo *col = cmd_info->col_info + idx;
	PyObject *col_desc = NULL; /* build description of column here */
	PyObject *type_obj = Py_None; /* type of this column */

	/* Get column description and allocate a buffer for the column
	 * data
	 */
	if (ct_describe(cmd_info->cmd, idx + 1, &col->fmt) != CS_SUCCEED) {
	    raise_exception_string(InterfaceError, "ct_describe() failed");
	    goto error;
	}
	/* Check if we need to (re)allocate a column buffer
	 */
	if (col->fmt.maxlength != col->buff_len && col->buff != NULL) {
	    Py_Free(col->buff);
	    col->buff = NULL;
	}

	/* Reduce the number of types that I have to handle
	 */
	switch (col->fmt.datatype) {
	case CS_LONGCHAR_TYPE:
	case CS_VARCHAR_TYPE:
	case CS_TEXT_TYPE:
	    col->fmt.datatype = CS_CHAR_TYPE;
	case CS_CHAR_TYPE:
	    if (col->fmt.maxlength > 64 * 1024)
		col->fmt.maxlength = 64 * 1024;
	    debug_msg(cmd_info->debug, "    CS_CHAR_TYPE\n");
	    type_obj = (PyObject*)&STRINGType;
	    break;

	case CS_IMAGE_TYPE:
	case CS_LONGBINARY_TYPE:
	case CS_VARBINARY_TYPE:
	    col->fmt.datatype = CS_BINARY_TYPE;
	case CS_BINARY_TYPE:
	    if (col->fmt.maxlength > 64 * 1024)
		col->fmt.maxlength = 64 * 1024;
	    debug_msg(cmd_info->debug, "    CS_BINARY_TYPE\n");
	    type_obj = (PyObject*)&BINARYType;
	    break;

	case CS_BIT_TYPE:
	case CS_TINYINT_TYPE:
	case CS_SMALLINT_TYPE:
	    col->fmt.datatype = CS_INT_TYPE;
	    col->fmt.maxlength = 4;
	case CS_INT_TYPE:
	    debug_msg(cmd_info->debug, "    CS_INT_TYPE\n");
	    type_obj = (PyObject*)&NUMBERType;
	    break;

	case CS_MONEY_TYPE:
	case CS_MONEY4_TYPE:
	case CS_NUMERIC_TYPE:
	case CS_DECIMAL_TYPE:
	case CS_REAL_TYPE:
	    col->fmt.datatype = CS_FLOAT_TYPE;
	    col->fmt.maxlength = 8;
	case CS_FLOAT_TYPE:
	    debug_msg(cmd_info->debug, "    CS_FLOAT_TYPE\n");
	    type_obj = (PyObject*)&NUMBERType;
	    break;

	case CS_DATETIME4_TYPE:
	    col->fmt.datatype = CS_DATETIME_TYPE;
	    col->fmt.maxlength = 8;
	case CS_DATETIME_TYPE:
	    debug_msg(cmd_info->debug, "    CS_DATETIME_TYPE\n");
	    type_obj = (PyObject*)&DATETIMEType;
	    break;

	case CS_BOUNDARY_TYPE:
	    raise_exception_string(InterfaceError, "CS_BOUNDARY_TYPE data not handled");
	    goto error;
	case CS_SENSITIVITY_TYPE:
	    raise_exception_string(InterfaceError, "CS_SENSITIVITY_TYPE data not handled");
	    goto error;
	}

	/* If required, build a tuple to describe the format of this
	 * column.  The format of this tuple is defined in the DB API
	 * 2.0 spec.
	 */
	if (new_desc != NULL) {
	    col_desc = Py_BuildValue("(s#,O,i,i,i,i,i)",
				     col->fmt.name, col->fmt.namelen,
				     type_obj,
				     0, col->fmt.maxlength,
				     col->fmt.precision, col->fmt.scale,
				     (col->fmt.status & CS_CANBENULL) != 0);
	    if (col_desc == NULL)
		goto error;
	    if (!set_tuple_idx(new_desc, idx, col_desc)) {
		new_desc = NULL;
		goto error;
	    }
	}

	/* Allocate a buffer for the column if necessary
	 */
	if (col->buff == NULL) {
	    col->buff = Py_Malloc(col->fmt.maxlength);
	    if (col->buff == NULL)
		goto error;
	    col->buff_len = col->fmt.maxlength;
	}

	/* Bind the buffer to the column
	 */
	debug_msg(cmd_info->debug, "    ct_bind()\n");
	if (ct_bind(cmd_info->cmd, idx + 1,
		    &col->fmt, col->buff,
		    &col->copied, &col->indicator) != CS_SUCCEED) {
	    raise_exception_string(InterfaceError, "ct_bind() failed");
	    goto error;
	}
    }
    /* Everything was successful, now return the format description if
     * necessary.
     */
    if (new_desc != NULL) {
	Py_DECREF(*desc);
	*desc = new_desc;
    }
    return 1;

error:
    if (new_desc != NULL) {
	Py_DECREF(new_desc);
    }
    cmd_abort_quietly(cmd_info);
    return 0;
}

/* Fetch a single row from the current command and return it as a
 * tuple.  If there are no more rows to retrieve, return NULL.
 *
 * On failure, set @fetch_failed non-zero, and return NULL.
 */
PyObject *cmd_fetch_row(CmdInfo *cmd_info, int *fetch_failed)
{
    CS_INT rows_read;		/* not used */
    CS_RETCODE ret_code;	/* Sybase return code */
    int idx;			/* iterate over columns */
    PyObject *row_tuple = NULL;	/* build new row tuple here */

    debug_msg(cmd_info->debug, "  cmd_fetch_row\n");

    *fetch_failed = 0;
    ret_code = ct_fetch(cmd_info->cmd,
			CS_UNUSED, CS_UNUSED, CS_UNUSED, &rows_read);
    switch (ret_code) {
    case CS_SUCCEED:
	/* Fetch was successful.  Number of rows have been
	 * fetched.
	 */
	debug_msg(cmd_info->debug, "    ct_fetch(): CS_SUCCEED\n");
	break;
    case CS_END_DATA:
	/* All rows have been fetched.
	 */
	debug_msg(cmd_info->debug, "    ct_fetch(): CS_END_DATA\n");
	return NULL;
    case CS_ROW_FAIL:
	/* Recoverable error while fetching a row.  Recoverable errors
	 * include memory allocation failures and conversion errors
	 * the occur while copying row values to program variables.
	 *
	 * Application can continue calling ct_fetch() to keep
	 * retrieving rows, or can call ct_cancel() to cancel the
	 * remaining results.
	 */
	debug_msg(cmd_info->debug, "    ct_fetch(): CS_ROW_FAIL\n");
	if (!cmd_info->is_eed) {
	    raise_exception(cmd_info->conn_info, NULL);
	    cmd_abort_quietly(cmd_info);
	}
	*fetch_failed = 1;
	return NULL;
    case CS_FAIL:
	/* Fetch failed.
	 */
	debug_msg(cmd_info->debug, "    ct_fetch(): CS_FAIL\n");
	if (!cmd_info->is_eed) {
	    raise_exception(cmd_info->conn_info, NULL);
	    cmd_abort_quietly(cmd_info);
	}
	*fetch_failed = 1;
	return NULL;
    case CS_CANCELED:
	debug_msg(cmd_info->debug, "    ct_fetch(): CS_CANCELED\n");
	if (!cmd_info->is_eed) {
	    raise_exception(cmd_info->conn_info, "ct_fetch cancelled!?");
	    cmd_abort_quietly(cmd_info);
	}
	*fetch_failed = 1;
	return NULL;
    case CS_PENDING:
	debug_msg(cmd_info->debug, "    ct_fetch(): CS_PENDING\n");
	raise_exception(cmd_info->conn_info, "ct_fetch returned CS_PENDING");
	cmd_abort_quietly(cmd_info);
	*fetch_failed = 1;
	return NULL;
    case CS_BUSY:
	debug_msg(cmd_info->debug, "    ct_fetch(): CS_BUSY\n");
	raise_exception(cmd_info->conn_info, "ct_fetch returned CS_BUSY");
	cmd_abort_quietly(cmd_info);
	*fetch_failed = 1;
	return NULL;
    }

    /* Only get down here when a row of data has been successfully
     * retrieved.
     *
     * Build a tuple for the row of data and then copy the columns
     * into the tuple
     */
    row_tuple = PyTuple_New(cmd_info->num_cols);
    if (row_tuple == NULL) {
	cmd_abort_quietly(cmd_info);
	*fetch_failed = 1;
	return NULL;
    }

    for (idx = 0; idx < cmd_info->num_cols; idx++) {
	ColInfo *col = cmd_info->col_info + idx;
	PyObject *col_data = NULL; /* actual column data */

	if (col->indicator == CS_NULLDATA) {
	    Py_INCREF(Py_None);
	    if (set_tuple_idx(row_tuple, idx, Py_None))
		continue;
	    /* Bogus.  Could not set tuple item.
	     */
	    cmd_abort_quietly(cmd_info);
	    *fetch_failed = 1;
	    return NULL;
	}
	switch (col->fmt.datatype) {
	case CS_CHAR_TYPE:
	case CS_BINARY_TYPE:
	    col_data = PyString_FromStringAndSize(col->buff, col->copied);
	    break;
	case CS_INT_TYPE:
	    col_data = PyInt_FromLong(*(int*)col->buff);
	    break;
	case CS_FLOAT_TYPE:
	    col_data = PyFloat_FromDouble(*(double*)col->buff);
	    break;
	case CS_DATETIME_TYPE:
	    col_data = PyInt_FromLong(CS_DATETIME_to_time(col->buff));
	    break;
	}
	if (col_data == NULL) {
	    cmd_abort_quietly(cmd_info);
	    Py_DECREF(row_tuple);
	    *fetch_failed = 1;
	    return NULL;
	}

	if (!set_tuple_idx(row_tuple, idx, col_data)) {
	    cmd_abort_quietly(cmd_info);
	    *fetch_failed = 1;
	    return NULL;
	}
    }

    return row_tuple;
}

/* Send parameters for a Sybase command
 */
int cmd_send_params(CmdInfo *cmd_info, PyObject *param_seq)
{
    int num_params;
    int idx;

    num_params = PySequence_Length(param_seq);
    for (idx = 0; idx < num_params; idx++) {
	PyObject *param;
	CS_DATAFMT fmt;

	param = PySequence_GetItem(param_seq, idx);
	Py_DECREF(param);
	memset(&fmt, 0, sizeof(fmt));

	if (PyInt_Check(param)) {
	    CS_INT value;

	    value = PyInt_AsLong(param);
	    fmt.status = CS_INPUTVALUE;
	    fmt.datatype = CS_INT_TYPE;
	    if (ct_param(cmd_info->cmd, &fmt, &value, sizeof(value), 0) != CS_SUCCEED) {
		raise_exception(cmd_info->conn_info, "ct_param failed");
		return 0;
	    }
	} else if (PyString_Check(param)) {
	    char *value;
	    unsigned int len;

	    value = PyString_AsString(param);
	    fmt.status = CS_INPUTVALUE;
	    fmt.datatype = CS_CHAR_TYPE;
	    len = PyString_Size(param);
	    if (ct_param(cmd_info->cmd, &fmt, value, len, 0) != CS_SUCCEED) {
		raise_exception(cmd_info->conn_info, "ct_param failed");
		return 0;
	    }
	} else {
	    raise_exception_string(ProgrammingError, "only integer and string parameters supported");
	    return 0;
	}
    }

    return 1;
}

/* Fetch one logical result of rows returned by the current command.
 * On success, returns a Python list of the rows in the logical
 * result.  On failure, raises a Python exception and returns NULL.
 */
PyObject *cmd_fetch_logical_result(CmdInfo *cmd_info)
{
    PyObject *logical_result;

    debug_msg(cmd_info->debug, "  cmd_fetch_logical_result\n");
    logical_result = PyList_New(0);
    if (logical_result == NULL) {
	cmd_abort_quietly(cmd_info);
	return NULL;
    }

    for (;;) {
	int fetch_failed;
	PyObject *row_tuple;

	row_tuple = cmd_fetch_row(cmd_info, &fetch_failed);
	if (row_tuple == NULL) {
	    if (fetch_failed) {
		Py_DECREF(logical_result);
		return NULL;
	    }
	    return logical_result;
	}

	/* Append the row tuple to our results list
	 */
	if (PyList_Append(logical_result, row_tuple) < 0) {
	    cmd_abort_quietly(cmd_info);
	    Py_DECREF(row_tuple);
	    Py_DECREF(logical_result);
	    return NULL;
	}
	Py_DECREF(row_tuple);
    }
}
