/**********************************************************************
Copyright 1999 by ITG Australia.

                        All Rights Reserved

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose and without fee is hereby granted,
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in
supporting documentation, and that the names of ITG Australia or ITGA
not be used in advertising or publicity pertaining to distribution of
the software without specific, written prior permission.

ITG AUSTRALIA DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
EVENT SHALL ITG AUSTRALIA BE LIABLE FOR ANY SPECIAL, INDIRECT OR
CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
PERFORMANCE OF THIS SOFTWARE.
**********************************************************************/
#ifndef __Sybase_h
#define __Sybase_h

#include <time.h>
#include <ctpublic.h>
#include <bkpublic.h>
#include <Python.h>
#include <structmember.h>

/* Cursor command execution is implemented via a state machine.
 */
typedef enum {
    STATE_INIT,			/* initialisation state */
    STATE_FETCH,		/* fetch rows */
    STATE_FINISHED,		/* command complete */
    STATE_CLOSED		/* cursor has been closed */
} CmdState;

/* A Sybase connection is needed before any operations can be
 * performed on the database.  A connection structure is contained in
 * both the Connect and Bulkcopy objects, which are created by the
 * Sybase.connect() and Sybase.bulkcopy() methods respectively.
 */
typedef struct {
    CS_CONNECTION *conn;	/* Sybase connection for this session */
    int is_bulkcopy;		/* is the connection for bulkcopy? */
    int debug;			/* print debug messages on this connection? */
    int is_conn_ok;		/* is connection established? */
} ConnInfo;

/* Sybase column description, data buffer, and status.  Used to
 * retrieve column data from Sybase.
 */
typedef struct {
    CS_DATAFMT fmt;		/* format of this data column */
    void *buff;			/* buffer for retrieving the data */
    CS_INT buff_len;		/* length of the allocated buffer */
    CS_INT copied;		/* length of data copied into buffer */
    CS_SMALLINT indicator;	/* status of copied data */
    union {
	CS_INT cs_int;
	CS_FLOAT cs_float;
    } v;
} ColInfo;

/* Sybase command structure.  Sybase commands are owned by a Sybase
 * connection.  Nearly all access to Sybase data requires the use of a
 * command.
 */
typedef struct {
    ConnInfo *conn_info;	/* owner connection */
    CS_COMMAND *cmd;		/* Sybase command; only one at a time */
    CmdState state;		/* current state of the command */
    int debug;			/* print debug messages; inherit from conn_info */
    int is_cursor;		/* is this command being used by a cursor? */
    int is_cursor_open;		/* is the cursor open? */
    int is_eed;			/* for handling Extended Error Data? */
    CS_INT num_cols;		/* number of columns in data currently
				 * being retrieved */
    ColInfo *col_info;		/* format of columns currently being
				 * retrieved */
} CmdInfo;

/* ----------------------------------------------------- */

/* Cursor - Python object
 *
 * Implements the Cursor Objects as specified in the Python DB API 2.0
 */
typedef struct {		/* Sybase connection instance object */
    PyObject_HEAD		/* python header: ref-count + &typeobject */

    CmdInfo cmd_info;		/* context for executing SQL commands */
    PyObject *desc;		/* description of results returned by cursor */
    int is_update;		/* is cursor for update? */
    int row_count;		/* number of rows in fetch */
    int array_size;		/* number of rows to fetch from server */
} CursorObj;

/* Connect - Python object
 *
 * Implements the Connection Objects as specified in the Python DB API
 * 2.0
 */
typedef struct {		/* Sybase connection instance object */
    PyObject_HEAD		/* python header: ref-count + &typeobject */

    ConnInfo conn_info;		/* connection to database */
    CmdInfo cmd_info;		/* context for executing SQL commands */
} ConnectObj;

/* Bulkcopy - Python object
 *
 * Implements Bulkcopy Objects (not in the Python DB API 2.0)
 */
typedef struct {		/* Sybase connection instance object */
    PyObject_HEAD		/* python header: ref-count + &typeobject */

    ConnInfo *conn_info;	/* owner connection */
    CS_BLKDESC *blk;		/* Sybase bulkcopy context */

    char *table;		/* name of table being copied into */
    int num_cols;		/* number of columns in the table */
    ColInfo *col_info;		/* format of columns currently being
				 * copied */
    int debug;			/* print debug messages; inherit from conn_info */
    int copy_in_progress;	/* is bulkcopy in progress? */
} BulkcopyObj;

/* ----------------------------------------------------- */
/* Sybase.c */

/* Exceptions as specified in the Python DB API 2.0
 */
extern PyObject *Warning;
extern PyObject *Error;
extern PyObject *InterfaceError;
extern PyObject *DatabaseError;
extern PyObject *DataError;
extern PyObject *OperationalError;
extern PyObject *IntegrityError;
extern PyObject *InternalError;
extern PyObject *ProgrammingError;
extern PyObject *NotSupportedError;

/* Data types as specified in the Python DB API 2.0
 */
extern PyTypeObject STRINGType;
extern PyTypeObject BINARYType;
extern PyTypeObject NUMBERType;
extern PyTypeObject DATETIMEType;
extern PyTypeObject ROWIDType;

/* Print a debug message if debugging has been turned on for @cmd_info
 */
void debug_msg(int debug_flg, char *fmt, ...);

/* Raise a Python exception of type @obj.  @obj should be one of the
 * exceptions defined above.  @str is transformed into a Python string
 * and is sent with the exception.
 */
void raise_exception_string(PyObject *obj, char *str);

/* Raise a Python exception of type Error (defined above).  All of the
 * Sybase messages on the connection @conn_info are retrieved, built
 * into a Python list, and sent with the exception.  If @msg is not
 * NULL, it is transformed into a Python string and supplied as the
 * first item in the list.
 */
void raise_exception(ConnInfo *conn_info, char *msg);

/* Checks the Sybase result code in @result and logs a debug message
 * describing @result.  If @result is OK, the function returns
 * non-zero.  If @result indicates an error, the function raises a
 * Python exception on @conn_info and aborts the current command (via
 * cmd_abort_quietly()), then returns zero.
 */
int handle_result_type(CmdInfo *cmd_info, CS_INT result);

/* A simple wrapper around the Sybase ct_results() function.  Calls
 * ct_results(), logs debug messages describing the result and then
 * returns the value returned by ct_results().  If the return value
 * indicated an error, the function raises a Python exception on
 * @conn_info and aborts the current command (via cmd_abort_quietly()).
 * @result returns the result returned by ct_results().
 */
CS_RETCODE wrap_ct_results(CmdInfo *cmd_info, CS_INT *result);

/* ----------------------------------------------------- */
/* bulkcopy.c */

/* Implement the Connect.bulkcopy() method (not in DB API 2.0 spec).
 * Creates a new bulkcopy object for loading data into @table.
 */
PyObject *bulkcopy_new(ConnInfo *conn_info, char *table, int debug);

/* ----------------------------------------------------- */
/* cmd.c */

/* Free the buffers that were allocated to retrieve column results
 * of the Sybase command in @cmd_info.
 */
void free_column_buffers(CmdInfo *cmd_info);

/* Check that the Sybase command in @cmd_info is valid.  Return
 * non-zero if valid, raise a Python exception and zero otherwise.
 */
int cmd_check(CmdInfo *cmd_info);

/* Initialise a Sybase command in @cmd_info on the Sybase connection
 * @conn_info.  Return non-zero if successful, raise a Python
 * exception and return zero upon failure.
 */
int cmd_init(CmdInfo *cmd_info, ConnInfo *conn_info, int debug, int is_cursor);

/* Free the Sybase command in @cmd_info.
 */
void cmd_free(CmdInfo *cmd_info);

/* Abort the Sybase command in @cmd_info without raising exceptions on
 * failure.
 */
void cmd_abort_quietly(CmdInfo *cmd_info);

/* Fetch one row from the Sybase command in @cmd_info and return it as
 * a Python tuple.  If there are no more results to be fetched, return
 * NULL.  When NULL is returned, @fetch_failed will contain a non-zero
 * value if an error was encountered (exception will also be raised).
 */
PyObject *cmd_fetch_row(CmdInfo *cmd_info, int *fetch_failed);

/* Send the parameters contained in the Python sequence @param_seq to
 * the Sybase command in @cmd_info.  Return non-zero on success.
 * Raise a Python exception and return zero on failure.
 */
int cmd_send_params(CmdInfo *cmd_info, PyObject *param_seq);

/* Bind the @cmd_info command results to a set of buffers.  if @desc
 * is non-null, return a description (as specified in the DB API 2.0
 * spec) of the command results in @desc.  Return non-zero on success.
 * Raise a Python exception and return zero on failure.
 */
int cmd_row_bind(CmdInfo *cmd_info, PyObject **desc);

/* Fetch one logical result of rows from @cmd_info.  On success,
 * returns a Python list of the rows in the logical result.  On
 * failure, raises Python exception and return NULL.
 */
PyObject *cmd_fetch_logical_result(CmdInfo *cmd_info);

/* ----------------------------------------------------- */
/* conn.c */

/* Initialise a connection in @conn_info to the Sybase server @dsn and
 * log in as @user with password @passwd.  If @debug is non-zero,
 * debug messages will be printed.  Returns non-zero on success, zero
 * on failure.
 */
int conn_init(ConnInfo *conn_info,
	      char *dsn, char *user, char *passwd, int bulkcopy, int debug);

/* Close and free the Sybase connection @conn_info.
 */
void conn_free(ConnInfo *conn_info);

/* Clear all Sybase messages on the @conn_info connection.
 */
void conn_clear_messages(ConnInfo *conn_info);

/* Check that the Sybase connection in @conn_info is valid.  Return
 * non-zero if valid, raise a Python exception and return zero
 * otherwise.
 */
int conn_check(ConnInfo *conn_info);

/* ----------------------------------------------------- */
/* connect.c */

/* Implement the Sybase.connect() method as per DB API 2.0 spec.
 */
PyObject *ConnectType_new(PyObject *module,
			  PyObject *args, PyObject *keyword_args);

/* ----------------------------------------------------- */
/* cursor.c */

/* Create a new Cursor object as per DB API 2.0 spec.
 */
PyObject *cursor_new(ConnInfo *conn_info, int update, int debug);

#endif
