/**********************************************************************
Copyright 1999 by ITG Australia.

                        All Rights Reserved

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose and without fee is hereby granted,
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in
supporting documentation, and that the names of ITG Australia or ITGA
not be used in advertising or publicity pertaining to distribution of
the software without specific, written prior permission.

ITG AUSTRALIA DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
EVENT SHALL ITG AUSTRALIA BE LIABLE FOR ANY SPECIAL, INDIRECT OR
CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
PERFORMANCE OF THIS SOFTWARE.
**********************************************************************/
#include "Sybase.h"

static char *module_name = "Sybase";

/* Exception objects as specified by DB API 2.0
 */
PyObject *Warning;
PyObject *Error;
PyObject *InterfaceError;
/* $Id: Sybase.c,v 1.2 1999/11/01 05:31:35 djc Exp $
 */
PyObject *DatabaseError;
PyObject *DataError;
PyObject *OperationalError;
PyObject *IntegrityError;
PyObject *InternalError;
PyObject *ProgrammingError;
PyObject *NotSupportedError;

/* ----------------------------------------------------- */

/* Declare all of the types required by the DB API 2.0
 */

PyTypeObject STRINGType = {	/* main python type-descriptor */
    /* type header */
    PyObject_HEAD_INIT(&PyType_Type)
    0,				/* ob_size */
    "STRING",			/* tp_name */
    sizeof(PyObject),		/* tp_basicsize */
    0				/* tp_itemsize */
};
PyTypeObject BINARYType = {	/* main python type-descriptor */
    /* type header */
    PyObject_HEAD_INIT(&PyType_Type)
    0,				/* ob_size */
    "BINARY",			/* tp_name */
    sizeof(PyObject),		/* tp_basicsize */
    0				/* tp_itemsize */
};
PyTypeObject NUMBERType = {	/* main python type-descriptor */
    /* type header */
    PyObject_HEAD_INIT(&PyType_Type)
    0,				/* ob_size */
    "NUMBER",			/* tp_name */
    sizeof(PyObject),		/* tp_basicsize */
    0				/* tp_itemsize */
};
PyTypeObject DATETIMEType = {	/* main python type-descriptor */
    /* type header */
    PyObject_HEAD_INIT(&PyType_Type)
    0,				/* ob_size */
    "DATETIME",			/* tp_name */
    sizeof(PyObject),		/* tp_basicsize */
    0				/* tp_itemsize */
};
PyTypeObject ROWIDType = {	/* main python type-descriptor */
    /* type header */
    PyObject_HEAD_INIT(&PyType_Type)
    0,				/* ob_size */
    "DATETIME",			/* tp_name */
    sizeof(PyObject),		/* tp_basicsize */
    0				/* tp_itemsize */
};

/* ----------------------------------------------------- */

/* Output debug messages for command if debugging is enabled
 */
void debug_msg(int debug_flg, char *fmt, ...)
{
    va_list ap;

    if (!debug_flg)
	return;

    va_start(ap, fmt);
    vfprintf(stderr, fmt, ap);
    va_end(ap);
}

/* Insert an object value into a dictionary
 */
static int dict_add_obj(PyObject *dict, char *key, PyObject *obj)
{
    if (PyDict_SetItemString(dict, key, obj))
	return 0;
    Py_DECREF(obj);
    return 1;
}

/* Build a string object and place it into a dictionary.  Return
 * non-zero on success, 0 on failure.
 */
static int dict_add_string(PyObject *dict, char *name, char *value, int len)
{
    PyObject *obj;

    if (len < 0)
	obj = PyString_FromString(value);
    else
	obj = PyString_FromStringAndSize(value, len);
    if (obj == NULL)
	return 0;
    if (PyDict_SetItemString(dict, name, obj)) {
	Py_DECREF(obj);
	return 0;
    }
    Py_DECREF(obj);
    return 1;
}

/* Build an integer object and place it into a dictionary.  Return
 * non-zero on success, 0 on failure.
 */
static int dict_add_int(PyObject *dict, char *name, int value)
{
    PyObject *obj;

    obj = PyInt_FromLong(value);
    if (obj == NULL)
	return 0;
    if (PyDict_SetItemString(dict, name, obj)) {
	Py_DECREF(obj);
	return 0;
    }
    Py_DECREF(obj);
    return 1;
}

/* Build a type object and place it into a dictionary.  Return
 * non-zero on success, 0 on failure.
 */
static int dict_add_type(PyObject *dict, char *name, PyTypeObject *type)
{
    Py_INCREF(type);
    if (PyDict_SetItemString(dict, name, (PyObject*)type))
	return 0;
    return 1;
}

/* Raise a Python exception as a plain string.
 */
void raise_exception_string(PyObject *obj, char *str)
{
    PyObject *str_obj = PyString_FromString(str);
    PyErr_SetObject(obj, str_obj);
}

/* Raise a Python exception as a list containing an optional string,
 * followed by a zero or more dictionaries describing the system
 * messages, followed by zero or more dictionaries describing client
 * messages.
 */
void raise_exception(ConnInfo *conn_info, char *msg)
{
    CS_INT num_msgs;		/* query Sybase for number of messages */
    PyObject *err;		/* exception object */
    PyObject *obj;		/* build exception components */
    int idx;			/* iterate over messages */

    err = PyList_New(0);
    if (msg != NULL) {
	debug_msg(conn_info->debug, "  raise %s\n", msg);
	obj = PyString_FromString(msg);
	PyList_Append(err, obj);
	Py_DECREF(obj);
    }

    /* Get host messages
     */
    if (ct_diag(conn_info->conn, CS_STATUS, CS_SERVERMSG_TYPE,
		CS_UNUSED, &num_msgs) != CS_SUCCEED)
	num_msgs = 0;

    for (idx = 1 ; idx <= num_msgs; idx++) {
	CS_SERVERMSG data;	/* query Sybase server message */

	if (ct_diag(conn_info->conn,
		    CS_GET, CS_SERVERMSG_TYPE, idx, &data) != CS_SUCCEED) {
	    char text[64];
	    sprintf(text, "get serv-msg %d failed", idx);
	    obj = PyString_FromString(text);
	    PyList_Append(err, obj);
	    Py_DECREF(obj);
	    continue;
	}

	obj = PyDict_New();
	dict_add_int(obj, "msgnumber", data.msgnumber);
	dict_add_int(obj, "state", data.state);
	dict_add_int(obj, "severity", data.severity);
	dict_add_string(obj, "text", data.text, data.textlen);
	dict_add_string(obj, "server", data.svrname, data.svrnlen);
	dict_add_string(obj, "proc", data.proc, data.proclen);
	dict_add_int(obj, "line", data.line);
	dict_add_string(obj, "sqlstate", data.sqlstate, data.sqlstatelen);

	debug_msg(conn_info->debug, " [msgnumber: %d\n", data.msgnumber);
	debug_msg(conn_info->debug, "  state:     %d\n", data.state);
	debug_msg(conn_info->debug, "  severity:  %d\n", data.severity);
	debug_msg(conn_info->debug, "  text:      %.*s\n", data.textlen, data.text);
	debug_msg(conn_info->debug, "  server:    %.*s\n", data.svrnlen, data.svrname);
	debug_msg(conn_info->debug, "  proc:      %.*s\n", data.proclen, data.proc);
	debug_msg(conn_info->debug, "  line:      %d\n", data.line);
	debug_msg(conn_info->debug, "  sqlstate:  %.*s]\n", data.sqlstatelen, data.sqlstate);

	/* Some messages contain extended error data which is
	 * retrieved as a logical result.
	 */
	if (data.status & CS_HASEED) {
	    CmdInfo eed;	/* command to obtain extended error data */
	    PyObject *val;	/* EED returned as Sybase logical result */

	    val = NULL;
	    memset(&eed, 0, sizeof(eed));
	    eed.conn_info = conn_info;
	    eed.is_eed = 1;
	    if (ct_diag(conn_info->conn, CS_EED_CMD, CS_SERVERMSG_TYPE,
			idx, &eed.cmd) == CS_SUCCEED) {
		if (cmd_row_bind(&eed, NULL))
		    val = cmd_fetch_logical_result(&eed);
	    }
	    if (val != NULL)
		dict_add_obj(obj, "eed", val);
	    free_column_buffers(&eed);
	}

	PyList_Append(err, obj);
	Py_DECREF(obj);
    }

    /* Get client messages
     */
    if (ct_diag(conn_info->conn, CS_STATUS, CS_CLIENTMSG_TYPE,
		CS_UNUSED, &num_msgs) != CS_SUCCEED)
	num_msgs = 0;

    for (idx = 1 ; idx <= num_msgs; idx++) {
	CS_CLIENTMSG data;

	if (ct_diag(conn_info->conn,
		    CS_GET, CS_CLIENTMSG_TYPE, idx, &data) != CS_SUCCEED) {
	    char text[64];
	    sprintf(text, "get client-msg %d failed", idx);
	    obj = PyString_FromString(text);
	    PyList_Append(err, obj);
	    Py_DECREF(obj);
	    continue;
	}

	obj = PyDict_New();
	dict_add_int(obj, "severity", data.severity);
	dict_add_int(obj, "msgnumber", data.msgnumber);
	dict_add_string(obj, "msgstring", data.msgstring, data.msgstringlen);
	dict_add_int(obj, "osnumber", data.osnumber);
	dict_add_string(obj, "osstring", data.osstring, data.osstringlen);
	dict_add_int(obj, "status", data.status);
	dict_add_string(obj, "sqlstate", data.sqlstate, data.sqlstatelen);

	debug_msg(conn_info->debug, " [severity: %d\n", data.severity);
	debug_msg(conn_info->debug, "  msgnumber: %d\n", data.msgnumber);
	debug_msg(conn_info->debug, "  msgstring: %.*s\n", data.msgstringlen, data.msgstring);
	debug_msg(conn_info->debug, "  osnumber: %d\n", data.osnumber);
	debug_msg(conn_info->debug, "  osstring: %.*s\n", data.osstringlen, data.osstring);
	debug_msg(conn_info->debug, "  status: %d\n", data.status);
	debug_msg(conn_info->debug, "  sqlstate:  %.*s]\n", data.sqlstatelen, data.sqlstate);

	/* Some messages contain extended error data which is
	 * retrieved as a logical result.
	 */
	if (data.status & CS_HASEED) {
	    CmdInfo eed;
	    PyObject *val;

	    val = NULL;
	    memset(&eed, 0, sizeof(eed));
	    eed.conn_info = conn_info;
	    eed.is_eed = 1;
	    if (ct_diag(conn_info->conn, CS_EED_CMD, CS_CLIENTMSG_TYPE,
			idx, &eed.cmd) == CS_SUCCEED) {
		if (cmd_row_bind(&eed, NULL))
		    val = cmd_fetch_logical_result(&eed);
	    }
	    if (val != NULL)
		dict_add_obj(obj, "eed", val);
	    free_column_buffers(&eed);
	}

	PyList_Append(err, obj);
	Py_DECREF(obj);
    }

    /* Clear messages
     */
    conn_clear_messages(conn_info);

    PyErr_SetObject(Error, err);
}

/* Call ct_results() and print a debug message.  For all error
 * results, raise a Python exception.  Returns the ct_results() return
 * value.
 */
CS_RETCODE wrap_ct_results(CmdInfo *cmd_info, CS_INT *result)
{
    CS_RETCODE ret_code;

    debug_msg(cmd_info->debug, "  wrap_ct_results\n");
    ret_code = ct_results(cmd_info->cmd, result);
    switch (ret_code) {
    case CS_SUCCEED:
	/* Now we need to query the result detail
	 */
	debug_msg(cmd_info->debug, "    ct_results(): CS_SUCCEED\n");
	break;
    case CS_END_RESULTS:
	/* Finished processing results
	 */
	debug_msg(cmd_info->debug, "    ct_results(): CS_END_RESULTS\n");
	break;
    case CS_FAIL:
	/* Try to clean up.
	 */
	debug_msg(cmd_info->debug, "    ct_results(): CS_FAIL\n");
	raise_exception(cmd_info->conn_info, NULL);
	cmd_abort_quietly(cmd_info);
	break;
    case CS_CANCELED:
    case CS_PENDING:
    case CS_BUSY:
	debug_msg(cmd_info->debug, "    ct_results(): CS_CANCELED or CS_PENDING or CS_BUSY\n");
	raise_exception_string(InterfaceError, "ct_results unexpected return value");
	cmd_abort_quietly(cmd_info);
	break;
    }
    return ret_code;
}

/* Output a debug message for the passed result code.  For all result
 * codes that indicate an error, raise a Python exception.
 */
int handle_result_type(CmdInfo *cmd_info, CS_INT result)
{
    switch (result) {
    case CS_CMD_DONE:
	/* STATUS: Results of a logical command have been
	 * completely processed.
	 */
	debug_msg(cmd_info->debug, "    ct_results(): result = CS_CMD_DONE\n");
	return 1;
    case CS_CMD_SUCCEED:
	/* STATUS: Success of a command that returns no data, such
	 * as a language command containing a Transact-SQL insert
	 * statement.
	 */
	debug_msg(cmd_info->debug, "    ct_results(): result = CS_CMD_SUCCEED\n");
	return 1;

    case CS_COMPUTE_RESULT:
	/* FETCH: Compute row results; single row of compute
	 * results.
	 */
	debug_msg(cmd_info->debug, "    ct_results(): result = CS_COMPUTE_RESULT\n");
	return 1;
    case CS_CURSOR_RESULT:
	/* FETCH: Cursor row results; zero or more rows of tabular
	 * data.
	 */
	debug_msg(cmd_info->debug, "    ct_results(): result = CS_CURSOR_RESULT\n");
	return 1;
    case CS_PARAM_RESULT:
	/* FETCH: Return parameter results; single row of return
	 * parameters.
	 */
	debug_msg(cmd_info->debug, "    ct_results(): result = CS_PARAM_RESULT\n");
	return 1;
    case CS_ROW_RESULT:
	/* FETCH: Regular row results; zero or more rows of
	 * tabular data.
	 */
	debug_msg(cmd_info->debug, "    ct_results(): result = CS_ROW_RESULT\n");
	return 1;
    case CS_STATUS_RESULT:
	/* FETCH: Stored procedure returned result status; single
	 * row containing a single status.
	 */
	debug_msg(cmd_info->debug, "    ct_results(): result = CS_STATUS_RESULT\n");
	return 1;

    case CS_COMPUTEFMT_RESULT:
	/* INFO: Compute format information.  Application can call
	 * ct_describe() and ct_res_info(), and ct_compute_info() to
	 * retrieve compute format information.
	 *
	 * Shouldn't get this unless we set CS_EXPOSE_FMTS
	 */
	debug_msg(cmd_info->debug, "    ct_results(): result = CS_COMPUTEFMT_RESULT\n");
	raise_exception_string(InterfaceError, "CS_COMPUTEFMT_RESULT not expected");
	cmd_abort_quietly(cmd_info);
	return 0;
    case CS_ROWFMT_RESULT:
	/* INFO: Row format information. Application can call
	 * ct_describe() and ct_res_info() to retrieve row format
	 * information.
	 *
	 * Shouldn't get this unless we set CS_EXPOSE_FMTS
	 */
	debug_msg(cmd_info->debug, "    ct_results(): result = CS_ROWFMT_RESULT\n");
	raise_exception_string(InterfaceError, "CS_ROWFMT_RESULT not expected");
	cmd_abort_quietly(cmd_info);
	return 0;
    case CS_CMD_FAIL:
	/* STATUS: Server encountered an error while executing a
	 * command.
	 */
	debug_msg(cmd_info->debug, "    ct_results(): result = CS_CMD_FAIL\n");
	raise_exception(cmd_info->conn_info, NULL);
	cmd_abort_quietly(cmd_info);
	return 0;
    case CS_MSG_RESULT:
	/* INFO: Application can call ct_res_info() to get the
	 * message's id.  Parameters associated with the message,
	 * if any, are returned as a separate parameter result
	 * set.
	 */
	debug_msg(cmd_info->debug, "    ct_results(): result = CS_MSG_RESULT\n");
	raise_exception_string(InterfaceError, "messages not supported");
	return 0;
    case CS_DESCRIBE_RESULT:
	/* An application can call ct_describe() or ct_dyndesc()
	 * to retrieve the information.
	 */
	debug_msg(cmd_info->debug, "    ct_results(): result = CS_DESCRIBE_RESULT\n");
	raise_exception_string(InterfaceError, "Dynamic SQL not supported");
	return 0;
    }
    raise_exception_string(InterfaceError, "Unexpected result type");
    return 0;
}

/* ---------------------------------------------------------------- */

static struct PyMethodDef Sybase_methods[] = {
    { "connect", (PyCFunction)ConnectType_new, METH_VARARGS | METH_KEYWORDS },
    { NULL, NULL }
};

/* ---------------------------------------------------------------- */

/* Build an exception object and place it into a dictionary.  Return
 * non-zero on success, 0 on failure.
 */
static int build_exception(PyObject **obj,
			   PyObject *dict, char *name, PyObject *base)
{
    char full_name[128];

    sprintf(full_name, "%s.%s", module_name, name);
    *obj = PyErr_NewException(full_name, base, NULL);
    if (*obj == NULL)
	return 0;
    if (PyDict_SetItemString(dict, name, *obj))
	return 0;
    return 1;
}

/* ---------------------------------------------------------------- */

/* Sybase module initialisation
 *
 * This is compliant with the DB API spec 2.0.
 */
void initSybase()
{
    PyObject *module;
    PyObject *dict;

    module = Py_InitModule(module_name, Sybase_methods);
    dict = PyModule_GetDict(module);
    if (!build_exception(&Warning, dict, "Warning", NULL)
	|| !build_exception(&Error, dict, "Error", NULL)
	|| !build_exception(&InterfaceError, dict, "InterfaceError", Error)
	|| !build_exception(&DatabaseError, dict, "DatabaseError", Error)
	|| !build_exception(&DataError, dict, "DataError", DatabaseError)
	|| !build_exception(&OperationalError, dict, "OperationalError", DatabaseError)
	|| !build_exception(&IntegrityError, dict, "IntegrityError", DatabaseError)
    	|| !build_exception(&InternalError, dict, "InternalError", DatabaseError)
    	|| !build_exception(&ProgrammingError, dict, "ProgrammingError", DatabaseError)
    	|| !build_exception(&NotSupportedError, dict, "NotSupportedError", DatabaseError)
	|| !dict_add_string(dict, "apilevel", "2.0", -1)
	|| !dict_add_int(dict, "threadsafety", 0)
	|| !dict_add_string(dict, "paramstyle", "format", -1)
	|| !dict_add_type(dict, "STRING", &STRINGType)
	|| !dict_add_type(dict, "BINARY", &BINARYType)
	|| !dict_add_type(dict, "NUMBER", &NUMBERType)
	|| !dict_add_type(dict, "DATETIME", &DATETIMEType)
	|| !dict_add_type(dict, "ROWID", &ROWIDType)) {
	char reason[64];

	sprintf(reason, "%s: import failed", module_name);
	PyErr_SetString(PyExc_ImportError, reason);
    }
}
