# -------------------------------------------------------------------------
# MODULE:      WidgetButton
#
# DESCRIPTION: 
#     Buttons implemented with widgets.
#
# AUTHOR:
#     Per Spilling, CWI, Amsterdam, <per@cwi.nl>

import Xm, X, Xmd

import vp

from WidgetObject import WidgetObject
from Control      import Control
from Datum        import Datum, ExternalDatum, XResourceShell
from MiscGraphic  import Label

debug = vp.FALSE

# -------------------------------------------------------------------------
# CLASS:         PushButton
#
# INHERITS FROM: (Control, Label) : WidgetObject : Graphic :\
#                (TreeNode, DatumDict) : Object 
#
# COLLABORATORS: Command
#
# DESCRIPTION: 
#     A concrete class which has specific support for collaboration with the
#     Command classes, but which can also be used on its own.
# 
#     Args can be the following: 
#		 - 'name'          : <string>
#		 - 'callback'      : <callback function/method instance>
#		 - 'command'       : <command instance>
#

class PushButton( Control, Label ):

	widget_class = Xm.PushButton

	# ------------------------------------------------------------------
	# Init methods

	def InitWidget( self ):
		#
		# The Graphic.Subscribe() method uses the event handler mechanism of
		# X (via the EventNotifier module). This mechanism doesn't work with
		# popup- and pulldown menus. Instead of the event handler mechanism 
		# we must therefore use the callback mechanism. The _ActivateCB() 
		# function is used to interface with the widget.
		#
		Control.InitWidget( self )
		self.w.AddCallback( 'activateCallback', self.ActivatedCB, None )


	# ------------------------------------------------------------------
	# Event notification methods:

	def ActivatedCB( self, w, client_data, call_data ):
		if debug: print 'PushButton.ActivatedCB called'
		self.ExecuteCallback('activate', self)


# -------------------------------------------------------------------------
# CLASS:         ArrowButton
#
# INHERITS FROM: (Control, WidgetObject) : Graphic :\
#                (TreeNode, DatumDict) : Object 
#
# COLLABORATORS: Command
#
# DESCRIPTION: 
#     A concrete class which has specific support for collaboration with the
#     Command classes, but which can also be used on its own.
# 
#     Args can be the following: 
#		 - 'name'          : <string>
#		 - 'callback'      : <callback function/method instance>
#		 - 'command'       : <command instance>
#
#     The direction of the arrow can be set with the X resource 
#     'arrowDirection'. The values can be: 
#        - X.ARROW_UP (default) 
#        - X.ARROW_DOWN
#        - X.ARROW_LEFT
#        - X.ARROW_RIGHT
#

class ArrowButton( Control, WidgetObject ):

	widget_class = Xm.ArrowButton

	# ------------------------------------------------------------------
	# Init methods

	def InitWidget( self ):
		#
		# The Graphic.Subscribe() method uses the event handler mechanism of
		# X (via the EventNotifier module). This mechanism doesn't work with
		# popup- and pulldown menus. Instead of the event handler mechanism 
		# we must therefore use the callback mechanism. The _ActivateCB() 
		# function is used to interface with the widget.
		#
		Control.InitWidget( self )
		self.w.AddCallback( 'activateCallback', self.ActivatedCB, None )


	# ------------------------------------------------------------------
	# Event notification methods:

	def ActivatedCB( self, w, client_data, call_data ):
		if debug: print 'ArrowButton.ActivatedCB called'
		self.ExecuteCallback( 'activate', self )



# -------------------------------------------------------------------------
# CLASS:         ToggleButton
#
# INHERITS FROM: (Control, Label : WidgetObject : Graphic :\
#                (TreeNode, DatumDict) : Object 
# 
# COLLABORATORS: Command, RadioGroup
#
# DESCRIPTION: 
#     A concrete class which has specific support for collaboration with the
#     Command classes and the RadioGroup, but which can also be used on its 
#     own.
#
#     Args can be any of the following: 
#         - 'name'          : <string>
#         - 'callback'      : <callback function/method instance>
#         - 'command'       : <Command instance>
#         - 'radio_group'   : <RadioGroup instance>
#         - 'use_indicator' : (vp.TRUE=default/vp.FALSE)
#

class ToggleButton( Control, Label ):

	widget_class  = Xm.ToggleButton

	# ------------------------------------------------------------------
	# Init methods

	def __init__( self, argdict = {} ):
		self.use_indicator = vp.TRUE
		self.radio_group   = None
		self.state         = vp.OFF
		Control.__init__( self, argdict )

		if self.radio_group != None: self.radio_group.Register( self )


	def CreateWidget( self ):
		Label.CreateWidget( self )
		if self.radio_group != None: 
			self.w.indicatorType = Xmd.ONE_OF_MANY
		self.SetIndicator( self.use_indicator )
		

	def InitWidget( self ):
		Control.InitWidget( self )
		self.w.AddCallback( 'valueChangedCallback', self.ValueChangedCB, None)
		self.Subscribe( vp.EXPOSE, self._InitSizeEH, None )


	def _InitSizeEH( self, target, client_data, xevent, e ):
		#
		# This is a hack to get the size set correctly
		#
		self.UnSubscribe( vp.EXPOSE )
		(w,h) = self.GetSize()
		self.w.SetValues({ 'width': w, 'height': h })
		

	# ------------------------------------------------------------------
	# Modify methods

	def SetIndicator( self, bool ):
		#
		# This method will remove or reinstall the indicator icon from the 
		# ToggleButton. Having no indicator can be useful if one wants to
		# use ToggleButtons in a 'toolchest'. The default is to have an 
		# indicator.
		# 
		if bool == vp.FALSE:
			self.SetXResources({
				  'indicatorOn'    : X.FALSE,
				  'recomputeSize'  : X.FALSE,
				  'alignment'      : Xmd.ALIGNMENT_CENTER,
				  'shadowThickness': 2,
				  'marginTop'      : 0,
				  'marginBottom'   : 0,
				  'marginLeft'     : 0,
				  'spacing'        : 0,
				  'shadowThickness': 2
				  })
		else:
			self.SetXResources({
				  'indicatorOn'  : X.TRUE,
				  'recomputeSize': X.FALSE,
				  'alignment'    : Xmd.ALIGNMENT_BEGINNING
				  })


	def SetToggle( self, state ):
		self.state = state
		if state == vp.ON:
			state = X.TRUE
		else:
			state = X.FALSE

		if self.IsRealized(): self.w.ToggleButtonSetState( state, X.FALSE )
		
		
	# ------------------------------------------------------------------
	# Access methods:

	def GetState( self ): return self.state

	# ------------------------------------------------------------------
	# Event notification methods:

	def ValueChangedCB( self, w, client_data, call_data ):
		if debug: print 'ToggleButton.ValueChanged called'

		if self.w.set == X.TRUE:
			self.SetToggle( vp.ON )
		else:
			self.SetToggle( vp.OFF )
		self.ExecuteCallback( 'activate', self )
			
