# -------------------------------------------------------------------------
# MODULE:      TextControls
#
# DESCRIPTION: 
#     Contains the TextList, TextField and LabeledTextField classes.
#
# AUTHOR:
#     Per Spilling, CWI, Amsterdam, per@cwi.nl

import Xm, Xmd, X

import vp

from WidgetObject import WidgetObject
from Control      import Control

from Box          import Box, Glue
from MiscGraphic  import Label

debug = vp.FALSE

# -------------------------------------------------------------------------
# CLASS:         TextList
#
# INHERITS FROM: (Control, WidgetObject) : Graphic : \
#                (TreeNode, DatumDict) : Object
#
# DESCRIPTION: 
#     This is a control class which enables one to select a text-item from a 
#     list. Items can be added to- and removed from the list dynamically, and
#     the TextList class will see to it that the list is kept in alphabetical
#     order. Scrollbars will be used if needed.
#

class TextList( Control, WidgetObject ):

	# ------------------------------------------------------------------
	# Init methods

	def __init__( self, argdict = {} ): 
		#
		# Args can be any of the following: 
		# - 'name'          : <string>
		# - 'callback'      : <callback function/method instance>
		# - 'command'       : <command instance>
		# - 'order'         : (SORTED=default, REVERSE_SORTED, UNSORTED)
		# - 'list'          : <python-list of strings>
		#
		self.list      = []
		self.order     = vp.SORTED
		self.selection = None
		Control.__init__( self, argdict ) 
		self.SetDefaultXRes()

		if self.list != []: self._Reorder()

		self.Subscribe( vp.EXPOSE, self._InitExposeEH, None )


	def SetDefaultXRes( self ):
		self.SetXResources({
			'listSizePolicy'        : Xmd.CONSTANT,
			'scrollBarDisplayPolicy': Xmd.STATIC
			})

	
	def CreateWidget( self ):
		#
		# Must override the default WidgetObject.CreateWidget() method
		# since a special creation function is used. The TextList is
		# actually constructed out of two widgets: a List widget and
		# a ScrolledWindow widget.
		#
		if debug: print 'TextList.CreateWidget called for', self.GetName()

		pw = self.parent.GetWidget()
		self.list_w = pw.CreateScrolledList( 'TextList', self.xresources )
		self.list_w.ManageChild()
		self.w = self.list_w.Parent()  # ScrolledWindow widget
		self.InitWidget()


	def InitWidget( self ):
		Control.InitWidget( self )
		self.list_w.AddCallback( 'browseSelectionCallback', 
			                     self.ItemSelectedCB, None )

		# Add text items to the list widget

		if self.list != None:
			n = 0
			for item in self.list:
				n = n + 1
				self.list_w.ListAddItem( item, n )
			self._ResetSize()


	def _ResetSize( self ):
		#
		# For some reason the list widget has a tendency to reszise itself
		# when the size of the list changes. 
		#
		self.w.width = self.GetWidth()
		self.w.height = self.GetHeight()


	def _InitExposeEH( self, target, client_data, xevent, e ):
		#
		# And for some reason the list widget has a tendency to reszise itself
		# when it is exposed for the first time.
		#
		self._ResetSize()
		self.UnSubscribe( vp.EXPOSE )


	# ------------------------------------------------------------------
	# Event notification methods:

	def ItemSelectedCB( self, w, client_data, cbs ):
		self.selection = self.list[ cbs.item_position - 1 ]
		
		if debug:
			print 'TextList.ItemSelected:', self.selection, 'was selected'

		if self.editmode == vp.FALSE:
			self.ExecuteCallback( 'activate', self )
	

	# ------------------------------------------------------------------
	# Misc. public methods:

	def AddItem( self, item ):
		#
		# Add one item
		#
		self.list.append( item )
		self._Reorder()

		print 'item =', item, 'list =', self.list

		n = self.list.index( item )
		self.list_w.ListAddItem( item, n + 1 ) 


	def AddList( self, item_list ):
		#
		# Add a list of items
		#
		self.list = self.list + item_list
		self._Reorder()

		for item in item_list:
			n = self.list.index( item )
			self.list_w.ListAddItem( item, n + 1 ) 
		

	def SetList( self, item_list ):
		#
		# Delete the current list and install a new one.
		#
		if type(item_list) == type([]):
			self.list = item_list
		else:
			self.list = []

		if self.IsRealized():
			self.w.UnmanageChild()
			self.list_w.ListDeleteAllItems( )
			if self.list != []:
				self._Reorder()
				for item in self.list:
					n = self.list.index( item )
					self.list_w.ListAddItem( item, n + 1 )

			self._ResetSize()
			self.w.ManageChild()


	def SetOrder( self, order ):
		#
		# order = SORTED, REVERSE_SORTED, UNSORTED 
		#
		if order != self.order:
			self.order = order
			self._Reorder()
		else:
			self.order = order


	def DeleteItem( self, item ):
		#
		# Delete one item from the list
		#
		if item in self.list:
			self.list.remove( item )
			self.list_w.ListDeleteItem( item ) 
			if item == self.selection: self.selection = None


	def GetSelection( self ):
		#
		# Return the selected item(s)
		#
		return self.selection


	# ------------------------------------------------------------------
	# Misc. private methods:

	def _Reorder( self ):
		if self.order == vp.SORTED:
			self.list.sort()
		elif self.order == vp.REVERSE_SORTED:
			self.list.sort()
			self.list.reverse()
			

# -------------------------------------------------------------------------
# CLASS:         CheckList
#
# INHERITS FROM: TextList : (Control, WidgetObject) : Graphic : \
#                (TreeNode, DatumDict) : Object
#
# DESCRIPTION: 
#     Similar to the TextList, but with the added functionality that items
#     can be checked on and off. A consequence of this is that the 
#     GetSelection() method will return a list instead of a single item.
#
#     NB! The CheckList does not (yet) handle multiple items with the same 
#     value.
#
		
class CheckList( TextList ):

	# ------------------------------------------------------------------
	# Init methods

	def __init__( self, argdict = {} ): 
		self.unformatted_list = []  # the 'list' attribute (see the TextList 
                                    # class) will contain the formatted list
		TextList.__init__( self, argdict )
		self.selection = []

		
	def InitWidget( self ):
		if self.selection != []:
			TextList.SetList( self, self._GetFormattedList() )
		TextList.InitWidget( self )


	# ------------------------------------------------------------------
	# Misc. public methods:

	def GetSelection( self ): return self.selection[:]

	def SetSelection( self, sel ):
		#
		# Check that it is a valid selection
		#
		if sel == None: 
			self.selection == []
		else:
			if type(sel) != type([]): sel = [sel]
			for item in sel: 
				if item not in self.unformatted_list: 
					return  # not valid selection
			self.selection = sel

		if self.IsRealized():
			TextList.SetList( self, self._GetFormattedList() )


	def SetList( self, item_list ):
		if type(item_list) == type([]):
			self.unformatted_list = item_list
		else:
			self.unformatted_list = []
		TextList.SetList( self, self._GetFormattedList() )


	def AddItem( self, item ): 
		self.unformatted_list.append( item )
		TextList.AddItem( self, '  ' + item )


	def AddList( self, item_list ):
		self.unformatted_list = self.unformatted_list + item_list
		TextList.SetList( self, self._GetFormattedList() )
		

	def DeleteItem( self, item ):
		if item in self.unformatted_list:
			i = self.unformatted_list.index( item )
			self.list_w.ListDeleteItem( self.list[i] ) 
			del self.unformatted_list[i]
			del self.list[i]
			if item in self.selection: 
				self.selection.remove( item )


	# ------------------------------------------------------------------
	# Event notification methods:

	def ItemSelectedCB( self, w, client_data, cbs ):
		#
		# Overridden from TextList; items can be selected or deselected
		#
		sel = self._GetStrippedItem(self.list[cbs.item_position-1])

		if self.selection == None: self.selection = []
		
		if sel in self.selection:         
			self.selection.remove( sel )
		else:                             
			self.selection.append( sel )

		TextList.SetList( self, self._GetFormattedList() )
		if self.editmode == vp.FALSE:
			self.ExecuteCallback( 'activate', self )
		

	# ------------------------------------------------------------------
	# Misc. private methods:

	def _GetStrippedItem( self, string ):
		i  = 0
		while string[i] == ' ' or string[i] == '*': i  = i + 1
		return string[i:]


	def _GetFormattedList( self ):
		formatted_list = []

		if self.selection == None: self.selection = []

		for item in self.unformatted_list:
			if item in self.selection:
				formatted_list.append( '* ' + item )
			else:
				formatted_list.append( '  ' + item )

		return formatted_list


	def _Reorder( self ):
		if not self.IsRealized():
			self.unformatted_list = self.list

		if self.order == vp.SORTED:
			self.unformatted_list.sort()
		elif self.order == vp.REVERSE_SORTED:
			self.unformatted_list.sort()
			self.unformatted_list.reverse()
		
		self.list = self._GetFormattedList()


# -------------------------------------------------------------------------
# CLASS:         TextField
#
# INHERITS FROM: (Control, WidgetObject) : Graphic : \
#                (TreeNode, DatumDict) : Object
#
# DESCRIPTION: 
#     A control class for text-input.
#

class TextField( Control, WidgetObject ):

	widget_class = Xm.Text

	# ------------------------------------------------------------------
	# Init methods

	def __init__( self, argdict = {} ): 
		#
		# Args can be any of the following: 
		# - 'name'          : <string>
		# - 'callback'      : <callback function/method instance>
		# - 'command'       : <command instance>
		# - 'characters'    : <num. of chars in the text-field> (10=default)
		# - 'text'          : <text to be displayed>
		#
		self.editable   = vp.TRUE
		self.characters = 10
		self.text       = ''
		Control.__init__( self, argdict )  
		self.SetDefaultXRes()


	def SetDefaultXRes( self ):
		self.SetXResources({ 'columns': self.characters })
		self.SetXResources({ 'highlightThickness': 0 })

		if self.editable:
			self.SetXResources({ 'editable': X.TRUE })
		else:
			self.SetXResources({ 'editable': X.FALSE })


	def InitWidget( self ):
		Control.InitWidget( self )
		self.w.TextSetString( self.text )
		self.w.AddCallback( 'activateCallback', self.ActivateCB, None )


	# ------------------------------------------------------------------
	# Event notification methods:

	def ActivateCB( self, w, client_data, call_data ):
		if debug:
			print 'TextField.Activated: ', self.w.TextGetString( )

		if self.editmode == vp.FALSE:
			self.ExecuteCallback( 'activate', self )


	# ------------------------------------------------------------------
	# Public methods:

	def SetText( self, text ):
		self.text = text

		if self.IsRealized():
			self.w.TextSetString( text )

	def GetText( self ):
		if self.IsRealized():
			return self.w.TextGetString( )
		else:
			return ''


# -------------------------------------------------------------------------
# CLASS:         LabeledTextField
#
# INHERITS FROM: (Control, WidgetObject) : (TreeNode, DatumDict) : Object
#
# DESCRIPTION: 
#     A control class for text-input.
#

class LabeledTextField( Box ):

	def __init__( self, argdict = {} ):
		#
		# Args can be any of:
		# - 'label'            : <label-string>
		# - 'callback'         : <callback function/method instance>
		# - 'command'          : <command instance>
		# - 'characters'       : <num. of chars in the text-field> (10=default)
		# - 'text'             : <text to be displayed>
		# - 'l_stretchability' : (vp.FIXED, vp.FIXED)=default
		# - 'tf_stretchability': (vp.ELASTIC, vp.ELASTIC)=default
		#
		argdict = self.MergeDefaults( argdict, {
			  'stretchability': (vp.ELASTIC,vp.FIXED)
			  })
		self.tf_argdict = {}   # args for textfield
		self.l_argdict  = {}   # args for label
		self.l_argdict['stretchability'] = (vp.FIXED,vp.FIXED)

		Box.__init__( self, argdict )

		self.label     = Label( self.l_argdict )
		self.textfield = TextField( self.tf_argdict )

		self.AddChild( self.label )
		self.AddChild( self.textfield )

		self.AddFunctionDatum('_label_changed_cb', self.LabelChangedCB)
		self.label.label.LinkTo( self._label_changed_cb )
		self.label.pixmap.LinkTo( self._label_changed_cb )


	def ProcessArgs( self, argdict ):
		#
		# This method is overridden from the Graphic class since the arguments
		# must be split between the label-part and the textfield-part of the
		# LabeledTextField class.
		#
		for key in argdict.keys():
			if key in ('callback', 'command', 'characters', 'text','editable'):
				self.tf_argdict[key] = argdict[key]
				del argdict[key]
			elif key == 'tf_stretchability':
				self.tf_argdict['stretchability'] = argdict[key]
				del argdict[key]
			elif key == 'bitmap':
				self.l_argdict[key] = argdict[key]
				del argdict[key]
			elif key == 'l_stretchability':
				self.l_argdict['stretchability'] = argdict[key]
				del argdict[key]

		Box.ProcessArgs( self, argdict )


	def LabelChangedCB( self, value ):
		if self.IsRealized():
			self.did_calculate_size = vp.FALSE
			self.DoLayout()

	
	# ------------------------------------------------------------------
	# Misc. public methods:

	def SetText( self, text ): 
		if hasattr( self, 'textfield' ):
			self.textfield.SetText( text )
		else:
			self.tf_argdict['text'] = text


	def GetText( self ): return self.textfield.GetText()


	def SetBitmap( self, filename ): 
		if hasattr( self, 'label' ):
			self.label.SetBitmap( filename )
		else:
			self.l_argdict['label'] = filename


	def SetName( self, string ): 
		Box.SetName( self, string )
		if hasattr( self, 'label' ):
			self.label.SetName( string )
		else:
			self.l_argdict['name'] = string
		
