# -------------------------------------------------------------------------
# MODULE:      Sketcher
#
# DESCRIPTION: 
#     Contains the (Point)Sketcher, RubberBandSketcher, and RubberRectSketcher
#     classes.
#
# AUTHOR:
#     Per Spilling, CWI, Amsterdam, <per@cwi.nl>
#

import X, vp, EventNotifier

from vp         import TRUE, FALSE, \
                       MOUSE_DOWN, MOUSE2_DRAG, MOUSE1_DRAG, MOUSE_UP

from Object     import Object
from Responder  import Responder
from Graphic    import Graphic
from Control    import Control

debug = vp.FALSE


# -------------------------------------------------------------------------
# CLASS:         Sketcher 
#
# INHERITS FROM: (Control, Responder, Object)
#
# COLLABORATORS: Canvas, Command
#
# DESCRIPTION: 
#     This is a class which does not create any (own) graphic. It is supposed
#     to be used to create new objects (shapes/symbols) in a Canvas. The
#     Sketcher class is used a superclass for other sketchers but can be used 
#     on its own aswell as a "point-sketcher" (for positioning a new object).
#

class Sketcher( Control, Responder, Object ):

	# ------------------------------------------------------------------
	# Initialization & finalization methods

	def __init__( self, argdict ):
		#
		# Args can be the following:
		# - 'canvas'  : the canvas the sketcher should collaborate with 
		#               (required argument)
		# - 'callback': <activate-callback function/method instance>
		# - 'command' : <command instance>
		#
		Control.__init__( self, argdict )

		if self.callback != None:
			Object.AddCallback( self, 'end_sketch', self.callback )

		self.gc         = None           # (X windows) graphical context
		self.color      = 0
		self.begin      = (0,0)
		self.end        = (0,0)
		self.old        = (0,0)
		self.prev_event = None

		# install event-handlers without subscribing

		self.SetEH( MOUSE_DOWN, self.MouseDownEH, None )
		self.SetEH( MOUSE1_DRAG, self.Mouse1DragEH, None )
		self.SetEH( MOUSE_UP, self.MouseUpEH, None )

		# check if the canvas has subscribed to the necessary events

		if not self.canvas.DidSubscribe( vp.MOUSE_DOWN ):
			self.canvas.Subscribe( vp.MOUSE_DOWN, self.MouseDownEH, None )

		if not self.canvas.DidSubscribe( vp.MOUSE1_DRAG ):
			self.canvas.Subscribe( vp.MOUSE1_DRAG, self.Mouse1DragEH, None )

		if not self.canvas.DidSubscribe( vp.MOUSE_UP ):
			self.canvas.Subscribe( vp.MOUSE_UP, self.MouseUpEH, None )


	# ------------------------------------------------------------------
	# Public access methods

	def GetGeometry( self ):
		if debug: 
			print 'Sketcher.GetGeometry: begin =',self.begin, 'end =', self.end
		xmin = min( [self.begin[0], self.end[0]] )
		ymin = min( [self.begin[1], self.end[1]] )
		xmax = max( [self.begin[0], self.end[0]] )
		ymax = max( [self.begin[1], self.end[1]] )

		return (xmin,ymin,xmax,ymax)


	# ------------------------------------------------------------------
	# Event handler methods:

	def MouseDownEH( self, target, void, xevent, e ):
		#
		# Create GC if needed, and set the initial mouse position 
		#
		if e == None: e = xevent       # XButtonEvent

		if debug: print 'Sketcher.MouseDownEH called'

		self.prev_event = MOUSE_DOWN
		self.begin = (e.x,e.y)

		if e.button == X.Button1: self.PrepareSketch()

		
	def Mouse1DragEH( self, target, void, xevent, e ):
		if self.prev_event == MOUSE_DOWN or self.prev_event == MOUSE1_DRAG:
			if e == None: e = xevent
			self.prev_event = MOUSE1_DRAG
			self.DrawSketch( e.x, e.y )       # subclass will do the drawing


	def MouseUpEH( self, target, void, xevent, e ):
		if self.prev_event == MOUSE_DOWN or self.prev_event == MOUSE1_DRAG:
			if e == None: e = xevent
			self.prev_event = MOUSE_UP
			self.end = (e.x,e.y)
			self.EndSketch( e.x, e.y )        # subclass will clean up

		
	# ------------------------------------------------------------------
	# Misc private methods:

	def PrepareSketch( self ):
		EventNotifier.GrabFocus( self )
		if self.gc == None:
			self.gc = self.canvas.w.CreateGC({})
			self.gc.function = X.GXxor                 # draw in xor mode
		self.gc.foreground = self.canvas.w.background  # set color

		
	def DrawSketch( self, x, y ): pass


	def EndSketch( self, x, y ):
		#
		# Should be overridden in subclass
		#
		EventNotifier.GrabFocus( None )
		self.ExecuteCallback( 'end_sketch', self )


# -------------------------------------------------------------------------
# CLASS:         RubberBandSketcher 
#
# INHERITS FROM: Sketcher : (Control, Graphic) : (TreeNode, DatumDict) : Object
#
# COLLABORATORS: Canvas, Command
#
# DESCRIPTION: 
#     This is a class which does not create any (own) widget. It is supposed
#     to be used to create new objects (shapes/symbols) in a Canvas. The
#     RubberBandSketcher class can be used to create all kinds of line-objects 
#     (plain line, arrows, etc.)
#

class RubberBandSketcher( Sketcher ):

	def __init__( self, argdict ):
		Sketcher.__init__( self, argdict )
		self.old_line = vp.FALSE


	def DrawSketch( self, x, y ):
		if self.old_line:
			self.gc.DrawLine( self.begin[0], self.begin[1], 
				              self.old[0], self.old[1] )
		
		self.gc.DrawLine( self.begin[0], self.begin[1], x, y )
		self.old_line = vp.TRUE
		self.old = (x, y)


	def EndSketch( self, x, y ):
		#
		# Clean up
		#
		if self.old_line:
			self.gc.DrawLine( self.begin[0], self.begin[1], 
				              self.old[0], self.old[1] )
		self.old_line = vp.FALSE
		Sketcher.EndSketch( self, x, y )
		

	def GetGeometry( self ): return (self.begin, self.end)


# -------------------------------------------------------------------------
# CLASS:         RubberRectSketcher 
#
# INHERITS FROM: Sketcher : (Control, Graphic) : (TreeNode, DatumDict) : Object
#
# COLLABORATORS: Canvas, Command
#
# DESCRIPTION: 
#     The RubberRectSketcher class be used to create all kinds of object which
#     can be specified by a bounding box, or it can be used for making a 
#     selection rectangle..
#

class RubberRectSketcher( Sketcher ):

	def __init__( self, argdict ):
		Sketcher.__init__( self, argdict )
		self.old_rect = vp.FALSE


	def DrawSketch( self, x, y ):
		if self.old_rect:
			x0, y0 = self.begin
			x1, y1 = self.old
			if x0 > x1: x0, x1 = x1, x0
			if y0 > y1: y0, y1 = y1, y0
			self.gc.DrawRectangle( x0, y0, x1 - x0, y1 - y0 )

		x0, y0 = self.begin
		x1, y1 = x, y
		if x0 > x1: x0, x1 = x1, x0
		if y0 > y1: y0, y1 = y1, y0
		self.gc.DrawRectangle( x0, y0, x1 - x0, y1 - y0 )
		self.old_rect = vp.TRUE
		self.old = (x, y)


	def EndSketch( self, x, y ):
		#
		# Clean up
		#
		if self.old_rect:
			x0, y0 = self.begin
			x1, y1 = self.old
			if x0 > x1: x0, x1 = x1, x0
			if y0 > y1: y0, y1 = y1, y0
			self.gc.DrawRectangle( x0, y0, x1 - x0, y1 - y0 )
		self.old_rect = vp.FALSE
		Sketcher.EndSketch( self, x, y )
		
