# -------------------------------------------------------------------------
# MODULE:      ObjectIO
#
# DESCRIPTION: 
#     Contains the File and Storable classes which implement object 
#     persistence.
#
# ACKNOWLEDGEMENT:
#     The design of these classes are inspired by the object persistence 
#     (activation and passivation) implemented in the ET++ framework.
#
# AUTHOR:
#     Per Spilling, CWI, Amsterdam, per@cwi.nl

import string, sys, TypeInfo

from vp       import TRUE, FALSE
from Mixin    import Mixin

EOF   = ''
debug = FALSE

# some global objects used for type checking

list = []  
dict = {}

# --------------------------------------------------------------------------
# CLASS:         File
#
# DESCRIPTION: 
#     A class which provides support for writing/reading objects to/from file.
#

class File:

	def __init__( self, fname, mode ):
		self.fname     = fname
		self.file      = open( fname, mode )

		if self.file == None: print 'File: unable to open file', fname

		self.type_dict = {}    # <type>: [<oid>,..]
		self.ref_dict  = {}    # <oid>: <ref>


	# ------------------------------------------------------------------
	# Copy the built-in file methods and add some new ones

	def Close( self ): self.file.close()
	def Flush( self ): self.file.flush()
	def Read( self, bytes ): return self.file.read( bytes )
	def ReadChar( self ): return self.file.read(1)

	def ReadWord( self ): 
		#
		# Remove whitespace and return the next word (token)
		#
		c = self.ReadChar()
		if c == EOF:
			return EOF
		else:
			while c in string.whitespace: 
				c = self.ReadChar()

			w = ''
			while c != EOF and c not in string.whitespace:
				w = w + c
				c = self.ReadChar()
			return w

	def ReadLine( self ): return self.file.readline()
	def ReadLines( self ): return self.file.readlines()
	def Seek( self, offset, whence=0 ): self.file.seek( offset, whence )
	def Tell( self ): return self.file.tell()
	def Write( self, str ): self.file.write( str )

		
	# ------------------------------------------------------------------
	# Methods which support generic object I/O
		
	def GetObjectRef( self, cname, index ):
		return self.ref_dict[self.type_dict[cname][index]]


	def GetIndex( self, obj ):
		#
		# Assumes that obj is a reference to either a list, a dict, or an
		# instance of a class. The (undocumented) *id* function of python
		# is used to check for indentity.
		#
		oid = id(obj)  # object identity

		if type( obj ) == type( list ):
			cname = 'list'
		elif type( obj ) == type( dict ):
			cname = 'dict'
		else:
			cname = obj.__class__.__name__

		if self.type_dict.has_key( cname ) and oid in self.type_dict[cname]:
			return self.type_dict[cname].index( oid )
		else:
			return None


	def MakeIndex( self, obj ):
		oid = id(obj)  # object identity

		if type( obj ) == type( list ):
			cname = 'list'
		elif type( obj ) == type( dict ):
			cname = 'dict'
		else:
			cname = obj.__class__.__name__

		if not self.type_dict.has_key( cname ):
			self.type_dict[cname] = [oid]
		elif not oid in self.type_dict[cname]:
			self.type_dict[cname].append( oid )

		self.ref_dict[oid] = obj

		return self.type_dict[cname].index( oid )


		
# --------------------------------------------------------------------------
# CLASS:         Storable
#
# DESCRIPTION: 
#     A protocol class which implements object input/output. Storable classes
#     should inherit from this class. The subclasses should override the 
#     following methods (see test/ObjectIO_test.py):
#
#     - _WriteTo( ofile )
#     - _ReadFrom( ifile )
#
#     and copy the LoadClass() method.
#
#     Storable provides support for storing basic data types, python lists, 
#     and instance of classes which comply to the Storable protocol.
#
# FILE FORMAT:
#     The File class is used when writing objects to a file. The File class 
#     maintains a dictionary of types and type-instances. Every type-instance
#     gets a unique table index number. This number is used to refer to 
#     different object instances in the file. The following format is used:
#
#     - object instances:  { <classname> #<inum>
#                              <subclass data>
#                          }
#
#     - object references: <classname> @<index>
#
#     Python lists, strings, and numbers are automatically written and read by
#     the Storable class. The following type-tags are used for these classes:
#
#     - lists:             list
#     - strings:           s 
#     - numbers:           n 
#
#     Storable will use indentation to make nested structures and lists more 
#     readable for humans. The default indentation step is 4 spaces. This can
#     be changed via the "indent_step" class variable.
#
#  EXAMPLES:
#     test/ObjectIO_test.py.
# 
#  TODO:
#     Provide support for storing python dictionaries.
#

class Storable( Mixin ):

	# ------------------------------------------------------------------
	# Public class attributes

	indent_step = 4     # number of spaces


	# ------------------------------------------------------------------
	# Private class attributes

	_indent    = 0      # current indent level
	_do_indent = TRUE


	# ------------------------------------------------------------------
	# Public methods

	def LoadClass( self, cname ): 
		#
		# This method must be copied in every storable class otherwise it
		# is impossible(?) to get at the class object.
		#
		return eval( cname )


	def WriteTo( self, file ):
		ix = file.GetIndex( self )

		if ix != None:                # ref to an object already printed
			if Storable._do_indent:
				file.Write( Storable._indent * ' ' )
			file.Write( self.GetClassName() + ' @' + `ix` + '\n' ) 
			Storable._do_indent = TRUE

		else:                         # print a new instance 
			ix = file.MakeIndex( self )
			if Storable._do_indent:
				file.Write( Storable._indent * ' ' )

			file.Write( '{ ' + self.GetClassName() + ' #' + `ix` + '\n' )
			self.Indent()
			Storable._do_indent = TRUE

			self._WriteTo( file )     # let subclass print specific things

			self.Dedent()
			file.Write( Storable._indent * ' ' + '}\n' )


	def WriteAttr( self, file, attr, val ):
		#
		# Assumes that attr is a string and val can either a basic type
		# or an object instance.
		#
		if debug: print 'Storable.WriteAttr: attr =', attr, 'val =', val

		file.Write( Storable._indent * ' ' + attr + ': ' )

		if TypeInfo.IsObject( val ) and hasattr( val, 'WriteTo' ):
			Storable._do_indent = FALSE
			val.WriteTo( file )

		elif type(val) == type(list):
			self._WriteList( file, val )

		elif type(val) == type(''):
			file.Write( 's ' + val + '\n' )    # s is type tag for strings

		elif type(val) == type(None):
			file.Write( `val` + '\n' )   

		else:
			file.Write( 'n ' + `val` + '\n' )  # n is type tag for numbers


	def ReadFrom( self, file ):
		obj = None
		w   = file.ReadWord()

		if w == EOF:
			return EOF

		if w == '{':                     # make a new object
			cname = file.ReadWord()      # class name

			if cname == 'list':
				obj = []                 # create list instance

				if debug:  print 'Storable.ReadFrom: list created'

				ix = file.MakeIndex( obj )
				w  = file.ReadWord()     # read index from file 
				count = file.ReadWord()  # number of items in list

				for n in range(eval(count)):
					item = self.ReadFrom( file )
					obj.append( item )

			else:
				class_obj = self.LoadClass( cname )
				obj = class_obj()        # create class instance

				if debug: 
					print 'Storable.ReadFrom:', obj.GetClassName(), 'created'

				ix = file.MakeIndex( obj )
				w  = file.ReadWord()     # read index from file 
				obj._ReadFrom( file )    # let subclass read specific things

			w = file.ReadWord()          # this should be the an }

			if w != '}':
				print 'Storable.ReadFrom: expected } at end of', obj

		elif w == 's':                   # string object
			obj = file.ReadWord()

		elif w == 'n':                   # numeric object
			obj = eval(file.ReadWord())

		elif w == 'None':                # None
			obj = None

		else:                            # ref to existing object
			nw  = file.ReadWord()
			ix = eval(nw[1:])  
			obj = file.GetObjectRef( w, ix )

		return obj


	def ReadAttr( self, file ):
		#
		# Assumes the following format: <attr>: <val> 
		#
		attr = file.ReadWord()
		val = self.ReadFrom( file )
		return ( attr[:len(attr)-1], val )


	# ------------------------------------------------------------------
	# Methods which should be overriden in the subclasses

	def _WriteTo( self, file ):
		#
		# To be overridden by subclass
		#
		pass


	def _ReadFrom( self, file ):
		#
		# To be overridden by subclass
		#
		pass


	# ------------------------------------------------------------------
	# Private methods

	def Indent( self ): 
		Storable._indent = Storable._indent + Storable.indent_step

	def Dedent( self ): 
		Storable._indent = Storable._indent - Storable.indent_step

	def _WriteList( self, file, list ):
		#
		# Generic way of writing a python list to file
		#
		if debug: print 'Storable._WriteList: list =', list

		ix = file.GetIndex( list )

		if ix != None:                   # ref to a list already printed
			file.Write( 'list @' + `ix` + '\n' )
		else:                            # first ref to this list
			ix = file.MakeIndex( list )
			count = len(list)
			file.Write( '{ list #' + `ix` + ' ' + `count` + '\n' )
			self.Indent()

			for item in list:
				if TypeInfo.IsObject( item ) and hasattr( item, 'WriteTo' ):
					item.WriteTo( file )

				elif type(item) == type(list):
					self._WriteList( file, None, item )

				elif type(item) == type(''):
					file.Write( Storable._indent * ' ' + 's ' + item + '\n' )

				elif type(item) == type(None):
					file.Write( Storable._indent * ' ' + `item` + '\n' )   

				else:
					file.Write( Storable._indent * ' ' + 'n ' + `item` + '\n' )

			self.Dedent()
			file.Write( Storable._indent * ' ' + '}\n' )
				

				
				
				
