# -------------------------------------------------------------------------
# MODULE:      ModelView
#
# DESCRIPTION:  
#     Contains the baseclasses for the Model-View framework: 
#     Model, EditorModel, View, EditorView.
#
# AUTHOR:
#     Per Spilling <per@cwi.nl> and Dirk Soede <soede@cwi.nl>, CWI, Amsterdam.

import vp

from Object import Object

debug       = vp.FALSE
debug_mv    = vp.FALSE
debug_final = vp.FALSE

# -------------------------------------------------------------------------
# CLASS:      Model
#
# INHERITS FROM: Object
#
# DESCRIPTION:  
#     Superclass of all models. 
# 

# -------------------------------------------------------------------------
# 'Class methods' for the Model class:

def AddViewClass( model_class, view_category, view_class ):
	mcn = model_class.__name__

	if not model_class.mv_view_class_dict.has_key( mcn ):
		model_class.mv_view_class_dict[mcn] = {} # make a new vc-dict
	model_class.mv_view_class_dict[mcn][view_category] = view_class


class Model( Object ):

	# ------------------------------------------------------------------
	# Class attributes.

	mv_view_class_dict = {}  # The dict of dict of view classes. Categories 
                             # of view classes should be added with the 
							 # 'AddViewClass' 'class-method'   
	
	# ------------------------------------------------------------------
	# Initialisation & destruction

	def __init__( self, editor ):
		#
		# 'mv_editor' is the EditorModel enclosing the models. Normally
		# this would be the application class. 
		#
		self.mv_editor = editor 
		self.mv_views  = {}

		if debug_mv: print self.GetClassName(), ': editor =', self.mv_editor

		if self.GetViewClassDict() != None:
			self.CreateViews()
		else:
			print self.GetClassName(), 'has not been assigned view classes!'


	def GetViewClassDict( self ):
		if Model.mv_view_class_dict.has_key( self.GetClassName() ):
			return Model.mv_view_class_dict[ self.GetClassName() ]
		else:
			return None


	def CreateViews( self ):
		#
		# Calls the constructor of every View class
		#
		vc_dict = self.GetViewClassDict()
		for view_category in vc_dict.keys():
			view_class = vc_dict[view_category]
			self.mv_views[view_category] = view_class(self, view_category)


	def RealizeView( self, category ): self.mv_views[category].Realize()

	def UnRealizeView( self, category ): self.mv_views[category].UnRealize()


	def UnRealizeAllViews( self ):
		for category in self.mv_views.keys():
			if self.mv_views[category].IsRealized():
				self.mv_views[category].UnRealize()


	def SynchronizeViewRealizations( self ):
		#
		# Enforce the correct realization on the views. The view itself will
		# evaluate the condition for realization.
		#
		if debug:
			print 'Model.SynchronizeViewRealizations called for', \
				  self.GetName(), 'view-categories =', self.mv_views.keys()

		for view_category in self.mv_views.keys():
			self.mv_views[view_category].SynchronizeRealization()


	def Finalize( self ):
		#
		# Cleanup all views
		#
		if debug_final:
			print 'Model.Finalize called for', self.GetClassName(), \
				  'mv_views =', self.mv_views

		for view_category in self.mv_views.keys():
			self.mv_views[view_category].Finalize()
			del self.mv_views[view_category]

		self.mv_views = None
		Object.Finalize( self )

		
	# ------------------------------------------------------------------
	# Access methods.

	def GetEditorView( self, view_category ):
		#
		# Return the editor-view where the views of 'view_category' are 
		# displayed and edited.
		#
		return self.mv_editor.mv_views[view_category]


	def GetView( self, view_category ):
		#
		# Return one particular view of the model
		#
		return self.mv_views[view_category]


	# ------------------------------------------------------------------
	# Misc public methods

	def SetImmediateMode( self, mode ):
		#
		# 'mode' off means no updates are done in the views. Used to 'compress'
		# the updates.
		#
		for view_category in self.mv_views.keys():
			self.mv_views[view_category].SetImmediateMode( mode )


	# ------------------------------------------------------------------
	# Communication with views.

	def NotifyChange( self, change_category, data ):
		#
		# Notify all views about a change in the model
		#
		for view_category in self.mv_views.keys():
			self.mv_views[view_category]._ModelChanged( change_category, data )
			

	def GraphicUpdateOn( self ):
		#
		# Affects the graphic of all views of the model
		#
		for view_category in self.mv_views.keys():
			self.mv_views[view_category].GraphicUpdateOn()


	def GraphicUpdateOff( self ):
		for view_category in self.mv_views.keys():
			self.mv_views[view_category].GraphicUpdateOff()



# -------------------------------------------------------------------------
# CLASS:      EditorModel
#
# INHERITS FROM: Model : Object
#
# DESCRIPTION:  
#     A special model class which is used as an editor for other model classes.
#

# -------------------------------------------------------------------------
# 'Class methods' for the EditorModel class:

def AddWindowClass( model_class, view_category, window_class ):
	mcn = model_class.__name__

	if not model_class.mv_window_class_dict.has_key( mcn ):
		model_class.mv_window_class_dict[mcn] = {} # make a new vc-dict
	model_class.mv_window_class_dict[mcn][view_category] = window_class


class EditorModel( Model ):

	# ------------------------------------------------------------------
	# Class attributes.

	mv_window_class_dict = {}


	# ------------------------------------------------------------------
	# Initialisation & destruction

	def __init__( self ):
		Model.__init__( self, None )
		self.edit_object = None
		self.mv_windows = {}
		self.CreateWindows()
		

	def RealizeView( self, category ):
		self.mv_views[category].SetWindow( self.mv_windows[category] )
		Model.RealizeView( self, category )


	def GetWindowClassDict( self ):
		if EditorModel.mv_window_class_dict.has_key( self.GetClassName() ):
			return EditorModel.mv_window_class_dict[ self.GetClassName() ]
		else:
			return None


	def CreateWindows( self ):
		#
		# Default window-creation method
		#
		wc_dict = self.GetWindowClassDict()
		if wc_dict != None:
			for category in wc_dict.keys():
				window_class = wc_dict[category]
				self.mv_windows[category] = window_class()


	# ------------------------------------------------------------------
	# Misc public methods

	def Load( self, model ):
		#
		# Load 'model', i.e. its views will be displayed in the EditorViews
		#
		if debug_mv:
			print 'EditorModel.Load: model =', model.GetName(), \
				  'editor-views =', self.mv_views.keys(), \
				  'view-class-dict =', self.GetViewClassDict().keys()

		if self.edit_object != None:
			self.edit_object.UnRealizeAllViews()
		self.edit_object = model

		for key in self.mv_views.keys():
			if model.mv_views.has_key( key ):
				self.mv_views[key].SetEditObjectView( model.mv_views[key] )

		self.edit_object.SynchronizeViewRealizations()


	def UnLoad( self, model ):
		#
		# UnLoad 'model', i.e. its views will be removed from the EditorViews
		#
		if debug_mv: print 'Model.UnLoad:', model

		if self.edit_object == model:
			self.edit_object.UnRealizeAllViews()

			for key in self.mv_views.keys():
				self.mv_views[key].SetEditObjectView( None )

			self.edit_object = None
		else:
			print 'Model.UnLoad: model not equal to edit-object!'


	def GetCurrentEditObject( self ): return self.edit_object

# -------------------------------------------------------------------------
# Class:     View
#
# INHERITS FROM: Object
#
# DESCRIPTION:  
#     Superclass of views. 
# 

class View( Object ):

	# ------------------------------------------------------------------
	# Initialisation & finalization.

	def __init__( self, model, view_category ):
		Object.__init__( self )
		self.mv_model         = model
		self.mv_view_category = view_category # own view category
		self.mv_immediate     = vp.TRUE	      # Immediate update mode.
		self.mv_need_update   = vp.FALSE
		self.mv_changes       = []
		self.mv_handlers      = {}            # change handlers
		self.mv_graphic       = None          # graphic created on demand
		self.visible          = vp.TRUE       # Graphic's visibility

	# ------------------------------------------------------------------
	# Realization of views.
	# 
	# Deals with the view's own graphic and the graphics it contains. These
	# methods should be overriden when a view contains subviews.

	def Realize( self ):
		#
		# Default implementation
		#
		self.MVCreateGraphic()
		if self.mv_graphic != None:
			self.GetGraphicParent().AddChild( self.mv_graphic )
		
	
	def UnRealize( self ): self.MVDeleteGraphic()


	def SynchronizeRealization( self ):
		#
		# Bring realization of 'self' in accordance with the realization 
		# status of the editor-view.
		#
		realization = self.GetEditorView().IsRealized()

		if realization != self.IsRealized():
			if self.IsRealized():
				self.UnRealize()
			else:
				self.Realize()


	def CreateGraphic( self ): 
		#
		# Must be implemented by subclasses
		#
		pass


	def MVCreateGraphic( self ):
		self.mv_graphic = self.CreateGraphic()      

		# Must test if 'mv_graphic' is None. This might be the case if the
		# subclass is doing something special like for instance using many
		# graphic objects to represent the view.

		if self.mv_graphic != None:            
			self.mv_graphic.mv_view = self     # to enable MV access  
		if not self.visible: self.Hide()

		
	def MVDeleteGraphic( self ):
		if self.mv_graphic != None:  
			self.mv_graphic.Finalize()
			self.mv_graphic.mv_view = None
			self.mv_graphic = None


	def Finalize( self ):
		if debug_final:
			print 'View.Finalize called for', self.mv_model.GetClassName()

		if self.IsRealized():
			self.MVDeleteGraphic()
		self.mv_model    = None
		self.mv_changes  = None
		self.mv_handlers = None
		
		Object.Finalize( self )


	# ------------------------------------------------------------------
	# Misc public methods.

	def Show( self ):
		self.visible = vp.TRUE
		if self.IsRealized(): self.mv_graphic.Show()


	def Hide( self ):
		self.visible = vp.FALSE
		if self.IsRealized(): self.mv_graphic.Hide()


	# ------------------------------------------------------------------
	# Change handling - public methods.

	def SetImmediateMode( self, bool ):
		if bool == vp.TRUE and self.mv_need_update:
			self.ProcessOldChanges()
		self.mv_immediate = bool


	def InstallChangeHandler( self, change_category, handler ):
		self.mv_handlers[change_category] = handler


	# ------------------------------------------------------------------
	# Change handling - private methods.

	def _ModelChanged( self, change_category, data ):
		if self.mv_immediate:
			self.ModelChanged( change_category, data )
		else:
			self.mv_need_update = vp.TRUE
			self.StoreChange( change_category, data )
		

	def ModelChanged( self, change_category, data ):
		if change_category in self.mv_handlers.keys():
			self.mv_handlers[change_category]( data )
		else:
			if debug:
				print 'View.ModelChanged: no handler for', change_category, '!'


	def StoreChange( self, change_category, data ):
		self.mv_changes.append( (change_category, data) )


	def ProcessOldChanges( self ):
		#
		# It is assumed that change categories without data can be merged
		# into a single change.
		#
		merged_changes = {}
		while len(self.mv_changes) > 0:
			change = change_category, data = self.mv_changes[0]
			if data == None:
				merged_changes[change_category] = vp.TRUE
			else:
				self.ModelChanged( change_category, data )
			self.mv_changes.remove( change )
		for change_category in merged_changes.keys():
			self.ModelChanged( change_category, None )
			
	# ------------------------------------------------------------------
	# Access methods:

	def GetChangeCategories( self ): return self.mv_handlers.keys()
		
	def GetGraphic( self ): return self.mv_graphic

	def GetGraphicParent( self ): return self.GetEditorView().GetGraphic()
		
	def GetEditorView( self ): 
		return self.mv_model.GetEditorView( self.mv_view_category )

	def GetEditor( self ): return self.mv_model.mv_editor


	# ------------------------------------------------------------------
	# Query methods:

	def IsRealized( self ):
		#
		# The view can either use the graphic to check whether it is realized
		# or not, or have its own realized attribute.
		#
		if hasattr( self, 'realized' ):
			return self.realized
		elif self.mv_graphic != None:
			return self.mv_graphic.IsRealized()
		else:
			return vp.FALSE


# -------------------------------------------------------------------------
# CLASS:      EditorView
#
# INHERITS FROM: View : Object
#
# DESCRIPTION:  
#     A generic top-level view. Uses a Canvas as graphic.
#   

class EditorView( View ):

	# ------------------------------------------------------------------
	# Initialisation.

	def __init__( self, model, view_category ):
		View.__init__( self, model, view_category )
		self.edit_object_view = None
		self.window           = None		 # Encapsulating window.


	def CreateGraphic( self ):
		from Canvas import Canvas
		return Canvas()


	def SetWindow( self, window ): 
		#
		# Called by EditorModel.RealizeView to indicate which window this
		# editor-view should use.
		#
		self.window = window


	def Realize( self ):
		if self.window != None:
			self.MVCreateGraphic()
			self.window.AddWorkArea( self.GetGraphic() )
			if self.edit_object_view != None: self.edit_object_view.Realize()
		else:
			print 'EditorView.Realize:', \
				  'the SetWindow() method must be called before Realize()!',


	def UnRealize( self ):
		if self.edit_object_view != None: self.edit_object_view.UnRealize()
		self.MVDeleteGraphic()


	# ------------------------------------------------------------------
	# Misc public methods

	def SetEditObjectView( self, view ): 
		#
		# Unrealize the current 'edit_object_view' is there is any. The
		# Model.Load method will call the SynchronizeViewRealizations method
		# of the model of the 'edit_object_view'.
		#
		if debug:
			print 'EditorView.SetEditObjectView called for', \
				  self.GetClassName(), 'view =', view

		if self.edit_object_view != None and \
			  self.edit_object_view.IsRealized(): 
			self.edit_object_view.mv_model.UnRealizeView(self.mv_view_category)

		self.edit_object_view = view


	# ------------------------------------------------------------------
	# Misc Model-View methods:

	def GraphicUpdateOn( self ): 
		self.GetGraphic().RedrawOn()
		self.GetGraphic().Show()


	def GraphicUpdateOff( self ): 
		self.GetGraphic().RedrawOff()
		self.GetGraphic().Hide()


