# -------------------------------------------------------------------------
# MODULE:      Misc
#
# DESCRIPTION: 
#     Contains various simple WidgetObject classes: Label and Separator.
#
# AUTHOR:
#     Per Spilling, CWI, Amsterdam, <per@cwi.nl>
#

import Xm, Xmd, X

import vp
from WidgetObject import WidgetObject
from Datum        import Datum, ExternalDatum, XResourceShell

debug = vp.FALSE


# -------------------------------------------------------------------------
# CLASS:         ScrolledViewPort 
#
# INHERITS FROM: WidgetObject : Graphic : (TreeNode,DatumDict) : Object
# 
# DESCRIPTION: 
#     A viewport with scrollbars.
#

class ScrolledViewPort( WidgetObject ):

	widget_class = Xm.ScrolledWindow

	def __init__( self, argdict = {} ):
		#
		# Arg can be:
		# - 'view' : <an instance of a View class>
		#
		self.view = None
		self.did_add_view = vp.FALSE
		WidgetObject.__init__( self, argdict )

		if not 'scrollingPolicy' in self.xresources.keys():
			self.SetXResources({ 'scrollingPolicy': Xmd.AUTOMATIC })

		if self.view != None:
			WidgetObject.AddChild( self, self.view )

		self.Subscribe( vp.RESIZE, self.ResizeEH, None )


	def CreateWidget( self ):
		self.w = self.parent.GetWidget().CreateScrolledWindow( 
			                              'ScrolledViewPort', self.xresources )
		self.w.ManageChild()
		self.InitWidget()


	def InitAddedNode( self, child ):
		if not self.did_add_view:
			WidgetObject.InitAddedNode( self, child )
			self.view = child
			self.did_add_view = vp.TRUE
		else:
			self.view.AddChild( child )
		

	# ------------------------------------------------------------------
	# Event handler methods:

	def ResizeEH( self, target, void, xevent, void ):
		WidgetObject.ResizeEH( self, target, void, xevent, void )
		
		if self.view != None:
			(w,h)   = self.view.GetSize()
			(hs,vs) = self.view.GetStretchability()

			if w < self.width.value and hs == vp.ELASTIC:
				self.view.width.Set( self.width.value )

			if h < self.height.value and vs == vp.ELASTIC:
				self.view.height.Set( self.height.value )
			

# -------------------------------------------------------------------------
# CLASS:         PanedViewPort 
#
# INHERITS FROM: WidgetObject : Graphic : (TreeNode,DatumDict) : Object
# 
# DESCRIPTION: 
#     A viewport with resizeable panes.
#

class PanedViewPort( WidgetObject ):

	widget_class = Xm.PanedWindow
	


# -------------------------------------------------------------------------
# CLASS:         Separator
#
# INHERITS FROM: WidgetObject : Graphic : (TreeNode, DatumDict) : Object
#
# DESCRIPTION: 
#

class Separator( WidgetObject ):

	widget_class = Xm.Separator

	def __init__( self, argdict = {} ):
		argdict = self.MergeDefaults( argdict, {
			'stretchability': ( vp.ELASTIC, vp.FIXED )
			})
		WidgetObject.__init__( self, argdict )


# -------------------------------------------------------------------------
# CLASS:         Label
#
# INHERITS FROM: WidgetObject : Graphic : (TreeNode, DatumDict) : Object 
#
# DESCRIPTION: 
#

class Label( WidgetObject ):

	widget_class = Xm.Label

	def __init__( self, argdict = {} ):
		#
		# Args can be:
		# - 'label' : <string>
		# - 'font'  : <fontname> default=None, i.e. the default font will
		#                        be used 
		#
		WidgetObject.__init__( self, argdict )


	def AddDatums( self ):
		self.AddDatum('label',
			          ExternalDatum(XResourceShell('labelString',self)))
		self.AddDatum('pixmap',
		               ExternalDatum(XResourceShell('labelPixmap',self)))
		WidgetObject.AddDatums( self )


		self._width.UpdateOnDatum( self.label )
		self._width.UpdateOnDatum( self.pixmap )
		self.AddFunctionDatum('_label_changed_cb', self.LabelChangedCB)
		self.label.LinkTo( self._label_changed_cb )
		self.pixmap.LinkTo( self._label_changed_cb )

		if debug:
			print 'Label.AddDatums called for', self.GetName()


	def ProcessArgs( self, argdict ):
		if 'font' in argdict.keys():
			self.SetFont( argdict['font'] )
			del ardict['font']
		WidgetObject.ProcessArgs( self, argdict )


	def CreateWidget( self ):
		#
		# The 'labelString' X resource cannot be given to the 
		# Xt.CreateManagedWidget() function for some reason. This means
		# that this resources can only be set after the widget has been
		# created and realized.
		# 
		if debug:
			print 'Label.CreateWidget called for', self.GetName(), \
				  'xresources =', self.xresources, \
				  'name =', self.label.Get()
		
		if self.xresources.has_key( 'labelString' ):
			tmp_dict = {}
			for key in self.xresources.keys():
				if key != 'labelString':
					tmp_dict[key] = self.xresources[key]

			self.w = self.parent.GetWidget().CreateWidget( self.GetName(), 
			                                               self.widget_class,
														   tmp_dict )
			tmp_dict = None
		else:
			self.w = self.parent.GetWidget().CreateWidget( self.GetName(), 
			                                               self.widget_class,
														   self.xresources )
			
		# determine the total_margin_width

		w = self.w

		if debug:
			print 'Label.CreateWidget:', \
				  'marginWidth =', w.marginWidth, \
				  'marginLeft =', w.marginLeft, \
				  'marginRight =', w.marginRight, \
				  'shadowThickness =', w.shadowThickness

		mw = w.marginWidth 
		ml = w.marginLeft
		mr = w.marginRight
		st = w.shadowThickness
		ht = w.hilightThickness

		if mw == None: mw = 0
		if ml == None: ml = 0
		if mr == None: mr = 0
		if st == None: st = 0
		if ht == None: ht = 2

		self.total_margin_width = 2*mw + ml + mr + 2*st + 2*ht

		if self.IsVisible():
			self.w.ManageChild()
		self.InitWidget()

		
	# ------------------------------------------------------------------
	# Modify methods

	def SetFont( self, fontname ):
		self.SetXResources({ 'fontList': fontname })
		self.LabelChangedCB( None )


	def SetBitmap( self, filename ):
		self.SetXResources({ 'labelType': Xmd.PIXMAP })
		self.pixmap.Set( filename )


	def SetName( self, string ):
		self.SetXResources({ 'labelType': Xmd.STRING })
		self.label.Set( string )
		self.GetName = self._GetName  # set method pointer


	# ------------------------------------------------------------------
	# Access methods:

	def _GetName( self ): 
		try:
			return self.label.value
		except:
			return self.GetClassName()


	# ------------------------------------------------------------------
	# Notification methods:

	def LabelChangedCB( self, value ):
		if self.IsRealized():
			self._width.UpdateInternal()  # internal width
			if self.width.value != None and self.height.value != None:
				self.natural_size = (self.width.value, self.height.value)
			self.ExecuteCallback( 'size_changed', self )





			
			


