# -------------------------------------------------------------------------
# MODULE:      Menu
#
# DESCRIPTION: 
#     Contains the PopupMenu, PulldownMenu, and MenuBar classes.
#
# ACKNOWLEDGEMENT:
#     Some ideas for the design of the these classes have been taken from
#     the book: "OOP w. C++ & Motif", D. Young, Prentice-Hall, 1992.
#
# AUTHOR:
#     Per Spilling, CWI, Amsterdam, per@cwi.nl

import X, Xm, vp

from Datum         import Datum
from WidgetObject  import WidgetObject
from WidgetButton  import PushButton
from MiscGraphic   import Separator

debug = vp.FALSE

# -------------------------------------------------------------------------
# CLASS:         Menu
#
# INHERITS FROM: WidgetObject : (Responder,TreeNode,DatumDict) : Object
#
# DESCRIPTION: 
#     An abstract class which provides the functionality for all menu classes.
#     
#     Args can be the following:
#        - 'title': <string>
#        - 'items': <list>
#
#     The items in the 'items' list must either be an instance of a command
#     class or a (<string>, <callback>) tuple.
#

class Menu( WidgetObject ):

	def __init__( self, argdict = {}):
		self.title_w = None
		self.items   = None
		self.enabled = vp.TRUE
		argdict = self.MergeDefaults( argdict, {'title': self.GetName()})
		WidgetObject.__init__( self, argdict )

		if self.items != None:
			idict = {}
			for item in self.items:
				self._AddItem( idict, item )
			self.items = idict


	def AddDatums(self):
		WidgetObject.AddDatums(self)
		self.AddFunctionDatum('title', self.SetTitleCB)


	def _AddItem( self, idict, item ):
		if type(item) == type(()):            # item is (<name>, <callback)
			idict[item[0]] = PushButton({
				  'name'    : item[0], 
				  'callback': item[1],
				  'enabled' : self.enabled
				  })
			self.AddChild( idict[item[0]] )

		elif item.IsA( Separator ):
			self.AddChild( item )

		else:                                 # item is a command object
			idict[item.GetName()] = PushButton({
				  'command': item, 
				  'enabled': self.enabled
				  })
			self.AddChild( idict[item.GetName()] )




	# ------------------------------------------------------------------
	# Public methods:

	def Enable( self ):
		self.enabled = vp.TRUE
		for key in self.items.keys(): self.items[key].Enable()


	def Disable( self ):
		self.enabled = vp.FALSE
		for key in self.items.keys(): self.items[key].Disable()


	def AddItem( self, menu_item ):
		if self.items == None: 
			self.items = {}
		self._AddItem( self.items, menu_item )


	def SetMenu( self, menu_items ):
		#
		# Replaces the current menu with a new one
		#
		for child in self.children[:]: child.Finalize()

		if menu_items != None:
			self.items = {}
			for item in menu_items: self._AddItem( self.items, item )


	def SetTitle( self, title ): self.title.Set( title )

	# ------------------------------------------------------------------
	# Callback methods:

	def SetTitleCB( self, title ):
		#
		# Called by the 'title' datum
		#
		if title != None and self.w != None: 
			self.title_w.labelString = title


# -------------------------------------------------------------------------
# CLASS:         PopupMenu
#
# INHERITS FROM: Menu : WidgetObject : (Responder,TreeNode,DatumDict) : Object
#
# COLLABORATORS: Command, PushButton
#
# DESCRIPTION: 
#     The PopupMenu class encapsulates the process of constructing popup
#     menus. 
#

class PopupMenu( Menu ):

	def CreateWidget( self ):
		if debug:
			print 'PopupMenu.CreateWidget: parent =',self.parent.GetClassName()

		self.w = self.parent.GetWidget().CreatePopupMenu( 
			                         'PopupMenu', self.xresources )
		self.title_w = self.w.CreateManagedWidget( self.title.Get(),
			                                       Xm.Label, 
												   self.xresources )
		void = self.w.CreateManagedWidget( 'sep', Xm.Separator,self.xresources)
		self.InitWidget()


	def Show( self, xevent ):
		#
		# The xevent argument is needed so that the menu can determine where
		# it must popup.
		#
		self.w.MenuPosition( xevent )
		self.w.ManageChild()


# -------------------------------------------------------------------------
# CLASS:         PulldownMenu
#
# INHERITS FROM: Menu : WidgetObject : (Responder,TreeNode,DatumDict) : Object
#
# COLLABORATORS: Command, PushButton
#
# DESCRIPTION: 
#     The PulldownMenu class encapsulates the process of constructing pulldown
#     menus. 
#

class PulldownMenu( Menu ):

	def CreateWidget( self ):
		#
		# Create a pulldown menu pane. The menu is represented by a
		# cascade button in the parent widget (MenuBar).
		#
		pw     = self.parent.GetWidget()
		self.w = pw.CreatePulldownMenu( self.title.Get(), self.xresources )

		if self.parent.IsA( MenuBar ):
			self.title_w = pw.CreateManagedWidget( self.title.Get(), 
				                                   Xm.CascadeButton, 
												   self.xresources )
			self.title_w.subMenuId = self.w

		self.InitWidget()


	def GetNaturalSize( self ):
		#
		# Return the nat. size of the title
		#
		if self.title_w != None:
			return (self.title_w.width, self.title_w.height)
		else:
			return (0, 0)


	def SetSize( self, w, h ):
		if self.title_w != None:
			Menu.SetSize( self, w, h )



# -------------------------------------------------------------------------
# CLASS:         MenuBar
#
# INHERITS FROM: WidgetObject : (Responder,TreeNode,DatumDict) : Object
#
# COLLABORATORS: Command, PushButton
#
# DESCRIPTION: 
#     The MenuBar class encapsulates the process of constructing a menubar 
#     with multiple menu panes. The children of the MenuBar should be 
#     PulldownMenu instances.
#

class MenuBar( WidgetObject ):

	def CreateWidget( self ):
		#
		# Using the XmMenuBar widget as the base widget
		#
		self.w = self.parent.GetWidget().CreateMenuBar( 
			                       'MenuBar', self.xresources )
		self.w.ManageChild()
		self.InitWidget()


	def GetNaturalSize( self ):
		w = 0
		h = 0

		for child in self.children:
			ns = child.GetNaturalSize()
			w = w + ns[0]
			h = max(h, ns[1])

		self.natural_size = (w,h)
		return (w,h)



#-------------------------------------------------------------------------
# CLASS:         OptionMenu
#
# INHERITS FROM: Menu : WidgetObject : (Responder,TreeNode,DatumDict) : Object
# 
# COLLABORATORS: Command, PushButton
#
# DESCRIPTION: 
#     A menu of mutually exclusive options.
#

class OptionMenu( Menu ):

	# ------------------------------------------------------------------
	# Init methods

	def __init__( self, argdict = {}):
		self.pulldown   = PulldownMenu()
		Menu.__init__( self, argdict )

		# The option menu widget doesn't calculate its real size until it
		# is realized (in X terms). In order to get this size an EXPOSE 
		# callback is used.

		self.Subscribe( vp.EXPOSE, self.ExposeEH, None )
		WidgetObject.AddChild( self, self.pulldown )


	def Realize( self ):
		Menu.Realize( self )
		self.w.subMenuId = self.pulldown.GetWidget()


	def CreateWidget( self ):
		self.w = self.parent.GetWidget().CreateOptionMenu( 
			                          'Menu', self.xresources )
		self.w.resizeHeight = X.TRUE
		self.w.resizeWidth  = X.TRUE
		self.w.ManageChild()
		self.InitWidget()


	def ExposeEH( self, target, void, xevent, void ):
		#
		# Only used to get the actual size of the label and menu-title
		#
		self.UnSubscribe( vp.EXPOSE )
		self.natural_size = (self.w.width, self.w.height)
		self.ExecuteCallback( 'size_changed', self )


	# ------------------------------------------------------------------
	# Misc public methods

	def AddChild( self, child ):
		#
		# Called by the AddItem method
		#
		self.pulldown.AddChild( child )




