# -------------------------------------------------------------------------
# MODULE:      Font 
#
# DESCRIPTION: 
#     Sjoerd's Font class modified for vpApp.
#
# USAGE:
#     Font.Init( widget )
#         Initializes the font module. 
#
#     Font.FindFont( font, size, unit ) -> font_object
#         This function returns a font object.  The font is a symbolic name 
#         for the font, the size is the size in points or pixels.  The actual 
#         size chosen may be smaller if the desired size is not available.  
#         This function returns a font object, the methods of which are 
#         described below.
#
# AUTHOR:
#     Sjoerd Mullender <sjoerd@cwi.nl> & Per Spilling <per@cwi.nl>, 
#     CWI, Amsterdam

import vp
from Object import Object

# -------------------------------------------------------------------------
# Global data: these are initialized in the Init() method.

_widget         = None    # must be set via the Init() method
_mmscreenwidth  = 0       # screen width in millimeters
_mmscreenheight = 0       # screen height in millimeters
_screenwidth    = 0       # screen width in pixels
_screenheight   = 0       # screen height in pixels
_dpi_x          = 0       # dots per inch on the screen 
_dpi_y          = 0     

# constants

PPI             = 720     # 0.1 points per inch 


# -------------------------------------------------------------------------
# CLASS:         _Font
#
# INHERITS FROM:  Object
#
# DESCRIPTION: 
#   __del__()
#       Close the font object and free its resources. => del <fobj>
#
#   GetStringSize(text) -> width, height
#       Return the dimensions in points or pixels (=default) of the box that 
#       the given text would occupy if displayed in the font the font object 
#       represents.
#
#   GetAscent( *unit ) -> ascent
#       Return the height of the ascent in points or pixels (=default).
#
#   GetDescent( *unit ) -> descent
#       Return the height of the descent in points or pixels (=default).
#
#   GetFontHeight( *unit ) -> fontheight
#       Return the height of the font in points or pixels (=default).
#

class _Font( Object ):

	# ------------------------------------------------------------------
	# Init & close methods

	def __init__( self, fontname ):
		self._unit = vp.PIXEL
		self._font = _widget.LoadQueryFont( fontname )


	def __del__( self ): del self._font


	# ------------------------------------------------------------------
	# Query methods:

	def IsClosed(self): return not hasattr(self, '_font')


	# ------------------------------------------------------------------
	# Access methods

	def GetFont( self ): return self._font


	def GetStringSize( self, str, *unit ):
		#
		# str  = the string for which the size should be computed
		# unit = the unit of measure (vp.PIXEL or vp.POINT)
		#
		import string
		if self.IsClosed(): raise error, 'font object already closed'

		if not unit == ():
			unit = unit[0]
		else:
			unit = vp.PIXEL

		strlist   = string.splitfields(str, '\n')
		maxwidth  = 0
		maxheight = len(strlist) * (self._font.ascent + self._font.descent)

		for str in strlist:
			width = self._font.TextWidth(str)   # width in pixels
			if width > maxwidth:
				maxwidth = width

		if unit == vp.PIXEL:
			return (maxwidth, maxheight)
		else:
			return (int(maxwidth * PPI/_dpi_x), int(maxheight * PPI/_dpi_y))


	def GetDescent( self, *unit ):
		if self.IsClosed(): raise error, 'font object already closed'

		if not unit == ():
			unit = unit[0]
		else:
			unit = vp.PIXEL

		if unit == vp.PIXEL:
			return self._font.descent
		else:
			return int(self._font.descent * PPI/_dpi_y)


	def GetAscent( self, *unit ):
		if self.IsClosed(): raise error, 'font object already closed'

		if not unit == ():
			unit = unit[0]
		else:
			unit = vp.PIXEL

		if unit == vp.PIXEL:
			return self._font.ascent
		else:
			return int(self._font.ascent * PPI/_dpi_y)


	def GetFontHeight( self, *unit ):
		if self.IsClosed(): raise error, 'font object already closed'

		if not unit == ():
			unit = unit[0]
		else:
			unit = vp.PIXEL

		if unit == vp.PIXEL:
			return self._font.ascent + self._font.descent
		else:
			return int((self._font.ascent + self._font.descent) * PPI/_dpi_y)


# -------------------------------------------------------------------------
# Public module-methods

font_dbase = {}
	
def Init( widget ):
	global _widget, _mmscreenwidth, _mmscreenheight, _screenwidth, \
		   _screenheight, _dpi_x, _dpi_y
	
	_widget         = widget    
	_mmscreenwidth  = widget.WidthMMOfScreen()
	_mmscreenheight = widget.HeightMMOfScreen()
	_screenwidth    = widget.WidthOfScreen()
	_screenheight   = widget.HeightOfScreen()
	_dpi_x          = int(_screenwidth * 25.4 / _mmscreenwidth + .5)
	_dpi_y          = int(_screenheight * 25.4 / _mmscreenheight + .5)


def FindFont( fontname ):
	global font_dbase
	
	if not fontname in font_dbase.keys():
		font_dbase[fontname] = _Font( fontname )
	return font_dbase[fontname]
		


