# -------------------------------------------------------------------------
# MODULE:      Dialog
#
# DESCRIPTION: 
#     Contains the Dialog class.
#
# AUTHOR:
#     Per Spilling, CWI, Amsterdam, per@cwi.nl

import Xm, Xmd, X
import vp

from vp            import TRUE, FALSE
from Object        import Object
from WidgetObject  import WidgetObject
from WidgetButton  import PushButton
from Canvas        import Canvas
from MiscGraphic   import Separator
from Box           import *
from Window        import Window

debug = FALSE

		
# -------------------------------------------------------------------------
# CLASS:         Dialog
#
# INHERITS FROM: WidgetObject : (TreeNode,DatumDict) : Object
#
# DESCRIPTION: 
#     An abstract class which is the base class for all Dialog classes. It
#     provides a standard layout for dialogs which looks like this:
#     
#              --------------------------------
#              |    Message/title area        |
#              --------------------------------
#              |                              |
#              |                              |
#              |    Work area                 |
#              |                              |
#              |                              |
#              --------------------------------
#              |    Button area               |
#              --------------------------------
#
#     The subclasses are responsible for defining the work-area.
#             

class Dialog( WidgetObject ):

	# ------------------------------------------------------------------
	# Initialization methods

	def __init__( self, argdict = {} ):
		WidgetObject.__init__( self, argdict )

		self.ok_cb       = None    
		self.cancel_cb   = None
		self.help_cb     = None
		self.tmp_dialog  = FALSE
		self.realized    = FALSE
		self.SetDefaultXRes()
		

	def Clone( self ): return self.__class__()

	def SetDefaultXRes( self ): pass


	def CreateWidget( self ):
		self.window = Window({ 
			  'window_type': vp.DIALOG_WINDOW, 
			  'layout'     :vp.MWC
			  })
		self.w = self.window.GetWidget()

		self.CreateDialogComponents()
		self.realized = TRUE


	def CreateDialogComponents( self ):
		self.window.AddWorkArea( self.CreateWorkArea() )
		self.window.AddMessageArea()
		self.window.AddCommandArea( self._CreateButtonArea() )


	def _CreateButtonArea( self ): # -> button-box
		self.help_button = PushButton({ 'name':'Help','callback':self.HelpCB })
		cb = PushButton({ 'name': 'Cancel', 'callback': self.CancelCB })
		ob = PushButton({ 'name': 'Ok',     'callback': self.OkCB }) 
		ob.SetXResources({ 'showAsDefault': X.TRUE })

		return EqualSizeBox({
			  'stretchability': (vp.ELASTIC, vp.FIXED ),
			  'child_list': [Glue( 5, vp.FIXED ),
			                 ob,
							 Glue( 5, vp.ELASTIC ),
							 cb,
							 Glue( 5, vp.ELASTIC ),
							 self.help_button,
							 Glue( 5, vp.FIXED )]} )


	def CreateWorkArea( self ):  # -> work_area
		#
		# This method should be overridden by subclasses
		#
		return Canvas({ 'size': (300, 200) })


	def IsVisible( self ):
		if hasattr( self, 'window'):
			return self.window.IsVisible()
		else:
			return self.w.IsManaged()


	# ------------------------------------------------------------------
	# Event handler methods:

	def ExposeEH( self, target, void, xevent, void ):
		if hasattr( self,  'topbox' ):
			self.Subscribe( vp.RESIZE, self.ResizeEH, None )
		else:
			self.UnSubscribe( vp.EXPOSE )
		

	def ResizeEH( self, target, void, xevent, void ):
		#
		# This method should be used when there is a topbox to be resized
		#
		if not hasattr( self, 'topbox'):
			return

		if self.w.width != self.width.value or \
			  self.w.height != self.height.value:
			WidgetObject.ResizeEH( self, target, void, xevent, void )
			self.topbox.SetSize( self.width.value-1, self.height.value-1 )


	# ------------------------------------------------------------------
	# Geometry methods:

	def SetSize( self, w, h ):
		WidgetObject.SetSize( self, w, h )
		if hasattr( self,  'topbox' ):
			self.topbox.SetSize( w-1, h-1 )


	# ------------------------------------------------------------------
	# Dialog management methods:

	def Post( self, message, OkCB, CancelCB, HelpCB ):
		dialog = self.GetDialog()
		dialog.SetMessage( message )
		dialog.SetOkCB( OkCB )
		dialog.SetCancelCB( CancelCB )
		dialog.SetHelpCB( HelpCB )
		dialog.Show()


	def GetDialog( self ):
		#
		# If there is a cached dialog and it is not in use, then return it.
		#
		if self.IsRealized() and not self.IsVisible():
			if debug:
				print 'Dialog.GetDialog: returning a cached dialog' 
			return self

		if not self.IsRealized():
			if debug:
				print 'Dialog.GetDialog: creating a new dialog'
			self.CreateWidget()
			return self

		else:
			#
			# Create a temporary dialog. Install callbacks to destroy it when
			# the user pops it down (only 1 dialog is cached in this 
			# implementation, but this could be altered if deemed useful)
			#
			if debug:
				print 'DialogManager.GetDialog: creating a temp dialog'
			tmp_dialog = self.Clone()   
			tmp_dialog.CreateWidget()
			tmp_dialog.SetTmpDialog()
			
		return tmp_dialog


	# ------------------------------------------------------------------
	# Misc. 'public' methods

	def SetMessage( self, message ):
		if hasattr( self, 'window'):
			self.window.SetMessage( message )

		elif self.w.IsSubclass( Xm.MessageBox ):
			self.w.messageString = message


	def SetTmpDialog( self ):
		#
		# This means that the dialog should destroy itself when popped-down
		#
		self.tmp_dialog = TRUE


	def SetOkCB( self, ok_cb ):
		self.ok_cb = ok_cb


	def SetCancelCB( self, cancel_cb ):
		self.cancel_cb = cancel_cb


	def SetHelpCB( self, help_cb ):
		#
		# If there is no help callback, then unmanage the corresponding button.
		#
		if help_cb == None:
			if self.w.IsSubclass( Xm.MessageBox ):
				hb = self.w.MessageBoxGetChild( Xmd.DIALOG_HELP_BUTTON )
				hb.UnmanageChild()

			elif self.w.IsSubclass( Xm.FileSelectionBox ):
				hb = self.w.FileSelectionBoxGetChild( 
					                              Xmd.DIALOG_HELP_BUTTON )
				hb.UnmanageChild()

			elif hasattr( self, 'help_button'):
				self.help_button.Hide()

		else:
			self.help_cb = help_cb
		

	# ------------------------------------------------------------------
	# 'Draw' methods:


	def Show(self):         
		if hasattr( self, 'window'):
			self.window.Show()
		else:
			if self.w and not self.w.IsManaged():
				self.w.ManageChild()


	def Hide(self):         
		if hasattr( self, 'window'):
			self.window.Hide()
		else:
			if self.w and self.w.IsManaged():
				self.w.UnmanageChild()

		if self.tmp_dialog:
			del self


	# ------------------------------------------------------------------
	# Callbacks methods; the w..CB() methods are used directly with the 
	# widgets while the others are used when PushButton-s are used.

	def OkCB( self, button ):
		if debug:
			print 'Dialog.OkCB called'
		self.Hide()

		if self.ok_cb:
			self.ok_cb( self )

	def wOkCB( self, w, client_data, call_data ):
		self.OkCB( None )


	def CancelCB( self, button ):
		if debug:
			print 'Dialog.CancelCB called'
		self.Hide()

		if self.cancel_cb:
			self.cancel_cb( self )

	def wCancelCB( self, w, client_data, call_data ):
		self.CancelCB( None )


	def HelpCB( self, button ):
		if debug:
			print 'Dialog.HelpCB called'
		if self.help_cb:
			self.help_cb( self )

	def wHelpCB( self, w, client_data, call_data ):
		self.HelpCB( None )

