# --------------------------------------------------------------------------
# MODULE:      ClassBrowser
#
# DESCRIPTION:
#     Contains some classes used to make a classbrowser application. The
#     classes are: ClassInfo, ClassDatabase, ClassBrowser
#
# AUTHOR:
#     Per Spilling, CWI, Amsterdam, per@cwi.nl
#

import sys, regex, Xmd, posix, posixpath
from newdir import is_function

import vp, ModelView

from TypeInfo         import IsClass
from Object           import Object
from TreeNode         import TreeNode
from OrnatedTreeNode  import OrnatedTreeNode
from Window           import Window, ScrolledWindow
from TextControls     import TextList, CheckList
from Frame            import Frame
from Box              import Box, EqualSizeBox, Glue
from Text             import Text
from ModelView        import EditorModel
from TreeMV           import TreeNodeModel, TreeNodeView, TreeEditorView
from WidgetButton     import PushButton
from MiscGraphic      import Label
from Menu             import PulldownMenu

debug = vp.FALSE

# --------------------------------------------------------------------------
# Some info on python internals:  PSRC = ~guido/src/python
# 
# function object attributes: (see ($PSRC)/Objects/funcobject.[ch])
#    func_code     - pointer to the code object
#    func_globals  - ??
#    func_name     - symbolic name
#
# code object attributes: (see $(PSRC)/Python/compile.[ch])
#    co_code       - instruction opcodes
#    co_consts     - list of immutable constant objects
#    co_names      - list of name used
#    co_filename   - the filename from which it was 'compiled'
#    co_name       - the name of the object
#
# The 'codehack' module can be used to get the line number from a code object:
#    getlineno(co)
#    
# class object attributes: 
#    __bases__     - a tuple of class objects
#    __dict__      - a dictionary with class attributes 
#    __name__      - string


# -------------------------------------------------------------------------
# Utility functions:

def IsPythonfile( fname ):
	if len(fname) > 3 and fname[len(fname)-3:] == '.py':
		return vp.TRUE
	else:
		return vp.FALSE


# -------------------------------------------------------------------------
# CLASS:         ClassInfo
#
# INHERITS FROM: TreeNodeModel : Model : Object
#
# DESCRIPTION:   
#     ClassInfo uses relations of OrnatedTreeNode in following way. When class
#     S is de the first superclass of A then A is a child of S. When T is a
#     superclass but not the first, there is an ornament link from T to A.
#

class ClassInfo( TreeNodeModel ):

	# ------------------------------------------------------------------
	# Initialization

	def __init__( self, class_obj, class_dbase ):
		self.class_obj   = class_obj
		self.class_dbase = class_dbase
		self.methods     = {}            # '<method-name>':<func-obj>
		self.ivars       = []            # instance variables
		self.did_init    = vp.FALSE

		if self.IsPseudoRoot():
			self.filename = 'root'
			self.name     = 'root'
		else:
			self.filename = self.GetFilenameFromClassObj( class_obj )
			self.name     = class_obj.__name__

		if debug: print 'adding', self.GetNameOfClass()

		TreeNodeModel.__init__(self, class_dbase, self.GetNameOfClass())
		if self.IsPseudoRoot():
			self.GetView( 'Graphical' ).Hide()


	def Initialize( self ):
		#
		# Determine the superclasses and incorporate them into the Tree
		# structure.
		#
		if self.IsPseudoRoot() or self.did_init == vp.TRUE:
			return

		bases = self.class_obj.__bases__
		if len(bases) > 0:					 # Primary superclass.
			superclass = self.class_dbase.GetClassInfo( bases[0] )
		else:
			superclass = self.class_dbase.GetRootClassInfo()

		superclass.AddChild( self )

		if len(bases) > 1:					 # Secondary superclasses.
			for base in bases[1:]:
				superclass = self.class_dbase.GetClassInfo( base )
				superclass.AddLink( self )
		#
		# Store methods names etc.
		# 
		mdict = self.class_obj.__dict__
		for entry in mdict.keys():
			#
			# Check if the entry is a function object or an instance var
			#
			if is_function( mdict[entry] ):
				if not entry in self.methods.keys():
					self.methods[entry] = mdict[entry]
			elif not entry in self.ivars:
				self.ivars.append( entry )

		self.did_init = vp.TRUE


	# ------------------------------------------------------------------
	# Public access methods:

	def GetBaseClassNames( self ):
		return self.class_dbase.GetClassInfoNames( self.GetAllSuperClasses() )


	def GetMethodNames( self ): return self.methods.keys()

	def GetSubClasses(self): return self.children + self.outlinks

	def GetPrimarySubClasses( self ): return self.children

	def GetSecondarySubClasses( self ): return self.outlinks


	def GetImmediateSuperClasses( self ):
		if self.parent != None and not self.parent.IsPseudoRoot():
			supers = [self.parent] + self.inlinks
		else:
			supers = []
		return supers


	def GetAllSuperClasses( self ):
		#
		# Duplicate classes caused by multiple inherited are eliminated.
		#
		total  = []
		supers = self.GetImmediateSuperClasses()
		for super in supers:
			if not super in total:
				total.append(super)
				so_supers = super.GetAllSuperClasses()
				for so_super in so_supers:
					if not so_super in total:
						total.append(so_super)
		return total		


	def GetNameOfClass(self):
		#
		# Get the name of the represented class.
		# 
		if self.IsPseudoRoot():
			return '__root__'
		else:
			return self.class_obj.__name__


	def IsPseudoRoot( self ):
		#
		# An artificial class is added as the root of all classes. This is
		# necessary for the hierarchy-view.
		# 
		return self.class_obj == None

	# ------------------------------------------------------------------
	# Private access methods:

	def GetFilenameFromClassObj( self, co ):
		fname = ''
		if co != None:
			for key in co.__dict__.keys():
				if is_function( co.__dict__[key] ):
					fname = co.__dict__[key].func_code.co_filename
					break

		return fname


	def GetMethodLocation( self, method_name ): # -> (filename,lineno)
		import codehack
		func_obj = self.methods[method_name]
		lineno   = codehack.getlineno( func_obj.func_code )
		return (self.filename, lineno)


# -------------------------------------------------------------------------
# Add view class to ClassInfo

ModelView.AddViewClass( ClassInfo, 'Graphical', TreeNodeView )


# -------------------------------------------------------------------------
# CLASS:         ClassDatabase
#
# INHERITS FROM: Object
#
# DESCRIPTION: 
#

class ClassDatabase( EditorModel ):

	# ------------------------------------------------------------------
	# Initialization

	def __init__( self, browser, files ):
		EditorModel.__init__( self )
		self.SetImmediateMode( vp.FALSE )
		self.browser      = browser
		self.dbase        = {}

		# Create a artificial root for the imported class structure.

		self.root_class   = ClassInfo( None, self )
		self.Load(self.root_class)

		if files != None: 
			if debug: print 'Load files:', files
			self.BuildDatabase( files )

		self.SetImmediateMode( vp.TRUE )
	

	def BuildDatabase( self, files ):
		for fname in files:
			if IsPythonfile( fname ): self._CreateClassInformation( fname )
		self._InitClassInformation()


	def _CreateClassInformation( self, fname ):
		#
		# Using Python itself to extract type information from the modules.
		# NB! This will create problems if 'active' modules are imported.
		#
		# Ideally one should use a Python parser which would not execute
		# anything.
		# 
		
		(path, modname) = self.GetModuleFromFilename( fname )
		if debug: print 'path =', path, 'modname =', modname

		if path != '' and path not in sys.path: sys.path.append( path )
		exec( 'import ' + modname )
		exec( 'module = ' + modname )
		mdict = module.__dict__

		for key in mdict.keys():
			if IsClass( mdict[key] ) and key not in self.dbase.keys():
				self.dbase[key] = ClassInfo( mdict[key], self )


	def _InitClassInformation( self ):
		for class_name in self.dbase.keys(): 
			self.dbase[class_name].Initialize()


	# ------------------------------------------------------------------
	# Private methods:

	def GetModuleFromFilename( self, fname ): 
		(path, fname) = posixpath.split(fname)
		if not hasattr( self, 'rxdot' ):
			self.rxdot = regex.compile( '\.' )
		return (path, fname[:self.rxdot.search(fname)])
			

	# ------------------------------------------------------------------
	# Public access methods:

	def GetRootClassInfo( self ): return self.root_class

	def GetClassInfo( self, data ):
		if type(data) == type('') and self.dbase.has_key(data):   # classname
			return self.dbase[data]

		elif IsClass( data ):                                     # classobject
			if not self.dbase.has_key( data.__name__ ):   
				self.dbase[data.__name__] = ClassInfo(data, self) # new entry
				self.dbase[data.__name__].Initialize()

			return self.dbase[data.__name__]

		else:
			return None


	def GetClassNames( self ): return self.dbase.keys()


	def GetClassInfoNames( self, class_infos ):
		names = []
		for class_info in class_infos:
			names.append(class_info.GetNameOfClass())
		return names


	def GetBaseClassNames( self, class_name ):
		if class_name in self.dbase.keys():
			return self.dbase[class_name].GetBaseClassNames()


	def GetMethodNames( self, class_name ):
		if class_name in self.dbase.keys():
			return self.dbase[class_name].GetMethodNames()


	def GetMethodLocation( self, class_name, method_name ):
		if class_name in self.dbase.keys():
			return self.dbase[class_name].GetMethodLocation( method_name )

	# ------------------------------------------------------------------
	# Callbacks.

	def ShowInDBaseCB( self, node ):
		self.browser.SetCurrentClass( node.GetNameOfClass() )
	

	def HideNodeCB( self, node ): node.view.Hide()


# -------------------------------------------------------------------------
# Add view class to ClassDatabase

ModelView.AddViewClass( ClassDatabase, 'Graphical', TreeEditorView )

		
# -------------------------------------------------------------------------
# CLASS:         ClassBrowser
#
# INHERITS FROM: Object
#
# DESCRIPTION: 
#

class ClassBrowser( Object ):

	# ------------------------------------------------------------------
	# Initialisation.

	def __init__( self, files ):
		#
		# code_viewer = vp.CODE_GNUCLIENT, vp.CODE_WINDOW or vp.CODE_NONE
		#
		self.code_update = vp.TRUE
		self.show_code   = None
		self.curr_class  = None   # the currently selected class
		self.curr_method = None   # the currently selected method
		self.curr_bases  = []     # the currently selected base classes
		self.ml_maxlen   = 0
		self.cwin        = None
		self.hwin        = None
		self.class_dbase = ClassDatabase( self, files )
		self.CreateDBaseWindow()


	def CreateDBaseWindow( self ):
		self.dwin = Window()
		self.dwin.AddWorkArea( self._CreateWorkArea() )

		fm = PulldownMenu({
			'title' : 'File', 
			'items': [
				('Load files...',      self.LoadFilesCB ),
				('Print hierarchy...', self.PrintHierCB),
				('Quit',               self.QuitCB)
				]
			})
		vm = PulldownMenu({
			'title' : 'Views', 
			'items': [
				('Emacs codeview',    self.ShowCodeEmacsCB),
				('Separate codeview', self.ShowCodeWindowCB),
				('No codeview',       self.ShowCodeNoneCB),
				('Show hierarchy...', self.ShowHierarchyCB)
				]
			})
		self.dwin.AddMenu( fm )
		self.dwin.AddMenu( vm )
		self.ShowCode( vp.CODE_NONE )

		print 'ready...'


	def _CreateWorkArea( self ):  # -> work-area
		#
		# define the class-, inheritance-, and method lists
		#
		list = self.class_dbase.GetClassNames()
		self.class_list = TextList({
			'name'    : 'Class list', 
			'list'    : list,
			'callback': self.ClassSelectedCB,
			'size'    : (200,300)
			})
		self.inherit_list = CheckList({
			'name'    : 'Inheritance list',
			'order'   : vp.UNSORTED,
			'callback': self.BaseSelectedCB,
			'size'    : (200,100)
			})
		self.method_list = TextList({
			'name'    : 'Method list',
			'callback': self.MethodSelectedCB,
			'size'    : (300,300)
			})

		# wrap the class list and the inheritance list together in a box and
		# put the method list in a separate box (Frame)

		cibox = Box({
			'alignment'     : vp.VCENTER,
			'stretchability': (vp.FIXED, vp.ELASTIC),
			'child_list'    : [
				Frame({ 
					'label': 'Class list:',
					'child': self.class_list 
					}),
				Frame({ 
					'label': 'Inheritance list:',
					'child': self.inherit_list 
					}),
				Glue( 7, vp.FIXED )
				]
			})
		mbox = Box({
			'alignment' : vp.VCENTER,
			'child_list': [
				Frame({
					'label': 'Method list:',
					'child': self.method_list
					}),
				Glue( 7, vp.FIXED )
				]
			})

		# wrap everthing together in the topbox

		return Box({
			'child_list': [
				Glue( 8,vp.FIXED ),
				cibox, 
				Glue( 8,vp.FIXED ),
				mbox, 
				Glue( 7,vp.FIXED )
				] 
			})


	def CreateCodeWindow( self ):
		self.hiliword    = None
		self.cpos        = 0
		self.cwin = Window({ 
			'name'       : 'CodeView',
			'size'       : (500, 300),
			'window_type': vp.DIALOG_WINDOW
			})
		self.cwin.AddCommandArea( Box({
			'child_list': [
				Glue( 10, vp.ELASTIC ),
				PushButton({ 
					'name'    : 'Dismiss', 
					'callback': self.HideCodeWinCB
					}),
				Glue( 10, vp.ELASTIC )
				]
			}))
		self.code_view = Text()      # scrollable text 24x80
		self.cwin.AddWorkArea( self.code_view )


	def CreateHierarchyWindow( self ):
		self.hwin  = ScrolledWindow({
			'name'       : 'ClassHierarchy',
			'size'       : (500, 500),
			'window_type': vp.DIALOG_WINDOW
			})
		b = Box({
			'child_list': [
				Glue( 10, vp.ELASTIC ),
				PushButton({ 
					'name'    : 'Dismiss', 
					'callback': self.HideHierarchyWinCB
					}),
				Glue( 10, vp.ELASTIC )
				]
			})
		self.hwin.AddCommandArea( b )

		view = self.class_dbase.GetView('Graphical')
		view.SetWindow( self.hwin )
		view.Realize()


	# ------------------------------------------------------------------
	# Callbacks

	def HideHierarchyWinCB( self, button ): self.hwin.Hide()
	def HideCodeWinCB( self, button ): self.cwin.Hide()
	def QuitCB( self, button ): vp.theQuitCommand.Execute()

	def PrintHierCB( self, button ):
		m = 'The hierarchy will be printed on stdout. Go on?'
		vp.theQuestionDialog.Post( m, self._PrintHierCB, None, None )


	def _PrintHierCB( self, button ):
		root = self.class_dbase.GetRootClassInfo()
		if root != None: root.PrintTree()


	def LoadFilesCB( self, button ): 
		if debug: print 'LoadFilesCB called'

		vp.theFileSelectionDialog.SetPattern( '*.py' )
		vp.theFileSelectionDialog.Post( '', self._LoadFilesCB, None, None )


	def _LoadFilesCB( self, file_dialog ):
		fname = file_dialog.GetFilename()
		if IsPythonfile( fname ): 
			#
			# A single file was selected
			#
			self.class_dbase.BuildDatabase( [fname] )
			self.class_list.SetList( self.class_dbase.GetClassNames() )

		else:
			#
			# Use the filename filter to select files
			#
			filter = file_dialog.GetFilenameFilter()
			(dir,pattern) = posixpath.split(filter)
			cwd = posix.getcwd()
			posix.chdir(dir)
			file = posix.popen( 'ls ' + pattern, 'r' )  # ls command
			posix.chdir(cwd)
			fnames = []
			for fname in file.readlines():
				fnames.append(fname[:len(fname)-1])     # strip null characters

			self.class_dbase.BuildDatabase( fnames )
			self.class_list.SetList( self.class_dbase.GetClassNames() )


	def ShowHierarchyCB( self, button ):
		if debug: print 'ShowHierarchyCB called'
		if self.hwin == None:
			vp.theQuestionDialog.Post( 
				  'This will take a while (maybe minutes). Go on?',
				  self._CreateAndShowHierarchyCB, None, None )
		else:
			self.hwin.Show()


	def _CreateAndShowHierarchyCB( self, button ): 
		self.CreateHierarchyWindow()
		self.hwin.Show()


	def ShowCodeNoneCB( self, button ):   self.ShowCode( vp.CODE_NONE )
	def ShowCodeEmacsCB( self, button ):  self.ShowCode( vp.CODE_GNUCLIENT )
	def ShowCodeWindowCB( self, button ): self.ShowCode( vp.CODE_WINDOW)


	def HelpCB( self, button ):
		if debug: print 'HelpCB called'

		if not hasattr( self, 'help_message' ): 
			  self.help_message = 'Help on commands:' 
		vp.theInfoDialog.Post( self.help_message, None, None, None )
		

	def BaseSelectedCB( self, check_list ):
		self.curr_bases = check_list.GetSelection()
		self.method_list.SetList( self._GetMethodList() )


	def ClassSelectedCB( self, text_list ):
		curr_class = text_list.GetSelection()
		self.SetCurrentClass( curr_class )


	def MethodSelectedCB( self, text_list ):
		if self.show_code == vp.CODE_WINDOW:
			if self.hiliword != None and self.cpos != -1:
				self.code_view.SetHighlight( self.cpos, 
				                             self.cpos+len(self.hiliword),
											 Xmd.HIGHLIGHT_NORMAL )

		(mn,cn) = self._GetMethodAndClassNames(text_list.GetSelection()) 
		self.curr_method = mn
		(fname,lineno) = self.class_dbase.GetMethodLocation(cn, mn)

		if fname != None:
			self._LoadFile( fname, lineno )
		else:
			print 'ClassBrowser: source file not found!'


	# ------------------------------------------------------------------
	# Private methods:

	def ShowCode( self, show_code ):
		if show_code > vp.CODE_WINDOW: show_code = vp.CODE_NONE

		if show_code == vp.CODE_WINDOW:
			if self.cwin == None: self.CreateCodeWindow()
			self.cwin.Show()
		else:
			if self.cwin != None: self.cwin.Hide()

		self.show_code = show_code


	def SetCurrentClass( self, curr_class ):
		if self.show_code == vp.CODE_WINDOW:
			#
			# unhilight previous hilighted word
			#
			if self.hiliword != None and self.cpos != -1:
				self.code_view.SetHighlight( self.cpos, 
					  self.cpos+len(self.hiliword), Xmd.HIGHLIGHT_NORMAL )

		# list the methods and the base classes

		self.curr_class = curr_class
		self.ml_maxlen  = 0 
		self.curr_bases = [curr_class]
		self.method_list.SetList( self._GetMethodList() )
		self.inherit_list.SetList( self._GetInheritList() )
		self.inherit_list.SetSelection( curr_class )
		curr_classinfo = self.class_dbase.GetClassInfo( curr_class )
		if  curr_classinfo.filename != None:
			self._LoadFile( curr_classinfo.filename, None )
		else:
			print 'ClassBrowser: source file not found!'
			

	def _LoadFile( self, fname, lineno ):
		if self.show_code ==  vp.CODE_WINDOW:
			self._LoadFileInCodeWindow( fname, lineno )
		elif self.show_code ==  vp.CODE_GNUCLIENT:
			self._LoadFileInEmacs( fname, lineno )		


	def _LoadFileInEmacs( self, fname, lineno ):
		prog = 'gnuclient'
		if lineno != None:
			args = ' -q ' + '+' + `lineno` + ' ' + fname
		else:
			#
			# search the file to get the line number
			#
			file = open( fname, 'r' )
			rxword = regex.compile('^class +' + '\<' + self.curr_class + '\>')

			lineno = 1
			found  = vp.FALSE
			line   = file.readline()
			while line != '':
				pos = rxword.search( line )
				if pos != -1:
					found = vp.TRUE
					break
				else:
					lineno = lineno + 1
					line = file.readline()
			file.close()
				
			if found:
				args = ' -q ' + '+' + `lineno` + ' ' + fname
			else:
				args = ' -q ' + ' ' + fname
		
		void = posix.system( prog + args )
			

	def _LoadFileInCodeWindow( self, fname, lineno ):
		#
		# load the file and hilight the class name
		#
		self.code_view.LoadFile( fname )
		if lineno == None:
			rxword = regex.compile('^class +' + '\<' + self.curr_class + '\>')
			pos    = rxword.search( self.code_view.GetString() )
		
			if pos != -1:
				self.cpos = pos + len('class ')
				self.hiliword = self.curr_class
				self.code_view.SetTopCharacter( pos )
				self.code_view.SetHighlight( self.cpos, 
				                             self.cpos+len(self.hiliword),
											 Xmd.HIGHLIGHT_SELECTED )
			else:
				self.cpos = -1
		else:
			self.code_view.ScrollTo( lineno )
			curr_line = self.code_view.GetLine( lineno )
			rxword    = regex.compile( self.curr_method )
			pos       = rxword.search( curr_line )

			if pos != -1:
				self.cpos = self.code_view.GetTopCharacter() + pos
				self.hiliword = self.curr_method
				self.code_view.SetHighlight( self.cpos, 
				                             self.cpos+len(self.hiliword),
											 Xmd.HIGHLIGHT_SELECTED )
			else:
				self.cpos = -1
			

	def _GetMethodList( self ):
		#
		# Get the method list of the class and its base classes. Format the 
		# list so that every entry has the same width (number of chars) and
		# with the method name placed in front of the class name. The methods 
		# of the base classes can be turned on and off.
		# 
		if self.ml_maxlen == 0:
			#
			# New list; determine the maxlength of of any methodname.
			#
			for word in self.class_dbase.GetMethodNames( self.curr_class ):
				self.ml_maxlen = max( [self.ml_maxlen,len(word)] )

			for cl_name in self.class_dbase.GetBaseClassNames(self.curr_class):
				list = self.class_dbase.GetMethodNames( cl_name )
				if list != None:
					for word in list:
						self.ml_maxlen = max( [self.ml_maxlen,len(word)] )
				else:
					print 'ClassBrowser: the method list of', cl_name, \
						  'is not available'

		# format the method list; the 'curr_bases' 

		mlist = []
		for cl_name in self.curr_bases:
			list = self.class_dbase.GetMethodNames( cl_name )
			if list != None:
				for word in list:
					padding = (self.ml_maxlen - len(word)) * ' '
					name = word + padding + '    ' + cl_name
					if name not in mlist:
						mlist.append( name )

		return mlist


	def _GetInheritList( self ):
		list = self.class_dbase.GetBaseClassNames( self.curr_class ) 
		list.insert( 0, self.curr_class )
		return list


	def _GetMethodAndClassNames( self, string ):
		#
		# Used to extract the method- and class name from the method list
		#
		i  = 0
		while string[i] != ' ': i  = i + 1
		mn = string[0:i]
		while string[i] == ' ': i = i + 1
		cn = string[i:]
		return (mn,cn)


	def _GetNameOfClass( self, string ):
		#
		# Used to extract the class name from the inheritance list
		#
		i  = 0
		while string[i] == ' ' or string[i] == '*': i  = i + 1
		return string[i:]
		

