# -------------------------------------------------------------------------
# MODULE:      Box
#
# DESCRIPTION: 
#     Contains the Box and Glue classes.
#
# AUTHOR:
#     Per Spilling, CWI, Amsterdam, per@cwi.nl

import X, Xt, Xm

import vp, EventNotifier

from vp            import TRUE, FALSE, HTOP, HCENTER, HBOTTOM, \
                          VLEFT, VCENTER, VRIGHT, ELASTIC, FIXED, \
                          HORIZONTAL, VERTICAL
from Graphic       import Graphic
from WidgetObject  import WidgetObject
from Canvas        import Canvas
from WidgetButton  import PushButton

debug = FALSE

# -------------------------------------------------------------------------
# CLASS:         Box
#
# INHERITS FROM: Canvas : (Composite,WidgetObject) : Graphic : \
#                (DatumDict, TreeNode) : Object
#
# DESCRIPTION: 
#     A layout class modelled after TeX's boxes and glue model. Children
#     can be layed out vertically or horisontally. The Box will determine the
#     size it needs based on the natural sizes of the children. If there is
#     extra space, then the Box will attempt to fill this space with those
#     children that are stretchable.
#

class Box( Canvas ):
	
	# ------------------------------------------------------------------
	# Init methods

	def __init__( self, argdict = {} ):
		#
		# Args can be:
		# - 'alignment' : [HTOP|HCENTER=default|HBOTTOM|VLEFT|VCENTER|VRIGHT]
		# - 'child_list': <python list of child instances>
		#
		self.child_list    = []
		self.alignment     = HCENTER
		Canvas.__init__( self, argdict )

		# private attributes

		self.ns_count           = 0      # number of non-stretchable children
		self.ns_height          = 0      # total non-stretchable height
		self.ns_width           = 0      # total non-stretchable width
		self.doing_layout       = FALSE
		self.do_layout_update   = TRUE
		self.did_calculate_size = FALSE
		self.did_initial_layout = FALSE

		for child in self.child_list:
			Canvas.AddChild( self, child )
		self.child_list = None


	def Realize( self ):
		Canvas.Realize( self )
		if not self.parent.IsA( Box ) and len(self.children) > 0: 
			self.DoLayout()


	def Finalize( self ):
		self.do_layout_update = FALSE
		Canvas.Finalize( self )

	# ------------------------------------------------------------------
	# Event handler methods:

	def ExposeEH( self, target, void, xevent, e ):
		if self.Redraw == self._RedrawWithControlPoints:
			self.Redraw = self._Redraw
			Canvas.ExposeEH( self, target, void, xevent, e )
			self.Redraw = self._RedrawWithControlPoints
		else:
			Canvas.ExposeEH( self, target, void, xevent, e )


	def ResizeEH( self, target, client_data, xevent, e ):
		#
		# This method is called when the window is created or resized 
		#
		if debug: 
			print 'Box.ResizeEH: old_sz =', self.GetSize(), \
				  'new sz =', self.w.width, self.w.height

		if self.w.width != self.width.value or \
			  self.w.height != self.height.value:
			Canvas.ResizeEH( self, target, client_data, xevent, e )
			self.DoLayout()


	def MouseDownEH( self, target, client_data, xevent, e ):
		Canvas.MouseDownEH( self, target, client_data, xevent, e )

		if self.show_cntrl_pnts and EventNotifier.event_focus == self:
			self._osize = self.GetSize()


	def MouseUpEH( self, target, client_data, xevent, e ):
		#
		# Must redo the layout if the size has changed
		#
		Canvas.MouseUpEH( self, target, client_data, xevent, e )

		if hasattr( self, '_osize' ) and self.show_cntrl_pnts and \
		   (self._osize[0] != self.width.value or \
			self._osize[1] != self.height.value):
			self.did_calculate_size = FALSE
			self.DoLayout()


	# ------------------------------------------------------------------
	#  Geometry methods

	def SetSize( self, width, height ):
		if debug:
			print 'Box.SetSize: old size =', self.GetSize(), \
				  'new size =', width, height

		if width != self.width.value or height != self.height.value:
			Canvas.SetSize( self, width, height )

			if self.IsRealized() and \
			   not self.parent.IsA( Box ) and len(self.children) > 0:
				self.DoLayout()
				self.did_initial_layout = TRUE


	# ------------------------------------------------------------------
	#  Child methods
		
	def InitAddedNode( self, child ):
		Canvas.InitAddedNode( self, child )

		if self.draggable == TRUE:
			#
			# Redirect drag-events to 'self'
			#
			child.Subscribe( vp.MOUSE_DOWN, self.MouseDownEH, None )
			child.Subscribe( vp.MOUSE2_DRAG, self.Mouse2DragEH, None )
			child.Subscribe( vp.MOUSE_UP, self.MouseUpEH, None )

		if self.IsRealized() and self.do_layout_update:
			self.did_calculate_size = FALSE
			self.DoLayout()


	def RemoveChild( self, child ):
		Canvas.RemoveChild( self, child )
		if self.IsRealized() and self.do_layout_update:
			self.DoLayout()


	# ------------------------------------------------------------------
	# Modifying methods

	def SetAlignment( self, new_alignment ):
		#
		# Check that it is an allowable alignment
		#
		if new_alignment < HCENTER or new_alignment > VRIGHT:
			print 'Box: bad alignment index!'
		else:
			self.alignment = new_alignment
			if self.IsRealized():
				self.DoLayout()


	# ------------------------------------------------------------------
	# 'private' layout methods
		
	def ChildSizeChangedCB( self, child ):
		self.did_calculate_size = FALSE

		if hasattr( self.parent, 'ChildSizeChangedCB' ):
			self.parent.ChildSizeChangedCB( self )
		else:
			self.DoLayout()

			
	def DoLayout( self ):
		#
		# This should be performed by the top-box
		#
		if not self.IsRealized(): 
			return
		elif self.parent.IsA( Box ):
			self.parent.DoLayout()
			return

		if not self.doing_layout:       # to avoid recursion
			self.doing_layout = TRUE
		else:
			return
		
		self.RedrawOff()
		if self.show_cntrl_pnts: self.parent.RedrawOff()

		(w,h) = self.GetNaturalSize()

		if w > self.width.value: Canvas.SetSize( self, w, self.height.value )
		if h > self.height.value: Canvas.SetSize( self, self.width.value, h )

		if debug: print 'Box.DoLayout(',self.GetName(),'): size needed =',w,h

		self.AlignChildren()
		self.doing_layout = FALSE

		self.RedrawOn()
		if self.show_cntrl_pnts: self.parent.RedrawOn()


	def AlignChildren( self ):
		self.w.UnmanageChild()          
		if self.alignment < VCENTER:
			self.SetHSizes()
			self.AlignHorizontal()
		else:
			self.SetVSizes()
			self.AlignVertical()

		# all children which are Boxes should also align their children

		for child in self.children:
			if child.IsA( Box ): child.AlignChildren()

		self.w.ManageChild()


	def CalculateSizeNeeded( self ):
		if self.alignment < VCENTER:
			self.natural_size = self.CalculateHSizeNeeded()
		else:
			self.natural_size = self.CalculateVSizeNeeded()

		(w,h) = self.natural_size
		bw    = 2*self.w.borderWidth
		self.natural_size = (w + bw, h + bw)

		self.did_calculate_size = TRUE


	def CalculateHSizeNeeded( self ):
		self.ns_count = 0
		self.ns_width = 0
		tot_w         = 0
		tot_h         = 0

		for child in self.children:
			(w,h) = child.GetNaturalSize()
			if debug:
				print 'Box.CalculateHSizeNeeded: ns of', child.GetName(), \
					  '=', w, h
			tot_w = tot_w + w
			tot_h = max( tot_h, h )

			if not child.IsElastic( HORIZONTAL ):
				self.ns_count = self.ns_count + 1
				self.ns_width = self.ns_width + w

		if debug:
			print 'Box.CalculateHSizeNeeded: ns_count =', self.ns_count, \
				  'ns_width =', self.ns_width, \
				  'tot size =', tot_w, tot_h

		return (tot_w, tot_h)


	def CalculateVSizeNeeded( self ):
		self.ns_count  = 0
		self.ns_height = 0
		tot_w          = 0
		tot_h          = 0

		for child in self.children:
			(w,h) = child.GetNaturalSize()
			tot_w = max( tot_w, w )
			tot_h = tot_h + h

			if not child.IsElastic( VERTICAL ):
				self.ns_count  = self.ns_count  + 1
				self.ns_height = self.ns_height + h
				
		return (tot_w, tot_h)


	def SetHSizes( self ):
		#
		# First determine the amount of stretch-width, and then set the 
		# new sizes
		#
		if self.ns_count != len(self.children):   
			tot_s_width = self.width.value - self.natural_size[0]
			s_width     = tot_s_width / (len(self.children) - self.ns_count)
		else:
			s_width = 0

		if debug: 
			print 'Box.SetHSizes: s_width =', s_width, \
				  'nw =', self.natural_size[0]

		for child in self.children:
			(hs, vs) = child.GetStretchability()
			(nw, nh) = child.GetNaturalSize()

			if hs == ELASTIC and vs == ELASTIC:
				child.SetSize( nw + s_width, self.height.value )
			elif vs == ELASTIC:
				child.SetSize( nw, self.height.value )
			elif hs == ELASTIC:
				child.SetSize( nw + s_width, nh )
			else:
				child.SetSize( nw, nh )

			if debug: 
				print 'Box.SetHSizes:', child.GetName(), 'nw =', nw, \
					  'new size =', child.GetSize()


	def SetVSizes( self ):
		#
		# First determine the amount of stretch-height, and then set the 
		# new sizes
		#
		if self.ns_count != len(self.children):
			tot_s_height = self.height.value - self.natural_size[1]
			s_height     = tot_s_height / (len(self.children) - self.ns_count)
		else:
			s_height = 0

		if debug: print 'Box.SetVSizes: s_height =', s_height

		for child in self.children:
			(hs, vs) = child.GetStretchability()
			(nw, nh) = child.GetNaturalSize()

			if hs == ELASTIC and vs == ELASTIC:  
				if debug:
					print 'Box.SetVSizes: child', child.GetName(), 'nh =', nh
				child.SetSize( self.width.value, nh + s_height )
			elif vs == ELASTIC:
				child.SetSize( nw, nh + s_height )
			elif hs == ELASTIC:
				child.SetSize( self.width.value, nh )
			else:
				child.SetSize( nw, nh )


	def AlignHorizontal( self ):
		delta_x = 0
		if self.alignment == HTOP:
			for child in self.children:
				child.SetPosition( delta_x, 0 )
				delta_x = delta_x + child.width.value

		elif self.alignment == HCENTER:
			sc = self.height.value/2
			for child in self.children:
				child.SetPosition( delta_x, sc - child.height.value/2 )
				delta_x = delta_x + child.width.value
		else:
			sb = self.height.value
			for child in self.children:
				child.SetPosition( delta_x, sb - child.height.value )
				delta_x = delta_x + child.width.value
				

	def AlignVertical( self ):
		delta_y = 0
		if self.alignment == VLEFT:
			for child in self.children:
				child.SetPosition( 0, delta_y )
				delta_y = delta_y + child.height.value
				
		elif self.alignment == VCENTER:
			sc = self.width.value/2
			for child in self.children:
				child.SetPosition( sc - child.width.value/2, delta_y )
				delta_y = delta_y + child.height.value

		else: # self.alignment == VRIGHT:
			sr = self.width.value
			for child in self.children:
				child.SetPosition( sr - child.width.value, delta_y )
				delta_y = delta_y + child.height.value


	def GetNaturalSize( self ):
		if not self.did_calculate_size:
			self.CalculateSizeNeeded()
		return self.natural_size


	def GetAlignment( self ):
		return self.alignment

		
# -------------------------------------------------------------------------
# CLASS:         EqualSizeBox
#
# INHERITS FROM: Box : Canvas : (Composite,WidgetObject) : Graphic : \
#                (DatumDict, TreeNode) : Object
#
# DESCRIPTION: 
#     A layout class like the Box class, the difference being that it will
#     make all its children equal in size to the largest child. 
#

class EqualSizeBox( Box ):

	def __init__( self, argdict = {} ):
		#
		# Args can be:
		# - 'alignment' : [HTOP|HCENTER=default|HBOTTOM|VLEFT|VCENTER|VRIGHT]
		# - 'child_list': <python list of child instances>
		#
		Box.__init__( self, argdict )

		# private attributes

		self.max_h       = 0        # max child natural height
		self.max_w       = 0        # max child natural width
		self.db_margin   = 0        # default button margin
		self.glue_w      = 0
		self.glue_h      = 0
		self.fill_count  = 0
		self.es_count    = 0        # equal size count


	def CalculateHSizeNeeded( self ):
		self.max_w      = 0
		self.max_h      = 0
		self.db_margin  = 0
		self.glue_w     = 0
		self.fill_count = 0
		self.es_count   = 0
		tot_w           = 0
		tot_h           = 0

		for child in self.children:
			(w,h) = child.GetNaturalSize()
			if not child.IsA( Glue ):
				self.es_count = self.es_count + 1

				if child.IsA( PushButton ) and \
					  child.w.showAsDefault == X.TRUE:
					self.db_margin = child.w.marginLeft
					self.max_w = max( self.max_w, w-2*self.db_margin )
					self.max_h = max( self.max_h, h-2*self.db_margin )
				else:
					self.max_w = max( self.max_w, w )
					self.max_h = max( self.max_h, h )
			else:
				self.glue_w   = self.glue_w + w
				tot_h         = max( tot_h, h )

				if not child.IsElastic( HORIZONTAL ):
					self.ns_count  = self.ns_count + 1
					self.ns_height = self.ns_width + w
				else:
					self.fill_count = self.fill_count + 1

		tot_h = max( tot_h, self.max_h ) + 2*self.db_margin
		tot_w = self.es_count*self.max_w + 2*self.db_margin + self.glue_w

		return (tot_w, tot_h)


	def CalculateVSizeNeeded( self ):
		self.max_w      = 0
		self.max_h      = 0
		self.db_margin  = 0
		self.glue_h     = 0
		self.fill_count = 0
		self.es_count   = 0 
		tot_w           = 0
		tot_h           = 0

		for child in self.children:
			(w,h) = child.GetNaturalSize()
			if not child.IsA( Glue ):
				self.es_count = self.es_count + 1

				if child.IsA( PushButton ) and child.w.showAsDefault == X.TRUE:
					self.db_margin = child.w.marginLeft
					self.max_w = max( self.max_w, w-2*self.db_margin )
					self.max_h = max( self.max_h, h-2*self.db_margin )
				else:
					self.max_w = max( self.max_w, w )
					self.max_h = max( self.max_h, h )
			else:
				self.glue_h   = self.glue_h + h
				tot_w         = max( tot_w, w )

				if not child.IsElastic( VERTICAL ):
					self.ns_count  = self.ns_count  + 1
					self.ns_height = self.ns_height + h
				else:
					self.fill_count = self.fill_count + 1

		tot_w = max( tot_w, self.max_w ) + 2*self.db_margin
		tot_h = self.es_count*self.max_h + 2*self.db_margin + self.glue_h

		return (tot_w, tot_h)

		
	def SetHSizes( self ):
		#
		# If there is elastic-glue then use it for filling, otherwise stretch
		# the "equal-size-children" to fill the available space.
		#
		if self.fill_count != 0:
			s_width = (self.width.value-self.natural_size[0]) / self.fill_count
			for child in self.children:
				if child.IsA( Glue ):
					(nw,nh)  = child.GetNaturalSize()
					if child.IsElastic( HORIZONTAL ):
						child.SetSize( nw + s_width, nh )
					else:
						child.SetSize( nw, nh )

				elif child.IsA( PushButton ) and \
					  child.w.showAsDefault == X.TRUE:
					w = self.max_w + 2*self.db_margin
					h = self.max_h + 2*self.db_margin
					child.SetSize( w, h )

				else:
					child.SetSize( self.max_w, self.max_h )
		else:
			s_width = (self.width.value-self.natural_size[0]) / self.es_count
			for child in self.children:
				if not child.IsA( Glue ):
					child.SetSize( s_width, self.height.value )


	def SetVSizes( self ):
		if self.fill_count != 0:
			s_height = (self.height.value-self.natural_size[1])/self.fill_count
			for child in self.children:
				if child.IsA( Glue ):
					(nw,nh)  = child.GetNaturalSize()
					if child.IsElastic( VERTICAL ):
						child.SetSize( nw, nh + s_height )
					else:
						child.SetSize( nw, nh )

				elif child.IsA( PushButton ) and \
					  child.w.showAsDefault == X.TRUE:
					w = self.max_w + 2*self.db_margin
					h = self.max_h + 2*self.db_margin
					child.SetSize( w, h )
					
				else:
					child.SetSize( self.max_w, self.max_h )
		else:
			s_height = (self.height.value-self.natural_size[1]) / self.es_count
			for child in self.children:
				if not child.IsA( Glue ):
					child.SetSize( self.width.value, s_height )



# -------------------------------------------------------------------------
# CLASS:         Glue
#
# INHERITS FROM: Graphic : TreeNode : Object
#
# DESCRIPTION: 
#     This is a class which does not create any widget. It can be used inside
#     layout views to control the distance between the visible objects.
#

class Glue( Graphic ):

	def __init__( self, natural_size, stretchable ):
		Graphic.__init__( self )
		self.SetNaturalSize( natural_size, natural_size )
		self.SetStretchability( stretchable, stretchable )
		self.realized = vp.TRUE


class HVGlue( Graphic ):
	
	def __init__( self, (ns_w, ns_h), (sh, sv) ):
		Graphic.__init__( self )
		self.SetNaturalSize( ns_w, ns_h )
		self.SetStretchability( sh, sv )
		self.realized = vp.TRUE
		

