# -------------------------------------------------------------------------
# MODULE:      Application
#
# DESCRIPTION: 
#     Contains the Application class.
#     
# USAGE:      
#     > from Application import Application
# 
# AUTHOR:
#     Per Spilling, CWI, Amsterdam, per@cwi.nl

import Xt, Xm, X, Xmd                    # Xt & Motif interface
import sys

import vp                       # global data for vpApp
from WidgetObject import WidgetObject

debug = vp.FALSE

# -------------------------------------------------------------------------
# CLASS:         Application
#
# INHERITS FROM: WidgetObject : TreeNode : Object
#
# DESCRIPTION: 
#     Only one instance of this class should exist in an application. This 
#     instance will be the root of all the top-level windows of a program 
#     and will be assigned to the global variable: "theApplication" (defined
#     in the vp module).
#

class Application( WidgetObject ):

	# ------------------------------------------------------------------
	# Initialization & destruction

	def __init__( self, appname='NoName' ):
		self.name     = appname
		WidgetObject.__init__( self )
		self.running  = vp.FALSE
		self.realized = vp.FALSE
		self.InitVP()


	def InitVP( self ):
		#
		# Init Xt and initialize global variables
		#
		self.CreateWidget()

		from MiscDialogs  import InfoDialog, QuestionDialog, \
			                     FileSelectionDialog
		from MiscCommands import UndoCommand, QuitCommand
		import Font
		Font.Init( self.w )

		vp.theApplication         = self
		vp.theInfoDialog          = InfoDialog()
		vp.theQuestionDialog      = QuestionDialog()
		vp.theFileSelectionDialog = FileSelectionDialog()
		vp.theUndoCommand         = UndoCommand()
		vp.theQuitCommand         = QuitCommand()

		fontname = '-*-courier-medium-r-normal-*-12-*-*-*-*-*-iso8859-1'
		vp.theDefaultFont = Font.FindFont( fontname ) 
										  
		if debug: print 'VP initialized'

		
	def CreateWidget( self ):
		#
		# Initialize Xt 
		#
		self.w = Xt.Initialize( self.name, [], [] )

		# Make the Application (toplevel) widget invisible (unmapped). This
		# is done so that all windows of an application are treated equally
		# as popup windows of the Application widget. The position of the
		# Application widget is also set because Motif will center a dialog
		# window over its parent.

		self.w.mappedWhenManaged = X.FALSE
		self.w.x      = 500      
		self.w.y      = 500
		self.w.width  = 1
		self.w.height = 1
		
		# Force the toplevel widget to exist so dialogs popped up from this
		# widget behave correctly

		self.w.RealizeWidget()
		self.realized = vp.TRUE

		# turn off drag-n-drop in Motif 1.2 since this creates problems for
		# vpApp drag and popup functionality.

		if Xm.Version >= 1002:
			xmdpy = self.w.GetDisplay( )
			xmdpy.dragInitiatorProtocolStyle = Xmd.DRAG_NONE
			xmdpy.dragReceiverProtocolStyle  = Xmd.DRAG_NONE


	def Quit( self ):
		#
		# Give the windows a chance to cleanup
		#
		if debug: print 'Application.__del__ called'

		for child in self.children:
			child.Finalize()

		sys.exit(0)


	# ------------------------------------------------------------------
	# "Draw" methods:

	def Show( self ):
		#
		# Show (popup) all registered windows of MAIN_WINDOW type; 
		# DIALOG_WINDOW-s must be popped-up explicitly.
		#
		from Window   import Window

		for win in self.children:
			if win.IsA( Window ) and win.window_type == vp.MAIN_WINDOW:
				win.Show()

		
	def Hide( self ):
		#
		# Hide (popdown) all registered windows
		#
		for w in self.children: 
			if w.w.IsManaged():
				w.Hide()
		

	# ------------------------------------------------------------------
	# Misc. public methods:

	def RemoveChild( self, child ):
		WidgetObject.RemoveChild( self, child )

		if len(self.children) == 0:
			self.Quit()


	def Run( self ):
		#
		# Make all registered windows visible (managed); then enter the
		# main event loop.
		#
		self.running = vp.TRUE
		self.Show()
		try:
			Xt.MainLoop()
		except SystemExit:
			if debug: print 'Application quitting'


	# ------------------------------------------------------------------
	# Access methods:

	def GetName( self ):
		return self.name


	# ------------------------------------------------------------------
	# Query method:

	def IsRunning( self ):
		return self.running


	def IsRealized( self ):
		return self.realized



		







