/*
 * Copyright (C) 1997-1998 by Dimitri van Heesch.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation under the terms of the GNU General Public License is hereby 
 * granted. No representations are made about the suitability of this software 
 * for any purpose. It is provided "as is" without express or implied warranty.
 * See the GNU General Public License for more details.
 *
 * This file is part of QdbtTabular 0.31.
 */

#include <stdio.h>
#include <qtooltip.h>
#include <qkeycode.h>
#include <qscrbar.h>

#include "qdbtlined.h"
#include "qdbttabular.h"
#include "qdbttable.h"


//-----------------------------------------------------------------------------

class QdbtDynaTip : public QToolTip
{
  public:
    QdbtDynaTip(QWidget *parent);
    virtual ~QdbtDynaTip();

  protected:
    void maybeTip( const QPoint &);
};

QdbtDynaTip::QdbtDynaTip(QWidget *parent) : QToolTip(parent)
{
}

QdbtDynaTip::~QdbtDynaTip()
{
  clear();
}

void QdbtDynaTip::maybeTip(const QPoint &pos)
{
  QString text;
  QRect r( ((QdbtTable*)parentWidget())->needsTip(pos,text) );
  if ( !r.isValid() ) return;
  tip(r,text); 
}

//-----------------------------------------------------------------------------

QdbtTable::QdbtTable(QWidget *p,const char *n) : QTableView(p,n)
{
  // use the base color as background (normally white)
  setBackgroundMode( PaletteBase );
  // set variable cell width
  QTableView::setCellWidth(0);
  // set variable cell height
  QTableView::setCellHeight(0); 
  // make the table clip its cells and use automatic scrollbars and 
  // smooth scrolling
  setMouseTracking(TRUE);

  setTableFlags( Tbl_clipCellPainting | 
                 Tbl_autoScrollBars | 
                 Tbl_smoothScrolling );
  // store a pointer to the parent for easy access, we assume that
  // only a QTabular is used as a parent.
  parent=(QdbtTabular *)p;
  
  // create a line edit widget
  cellEditor = new QdbtLineEdit(this);
  // hide the editor
  cellEditor->move(-1,0);
  cellEditor->resize(1,1);
  cellEditor->setFocusPolicy( NoFocus );
  //cellEditor->setEnabled(FALSE);

  // synchronize the position of the scrollbar with that of the header
  connect((QWidget *)horizontalScrollBar(),SIGNAL(sliderMoved(int)),
          SIGNAL(setHeaderOffset(int)));
  connect((QWidget *)horizontalScrollBar(),SIGNAL(valueChanged(int)),
          SIGNAL(setHeaderOffset(int)));

  // stop editing when return is pressed
  connect(cellEditor,SIGNAL(returnPressed()),SLOT(cellEditDone()));

  // start with only a invisible rubber length column
  // this is needed to keep the table from moving out of sync with the header
  // if the head-button is resized and the table's x-offset is >0
  columnWidths.resize(1);
  columnWidths[0]=0;
  setNumCols(0);

  dragging=FALSE;
  timerRunning=FALSE;
  editingEnabled=FALSE;
  selectRows=TRUE;

  // init keyboard focus policy and variables
  setFocusPolicy( TabFocus );
  currentFocusRow=currentFocusCol=-1;
  focusActive=FALSE;
  
  // init tooltip stuff
  tip=0;
  enableTooltips();
}

QdbtTable::~QdbtTable()
{
  // stop editing, before the widget is destroyed
  if (editingEnabled) cellEditDone();
}


void QdbtTable::paintEvent(QPaintEvent *e)
{
  QTableView::paintEvent(e);
  if (editingEnabled) // set the correct location/size of the edit field
  {
    int rowOffset=0,colOffset=0;
    rowOffset=cellYOffset(editingRow)-yOffset();
    colOffset=cellXOffset(editingCol)-xOffset();
    QdbtTableCell *cell = cells(editingRow,editingCol);
    QRect textArea = cell->getEditArea(columnWidths[editingCol]);
    textArea.moveBy(colOffset,rowOffset);
    int w=textArea.width();
    //int x=textArea.x();
    if (w<=0)
    {
      cellEditor->setGeometry(-1,0,1,1);
      cellEditor->setFocusPolicy( NoFocus );
      setFocus();
    }
    else
    {
      cellEditor->setFocusPolicy( StrongFocus );
      cellEditor->setGeometry(textArea.x()-4,textArea.y(),
                              textArea.width()+8,textArea.height());
    }
    cellEditor->setSelected(rowInfo[editingRow].selected);
  }
}

void QdbtTable::paintCell(QPainter *p,int r,int c)
{
  //printf("QdbtTable::paintCell(%d,%d)\n",r,c);

  // last column is a rubber length, do not draw 
  if (c==cells.cols()) return; 
  // get the cell info from the cells array
  QdbtTableCell *cell=cells(r,c);
  // if info available then draw it
  if (cell) 
  {
    bool editing=(editingEnabled && r==editingRow && c==editingCol);
    int  focusType;
    if (focusActive && r==currentFocusRow && c==currentFocusCol)
      focusType=QdbtTableCell::Bar;
    else
      focusType=QdbtTableCell::NoBar;
    cell->paint(p,columnWidths[c],rowInfo[r].height,
                rowInfo[r].selected,selectRows,editing,focusType);
  }
  else // empty cell
  {
    if (rowInfo[r].selected)     // selected empty cell
      p->fillRect(0,0,columnWidths[c],rowInfo[r].height,black);
    else                  // unselected empty cell
      p->eraseRect(0,0,columnWidths[c],rowInfo[r].height);
  }
}

void QdbtTable::setColumnWidth(int c,int w)
{
//  printf("QdbtTable::setColumnWidth(%d,%d)\n",c,w);
  if (c>=0 && c<cells.cols()) columnWidths[c]=w;
}

int QdbtTable::cellWidth(int c)
{
//  printf("QdbtTable::cellWidth(%d)\n",c);
  if (c>=0 && c<cells.cols()) return columnWidths[c]; else return 0;
}

int QdbtTable::cellHeight(int r)
{
//  printf("QdbtTable::cellHeight(%d)\n",r);
  if (r>=0 && r<cells.rows())  return rowInfo[r].height; else return 0;
}

int QdbtTable::cellXOffset(int c)
{
  if (c<0 || c>=cells.cols()) return -1;
  int i,offset=0;
  for (i=0;i<c;i++) offset+=columnWidths[i];
  return offset;
}

int QdbtTable::cellYOffset(int r)
{
  if (r<0 || r>=cells.rows()) return -1;
  int i,offset=0;
  for (i=0;i<r;i++) offset+=rowInfo[i].height;
  return offset;
}


void QdbtTable::insertCol(int col)
{
  //printf("QdbtTable::insertCol()\n");
  ASSERT(col>=-1 && col<=cells.cols());
  if (editingEnabled && col<=editingCol) editingCol++;
  int lnCols=cells.cols();
  if (col==-1) col=lnCols;
  cells.insertCol(col);
  lnCols++;
  columnWidths.resize(lnCols+1);
  int i; for (i=lnCols;i>col;i--) columnWidths[i]=columnWidths[i-1];
  columnWidths[col]=0;
  bool au=autoUpdate();
  setAutoUpdate(FALSE);
  if (focusActive && col<=currentFocusCol) // move focus if needed
  {
    currentFocusCol++;
    resetFocus(currentFocusRow,currentFocusCol,TRUE,FALSE);
  }
  setNumCols(lnCols);
  setAutoUpdate(au);
  updateTable();
}

void QdbtTable::removeCol(int col)
{
  ASSERT(col>=0 && col<cells.cols());
  //printf("QdbtTable::removeCol()\n");
  if (editingEnabled)
  {
    if (col==editingCol) cellEditDone();
    else if (col<editingCol) editingCol--;
  }
  cells.removeCol(col);
  int lnCols=cells.cols();
  int i; for (i=col;i<lnCols;i++) columnWidths[i]=columnWidths[i+1];
  columnWidths.resize(lnCols+1);
  bool au=autoUpdate();
  setAutoUpdate(FALSE);
  if (focusActive)
  {
    if (lnCols==0) // remove focus if no columns are left
    {
      currentFocusRow=currentFocusCol=-1;
      focusActive=FALSE;
    }
    else if (col<=currentFocusCol) // move focus if needed
    {
      if (currentFocusCol>0) currentFocusCol--;
      resetFocus(currentFocusRow,currentFocusCol,TRUE,FALSE);
    }
  }
  setNumCols(lnCols);
  setNumRows(cells.rows());
  setAutoUpdate(au);
  updateTable();
}

void QdbtTable::insertRow(int row,bool selectable)
{
  //printf("QdbtTable::insertRow()\n");
  if (editingEnabled && row<=editingRow) editingRow++;
  int lnRows=cells.rows();
  if (row==-1) row=lnRows;
  cells.insertRow(row);
  lnRows++;
  //printf("NumRows=%d\n",lnRows);
  rowInfo.resize(lnRows);
  int i; for (i=lnRows-1;i>row;i--) rowInfo[i]=rowInfo[i-1];
  rowInfo[row].height=fontMetrics().lineSpacing()+1;
  rowInfo[row].selected=FALSE;
  rowInfo[row].selectable=selectable;
  bool au=autoUpdate();
  setAutoUpdate(FALSE);
  if (focusActive && row<=currentFocusRow) // move focus if needed
  {
    currentFocusRow++;
    resetFocus(currentFocusRow,currentFocusCol,TRUE,FALSE);
  }
  setNumRows(lnRows);
  setAutoUpdate(au);
  updateTable();
}

void QdbtTable::removeRow(int row)
{
  //printf("QdbtTable::removeRow()\n");
  if (editingEnabled)
  {
    if (row==editingRow) cellEditDone();
    else if (row<editingRow) editingRow--;
  }
  cells.removeRow(row);
  int lnRows=cells.rows();
  int i; for (i=row;i<lnRows;i++) rowInfo[i]=rowInfo[i+1];
  rowInfo.resize(lnRows);
  bool au=autoUpdate();
  setAutoUpdate(FALSE);
  if (focusActive)
  {
    if (lnRows==0) // remove focus if no columns are left
    {
      currentFocusRow=currentFocusCol=-1;
      focusActive=FALSE;
    }
    else if (row<=currentFocusRow) // move focus if needed
    {
      if (currentFocusRow>0) currentFocusRow--;
      resetFocus(currentFocusRow,currentFocusCol,TRUE,FALSE);
    }
  }
  setNumCols(cells.cols());
  setNumRows(lnRows);
  setAutoUpdate(au);
  updateTable();
}

void QdbtTable::setDimensions(int rows,int cols)
{
  int i;
  //printf("QdbtTable::setDimensions(%d,%d)\n",rows,cols);
  ASSERT(rows>=0 && cols>=0);
  if (editingEnabled && (rows<=editingRow || cols<=editingCol)) cellEditDone();
  if (focusActive && (rows<=currentFocusRow || cols<=currentFocusCol))
  {
    currentFocusRow=currentFocusCol=-1;
    focusActive=FALSE;
  }
  int oldCols=cells.cols();
  int oldRows=cells.rows();
  cells.resize(rows,cols);
  columnWidths.resize(cols+1);
  rowInfo.resize(rows);
  // set the initial size of any new columns to 0
  for (i=oldCols+1;i<cols+1;i++) columnWidths[i]=0;
  // set the initial size of any new rows to the font height
  for (i=oldRows;i<rows;i++) 
  { 
    rowInfo[i].height=fontMetrics().lineSpacing()+1;
    rowInfo[i].selected=FALSE;
    rowInfo[i].selectable=TRUE;
  }
  bool au=autoUpdate();
  setAutoUpdate(FALSE);
  setNumRows(rows);
  setNumCols(cols);
  setAutoUpdate(au);
  updateTable();
}

void QdbtTable::changeCell(const QdbtTableCell *tc,int row,int col)
{
  //printf("QdbtTable::changeCell(%p,%d,%d)\n",tc,row,col);

  // If the user was editing the cell, we just stop him/her and take over
  if (editingEnabled && row==editingRow && col==editingCol) cellEditDone();

  QdbtTableCell *oldCell = cells(row,col);

  // create a new one with the specified properties
  // this ensures each cell is unique, otherwise it would be
  // easily possible to two cells in the table point to the same data, which
  // does not work if the user can edit the cells
  if (tc)
    cells(row,col) = new QdbtTableCell(*tc);
  else
    cells(row,col) = 0;

  // delete the current cell
  delete oldCell;
  
  
  int c,maxRowHeight=0;
  for (c=0;c<cells.cols();c++)
  {
    QdbtTableCell *t=cells(row,c);
    if (t)
    {
      int rh = t->heightHint(fontMetrics());
                 //QMAX(t->text() ? fontMetrics().lineSpacing()+1 : 0,
                 // t->pixmap().height()
                 //);
      if (rh>maxRowHeight) maxRowHeight=rh;
    }
  }
  if (maxRowHeight!=rowInfo[row].height) // row height changed by the new cell?
  {
    rowInfo[row].height=maxRowHeight;
    updateTable();
  }
  else
  {
    updateCell(row,col,FALSE);
  }
}

void QdbtTable::resizeEvent(QResizeEvent *e)
{
  QTableView::resizeEvent(e);
  QTableView::resizeEvent(e);
  setTableOffset(-xOffset());
  emit setHeaderOffset(xOffset());
}

void QdbtTable::setTableOffset(int pos)
{
  //printf("QdbtTable::setTableOffset(%d)\n",pos);
  if (-pos==xOffset()) return;
  setXOffset(-pos);
}

void QdbtTable::updateTable()
{
  if (autoUpdate() && isVisible())
  {
    updateTableSize();
    repaint();
    parent->repaint();
  }
}

void QdbtTable::updateRow(int row)
{
  //printf("QdbtTable:updateRow(%d)\n",row);
  int i;for (i=0;i<cells.cols()+1;i++) updateCell(row,i,FALSE);
  if (selectRows) emit selected(row,rowInfo[row].selected);
}

void QdbtTable::mousePressEvent(QMouseEvent *e)
{
  // send the mouse event to the parent (so it can be re-implemented
  // by the user)
  QMouseEvent pe(Event_MouseButtonPress,mapToParent(e->pos()),
                 e->button(),e->state());
  parent->mousePressEvent(&pe);

  int rowClicked=findRow(e->y());
  int colClicked=findCol(e->x());
  //printf("QdbtTable::mousePressEvent(%d)\n",rowClicked);
  if (rowClicked==-1) return;

  int but=e->button();
  if (but&LeftButton) // select a cell
  {
    if (selectRows) // select one row at a time
    {
      bool oldSelected=rowInfo[rowClicked].selected;
      if (rowInfo[rowClicked].selectable) 
        rowInfo[rowClicked].selected=!rowInfo[rowClicked].selected;
      if (oldSelected!=rowInfo[rowClicked].selected) updateRow(rowClicked); 
      startDragRow=rowClicked;
      dragging=TRUE;
    }
    else            // select one cell at a time
    {
      if (colClicked==-1) return;
      QdbtTableCell *c=(QdbtTableCell *)cell(rowClicked,colClicked);
      if (c->isSelectable()) 
      {
        c->setSelected(!c->isSelected());
        updateCell(rowClicked,colClicked,FALSE);
      }
      emit cellSelected(rowClicked,colClicked,c->isSelected());
      startDragRow=rowClicked;
      startDragCol=colClicked;
      dragging=TRUE;
    }
    //resetFocus(rowClicked,colClicked,TRUE,TRUE);
  }
  else if (but&MidButton) // edit a cell
  {
    if (colClicked<0 || colClicked>=cells.cols()) return; // not a valid column
    const QdbtTableCell *clickedCell = cell(rowClicked,colClicked);
    if (clickedCell && clickedCell->isEditable()) 
    {
      editCell(rowClicked,colClicked);
      resetFocus(rowClicked,colClicked,TRUE,TRUE);
    }
  }
}

void QdbtTable::editCell(int row,int col)
{
  //printf("editCell(%d,%d)\n",row,col);
  if (editingEnabled) // the user was already editing another cell
  {
    if (row==editingRow && col==editingCol) return; 
        // we are already editing this cell
    editingEnabled=FALSE;
    updateCell(editingRow,editingCol,FALSE); // update old cell
  }
  if (row==-1 || col==-1) return;
  const QdbtTableCell *c = cell(row,col);
  editingEnabled=TRUE;    // turn on editing
  editingRow=row;         // remember the cell
  editingCol=col;
  cellEditor->setEnabled(TRUE);
  cellEditor->setFocusPolicy( StrongFocus );
  cellEditor->setFocus(); // set the focus
  cellEditor->setColor(c->color(),
       c->background().isValid() ? c->background() : backgroundColor());
  updateCell(row,col,FALSE);
  cellEditor->setText(c->text());
  cellEditor->selectAll();
}

void QdbtTable::mouseDoubleClickEvent(QMouseEvent *e)
{
  // send the mouse event to the parent (this can be re-implemented
  // by the user)
  QMouseEvent pe(Event_MouseButtonDblClick,mapToParent(e->pos()),
                 e->button(),e->state());
  parent->mouseDoubleClickEvent(&pe);

  // find the row that is clicked
  int rowClicked=findRow(e->y());
  if (rowClicked==-1) return;
  if (e->button()&LeftButton) 
  {
    if (selectRows)
      emit activated(rowClicked);
    else
    {
      int colClicked=findCol(e->x());
      if (colClicked!=-1) emit cellActivated(rowClicked,colClicked);
    }
  }
  mousePressEvent(e);
}

void QdbtTable::markRange(int startRow,int endRow)
{
  //printf("QdbtTable::markRange startRow=%d endRow=%d\n",startRow,endRow);
  bool rangeSelected=rowInfo[startRow].selected;
  int minRow,maxRow;
  if (startRow<=endRow) { minRow=startRow; maxRow=endRow; }
                   else { minRow=endRow; maxRow=startRow; }
  int i;
  if (minRow<0) minRow=0;
  if (maxRow>=cells.rows()) maxRow=cells.rows()-1;
  for (i=minRow;i<=maxRow;i++)
  {
    bool visSelected=rowInfo[i].selected;
    if (rowInfo[i].selectable) rowInfo[i].selected=rangeSelected;
    if (visSelected!=rowInfo[i].selected) updateRow(i);
  }
}

void QdbtTable::markBlock(int startRow,int startCol,int endRow,int endCol)
{
  //printf("QdbtTable::markBlock startRow=%d startCol=%d endRow=%d endCol=%d\n",
  //                 startRow,startCol,endRow,endCol);
  bool blockSelected=cells(startRow,startCol)->isSelected();
  int minRow,minCol,maxRow,maxCol;
  if (startRow<=endRow) { minRow=startRow; maxRow=endRow; }
                   else { minRow=endRow;   maxRow=startRow; }
  if (startCol<=endCol) { minCol=startCol; maxCol=endCol; }
                   else { minCol=endCol;   maxCol=startCol; }
  if (minRow<0) minRow=0;
  if (minCol<0) minCol=0;
  if (maxRow>=cells.rows()) maxRow=cells.rows()-1;
  if (maxCol>=cells.cols()) maxCol=cells.cols()-1;
  int r,c;
  if (minCol<0 || minRow<0) return; // be paranoid
  for (c=minCol;c<=maxCol;c++)
    for (r=minRow;r<=maxRow;r++)
    {
       QdbtTableCell *cell=(QdbtTableCell *)cells(r,c);
       bool visSelected=cell->isSelected();
       if (cell->isSelectable()) cell->setSelected(blockSelected);
       if (visSelected!=cell->isSelected()) 
       {
         updateCell(r,c,FALSE);
         emit cellSelected(r,c,blockSelected);
       }
    }
}

void QdbtTable::mouseMoveEvent(QMouseEvent *e)
{
  QMouseEvent pe(Event_MouseMove,mapToParent(e->pos()),
                 e->button(),e->state());
  parent->mouseMoveEvent(&pe);
  mouseAt=e->pos();
  if (dragging)
  {
    if (selectRows) // select rows
    {
      int rowClicked=findRow(e->y());
      if (rowClicked!=-1)
      {
        if (timerRunning) // stop running timer
        {
          killTimers();
          timerRunning=FALSE;
        }
        markRange(startDragRow,rowClicked); 
      }
      else
      {
        int y=e->y();
        int yTop=0;
        int yBottom=QMIN(totalHeight()-yOffset(),viewHeight());
        //printf("y=%d yTop=%d yBottom=%d\n",y,yTop,yBottom);
        if (y<yTop)
        {
          markRange(startDragRow,topCell());
          scrollDirection=ScrollUp;
        }
        else if (y>=yBottom)
        {
          markRange(startDragRow,lastRowVisible());
          scrollDirection=ScrollDown;
        }
        if (!timerRunning)
        {
          startTimer(75);
          timerRunning=TRUE;
        }
      }
    }
    else // select cells
    {
      int rowClicked=findRow(e->y());
      int colClicked=findCol(e->x());
      if (rowClicked!=-1 && colClicked!=-1)
      {
        if (timerRunning) // stop running timer
        {
          killTimers();
          timerRunning=FALSE;
        }
        markBlock(startDragRow,startDragCol,rowClicked,colClicked); 
      }
      else
      {
        int x=e->x();
        int y=e->y();
        int xTop=0;
        int yTop=0;
        int xBottom=QMIN(totalWidth()- xOffset(),viewWidth());
        int yBottom=QMIN(totalHeight()-yOffset(),viewHeight());
        //printf("x=%d y=%d xTop=%d yTop=%d xBottom=%d yBottom=%d\n",
        //        x,y,xTop,yTop,xBottom,yBottom);
        //printf("rowClicked=%d colClicked=%d\n",rowClicked,colClicked);
        scrollDirection=NoScroll;
        int endDragRow=rowClicked,endDragCol=colClicked;
        if (y<yTop)
        {
          endDragRow=topCell();
          scrollDirection|=ScrollUp;
        }
        else if (y>=yBottom)
        {
          endDragRow=lastRowVisible();
          scrollDirection|=ScrollDown;
        }
        if (x<xTop)
        {
          endDragCol=leftCell();
          scrollDirection|=ScrollLeft;
        }
        if (x>=xBottom)
        {
          endDragCol=lastColVisible();
          scrollDirection|=ScrollRight;
        }
        //printf("startDragRow=%d startDragCol=%d endDragRow=%d endDragCol=%d\n",
        //         startDragRow,startDragCol,endDragRow,endDragCol);
        if (scrollDirection) 
          markBlock(startDragRow,startDragCol,endDragRow,endDragCol);
        if (!timerRunning)
        {
          startTimer(75);
          timerRunning=TRUE;
        }
      }
    }
  }
}

void QdbtTable::timerEvent(QTimerEvent *)
{
  //printf("QdbtTable::timerEvent() scrollDirection=%d\n",scrollDirection);

  if (selectRows)
  {
    int yTop=0;
    int yBottom=QMIN(totalHeight()-yOffset(),viewHeight());
    int y=mouseAt.y();
    int yDist=0;
    if (y<yTop) 
      yDist=yTop-y;
    else if (y>yBottom)
      yDist=y-yBottom;
    int scrollSpeed=QMIN(yDist>>2,8)+1; 
    int delay=QMAX(10,50-yDist*2);

    if (scrollDirection==ScrollDown)
    {
      int yBottom=QMAX(0,totalHeight()-viewHeight());
      int y=yOffset()+scrollSpeed;
      setYOffset(QMIN(y,yBottom));
      markRange(startDragRow,lastRowVisible());
    }
    if (scrollDirection==ScrollUp)
    {
      int y=QMAX(0,yOffset()-scrollSpeed);
      setYOffset(y);
      markRange(startDragRow,topCell());
    }
    killTimers();
    startTimer(delay);
  }
  else // select a block of cells
  {
    int xTop=0,yTop=0;
    int xBottom=QMIN(totalWidth()-xOffset(),viewWidth());
    int yBottom=QMIN(totalHeight()-yOffset(),viewHeight());
    int x=mouseAt.x(),y=mouseAt.y();
    int xDist=0,yDist=0;

    if (y<yTop) 
      yDist=yTop-y;
    else if (y>yBottom)
      yDist=y-yBottom;

    if (x<xTop)
      xDist=xTop-x;
    else if (x>xBottom)
      xDist=x-xBottom;
    
    int scrollSpeedX=QMIN(xDist>>2,8)+1; 
    int scrollSpeedY=QMIN(yDist>>2,8)+1; 
    int delay=QMAX(10,50-QMAX(xDist,yDist)*2);

    int endDragRow=startDragRow,endDragCol=startDragCol;

    if (scrollDirection&ScrollDown)
    {
      int yBottom=QMAX(0,totalHeight()-viewHeight());
      int y=yOffset()+scrollSpeedY;
      setYOffset(QMIN(y,yBottom));
      endDragRow=lastRowVisible();
    }
    if (scrollDirection&ScrollUp)
    {
      int y=QMAX(0,yOffset()-scrollSpeedY);
      setYOffset(y);
      endDragRow=topCell();
    }
    if (scrollDirection&ScrollRight)
    {
      int xBottom=QMAX(0,totalWidth()-viewWidth());
      int x=xOffset()+scrollSpeedX;
      setXOffset(QMIN(x,xBottom));
      endDragCol=lastColVisible();
    }
    if (scrollDirection&ScrollLeft)
    {
      int x=QMAX(0,xOffset()-scrollSpeedX);
      setXOffset(x);
      endDragCol=leftCell();
    }
    //printf("startDragRow=%d startDragCol=%d endDragRow=%d endDragCol=%d\n",
    //         startDragRow,startDragCol,endDragRow,endDragCol);
    
    if (scrollDirection) 
      markBlock(startDragRow,startDragCol,endDragRow,endDragCol);
    killTimers();
    startTimer(delay);
  } 
}

void QdbtTable::exposeCell(int row,int col)
{
  int xto=xOffset();
  int yto=yOffset();
  int tw=viewWidth();
  int th=viewHeight();
  int xco=cellXOffset(col);
  int yco=cellYOffset(row);
  int cw=columnWidth(col);
  int ch=rowHeight(row);
  //int maxx=QMAX(totalWidth()-xto,viewWidth());
  //int maxy=QMAX(totalHeight()-yto,viewHeight());
  //printf("QdbtTable::exposeCell(%d,%d) xto=%d yto=%d tw=%d th=%d xco=%d yco=%d cw=%d ch=%d maxx=%d maxy=%d\n",
  //       row,col,xto,yto,tw,th,xco,yco,cw,ch,maxx,maxy);
  
  int dx=0,dy=0;
  if (xto>xco) // move right
  {
    dx=xto-xco;
  }
  else if (xto+tw<xco+cw) // move left 
  {
    dx=QMIN(xco-xto,xto+tw-xco-cw);
  }
  if (yto>yco) // move down 
  {
    dy=yto-yco;
  }
  else if (yto+th<yco+ch) // move up 
  {
    dy=QMIN(yco-yto,yto+th-yco-ch);
  }
  //printf("Result dx=%d dy=%d x=%d y=%d\n",dx,dy,x,y);
  setXOffset(xto-dx);
  setYOffset(yto-dy);
}

void QdbtTable::resetFocus(int row,int col,bool enable,bool expose)
{
  //printf("resetFocus(%d,%d)\n",currentFocusRow,currentFocusCol);
  ASSERT(row>=0 && row<cells.rows() && col>=0 && col<cells.cols());
  int oldFocusRow=currentFocusRow;
  int oldFocusCol=currentFocusCol;
  currentFocusRow=row;
  currentFocusCol=col;
  focusActive=enable;
  if (oldFocusRow!=-1 && oldFocusCol!=-1)
  {
    if (selectRows)
      updateRow(oldFocusRow);
    else
      updateCell(oldFocusRow,oldFocusCol,FALSE);
  }
  if (focusActive)
  {
    if (selectRows)
      updateRow(currentFocusRow);
    else
      updateCell(currentFocusRow,currentFocusCol,FALSE);
  }
  if (expose) exposeCell(row,col);
}

void QdbtTable::keyPressEvent(QKeyEvent *e)
{
  // keys should have no effect on an empty table
  if (!focusActive || cells.rows()==0 || cells.cols()==0) return;
  int focusRow=currentFocusRow;
  int focusCol=currentFocusCol;
  //printf("Focus (%d,%d)\n",focusRow,focusCol);
  switch(e->key())
  {
    case Key_Up:
      //printf("Up pressed\n");
      if (focusRow>0) resetFocus(focusRow-1,focusCol,TRUE,TRUE);
      break;
    case Key_Down:
      //printf("Down pressed\n");
      if (focusRow<cells.rows()-1) resetFocus(focusRow+1,focusCol,TRUE,TRUE);
      break;
    case Key_Left:
      //printf("Left pressed\n");
      if (focusCol>0) resetFocus(focusRow,focusCol-1,TRUE,TRUE);
      break;
    case Key_Right:
      //printf("Right pressed\n");
      if (focusCol<cells.cols()-1) resetFocus(focusRow,focusCol+1,TRUE,TRUE);
      break;
    case Key_Space:
      //printf("Space pressed\n");
      if (focusRow!=-1 && focusCol!=-1) 
      {
        if (selectRows) // select one row at a time
        {
          bool oldSelected=rowInfo[focusRow].selected;
          if (rowInfo[focusRow].selectable) 
            rowInfo[focusRow].selected=!rowInfo[focusRow].selected;
          if (oldSelected!=rowInfo[focusRow].selected) updateRow(focusRow); 
        }
        else
        {
          QdbtTableCell *c=(QdbtTableCell *)cell(focusRow,focusCol);
          if (c->isSelectable()) 
          {
            c->setSelected(!c->isSelected());
            updateCell(focusRow,focusCol,FALSE);
          }
          emit cellSelected(focusRow,focusCol,c->isSelected());
        }
      }
      break;
    case Key_Enter:
    case Key_Return:
      if (focusRow!=-1 && focusCol!=-1)
      {
        const QdbtTableCell *selectedCell = cell(focusRow,focusCol);
        if (selectedCell && selectedCell->isEditable()) 
          editCell(focusRow,focusCol);
      }
      break;
    default:
      e->ignore();
  }
}

void QdbtTable::keyReleaseEvent(QKeyEvent *)
{
}

void QdbtTable::focusInEvent(QFocusEvent *)
{
  //printf("Focus in event\n");
  if (cells.rows()==0 || cells.cols()==0) return; // no cells
  int r=currentFocusRow,c=currentFocusCol;
  if (r==-1 || !rowIsVisible(r)) r=topCell();
  if (c==-1 || !colIsVisible(c)) c=leftCell();
  // avoid moving the table, because it is annoying for the user in some cases.
  if (r>=0 && r<cells.rows() && c>=0 && c<cells.cols())
    resetFocus(r,c,TRUE,FALSE); 
}

void QdbtTable::focusOutEvent(QFocusEvent *)
{
  //printf("Focus out event\n");
  if (currentFocusRow>=0 && currentFocusRow<cells.rows() &&
      currentFocusCol>=0 && currentFocusCol<cells.cols()) 
    resetFocus(currentFocusRow,currentFocusCol,FALSE,FALSE);
}

void QdbtTable::mouseReleaseEvent(QMouseEvent *e)
{
  QMouseEvent pe(Event_MouseButtonRelease,mapToParent(e->pos()),
                 e->button(),e->state());
  parent->mouseReleaseEvent(&pe);
  dragging=FALSE;
  if (timerRunning) killTimers();
  timerRunning=FALSE;
}

int QdbtTable::columnWidth(int col) const
{
  ASSERT(col>=0 && col<cells.cols());
  return columnWidths[col];
}

int QdbtTable::columnWidthHint(int col) 
{
  ASSERT(col>=0 && col<cells.cols());  
  int i,maxWidth=0;
  for (i=0;i<cells.rows();i++) 
  {
    QdbtTableCell *cell=cells(i,col);
    if (cell)
    {
      int w=cell->widthHint(fontMetrics());
      if (w>maxWidth) maxWidth=w;
    }
  }
  return maxWidth;
}

int QdbtTable::rowHeight(int row) const
{
  ASSERT(row>=0 && row<cells.rows());
   return rowInfo[row].height;
}

const QdbtTableCell *QdbtTable::cell(int row,int col)
{
  ASSERT(row>=0 && row<cells.rows() && col>=0 && col<cells.cols());
  return cells(row,col);
}

bool QdbtTable::rowSelected(int row) const
{
  ASSERT(row>=0 && row<cells.rows());
  return rowInfo[row].selected;
}

void QdbtTable::setRowSelected(int row,bool status) 
{
  ASSERT(row>=0 && row<cells.rows());
  bool oldStatus=rowInfo[row].selected;
  rowInfo[row].selected=status; 
  if (oldStatus!=status) updateRow(row);
}

void QdbtTable::cellEditDone()
{
  if (cellEditor->geometry()==QRect(-1,0,1,1)) return; // editor was hidden
  ASSERT(editingEnabled); // editor visible => must be editing
  // copy text from editor to cell
  QdbtTableCell *c=(QdbtTableCell *)cell(editingRow,editingCol);
  // if the text has changed then set the new text and emit a signal
  if (strcmp(c->text(),cellEditor->text())) 
  {
    c->setText(cellEditor->text());
    emit cellEdited(editingRow,editingCol);
  }
  // clear the editor (to set it offset to 0)
  cellEditor->setText(0);
  // hide the editor, we do not use hide() here, because that refreshes
  // the entire table which causes flashing
  cellEditor->setFocusPolicy( NoFocus );
  setFocus();
  cellEditor->setGeometry(-1,0,1,1);
  cellEditor->setEnabled(FALSE);
  // editing is now disabled (to avoid unnecessairy updates)
  editingEnabled=FALSE;  
  // expose the new contents of the cell
  updateCell(editingRow,editingCol,FALSE);
}
 
void QdbtTable::setFont(const QFont &font)
{
  QTableView::setFont(font);
  cellEditor->setFont(font); // set font for the cellEditor as well
  int r,c;
  for (r=0;r<numRows();r++)
  {
    int maxRowHeight=0;
    for (c=0;c<cells.cols();c++)
    {
      QdbtTableCell *t=cells(r,c);
      if (t)
      {
        int rh = t->heightHint(fontMetrics());
        if (rh>maxRowHeight) maxRowHeight=rh;
      }
    }
    rowInfo[r].height=maxRowHeight;
  }
  updateTable(); 
}

//#include <iostream.h>

// pass on table enter events
void QdbtTable::enterEvent(QEvent *e)
{
//cout << "before "<< parent << endl;
  //if (parent)
  //   parent->enterTableEvent(e); 
//cout << "after " << endl;   
}

// pass on table leave events
void QdbtTable::leaveEvent(QEvent *e)
{
  //if (parent)
  //   parent->leaveTableEvent(e);
}

// This member is called by the dynamic tooltip class
// If the mouse is on no cell or a completely visible cell it returns 
// on invalid rectangle. Otherwise it returns the area of the
// partly visible cell and `text' contains the full text of the cell
QRect QdbtTable::needsTip(const QPoint &pos, QString &text)
{
  int row=findRow(pos.y());
  int col=findCol(pos.x());
  if (row!=-1 && col!=-1)
  {
    const QdbtTableCell *c=cell(row,col);
  
    if (c && c->text() && strlen(c->text())>0)
    {
      int width = c->getTextArea().width();          // actual width
      int minWidth = fontMetrics().width(c->text()); // minimal width
      if (width<minWidth) 
      {
        int rowOffset=cellYOffset(row)-yOffset();
        int colOffset=cellXOffset(col)-xOffset();
        text=c->text();
        return QRect(colOffset,rowOffset,columnWidths[col],
                     rowInfo[row].height);
      }
    }
  }
  return QRect(0,0,-1,-1);
}

void QdbtTable::enableTooltips()
{
  if (!tip) tip=new QdbtDynaTip(this); 
}

void QdbtTable::disableTooltips()
{
  delete tip;
  tip=0;
}

void QdbtTable::setRowSelectable(int row,bool enable)
{
  ASSERT(row>=0 && row<=cells.rows());
  rowInfo[row].selectable=enable;
}

bool QdbtTable::rowSelectable(int row) const
{
  ASSERT(row>=0 && row<=cells.rows());
  return rowInfo[row].selectable;
}

void QdbtTable::exposeColumn(int col)
{
  //printf("QdbtTable::exposeColumn(%d)\n",col);
  int xto=xOffset();
  int tw=viewWidth();
  int xco=cellXOffset(col);
  int cw=columnWidth(col);
  
  int dx=0;
  if (xto>xco) // move right
  {
    dx=xto-xco;
  }
  else if (xto+tw<xco+cw) // move left 
  {
    dx=QMIN(xco-xto,xto+tw-xco-cw);
  }
  setXOffset(xto-dx);
}

void QdbtTable::setPalette(const QPalette &p)
{
  QTableView::setPalette(p);
}


