/*
 * Copyright (C) 1997-1998 by Dimitri van Heesch.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation under the terms of the GNU General Public License is hereby 
 * granted. No representations are made about the suitability of this software 
 * for any purpose. It is provided "as is" without express or implied warranty.
 * See the GNU General Public License for more details.
 *
 * This file is part of QdbtTabular 0.31.
 */

#include <stdio.h>
#include "qdbtlined.h"
#include "qdbttabcell.h"
#include "qdbtutil.h"
#include <qimage.h>
#include <qbitmap.h>

const int bw = 4; // border width between pixmap and text and between cells

/*!
  \class QdbtTableCell qdbttabcell.h
  \brief The QdbtTableCell class represents a cell in a QdbtTabular object.

  This class represents a cell of a QdbtTabular.
  A cell consists of a text (default the empty string) and a pixmap 
  (default the null Pixmap). 
  
  The cell itself is not aware of its width and height, but can compute
  its preferred size given the font metrics that are used for the text.
  
  The alignment of the pixmap with respect to the text can be controlled
  using setPixmapAlignment(). The alignment of the
  cell as a whole can be controlled using setAlignment().
  
  Suppose you have table widget called \c tabular.
  To change some properties of the cell at location (\e row,\e col) 
  in this table use the following code:
  \code
  // create a local copy of the cell
  QdbtTableCell cell(*tabular->cell(row,col));

  // change some properties of the cell

  .
  .
  .

  // update the cell at (row,col) in the table
  tabular->changeCell(&cell,row,col);
  \endcode
*/

/*!
  Constructs a cell with the following properties:
  \arg The cell has an empty text string.
  \arg The cell has a null pixmap.
  \arg The color of the text is set to black.
  \arg The background of the cell has the same color as 
       the background of the table.
  \arg The cell's text is not editable.
  \arg The cell is not selected.
  \arg The cell is selectable.
  \arg The pixmap is aligned to the left of the text.
  \arg The pixmap-text pair is aligned to the left.
*/

QdbtTableCell::QdbtTableCell()
{
  col=black;
  align=AlignLeft;
  editable=FALSE;
  selectable=TRUE;
  selected=FALSE;
  pmapAlign=AlignLeft;
  tr.setRect(0,0,0,0);
}

/*!
  Constructs a cell with the following properties:<br>
  \arg The cell has a text string \a t. 
       Use a null pointer to define no text.
  \arg The cell's pixmap is set to \a p. 
       Use a null pointer to define no pixmap.
  \arg The color is set to \a c.
  \arg The background of the cell has the same color as the 
       background of the table.
  \arg The ability to edit a cell is set to \a e. 
  \arg The cell is not selected.
  \arg The cell is selectable.
  \arg The pixmap is aligned to the left of the text.
  \arg The pixmap-text pair is aligned according to the value of \a a.
       Possible values are \c AlignLeft, \c AlignRight and \c AlignCenter.
*/

QdbtTableCell::QdbtTableCell(const char *t,const QPixmap *p,
                             const QColor &c,int a,bool e)
{
  //printf("QdbtTableCell(%s)\n",t);
  txt=t ? ((QString)t).stripWhiteSpace() : QString("");
  pmap=p ? pmap=*p : QPixmap();
  col=c;
  align=a;
  editable=e;
  selectable=TRUE;
  selected=FALSE;
  pmapAlign=AlignLeft;
  tr.setRect(0,0,0,0);
}

/*!
  Destroys the cell
*/

QdbtTableCell::~QdbtTableCell()
{
  //printf("QdbtTableCell::~QdbtTableCell()\n");
}

/*!
  Returns the preferred width of the cell. This is
  not the actual cell width, but the width that is required to
  fit both text and pixmap in the cell.
  \sa heightHint().
*/

int QdbtTableCell::widthHint(const QFontMetrics &fm) const
{
  int width;
  if (pmapAlign==AlignLeft || pmapAlign==AlignRight)
  {
    width=2*bw;
    if (!pmap.isNull()) width+=pmap.width();
    if (!txt.isNull())  width+=fm.width(txt);
    if (!pmap.isNull() && !txt.isNull()) width+=bw;
  }
  else // pmapAlign==AlignTop || pmapAlign==AlignBottom
  {
    width=2*bw+QMAX(pmap.width(),fm.width(txt));
  }
  return width;
}

/*!
  Returns the preferred height of the cell. This is
  not the actual cell height, but the height that is required to
  fit both text and pixmap in the cell.
  \sa widthHint().
*/

int QdbtTableCell::heightHint(const QFontMetrics &fm) const
{
  int height=0;
  bool hasPix = !pmap.isNull();
  bool hasTxt = TRUE; //txt.length()>0;
  if (pmapAlign==AlignLeft || pmapAlign==AlignRight)
  {
    if (hasPix) height=pmap.height();
    if (hasTxt) height=QMAX(height,fm.lineSpacing()+3);
  }
  else // pmapAlign==AlignTop || pmapAlign==AlignBottom
  {
    if (hasPix) height+=pmap.height(); 
    if (hasTxt) height+=fm.lineSpacing()+3;
    if (hasPix || hasTxt) height+=2*bw;  // add extra border
    if (hasPix && hasTxt) height+=bw;    // add room to separate
  }
  return height;
}

/*!
  Returns the area inside the cell, where the text is being drawn.
  The coordinates of the rectangle are in pixels and relative to the cell.
  The top left of the cell has coordinates (0,0).
*/

QRect QdbtTableCell::getTextArea() const
{
  return tr;
}

/*!
  Returns the area inside the cell, where the edit focus may be placed.
  The coordinates of the rectangle are in pixels and relative to the cell.
  The top left of the cell has coordinates (0,0).
*/

QRect QdbtTableCell::getEditArea(int width) const
{
  if ((align==AlignLeft && pmapAlign==AlignRight)
     ||(align==AlignRight && pmapAlign==AlignLeft)) return tr;
  if (pmap.isNull() || pmapAlign==AlignTop || pmapAlign==AlignBottom)
  {
    QRect er=tr;
    er.setX(bw);
    er.setWidth(width-2*bw);
    return er;
  }
  if (pmapAlign==AlignLeft)
  {
    QRect er=tr;
    er.setWidth(width-bw-tr.x());
    return er;
  }
  if (pmapAlign==AlignRight)
  {
    QRect er=tr;
    er.setX(bw);
    return er;
  }
  return tr;
}

/*!
  This function is called whenever the cell needs to be repainted.
  The width of the cell is \a w and the height is \a h.
  The painter \a p automatically clips against the area of the cell.
  The boolean \a rowSelected indicates if the row, that this
  cell belongs to, is selected. This should only have an effect if
  \a selectByRow is \c TRUE. The boolean
  \a editing indicates if the cell is currently being edited.
  The \a focusBar argument specify the the type of focus that needs to
  be drawn. Possible values are
  \arg \c Bar: draws a focus around the cell
  \arg \c NoBar: draw no focus.

*/

void QdbtTableCell::paint(QPainter *p,int w,int h,bool rowSelected,
                          bool selectByRow, bool editing,int focusBar)
{
  bool   select = selectByRow   ? rowSelected : selected;
  QColor fgCol  = col.isValid() ? col         : p->pen().color();
  QColor bgCol  = bg.isValid()  ? bg          : p->backgroundColor();
  QColor cf     = select        ? bgCol       : fgCol; // foreground color
  QColor cb     = select        ? fgCol       : bgCol; // background color
  QRect pr(0,0,0,0); // pixmap area
  QFontMetrics fm=p->fontMetrics();

  if (!pmap.isNull())
  {
    pr.setSize(pmap.size());
  }

  bool hasPix = !pmap.isNull();
  bool hasTxt = TRUE; // txt.length()>0;
  int ps  = hasPix ? 2*bw : 0; // border around pixmap
  int ts  = hasTxt ? 2*bw : 0; // border around text
  int hps = hasPix ? bw : 0;
  int hts = hasTxt ? bw : 0;
  int bs  = (hasPix && hasTxt) ? -bw : 0; // remove 1 border if needed
  int ma  = ps + ts + bs;               // total margin
  int tw  = hasTxt ? fm.width(txt) : 0; // text width
  int th  = fm.lineSpacing()+3;         // height of the text
  int pw  = hasPix ? pmap.width()  : 0; // pixmap width
  int ph  = hasPix ? pmap.height() : 0; // pixmap height
  int ta  = QMAX(0,QMIN(w-ts-hps-pw,tw)); // horizontal room available for text
  int minWidth=0;

  switch(pmapAlign) 
  {
    case AlignLeft:
      minWidth = pw+tw+ma;
      tr.setRect(pw+hts+hps, h>th ? (h-th)/2 : 0,ta,th);
      pr.moveBy(hps, h>ph ? (h-ph)/2 : 0);
      break;
    case AlignRight:
      minWidth = pw+tw+ma;
      tr.setRect(QMIN(minWidth,w)-hps-pw-ta-hts, h>th ? (h-th)/2 : 0,ta,th);
      pr.moveBy(QMIN(minWidth,w)-hps-pw, h>ph ? (h-ph)/2 : 0);
      break;
    case AlignTop:
      minWidth = QMIN(QMAX(pw+ps,tw+ts),w);
      tr.setRect((minWidth-QMIN(tw,w-ts))/2, hps+ph+hts, QMIN(tw,w-ts),h-ph-ma); 
      pr.moveBy( (minWidth-QMIN(pw,w))/2, hps);
      break;
    case AlignBottom:
      minWidth = QMIN(QMAX(pw+ps,tw+ts),w);
      tr.setRect((minWidth-QMIN(tw,w-ts))/2, hts, QMIN(tw,w-ts), h-ph-ma);
      pr.moveBy((minWidth-QMIN(pw,w))/2, h-ph-hps);
      break;
    default:
      warning("Warning: Unknown pixmap alignment\n");
      tr.setRect(0,0,0,0);
  }
 
  if (minWidth<w)
  {
    if (align==AlignRight)
    {
      int indent=w-minWidth;
      pr.moveBy(indent,0);
      tr.moveBy(indent,0);
    }
    else if (align==AlignCenter)
    {
      int indent=(w-minWidth)/2;
      pr.moveBy(indent,0);
      tr.moveBy(indent,0);
    }
  }

  // now we can actually start painting the cell
  
  QPen orgPen = p->pen();
  p->setPen(cf);

  // clear cell
  p->fillRect(0,0,w,h,cb);
  
  if (hasPix) // paint pixmap
  {
    if (!select) // draw normal
    {
      p->drawPixmap(pr.topLeft(),pmap);
    }
    else // draw with inverted colors
    {
      if (pmap.depth()==32) // a true color pixmap
      {
        RasterOp oldRop = p->rasterOp();
        p->setRasterOp(select ? NotCopyROP : CopyROP);  // select normal/inverse
        p->drawPixmap(pr.topLeft(),pmap); 
        p->setRasterOp( oldRop ); 
      }
      else // a palette based pixmap
      {
        QBitmap mask(*pmap.mask());
        QImage image=pmap.convertToImage().convertDepth(32);
        register int i,l=image.numBytes()>>2;
        register uint *b=(uint *)image.bits();
        for (i=0;i<l;i++) *b++=~(*b); 
        QPixmap invert;
        invert.convertFromImage(image);
        invert.setMask(mask);
        p->drawPixmap(pr.topLeft(),invert);
      }
    }
  }
  
  // paint text
  if (!editing)
  {
    int a;
    if (hasPix && pmapAlign==AlignLeft || pmapAlign==AlignRight)
      a = pmapAlign;
    else
      a = (align==AlignCenter) ? AlignLeft : align;
    QString displayTxt=truncateString(p,txt,tr.width(),a);
    p->drawText(tr,AlignVCenter|align,displayTxt); 
  }

  // paint focus if needed
  if (focusBar!=NoBar) 
  {
    BGMode bgMode=p->backgroundMode();
    p->setBackgroundMode(OpaqueMode);
    p->setPen(QPen(black,1,DashLine));
    p->drawRect(0,0,w,h);
    p->setBackgroundMode(bgMode);
  }
  p->setPen(orgPen);
}

/*!  
  \fn void QdbtTableCell::setText(const char *text) 
  Set the text of the cell to \a text.
  \sa text().
*/

/*!
  \fn const char *QdbtTableCell::text() const 
  Returns the text string of the cell.<p>
  \sa setText().
*/

/*!
  \fn void QdbtTableCell::setPixmap(const QPixmap *pixmap) 
  Set the pixmap of the cell to <em>pixmap</em>.
  \sa pixmap().
*/

/*!
  \fn const QPixmap &QdbtTableCell::pixmap() const
  Returns the pixmap of the cell.
  \sa setPixmap().
*/

/*!
  \fn void QdbtTableCell::setColor(const QColor &color) 
  Sets the color of the text to \a color.
  \sa color().
*/

/*!
  \fn const QColor &QdbtTableCell::color() const 
  Returns the color of the cell's text.<p>
  \sa setColor().
*/

/*!
  \fn void QdbtTableCell::setBackground(const QColor &color) 
  Sets the color of the background of the cell.

  If an invalid color is supplied (the default), then the background
  color of the table will be used.
  \sa background().
*/

/*!
  \fn const QColor &QdbtTableCell::background() const 
  Returns the background color of the cell.
  \sa setBackground().
*/

/*!
  \fn void QdbtTableCell::setEditable(bool state)
  Sets whether or not the cell may be edited by the application user.

  If \a state is \c TRUE the cell may be edited.

  If \a state is \c FALSE the cell may not be edited.

  \sa isEditable().
*/

/*!
  \fn bool QdbtTableCell::isEditable() const 
  Returns \c TRUE if the cell may be edited by the user.
  \sa setEditable().
*/

/*!
  \fn void QdbtTableCell::setSelectable(bool enable)
  Sets whether or not the cell may be selected by the user.

  If \a enable is \c TRUE the cell may be
  selected (highlighted or unhighlighted) by the user.
  
  If \a enable is \c FALSE the cell can only
  be highlighted from within the application.
  
  \sa isSelectable().
*/

/*!
  \fn bool QdbtTableCell::isSelectable() const
  Returns \c TRUE if the cell is currently highlighted.
  \sa setSelected().
*/

/*!
  \fn void QdbtTableCell::setSelected(bool enable)
  Sets whether or not the cell should be displayed as being selected.

  If \a enable is \c TRUE the cell is
  highlighted (drawn in inverse).

  If \a enable is \c FALSE the cell is
  unhighlighted (drawn normal).
  \sa isSelected().
*/

/*!
  \fn bool QdbtTableCell::isSelected() const
  Returns \c TRUE if the cell is currently highlighted.
  \sa setSelected().
*/

/*!
  \fn void QdbtTableCell::setAlignment(int alignment)
  Set the alignment of the text-pixmap pair.
  Possible values are:
  \arg \c AlignLeft: The pixmap and text are aligned to the left.
  \arg \c AlignCenter: The pixmap and text are centered.
  \arg \c AlignRight: The pixmap and text are aligned to the right.

  \sa alignment().
*/

/*!
  \fn int QdbtTableCell::alignment() const
  Returns the alignment of the text-pixmap pair.
  \sa alignment().
*/

/*!
  \fn void QdbtTableCell::setPixmapAlignment(int alignment) 
  Sets the alignment of the pixmap with respect to the text.
  Possible values are:
  \arg \c AlignLeft: The pixmap is displayed to the left of the text.
  \arg \c AlignRight: The pixmap is displayed to the right of the text.
  \arg \c AlignTop: The pixmap is displayed above the text.
  \arg \c AlignBottom: The pixmap is displayed below the text.

  \sa pixmapAlignment().
*/

/*!
  \fn int QdbtTableCell::pixmapAlignment() const
  Returns the align of the pixmap with respect to the text.
  Possible values are: \c AlignLeft, \c AlignRight,
  \c AlignTop, \c AlignBottom.

  \sa setPixmapAlignment().
*/

