/*  
	$Id: etreelist.cpp,v 1.2 1998/07/10 11:53:35 knuddel Exp $
	
	etreelist V0.9
	
	Requires the Qt widget libraries, available at no cost at 
    http://www.troll.no
       
    Copyright (C) 1998 Bernd Schumacher & Othmar Ehrhardt
                       B.Schumacher@eIT.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU Library General Public License as
	published by  
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
*/

#include "etreelist.h"

static uint ident=12;


ETreeList::ETreeList (	const char* rootText, QPixmap& rootIcon, 
						QPixmap& parentIcon, QPixmap& childIcon,
						QPixmap& parentOpenIcon, bool openFolders, 
						QWidget* parent, const char* name, WFlags f)
			: QListBox(parent, name, f)
{
	pIcon=parentIcon;
	cIcon=childIcon;
	doOpenFolder=openFolders;
	if ( parentOpenIcon.isNull() ) poIcon=parentIcon;
	else poIcon=parentOpenIcon;
	setMultiSelection( FALSE );
	allItems=new QList <ETreeListItem>;
	allItems->setAutoDelete( TRUE );
	rootItem=new ETreeListItem( "/", rootText, rootIcon, FALSE, TRUE, -1, FALSE );
	sortedItems= new QStrList( FALSE );	//no deep copy (Pointers to ETreeListItem->getPathID() )
	QListBox::insertItem( rootItem, 0 );
	QObject::connect( this, SIGNAL(selected(int)), SLOT(itemSelected(int)) );
}

ETreeList::~ETreeList()
{
}

void ETreeList::keyPressEvent( QKeyEvent *e )
{
    if ( numRows() == 0 ) return;
	if ( currentItem() < 0 ) setCurrentItem( topItem() );
    int pageSize, delta;
	int currentIndex=currentItem();
	ETreeListItem *tmp=(ETreeListItem *) item ( currentIndex );
	int indexAllItems=sortedItems->find ( tmp->getPathID() );
	if ( indexAllItems>-1 ) tmp=allItems->at( indexAllItems );	
	else tmp=0; 	//root item is current
    switch ( e->key() ) {
    case Key_Up:
		if ( currentItem() > 0 ) {
	   		setCurrentItem( currentItem() - 1 );
	  		if ( currentItem() < topItem()	) setTopItem( currentItem() );
		}
	break;
    case Key_Down:
		if ( currentItem() < (int)count() - 1 ) {
		    bool a = autoUpdate();
	    	bool u = FALSE;
	    	setAutoUpdate( FALSE );
	    	setCurrentItem( currentItem() + 1 );
	    	while ( currentItem() < (int)count() - 1
				 		&& currentItem() >= lastRowVisible() ) {
				setTopItem( topItem() + currentItem() + 1 - lastRowVisible() );
				u = TRUE;
		    }
		    setAutoUpdate( a );
	    	if ( u ) repaint();
		}
		break;
    case Key_Next:
		if ( style() == MotifStyle ) {
	    	delta = currentItem() - topItem();
	    	pageSize = QMAX( 1, lastRowVisible() - 1 - topItem() );
	    	setTopItem( QMIN( topItem() + pageSize, (int)count() - 1 ) );
	    	setCurrentItem( QMIN( topItem() + delta, (int)count() - 1 ) );
		} else {
	    	pageSize = QMAX( 1, lastRowVisible() - 1 - topItem() );
	    	setTopItem( QMIN(currentItem(),(int)count()-pageSize+1) );
	    	setCurrentItem( QMIN(lastRowVisible(), (int)count()-1) );
		}
		break;
    case Key_Prior:
		if ( style() == MotifStyle ) {
	    	delta = currentItem() - topItem();
	    	pageSize = QMAX( 1, lastRowVisible() - 1 - topItem() );
	    	setTopItem( QMAX( topItem() - pageSize, 0 ) );
	    	setCurrentItem( QMAX( topItem() + delta, 0 ) );
		} else {
		    pageSize = QMAX( 1, lastRowVisible() - topItem() );
		    setTopItem( QMAX(0,currentItem()-pageSize+1) );
	    	setCurrentItem( topItem() );
		}
		break;
	case Key_Minus:
		if ( tmp!=0 && tmp->isExpanded() && tmp->isParent() ) {
			tmp->setExpand( FALSE );
			updateTree(tmp->getPath());
			emit collapsedItem ( tmp->getPath() );
		}
		return;
	case Key_Plus:
		if ( tmp!=0 && !tmp->isExpanded() && tmp->isParent() && hasChild(tmp->getPathID()) ) {
			tmp->setExpand( TRUE );
			updateTree(tmp->getPath());
			emit expandedItem( tmp->getPath() );
		}
		return;
    case Key_Return:
    case Key_Enter:
		if ( currentItem() >= 0 ) {
			itemSelected( currentItem() );
		}
		return;
    default:
		return;
		break;
	}
	ETreeListItem *tmpItem=(ETreeListItem *) item( currentItem());
	emit highlightedItem( tmpItem->getPath() );
}
void ETreeList::itemSelected ( int index )
{
	ETreeListItem *tmp=(ETreeListItem *) item ( index );	
	emit selectedItem ( tmp->getPath() );
}

const QString ETreeList::getParentID( const char* path )
{
	QString buffer=path, newPath=path;
	buffer.replace( QRegExp ( "/$" ), "" ); 	//strip last slash
	buffer.replace( QRegExp ( ".*/" ), "" );
	buffer+="$"; 	//For RegExp -> to join with end of line
	newPath.replace( QRegExp( buffer.data() ), "" );
	newPath.replace( QRegExp ( "/$" ), "" ); 	//strip last slash
	if ( newPath.isEmpty() ) return QString("/");
	return newPath;
}

void ETreeList::insertItem( ETreeListItem *it )
{
	QString pathID=it->getPathID(), pID=getParentID( pathID.data() );
	sortedItems->inSort( it->getPathID() );
	it->setLastChild( TRUE );
	uint i=sortedItems->find( pathID.data() );
	for ( uint z=i+1; z<sortedItems->count(); z++ ) {
		if ( pID==getParentID(sortedItems->at( z )) ) {
			it->setLastChild( FALSE );
			break;
		}
	}
	for ( int p=i-1; p>-1; p-- ) {
		if ( pID==getParentID(sortedItems->at( p )) ) {
			ETreeListItem *tmp=allItems->at( p );
			tmp->setLastChild ( FALSE );
			break;			
		}
	}
	allItems->insert( i, it );
}

void ETreeList::removePath( const char* path )
{
	QString p=path, pathID;
	if ( p=="/" ) {
		sortedItems->clear();
		allItems->clear();
//		updateTree();	//Maybe its better under the control of the coder?
		return;			//So I decided to switch it off
	}
	bool found=FALSE, wasParent=FALSE;
	int index=0;
	for ( int i=allItems->count()-1; i>=0; i-- ) {
        ETreeListItem *tmp=allItems->at( i );
		if ( p==tmp->getPath() ) {
			found=TRUE;
			index=i; pathID=tmp->getPathID();
			wasParent=tmp->isParent();
			sortedItems->remove( tmp->getPathID() );
			allItems->remove( tmp ); 
			break;
		}	
	}
	if ( !found ) { 
		warning ( "Item (%s) not found for removing", path ); 
		return;
	}
	if ( wasParent ) {
		for ( int u=sortedItems->count()-1; u>=index; u-- ) {
			if ( pathID==getParentID( sortedItems->at( u )) ) {
				sortedItems->remove( u );
				allItems->remove( u );
			} 
		}
	}
}

void ETreeList::setCurrentPath( const char* path )
{
	QString p=path;
	p.replace( QRegExp("/$"), "" );
	ETreeListItem *tmp;
	for ( int i=count()-1; i>=0; i-- ) {
		tmp=(ETreeListItem *) item( i );
		if  ( p==tmp->getPath() ) {
			setCurrentItem( i );
			emit highlightedItem( tmp->getPath() );
			break;
		}
	}
}

void ETreeList::insertPath( const char* pathID, const char* text, QPixmap& icon, 
							bool parent, bool expand, bool change )
{
	QString pID=pathID; 
	pID.replace ( QRegExp("/$"), "" );	//strip last slash (if present)
	if ( !pID.contains( QRegExp("^/") )) return;	//return if invalid path
	int lev=pID.contains("/");
	if ( lev==1 ) {						//exit recursion on root level
		if ( sortedItems->find( pID.data() )>-1 ) return;
		ETreeListItem *tmpItem = new ETreeListItem( pID.data(), text, icon,
								change, parent, lev-1, expand );		
		insertItem( tmpItem );
		return;
	}	
	QString buffer=getParentID(pID.data());
	buffer.replace ( QRegExp( ".*/" ), "" );
	QString newText=buffer.right ( buffer.length()-1 );
	buffer=getParentID( pID.data() );
	insertPath ( buffer.data(), newText.data(), pIcon, TRUE, expand, TRUE );
	if ( sortedItems->find( pID.data() )>-1 ) return;
	ETreeListItem *tmpItem = new ETreeListItem( pID.data(), text, icon, 
								change, parent, lev-1, expand );		
	insertItem( tmpItem );
}

void ETreeList::insertChild( const char* path, const char* text, QPixmap& icon, bool expand )
{
	QString buffer=path, pathID;
	buffer.replace ( QRegExp ("/$"), "" );	//strip last slash
	int i=buffer.contains("/");	
	if ( i==0 ) {
		pathID=QString("/2")+text;			//item is in root path
	} else { 
		buffer.replace ( "/", "/1" );		//prepare path (for sorting)
		pathID=buffer+QString("/2")+text;	//2 for child item
	}
	if ( icon.isNull() ) icon=cIcon;
	insertPath ( pathID.data(), text, icon, FALSE, expand, FALSE );
}

void ETreeList::insertParent( const char* path, const char* text, QPixmap& icon, bool expand )
{
	QString buffer=path, pathID;
	buffer.replace ( QRegExp ("/$"), "" );	//strip last slash
	int i=buffer.contains("/");	
	if ( i==0 ) {
		pathID=QString("/1")+text;			//item is in root path
	} else { 
		buffer.replace ( "/", "/1" );		//prepare path (for sorting)
		pathID=buffer+QString("/1")+text;	//1 for parent item
	}
	bool chgIcon=TRUE;
	if ( icon.isNull() ) icon=pIcon;
	else chgIcon=FALSE;
	insertPath ( pathID.data(), text, icon, TRUE, expand, chgIcon );	
}

void ETreeList::mousePressEvent( QMouseEvent *e )
{
	int itemRow = findItem( e->pos().y() );
	if ( itemRow<0 ) return;
	ETreeListItem *itemClicked=(ETreeListItem *) item( itemRow );
	int lev=itemClicked->getLevel();
	int dx=ident/2+ident*lev;
	int dy=QMAX( itemClicked->pixmap()->height()+1, fontMetrics().lineSpacing()+1 )/2;
	if (dy%2==1) dy++;
	QRect inBox ( dx-1, dy-4, 9, 9 );				//Branchbox coordinates
	int *yCell=&dy;
	if ( QListBox::itemYPos( itemRow, yCell ) ) {
		ETreeListItem *tmpItem;
		int y=e->pos().y()-*yCell-2;				//calculate yPos in Row
		QPoint p ( e->pos().x()-2, y );
		int indexAllItems=sortedItems->find ( itemClicked->getPathID() );
		if ( indexAllItems<0 ) {	//root item is highlighted
			setCurrentItem( 0 );
			emit highlightedItem ( "/" );
		} else {	
			tmpItem=allItems->at( indexAllItems );
			if ( inBox.contains( p ) && itemClicked->isParent() &&
								hasChild( itemClicked->getPathID() )) {
				tmpItem->setExpand( !tmpItem->isExpanded() );
				if ( tmpItem->isExpanded() ) emit expandedItem ( tmpItem->getPath() );
				else emit collapsedItem( tmpItem->getPath() );
				updateTree(tmpItem->getPath());
			} else {
				if ( itemRow != -1 ) {
					setCurrentItem( itemRow );
//					toggleCurrentItem();
 				} else if ( contentsRect().contains( e->pos() ) &&
					lastRowVisible() >= (int) count() ) {
					setCurrentItem( count()-1 ); 
//					toggleCurrentItem();
 				}		
				emit highlightedItem ( tmpItem->getPath() );
			}
		}
	}
}

void ETreeList::paintCell( QPainter *p, int row, int )
{
    ETreeListItem *tli=(ETreeListItem*) item( row );
    if ( !tli ) return;
    QColorGroup g = colorGroup();
    QRect br=tli->boundingRect( p );			//bounding Rect for Item Text
	if ( isSelected( row ) ) {
		QColor	 fc;							//fill color
		if ( style() == WindowsStyle )
			fc = QApplication::winStyleHighlightColor();
		else fc = g.text();
		p->fillRect( br, fc );
		p->setPen( style() == WindowsStyle ? white : g.base() );
		p->setBackgroundColor( fc );
	} else {
		p->setBackgroundColor( g.base() );
		p->setPen( g.text() );
    }
    tli->paint( p );
    p->save();
    QString pID=getParentID( tli->getPathID() );
    p->setPen(darkGray);
    drawBranch( p, pID.data() );
	if ( hasChild(tli->getPathID()) ) {			//draw branch-boxes
	    if ( tli->isParent() ) {
    		p->setPen(black);
			p->setPen(SolidLine);
			p->setBrush( QBrush ( g.base() ) );
			int lev=tli->getLevel();
			int dx=ident/2+ident*lev;
			int dy=tli->height(p)/2;
			p->drawRect ( dx-1, dy-4, 9, 9 );
			p->drawLine ( dx+1, dy, dx+5, dy );
			if ( !tli->isExpanded() ) p->drawLine ( dx+3, dy-2, dx+3, dy+2 ); 
		}
    }
    p->restore();
    if ( currentItem() == row && hasFocus() ) {
	if ( style() == WindowsStyle ) {
	    p->drawWinFocusRect( br.x()+1, br.y()+1 , br.width()-2, br.height()-2,
				 QApplication::winStyleHighlightColor() );
	} else {
	    if ( isSelected( row ) )
		p->setPen( g.base() );
	    else
		p->setPen( g.text() );
	    p->setBrush( NoBrush );
	    p->drawRect( br.x()+1, br.y()+1 , br.width()-2, br.height()-2 );
	}
    }
    p->setBackgroundColor( g.base() );
    p->setPen( g.text() );
}

void ETreeList::drawBranch( QPainter *p, const char* path )
{
	QString pathID=path;
	if ( pathID=="/" ) return;
	uint z=sortedItems->find( path );
	ETreeListItem *tmp=allItems->at ( z );
	int lev=tmp->getLevel();
	if ( !tmp->isLastChild() ) {
		for ( int i=0; i<height(); i=i+2 ) 
			p->drawPoint( ident/2+3+ident*lev, i );
	}
	pathID=getParentID( path );
	drawBranch ( p, pathID.data() );	//do recursivly drawBranch from right to left
}

void ETreeList::updateTree( const char* path="/" )
{
	bool aUp=autoUpdate();
	setAutoUpdate(FALSE);
	setUpdatesEnabled( FALSE );
	ETreeListItem *changedItem=0;
	int pos=0;
	for ( int i=count()-1; i>0; i-- ) {	//must be grater zero cause of rootItem
		ETreeListItem* tmp=(ETreeListItem *) item ( i );
		QString p=tmp->getPath();
		if ( p==path ) {
			pos=i;						//item Position in QListBox 
			int index=sortedItems->find( tmp->getPathID() );
			changedItem=allItems->at( index );
			if ( showOpenFolder() && changedItem->changeIcon() ) {
				QPixmap pix=*(changedItem->pixmap());				
				if (changedItem->isExpanded()) pix=poIcon;
				else pix=pIcon;
				ETreeListItem *newItem=new ETreeListItem( 
					changedItem->getPathID(), changedItem->text(),
					pix, TRUE, changedItem->isParent(), 
					changedItem->getLevel(), changedItem->isExpanded()
				);
				newItem->setLastChild( changedItem->isLastChild() );
				QListBox::insertItem( newItem, pos );
				QListBox::removeItem( pos+1 );
			} else tmp->setExpand( changedItem->isExpanded() );
		} else if ( p.contains( path ) ) {		//deleting all Items,
	 		QListBox::removeItem(i);			//which contain Path
		}
	}
	if ( changedItem!=0 ) updateList( changedItem->getPathID(), pos );
	else updateList ( "/", 0 );
	setAutoUpdate(aUp);
	setUpdatesEnabled( TRUE );
	repaint();
}

int ETreeList::updateList ( const char* path, int pos )
{ 							//update recursivly from left to right
	if ( sortedItems->count()==0 ) return 0;	//return if list is empty
	QString pathID=path;
	int parentIndex;
	bool drawBranch;
	if ( pathID=="/" ) {
		parentIndex=-1;
		drawBranch=TRUE;
	} else {
		parentIndex=sortedItems->find( path );
		ETreeListItem *helper=allItems->at( parentIndex );
		drawBranch=helper->isExpanded();
	}
	ETreeListItem *tmp=allItems->at( parentIndex+1 );
	int offset=1, inserted=0;
	while ( tmp!=0 ) {
		QString pID=getParentID( tmp->getPathID() );
		if ( pathID==pID && drawBranch ) {
			QPixmap pix=*(tmp->pixmap());
			if ( showOpenFolder() && tmp->changeIcon() ) {
				if ( tmp->isExpanded() ) pix=poIcon;
				else pix=pIcon;
			}
			ETreeListItem *newItem=new ETreeListItem( 
					tmp->getPathID(), tmp->text(), pix, tmp->changeIcon(), 
					tmp->isParent(), tmp->getLevel(), tmp->isExpanded()
			);
			newItem->setLastChild( tmp->isLastChild() );
			QListBox::insertItem( newItem, pos+offset+inserted );
			if ( tmp->isParent() && tmp->isExpanded() && hasChild( tmp->getPathID() ) )
				inserted=inserted+updateList( tmp->getPathID(), pos+offset+inserted );
			if ( tmp->isLastChild() ) {
				return inserted+offset;
			}
			offset++;
		}
		tmp=allItems->next();
	}
	return inserted+offset;
}

bool ETreeList::hasChild( const char* path )
{
	uint z=sortedItems->find( path );
	if ( z==sortedItems->count()-1 ) return FALSE; 		//if is last item
	QString pathID=path;
	QString pID=getParentID( sortedItems->at( z+1 ) );
	if ( pathID==pID ) return TRUE;
	return FALSE;		
}

ETreeListItem::ETreeListItem( const char* path, const char *s, 
				const QPixmap& p, bool chIcon, bool isParent,
				int lev, bool expand )
	: QListBoxItem()
{ 
	setText( s ); 
	pm=p;
	parentItem=isParent;
	chgIcon=chIcon;
	itemPath=path;
	level=lev;
	doExpand=expand;
}


const char *ETreeListItem::getPathID()
{
	// call () operator
	return itemPath;
};


ETreeListItem::~ETreeListItem()
{
}

const QString ETreeListItem::getPath()
{
	QString buffer=itemPath.data();
	buffer.replace ( "/.", "/" );
	return buffer.data();
}

void ETreeListItem::paint( QPainter *p )
{
	p->save();	QPen pen;
	p->setPen( darkGray );
	if (level!=-1) {	//not rootItem
		if ( isLastChild() ) {
			for ( int i=0; i<height( p )/2; i=i+2 ) 
				p->drawPoint( ident/2+3+ident*level, i );
		} else {
			for ( int i=0; i<height( p ); i=i+2 ) 
				p->drawPoint( ident/2+3+ident*level, i );
		}
		for ( uint i=ident/2+4+ident*level; i<ident+6+ident*level; i=i+3 )
			p->drawPoint( i, height( p )/2 );
	}
	p->restore();
	p->drawPixmap( 3+ident+level*ident+(ident-pm.width())/2, 1, pm );
	QFontMetrics fm = p->fontMetrics();
        int yPos;                       // vertical text position
        if ( pm.height() < fm.height() )
        	yPos = fm.ascent() + fm.leading()/2;
        else
        	yPos = pm.height()/2 - fm.height()/2 + fm.ascent();
//						pm space branch
      	p->drawText( ident+8+ident+level*ident, yPos, text() );
}

int ETreeListItem::height(const QListBox *lb ) const
{
	int i=QMAX( pm.height()+1, lb->fontMetrics().lineSpacing()+1 );
	if (i%2==1) i++;	//to fit dotted line
	return i;
}

int ETreeListItem::height(const QPainter *p ) const
{
	int i=QMAX( pm.height()+1, p->fontMetrics().lineSpacing()+1 );
	if (i%2==1) i++;	//to fit dotted line
	return i;
}

int ETreeListItem::width(const QListBox *lb ) const
{
	if ( (uint)pm.width()>ident ) ident=pm.width();	
//	       pixmap  space  text			   		branch	   			space
	return ident+10+lb->fontMetrics().width( text() )+level*ident+ident+3;
}

QRect ETreeListItem::boundingRect( QPainter *p )
{
	QFontMetrics fm = p->fontMetrics();
	int rectX=ident+8+level*ident+ident-4;
	int rectY;
        if ( pm.height() < fm.height() )
        	rectY = 0;
        else
        	rectY = pm.height()/2 - fm.height()/2;
	int rectW=fm.width(text())+7;
	int rectH=fm.height()+1;
	return QRect ( rectX, rectY, rectW, rectH );
}
