/***********************************************************
Copyright 1991, 1992, 1993, 1994 by Stichting Mathematisch Centrum,
Amsterdam, The Netherlands.

                        All Rights Reserved

Permission to use, copy, modify, and distribute this software and its 
documentation for any purpose and without fee is hereby granted, 
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in 
supporting documentation, and that the names of Stichting Mathematisch
Centrum or CWI not be used in advertising or publicity pertaining to
distribution of the software without specific, written prior permission.

STICHTING MATHEMATISCH CENTRUM DISCLAIMS ALL WARRANTIES WITH REGARD TO
THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
FITNESS, IN NO EVENT SHALL STICHTING MATHEMATISCH CENTRUM BE LIABLE
FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT
OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

******************************************************************/


/* Cmap objects */

#include "Python.h"
#include "import.h"

PyObject *format_rgb, *format_rgb_b2t, *format_map, *format_map_b2t,
	*format_xmap;

typedef struct {
	PyObject_HEAD
	long	*map;
	int	size;
} cmapobject;

static PyObject *errobject;

staticforward PyTypeObject Cmaptype;

#define is_cmapobject(v)		((v)->ob_type == &Cmaptype)

static char doc_cmap[] =
  "This sequence object contains a colormap.";

static cmapobject *
newcolormapobject(size)
    int  size;
{
	cmapobject *xp;
	
	xp = PyObject_NEW(cmapobject, &Cmaptype);
	if (xp == NULL)
		return NULL;
	xp->size = size;
	if( (xp->map = (long *)malloc(size*sizeof(long))) == NULL) {
	    PyErr_NoMemory();
	    return NULL;
	}
	memset(xp->map, 0, size*sizeof(long));
	return xp;
}

static int
initcolormap(self, map)
    cmapobject *self;
    long *map;
{
    memcpy(self->map, map, self->size*sizeof(long));
}

static int
initrgbcolormap(self, rmap, gmap, bmap)
    cmapobject *self;
    unsigned char *rmap, *gmap, *bmap;
{
    register int i;
 
    for(i=0; i<self->size; i++)
	self->map[i] = rmap[i] | (gmap[i]<<8) | (bmap[i]<<16);
}

/* Cmap methods */

static void
cmap_dealloc(xp)
	cmapobject *xp;
{
	free(xp->map);
	PyMem_DEL(xp);
}

static char doc_map[] = "Convert an 8-bit image to RGB using this colormap.\n"
  "Args: data, width, source_fmt, dst_format. Returns new image";

static PyObject *
cmap_map(self, args)
    cmapobject *self;
    PyObject *args;
{
    PyObject *rv;
    long *map, *rvdata;
    unsigned char *data, pixel;
    int mapsize, datasize;
    int x, y;
    int srcwidth, dstwidth, height;
    PyObject *srcfmt, *dstfmt;
    int ok;

    /*
    ** We need the width here so we can correct for the 32bit alignment
    ** of the 8-bit source image
    */
    if ( !PyArg_ParseTuple(args, "s#iOO", &data, &datasize, &dstwidth,
		  &srcfmt, &dstfmt) )
	return NULL;
    ok = 0;
    if ( (srcfmt == format_map || srcfmt == format_xmap)
	&& dstfmt == format_rgb )
	ok = 1;
    if ( srcfmt == format_map_b2t && dstfmt == format_rgb_b2t )
	ok = 1;
    if (!ok) {
	PyErr_SetString(errobject, "Unsupported colormap conversion");
	return NULL;
    }
    if ( srcfmt == format_xmap )
	srcwidth = dstwidth;
    else
	srcwidth = (dstwidth+3) & ~3;
    height = datasize/srcwidth;
    if( (rv = PyString_FromStringAndSize(0, height*dstwidth*4)) == NULL)
	return NULL;
    rvdata = (long *)PyString_AsString(rv);
    map = self->map;
    mapsize = self->size;
    for(y=0; y<height; y++) {
	for(x=0; x<dstwidth; x++) {
	    pixel = data[x+y*srcwidth];
	    if ( (int)pixel >= mapsize ) {
		PyErr_SetString(PyExc_IndexError, "colormap index out of range");
		return NULL;
	    }
	    rvdata[x+y*dstwidth] = map[pixel];
	}
    }
    return rv;
}

static char doc_dither[] = "Dither a 24-bit image to 8 bits using this map.\n"
  "Args: data, width, height, source_format, floyd\n"
  "floyd (optional) specifies (slow) Floyd-Steinberg error diffusion\n"
  "Returns: newdata, newformat (currently always imgformat.xcolormap)";

static PyObject *
cmap_dither(self, args)
    cmapobject *self;
    PyObject *args;
{
    PyObject *rv;
    unsigned char *data;
    long *rvdata;
    int datasize;
    int i;
    PyObject *format;
    int floyd = 0;
    int width, height;

    if ( !PyArg_ParseTuple(args, "s#iiO|i", &data, &datasize, &width, &height, &format,
		  &floyd) )
	return NULL;
    if ( format != format_rgb ) {
	PyErr_SetString(errobject, "Unsupported source image format");
	return NULL;
    }
    if ( width*height*4 != datasize ) {
	PyErr_SetString(errobject, "Incorrect sized source image data");
	return NULL;
    }
    if( (rv = PyString_FromStringAndSize(0, width*height)) == NULL)
	return NULL;
    rvdata = (long *)PyString_AsString(rv);
    i = mppm_dither(data, width, height, self->map, self->size, floyd, rvdata);
    if ( i < 0 ) {
	PyErr_NoMemory();
	return NULL;
    }
    Py_INCREF(format_xmap);
    return Py_BuildValue("OO", rv, format_xmap);
}

static char doc_map8[] =
  "Map the 8-bit string argument to another 8-bit string using the\n"
  "'red' entry in this colormap";
 
static PyObject *
cmap_map8(self, args)
    cmapobject *self;
    PyObject *args;
{
    PyObject *rv;
    long *map;
    unsigned char *data, *rvdata, pixel;
    int mapsize, datasize;
    int i;

    if ( !PyArg_ParseTuple(args, "s#", &data, &datasize) )
	return NULL;
    if( (rv = PyString_FromStringAndSize(0, datasize)) == NULL)
	return NULL;
    rvdata = (unsigned char *)PyString_AsString(rv);
    map = self->map;
    mapsize = self->size;
    for(i=0; i<datasize; i++) {
	pixel = data[i];
	if ( (int)pixel >= mapsize ) {
	    PyErr_SetString(PyExc_IndexError, "colormap index out of range");
	    return NULL;
	}
	rvdata[i] = map[pixel];
    }
    return rv;
}

static char doc_getmacmapdata[] =
  "Get a string containing colormap info in Macintosh format";

static PyObject *
cmap_getmacmapdata(self, args)
    cmapobject *self;
    PyObject *args;
{
    PyObject *rv;
    long *map, pixel;
    unsigned short *rvdata;
    int mapsize;
    int i;

    if ( !PyArg_ParseTuple(args, "") )
	return NULL;
    map = self->map;
    mapsize = self->size;
    if( (rv = PyString_FromStringAndSize(0, mapsize*4*sizeof(short))) == NULL)
	return NULL;
    rvdata = (unsigned short *)PyString_AsString(rv);
    for(i=0; i<mapsize; i++) {
	pixel = map[i];
	*rvdata++ = i;
	*rvdata++ = ((pixel<<8) & 0xff00) | ( pixel     &0xff);
	*rvdata++ = ( pixel     & 0xff00) | ((pixel>> 8)&0xff);
	*rvdata++ = ((pixel>>8) & 0xff00) | ((pixel>>16)&0xff);
    }
    return rv;
}


static struct PyMethodDef cmap_methods[] = {
	{"map",			(PyCFunction)cmap_map,	1,	doc_map},
	{"map8",		(PyCFunction)cmap_map8,	1,	doc_map8},
	{"dither",		(PyCFunction)cmap_dither,	1,
     						     	doc_dither},
	{"getmacmapdata",	(PyCFunction)cmap_getmacmapdata, 1,
     							doc_getmacmapdata},
	{NULL,		NULL}		/* sentinel */
};

static PyObject *
cmap_getattr(xp, name)
	cmapobject *xp;
	char *name;
{
        /* The next two are mainly for reasonably-easy interfacing from
	** the C-modules that need to access colormap data.
	*/
        if ( strcmp(name, "_map_as_string") == 0 )
	    return PyString_FromStringAndSize((char *)xp->map,
					xp->size*sizeof(long));
        if ( strcmp(name, "_map_length") == 0 )
	    return PyInt_FromLong(xp->size);
	if ( strcmp(name, "__doc__") == 0 )
	    return PyString_FromString(doc_cmap);
	return Py_FindMethod(cmap_methods, (PyObject *)xp, name);
}

/* Functions to access a colormap as a sequence */

static int
cmap_length(a)
	cmapobject *a;
{
	return a->size;
}

static PyObject *
cmap_item(self, i)
	cmapobject *self;
	int i;
{
    long pixel;
    
    if ( i < 0 || i >= self->size ) {
	PyErr_SetString(PyExc_IndexError, "colormap index out of range");
	return NULL;
    }
    pixel = self->map[i];
    return Py_BuildValue("(iii)", pixel&0xff, (pixel>>8)&0xff, (pixel>>16)&0xff);
}

static PyObject *
cmap_slice(a, ilow, ihigh)
	cmapobject *a;
	int ilow, ihigh;
{
    PyErr_SetString(errobject, "Operation not supported on colormap");
    return NULL;
}

static PyObject *
cmap_concat(a, bb)
	cmapobject *a;
	PyObject *bb;
{
    PyErr_SetString(errobject, "Operation not supported on colormap");
    return NULL;
}

static PyObject *
cmap_repeat(a, n)
	cmapobject *a;
	int n;
{
    PyErr_SetString(errobject, "Operation not supported on colormap");
    return NULL;
}

static int
cmap_ass_slice(a, ilow, ihigh, v)
	cmapobject *a;
	int ilow, ihigh;
	PyObject *v;
{
    PyErr_SetString(errobject, "Operation not supported on colormap");
    return -1;
}

static int
cmap_ass_item(a, i, v)
	cmapobject *a;
	int i;
	PyObject *v;
{
        int red, green, blue;
	long rgb;
	
	if (i < 0 || i >= a->size) {
	    PyErr_SetString(PyExc_IndexError, "Colormap assignment index out of range");
	    return -1;
	}
	if (!PyArg_ParseTuple(v, "iii", &red, &green, &blue))
	    return -1;
	rgb = (red&0xff) | ((green&0xff)<<8) | ((blue&0xff)<<16);
	a->map[i] = rgb;
	return 0;
}

static PySequenceMethods cmap_as_sequence = {
	(inquiry)cmap_length,			/*sq_length*/
	(binaryfunc)cmap_concat,		/*sq_concat*/
	(intargfunc)cmap_repeat,		/*sq_repeat*/
	(intargfunc)cmap_item,			/*sq_item*/
	(intintargfunc)cmap_slice,		/*sq_slice*/
	(intobjargproc)cmap_ass_item,		/*sq_ass_item*/
	(intintobjargproc)cmap_ass_slice,	/*sq_ass_slice*/
};


static PyTypeObject Cmaptype = {
	PyObject_HEAD_INIT(&PyType_Type)
	0,			/*ob_size*/
	"imgcmap",		/*tp_name*/
	sizeof(cmapobject),	/*tp_basicsize*/
	0,			/*tp_itemsize*/
	/* methods */
	(destructor)cmap_dealloc, /*tp_dealloc*/
	0,			/*tp_print*/
	(getattrfunc)cmap_getattr, /*tp_getattr*/
	0,			 /*tp_setattr*/
	0,			/*tp_compare*/
	0,			/*tp_repr*/
	0,			/*tp_as_number*/
	&cmap_as_sequence,	/*tp_as_sequence*/
	0,			/*tp_as_mapping*/
	0,			/*tp_hash*/
};

static char doc_new[] = "Create a new colormap from a 4-byte-pixel string\n";

static PyObject *
cmap_new(self, args)
	PyObject *self;
	PyObject *args;
{
        long *colormap;
	int size;
	cmapobject *obj;
	
	if (!PyArg_ParseTuple(args, "s#", &colormap, &size))
	    return NULL;
	if ( size & 3 ) {
	    PyErr_SetString(errobject, "colormap entries are 4 bytes each");
	    return NULL;
	}
	if ((obj = newcolormapobject(size/4)) == NULL)
	    return NULL;
	initcolormap(obj, colormap);
	return (PyObject *)obj;
}

static char doc_new3[] = "Create a new colormap from 3 1-byte-pixel strings";

static PyObject *
cmap_new3(self, args)
	PyObject *self;
	PyObject *args;
{
        char *rmap, *gmap, *bmap;
	int rlen, glen, blen;
	cmapobject *obj;
	
	if (!PyArg_ParseTuple(args, "s#s#s#", &rmap, &rlen, &gmap, &glen, &bmap, &blen))
	    return NULL;
	if ( rlen != glen || rlen != blen ) {
	    PyErr_SetString(errobject, "colormaps should all be same size");
	    return NULL;
	}
	if ((obj = newcolormapobject(rlen)) == NULL)
	    return NULL;
	initrgbcolormap(obj, rmap, gmap, bmap);
	return (PyObject *)obj;
}

static PyObject *
internal_newcmap(map, size)
        long *map;
        int size;
{
        cmapobject *rv;

	if ( (rv=newcolormapobject(size)) == NULL)
	    return NULL;
	initcolormap(rv, map);
	return (PyObject *)rv;
}

static char doc_fromimage[] =
  "Create a reasonable colormap for an RGB image, if possible\n"
  "Args: data, width, height, source_format, mapsize\n"
  "mapsize is optional (default: 256) and gives the size of the map";

static PyObject *
cmap_fromimage(self, args)
	PyObject *self;
	PyObject *args;
{
	long *image;
	int width, height;
	PyObject *format;
	cmapobject *obj;
	int maplen = 256;
	int size, rv;
	
	if (!PyArg_ParseTuple(args, "s#iiO|i", &image, &size, &width, &height,
		     &format, &maplen))
	    return NULL;
	if ( format != format_rgb ) {
	    PyErr_SetString(errobject, "Unsupported source image format");
	    return NULL;
	}
	if ( size != width*height*4 ) {
	    PyErr_SetString(errobject, "Incorrect size image");
	    return NULL;
	}
	if ((obj = newcolormapobject(maplen)) == NULL)
	    return NULL;
	rv = mppm_genmap(image, width, height, obj->map, maplen);
	if ( rv < 0 ) {
	    PyErr_NoMemory();
	    Py_DECREF(obj);
	    return NULL;
	}
	if ( rv == 0 ) {
	    PyErr_SetString(errobject, "Too many colors in source image");
	    Py_DECREF(obj);
	    return NULL;
	}
	return (PyObject *)obj;
}


/* List of functions defined in the module */

static struct PyMethodDef cmap_module_methods[] = {
	{"new",		cmap_new,	1,	doc_new},
	{"new3",	cmap_new3,	1,	doc_new3},
	{"fromimage",	cmap_fromimage,	1,	doc_fromimage},
	{NULL,		NULL}		/* sentinel */
};


/* Initialization function for the module (*must* be called initimgcmap) */
static char doc_imgcolormap[] = "Support for colormap objects";

void
initimgcolormap()
{
	PyObject *m, *d, *x, *formatmodule, *formatdict;

	/* Create the module and add the functions */
	m = Py_InitModule("imgcolormap", cmap_module_methods);

	/* Add some symbolic constants to the module */
	d = PyModule_GetDict(m);
	errobject = PyString_FromString("imgcmap.error");
	PyDict_SetItemString(d, "error", errobject);
	PyDict_SetItemString(d, "_C_newmap",
		   PyInt_FromLong((long)&internal_newcmap)); /* ignore  warn */
	x = PyString_FromString(doc_imgcolormap);
	PyDict_SetItemString(d, "__doc__", x);

	/* Get supported formats */
	if ((formatmodule = PyImport_ImportModule("imgformat")) == NULL)
	    Py_FatalError("imgxxx depends on imgformat");
	if ((formatdict = PyModule_GetDict(formatmodule)) == NULL)
	    Py_FatalError("imgformat has no dict");

	format_rgb = PyDict_GetItemString(formatdict,"rgb");
	format_rgb_b2t = PyDict_GetItemString(formatdict,"rgb_b2t");
	format_map = PyDict_GetItemString(formatdict,"colormap");
	format_map_b2t = PyDict_GetItemString(formatdict,"colormap_b2t");
	format_xmap = PyDict_GetItemString(formatdict, "xcolormap");

	/* Check for errors */
	if (PyErr_Occurred())
		Py_FatalError("can't initialize module imgcmap");
}
