"""
A Field and Table class

This file implements a relatively generic Field and Table class. Currently
they are utilized as an ASCII file translation method whereby files of 
delimited, fixed width and comma separated values can be mixed and matched
for reading and writing.

The Field class contains methods for defining length, type and formatters.

The Table class collects Field definitions and exposes methods of parsing
strings from and to the various supported files. There are also methods
of handling a few SQL type statements.

(c)1998 Jim Knight - Digital Artisans Inc.
<jimbag@kw.igs.net>
"""
# ------------------------------------------------------------------------
# History:
# --------
# $Log: db.py,v $
# Revision 1.2  1998/02/11 19:09:55  jimbag
# Added limited SQL support
#
# Revision 1.1  1998/01/23 07:43:00  jimbag
# Initial revision
#
# -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=

import string

# this has nothing to do with the RCS version
__version__ = '0.1'

# ========================================================================
# base Field class definition
class Field:
    # --------------------------------------------------------------------
    def __init__( self, name, **kw ):
	"""
	class Field
	
	Arguments
	name    -- the field name
	
	Keyword arguments
	width   -- the field width (default 0 (variable))
	value   -- the current value
	default -- the default value if none is supplied
	type    -- 'C' for character or 'N' for numeric

	"""
	self.name = name
	self.width = 0
	self.value = None       # this should always be a string
	self.default = None
	self.type = 'C'         # field type. Mainly used for CSV stuff
	self.pad = 0            # false ( not used yet )
	for item in kw.keys():
	    if self.__dict__.has_key( item ):
		self.__dict__[item] = kw[item]
	    else:
		raise KeyError,`item`+' is not a valid member of Field class'
	if self.value == None:
	    self.value = self.default

    # --------------------------------------------------------------------
    # print the field definition
    def __repr__( self ):
	"""
	Field.__repr__()

	"""
	s = "Field( '%s', type='%s'" % (self.name, self.type)
	if self.width > 0:
	    s = s + ", width=%d" % self.width
	s = s + ' )'
	return s

    # --------------------------------------------------------------------
    # return the formatted field value    
    def __str__( self ):
	"""
	Field.__str__()

	"""
	return str(self.format( self.value ))

    # --------------------------------------------------------------------
    # return the len of the field
    def __len__( self ):
	"""
	Field.__len__()

	"""
	return self.width

    # --------------------------------------------------------------------
    # Assign a value to a field
    def assign( self, value ):
	"""
	Field.assign( value )
	
	Assign a value to the field. If no value is supplied then the
	default value is used. If the field has a width then the
	value is truncated to fit.

	"""
	if value == None:
	    value = self.default 
	else:
	    if self.width > 0:
		if len(value) > self.width:
		    value = value[:self.width]
	self.value =  value 

    # --------------------------------------------------------------------
    # This is just a placeholder for derived classes to override
    # this version strips leading and trailing whitespace
    def format( self, value ):
	"""
	Field.format( value )

	"""
	if value:
	    return string.strip(value)
	else:
	    return value

# ========================================================================
# These are derived classes for handling whatever you invent.
# They override the formatter in the base Field class
# ========================================================================
# An integer field
class FieldInt( Field ):
    def __init__( self, name, **kw ):
	kw['type'] = 'N'
	apply( Field.__init__, (self, name),  kw )

    # --------------------------------------------------------------------
    def format( self, value ):
	return string.atoi( value )

# ========================================================================
# a float field
class FieldFloat( Field ):
    def __init__( self, name, **kw ):
	kw['type'] = 'N'
	apply( Field.__init__, (self, name), kw )

    # --------------------------------------------------------------------
    def format( self, value ):
	return string.atof( value )

# ========================================================================
# currency
class FieldDollar( Field ):
    # --------------------------------------------------------------------
    def format( self, value ):
	val = string.atof( value ) / 100
	return '$%.02f' % val


# ========================================================================
# A table class
class Table:
    # --------------------------------------------------------------------
    def __init__( self, name, **kw ):
	"""
	class Table

	"""
	self.name = name
	for item in kw.keys():
	    if self.__dict__.has_key( item ):
		self.__dict__[item] = kw[item]
	    else:
		raise KeyError,`item`+' is not a valid member of Table class'
	# the field names for lookups
	self.fldkey = {}
	# fields in order of addition for sequential access
	self.fldseq = []


    # --------------------------------------------------------------------
    # add a field to the table
    def addField( self, fld ):
	"""
	Table.addField( Field fld )

	"""
	self.fldkey[fld.name] = fld
	self.fldseq.append( fld )

    # --------------------------------------------------------------------
    # print out all the fields
    def __str__( self ):
	"""
	Table.__str__()

	"""
	s = ''
	for fld in self.fldseq:
	    s = s + '%s(%d%c): [%s]\n' % (fld.name, fld.width, fld.type, fld)
	return s

    # --------------------------------------------------------------------
    # set a field value
    def __setitem__( self, name, value ):
	"""
	Table[ name ] = value

	"""
	if self.fldkey.has_key( name ):
	    self.fldkey[name].assign( value )

    # --------------------------------------------------------------------
    # get a field value
    def __getitem__( self, name ):
	"""
	Table[ name ]

	"""
	if self.fldkey.has_key( name ):
	    return str(self.fldkey[name])


    # --------------------------------------------------------------------
    # return the length of all the fields
    def __len__( self ):
	"""
	Table.__len__()

	"""
	n = 0
	for fld in self.fldseq:
	    n = n + len( fld )
	return n

    # --------------------------------------------------------------------
    # Add a table to this table
    def __add__( self, tbl ):
	"""
	Table.__add__()
	
	"""
	for fld in tbl.fldseq:
	    if self.fldkey.has_key( fld.name ):
		raise KeyError,`fld.name`+' is duplicated in this instance'
	    self.addField( fld )
	return self

    # --------------------------------------------------------------------
    # private method of getting field names from array
    # returns all the field names if no arg provided
    def _get_field_names( self, fields = None ):
	"""
	Table._get_field_names( fields )

	"""
	rflds = []
	if fields == None:
	    # use all the fields in the table
	    for fld in self.fldseq:
		rflds.append( fld.name )
	else:
	    # use the passed in fields
	    rflds = fields
	return rflds

    # --------------------------------------------------------------------
    # read a fixed length string and populate the field array
    def read_fixed( self, line, fields = None ):
	"""Table.read_fixed( line, fields=None )

	Reads a string of fixed length fields and
	sets the values of the fields to the values read. Uses
	the length of the field to determine the fixed length to read.
	If an array of field names is supplied in 'fields' just
	the named fields are populated.
	"""
        rflds = self._get_field_names( fields )
	# counters
	offs = 0
	n = 0
	# bomb thru the fields
	for field in rflds:
	    wide = len( self.fldkey[field] )
	    if wide > 0:
		val = line[offs:offs+wide]
		offs = offs + wide
		self.fldkey[field].assign( val )
		n = n + 1
	    # if this is a variable length field
	    # then read to the end of the string and
	    # bail out
	    else:
		val = line[offs:-1]
		self.fldkey[field].assign( val )
		n = n + 1
		break
	# return the number of fields converted
	return n


    # --------------------------------------------------------------------
    # return a fixed length string of the given fields
    def write_fixed( self, fields = None ):
	"""
	Table.write_fixed( fields )

	"""
	rflds = self._get_field_names( fields )
	s = ''
	for fld in rflds:
	    flen = self.fldkey[fld].width
	    # if it's a zero length field make it the len of the value
	    if not flen:
		flen = len(self.fldkey[fld].value)
	    # see if it's a numeric field type
	    if self.fldkey[fld].type == 'N':
		s = s + '%s' % string.zfill( str( self.fldkey[fld] ), flen )
	    else:
		s = s + '%s' % string.ljust( str( self.fldkey[fld] ), flen )
	return s
	

    # --------------------------------------------------------------------
    # read a delimited string and fill the field values with
    # the parsed values
    def read_delim( self, line, delim = ':', fields = None ):
	"""
	Table.read_delim( line, delim = ':', fields = None )

	"""
	items = string.splitfields( line, delim )
	rflds = self._get_field_names( fields )
	n = 0
	for fld in rflds:
	    if n < len( items ):
		self.fldkey[fld].assign( items[n] )
		n = n + 1
	return n

    # --------------------------------------------------------------------
    # return a delimited string of the field values
    def write_delim( self, delim = ':', fields = None ):
	"""
	Table.write_delim( delim = ':', fields = None )

	"""
	rflds = self._get_field_names( fields )
	s = ''
	first = 1
	for fld in rflds:
	    if not first:
		s = s + '%s' % delim
	    s = s + '%s' % str(self.fldkey[fld])
	    first = 0
	return s

    # --------------------------------------------------------------------
    # read a string of comma separated values and plug into the fields
    def read_csv( self, line, fields = None ):
	"""
	Table.read_csv( line, fields = None )

	"""
	rflds = self._get_field_names( fields )
	# some handy flags
	infld = 0    # are we in a field or not
	field = ''   # the current field we are gathering
	nflds = 0    # the number of fields collected
        n = 0        # the current offset into rflds
	for ch in line:
	    if ch == ',':
		# we are done collecting
		if not infld:
		    self.fldkey[rflds[n]].assign( field )
		    field = ''          # blank the storage
		    nflds = nflds + 1   # total fields
		    n = n + 1           # next field
		else:
		    field = field + ch
	    # it's a quote so flip the flag
	    elif ch == '"':
		if infld:
		    infld = 0
		else:
		    infld = 1
	    # just collect the field
	    else:
		field = field + ch
	return nflds + 1

    # --------------------------------------------------------------------
    # return a string of field values as comma separated values
    def write_csv( self, fields = None ):
	"""
	Table.write_csv( fields = None )

	"""
	rflds = self._get_field_names( fields )
	s = ''
	first = 1
	for fld in rflds:
	    if not first:
		s = s + ','
	    # see if it's a numeric field type
	    if self.fldkey[fld].type == 'N':
		s = s + '%s' % str( self.fldkey[fld] )
	    else:
		s = s + '"%s"' % str( self.fldkey[fld] )
	    first = 0
	return s

    # --------------------------------------------------------------------
    def create_sqlinsert( self, fields = None ):
	"""
	Table.create_sqlinsert()

	"""
	rflds = self._get_field_names( fields )
	s = 'INSERT INTO ' + self.name + ' ( '
	vals = ' VALUES ( '
	first = 1
	for fld in rflds:
	    # Fields prefaced with an underscore do not make it
	    # into the list. This is a quick way of eliminating
	    # fields from the list without passing a huge
	    # string of fields to this function
	    if fld[0] <> '_':
		if not first:
		    s = s + ', '
		    vals = vals + ','
		s = s + '%s' % fld
		if self.fldkey[fld].type == 'N':
		    if not str(self.fldkey[fld]):
			vals = vals + '0'
		    else:
			vals = vals + '%s' % str( self.fldkey[fld] )
		else:
		    vals = vals + "'%s'" % string.replace( str( self.fldkey[fld] ), "'", " " )
		first = 0
	return s + ') ' + vals + ' )'

    # --------------------------------------------------------------------
    def create_sqlupdate( self, fields = None ):
	"""
	Table.create_sqlupdate()

	"""
	rflds = self._get_field_names( fields )
	s = 'UPDATE ' + self.name + ' SET '
	first = 1
	for fld in rflds:
	    # Fields prefaced with an underscore do not make it
	    # into the list. This is a quick way of eliminating
	    # fields from the list without passing a huge
	    # string of fields to this function
	    if fld[0] <> '_':
		if not first:
		    s = s + ', '
		s = s + '%s=' % fld
		if self.fldkey[fld].type == 'N':
		    if not str(self.fldkey[fld]):
			s = s +'0'
		    else:
			s = s + '%s' % str( self.fldkey[fld] )
		else:
		   s = s + "'%s'" % string.replace( str( self.fldkey[fld] ), "'", " " )
		first = 0
	return s + ' '

    # --------------------------------------------------------------------
    def create_sqlschema( self, fields = None ):
	"""
	Table.create_sqlschema()

	"""
	# There are a few obvious problems with this. First, all the
	# possible data types for all SQL engines are not represented.
	# Pick your engine and make appropriate changes.
	# Second, the character to terminate a query is not the same
	# across engines. Waddaya gonna do? *shrug*
	rflds = self._get_field_names( fields )
	s = 'CREATE TABLE ' + self.name + ' (\n'
	for fld in rflds:
	    if fld[0] <> '_':
		s = s + '\t%s ' % (fld)
		if self.fldkey[fld].type == 'N':
		    s = s + 'int ,\n'
		else:
		    s = s + 'char(%d),\n' % (self.fldkey[fld].width)
	# gotta eat the last newline and comma.
	s = s[:-2] + '\n'
	return s + '\t)/g'

# =========================[ End Listing ]================================
