#include <stdio.h>

#include <sybfront.h>
#include <sybdb.h>
#include <syberror.h>

#include "allobjects.h"
#include "modsupport.h"

static object	*SybaseError;	/* exception sybase.error */

typedef struct {
	OB_HEAD
	LOGINREC	*login;		/* login record */
	DBPROCESS	*dbproc;	/* database process struct */
} sybdbobject;

typedef struct {
	int			sybmsgno;
	char		*sybmsgtext; 
} syberrobject;

extern typeobject SybDbtype; /* Forward */

int msg_handler();			
int err_handler();

int err_handler(dbproc, severity, dberr, oserr, dberrstr, oserrstr)
    DBPROCESS		*dbproc;
    int             severity;
    int             dberr;
    int             oserr;
    char            *dberrstr;
    char            *oserrstr;
{
	object *errtuple;
	if ( (dbproc == NULL) || ( DBDEAD(dbproc) )) {
 		errtuple = Py_BuildValue("(is)", dberr, (char *)dberrstr);
    	PyErr_SetObject(SybaseError, errtuple);
	}
	return(INT_CANCEL);
}

int msg_handler(dbproc, msgno, msgstate, severity, msgtext, srvname, procname, line)
	DBPROCESS		*dbproc;
	DBINT			msgno;
	int				msgstate;
	int				severity;
	char			*msgtext;
	char			*srvname;
	char			*procname;
	DBUSMALLINT		line;
{
	if ( msgno != 5701 && msgno != 5703 ) 
	{
		object			*errtuple;

		if (dbproc == NULL ) {
 			errtuple = Py_BuildValue("(is)", msgno, (char *)msgtext);
    		PyErr_SetObject(SybaseError, errtuple);
		}
		else {
			syberrobject *syberr=malloc(sizeof(syberrobject));
			if(syberr==NULL)
			{	/* should do something else here */
				return(-1);	
			}
			syberr -> sybmsgno = msgno;
			syberr->sybmsgtext=malloc(strlen(msgtext)+1);
			if(syberr->sybmsgtext==NULL)
			{	/* should do something else here too */
				free(syberr); 
				return(-1);
			}
			strcpy(syberr->sybmsgtext, msgtext);

			if(dbgetuserdata(dbproc)!=0)
				free(dbgetuserdata(dbproc));
			dbsetuserdata(dbproc, syberr);
		}
	}
	return(0);
}

static sybdbobject *
newsybdbobject(char *user, char *passwd, char *server)
{
	sybdbobject		*self;
	syberrobject	*syberr;

	self = NEWOBJ(sybdbobject, &SybDbtype);
	if (self != NULL) {

		if (dbinit() == FAIL)
		{
			DECREF(self);
        	return NULL;
		}

		dberrhandle(err_handler);
		dbmsghandle(msg_handler);

		self->login = dblogin();
		if (user) {
			(void)DBSETLUSER(self->login, user);
		}
		if (passwd) {
			(void)DBSETLPWD(self->login, passwd);
		}
		self->dbproc = dbopen(self->login, server);
		dbloginfree(self->login);
		if(!self->dbproc) {
			DECREF(self);
			return NULL;
		}
	}
	return self;
}

/* OBJECT FUNCTIONS: sybdb */

/* Common code for returning pending results */
static object *
getresults (DBPROCESS *dbp)
{
	object *results;
	object *list;
	object *tuple;
	object *o;
	int retcode;
	int cols;
	int *fmt;
	int i;
	DBCHAR	dateinfo[32]; /* BAD Assumption on converted date */
	int	len;

	results = newlistobject(0);
	if(results==NULL)
		return NULL;
	while ((retcode = dbresults(dbp)) != NO_MORE_RESULTS) 
	{
		if (retcode == SUCCEED && DBROWS(dbp) == SUCCEED) 
		{
			list = newlistobject(0);
			if(list==NULL)
			{
				DECREF(results);
				return NULL;
			}
			cols = dbnumcols(dbp);
			fmt = (int *)malloc(sizeof(int) * cols);
			if(fmt==NULL)
			{
				PyErr_NoMemory();
				DECREF(list);
				DECREF(results);
				return NULL;
			}
			for (i = 1; i <= cols; i++) 
			{
				int xfmt;
				switch(xfmt=dbcoltype(dbp, i)) 
				{
					case SYBCHAR:
					case SYBTEXT:
					case SYBIMAGE:
					case SYBINT1:
					case SYBINT2:
					case SYBINT4:
					case SYBFLT8:
					case SYBDATETIME:
					case SYBBIT:
						fmt[i-1] = xfmt;
						break;
					default:
						PyErr_SetString(PyExc_TypeError, "unexpected column type in returned data");
						free(fmt);
						DECREF(list);
						DECREF(results);
						return NULL;
				}
			}
			while (dbnextrow(dbp) != NO_MORE_ROWS) 
			{
				tuple = newtupleobject(cols);
				if(tuple==NULL)
				{
					free(fmt);
					DECREF(list);
					DECREF(results);
					return NULL;
				}
				for (i = 1; i <= cols; i++) 
				{
					switch(fmt[i-1])
					{
						case SYBCHAR:
						case SYBTEXT: /* a hack */
						case SYBIMAGE:
						{
							char *dat;

							dat=(char *)dbdata(dbp, i);
							if(dat==NULL && (dbdatlen(dbp, i)==0))
							{
								INCREF(None);
								o=None;
							}
							else
								o=newsizedstringobject(dat, dbdatlen(dbp, i));
							break;
						}
						case SYBINT1:
						case SYBBIT:
						{
							char *dat;

							dat=(char *)dbdata(dbp, i);
							if(dat==NULL && (dbdatlen(dbp, i)==0))
							{
								INCREF(None);
								o=None;
							}
							else
								o=newintobject((long)*dat);
							break;
						}
						case SYBINT2:
						{
							short *dat;

							dat=(short *)dbdata(dbp, i);
							if(dat==NULL && (dbdatlen(dbp, i)==0))
							{
								INCREF(None);
								o=None;
							}
							else
								o=newintobject((long)*dat);
							break;
						}
						case SYBINT4:
						{
							int *dat; /*Assumption on size of int - BAD!*/

							dat=(int *)dbdata(dbp, i);
							if(dat==NULL && (dbdatlen(dbp, i)==0))
							{
								INCREF(None);
								o=None;
							}
							else
								o=newintobject((long)*dat);
							break;
						}
						case SYBFLT8:
						{
							double *dat;

							dat=(double *)dbdata(dbp, i);
							if(dat==NULL && (dbdatlen(dbp, i)==0))
							{
								INCREF(None);
								o=None;
							}
							else
								o=newfloatobject(*dat);
							break;
						}
						case SYBDATETIME:
						{
							BYTE  *dat;

							dat=(BYTE *)dbdata(dbp, i);
							if(dat==NULL && (dbdatlen(dbp, i)==0))
							{
								INCREF(None);
								o=None;
							}
							else
								if(dbconvert(dbp, SYBDATETIME, dat,
									(dbdatlen(dbp,i)), SYBCHAR, dateinfo,
									(DBINT) - 1)==FAIL)
										o=NULL;
								else
									o=newstringobject((char *)dateinfo);
							break;
						}
					}
					if(o==NULL)
					{
						free(fmt);
						DECREF(list);
						DECREF(results);
						return NULL;
					}
					if(settupleitem(tuple, i-1, o)==-1)
					{
						free(fmt);
						DECREF(list);
						DECREF(results);
						return NULL;
					}
				}
				if(addlistitem(list,tuple)==-1)
				{
					free(fmt);
					DECREF(tuple);
					DECREF(list);
					DECREF(results);
					return NULL;
				}
				DECREF(tuple);
			}
			free(fmt);
			if(addlistitem(results,list)==-1)
			{
				DECREF(list);
				DECREF(results);
				return NULL;
			}
			DECREF(list);
		}
	} 
	return (results);
}

static object *
sybdb_sql (self, args)
	sybdbobject *self;
	object *args;
{
	char			*sql;
	DBPROCESS		*dbp;
	object			*errtuple;
	object			*results;
	syberrobject	*syberr;

	if (!getargs (args, "s", &sql)) {
    	return NULL;
	}

	dbcancel(self->dbproc);
	dbcmd(self->dbproc, sql);

	if ( dbsqlsend(self->dbproc) == FAIL ) { 
		syberr = (syberrobject *)dbgetuserdata(self->dbproc);
 		errtuple = Py_BuildValue("(is)", syberr->sybmsgno, syberr->sybmsgtext);
    	PyErr_SetObject(SybaseError, errtuple);
		free(syberr->sybmsgtext); free(syberr); dbsetuserdata(self->dbproc, 0);
		return (NULL);
	}

	if ( dbsqlok(self->dbproc) == FAIL ) {
		syberr = (syberrobject *)dbgetuserdata(self->dbproc);
 		errtuple = Py_BuildValue("(is)", syberr->sybmsgno, syberr->sybmsgtext);
    	PyErr_SetObject(SybaseError, errtuple);
		free(syberr->sybmsgtext); free(syberr); dbsetuserdata(self->dbproc, 0);
		return NULL;
	}
	else {
		results = getresults(self->dbproc);
		if(results==NULL)
			return NULL;
		if(!is_listobject(results)) 
		{
 			errtuple=Py_BuildValue("(is)", -1, "SQL result set - bad format!");
    		PyErr_SetObject(SybaseError, errtuple);
			return NULL; 
		}
		return (results);
	}
}

static object *
sybdb_sp (self, args)
	object *self;
	object *args;
{
	char *sp;
	DBPROCESS *dbp;
	object *spargs;
	object *sparg;
	object *results;
	object *r;
	object *errtuple;
	int spargcnt;
	int i;
	int retstatus;
	syberrobject *syberr;

	dbp = ((sybdbobject *)self)->dbproc;

	if (!getargs (args, "(sO)", &sp, &spargs)) {
		return NULL;
	}

	dbcancel(dbp);
	dbrpcinit(dbp, sp, 0);

	if (is_tupleobject(spargs)) 
	{
		spargcnt=gettuplesize(spargs);
		for (i=0; i < spargcnt; i++) 
		{
			sparg = gettupleitem(spargs,i);
			if (is_intobject(sparg)) {
				int i;
				i = getintvalue(sparg);
				dbrpcparam(dbp, NULL, 0, SYBINT4, -1, -1, &i);
			}
			else if (is_floatobject(sparg)) {
				double i;
				i = getfloatvalue(sparg);
				dbrpcparam(dbp, NULL, 0, SYBFLT8, -1, -1, &i);
			}
			else if (is_stringobject(sparg)) {
				dbrpcparam(dbp, NULL, 0, SYBCHAR, -1, getstringsize(sparg), getstringvalue(sparg));
			}
			else {
				errtuple = Py_BuildValue("(is)", -1, "Could not handle parameters to procedure.");
				PyErr_SetObject (SybaseError, errtuple);
				return NULL;
			}
		}
	}
	else if (spargs != None) {
		errtuple = Py_BuildValue("(is)", -1, "Could not handle parameters to procedure.");
		PyErr_SetObject (SybaseError, errtuple);
		return NULL;
	}

	dbrpcsend(dbp);
	if (dbsqlok(dbp) == FAIL) {
		syberr = (syberrobject *)dbgetuserdata(dbp);
 		errtuple = Py_BuildValue("(is)", syberr->sybmsgno, syberr->sybmsgtext);
    	PyErr_SetObject(SybaseError,errtuple);
		free(syberr->sybmsgtext); free(syberr); dbsetuserdata(dbp, 0);
		return (NULL);
	}
	results = getresults(dbp);
	if(results == NULL)
		return NULL;
	retstatus = dbretstatus(dbp);

	r = Py_BuildValue("(iO)", retstatus, results);
	DECREF(results);
	return (r);
}

static struct methodlist sybdb_methods[] = {
	{"sql",			sybdb_sql},
	{"sp",			sybdb_sp},
	{NULL,			NULL}		/* sentinel */
};

static void
sybdb_dealloc(self)
	sybdbobject *self;
{
	syberrobject *p;
	if (self && self->dbproc) 
			dbclose(self->dbproc);
	DEL(self);
}

static object *
sybdb_getattr(self, name)
	sybdbobject *self;
	char		*name;
{
	return findmethod(sybdb_methods, (object *)self, name);
}

typeobject SybDbtype = {
		OB_HEAD_INIT(&Typetype)
		0,
		"sybdb",
		sizeof(sybdbobject),
		0,
		sybdb_dealloc,		/*tp_dealloc*/
		0,					/*tp_print*/
		sybdb_getattr,		/*tp_getattr*/
		0,			/*tp_setattr*/
		0,			/*tp_compare*/
		0,			/*tp_repr*/
		0,			/*tp_as_number*/
		0,			/*tp_as_sequence*/
		0,			/*tp_as_mapping*/
};

/* MODULE FUNCTIONS: sybase */

static object *
sybase_new (self, args)
	sybdbobject *self;		/* Not used */
	object *args;
{
	char *user, *passwd, *server;
	object *db, *errtuple;

	if (!getargs (args, "(zzz)", &user, &passwd, &server)) {
		return NULL;
	}

	db = (object *) newsybdbobject(user, passwd, server);
	if (!db) {
		/* the dberr, dbmsgerr object is set in the err_handler */
		return NULL;
	}
	return db;
}

/* List of module functions */
static struct methodlist sybase_methods[]=
{
	{"new", sybase_new},
	{NULL, NULL}			/* sentinel */
};

/* Module initialisation */
void initsybase ()
{
	object *m, *d;

	/* Create the module and add the functions */
	m = initmodule ("sybase", sybase_methods);

	/* Add some symbolic constants to the module */
	d = getmoduledict (m);

	SybaseError = newstringobject ("sybase.error");
	if (SybaseError == NULL || dictinsert (d, "error", SybaseError) != 0) {
		fatal ("can't define sybase.error");
	}

	/* Check for errors */
	if (err_occurred ()){
		fatal ("can't initialize module sybase");
	}
}

