/*
  ifxdbmodule.ec
  $Id: ifxdbmodule.ec,v 1.11 1997/03/11 15:00:56 bertil Exp $  
*/

#include <time.h>
#define loc_t temp_loc_t
#include <ctype.h>
#undef loc_t

#ifdef WIN32
#include <value.h>
#include <sqlproto.h>
#else
#include <values.h>
#endif

#include <sqltypes.h>
#include <locator.h>
#include <datetime.h>

#include "Python.h"
#include "longobject.h"
#include "dbi.h"

EXEC SQL include sqlda.h;

/* These are global due to implementation of Informix esql.
   They are set, whenever required, to various names to allow
   for multiple database connections, cursors and queries.
*/
EXEC SQL BEGIN DECLARE SECTION;
static char *connectionName;
static char *cursorName;
static char *queryName;
EXEC SQL END DECLARE SECTION;

/* Create an sqlda object
*/
static struct sqlda *newBinding()
{
  struct sqlda *da = malloc(sizeof(struct sqlda));
  da->sqld = 0;
  da->sqlvar = 0;
  return da;
}


static PyObject *ifxdbError;

/* NOT In USE: typedef PyObject * (* CopyFcn)(const void *); */

/* Define and handle connection object
*/
typedef struct
{
  PyObject_HEAD
  char name[20];
  int has_commit;
} connectionObject;

/* pointer cast */
static connectionObject *connection(PyObject *o)
{
 return  (connectionObject *) o;
}
/* === */

static void setConnectionName(connectionObject*);
static void setConnection(connectionObject*);
static void ifxdbPrintError(const char *);
static int sqlCheck(const char *);

/* Define and handle cursor object
*/
typedef struct
{
  PyObject_HEAD
  PyObject *my_conx;
  PyObject *description;  
  int open;
  char cursorName[20];
  char queryName[20];
  struct sqlda *daIn;
  struct sqlda *daOut;
  int *originalType;
  char *inputBuffer;
  char *outputBuffer;
} cursorObject;

/* pointer cast */
static cursorObject *cursor(PyObject *o)
{
 return  (cursorObject *) o;
}

static void setCursorName(cursorObject *cur)
{
  cursorName = cur->cursorName;
  queryName = cur->queryName;
}
/* === */

/* XXX
*/
static void cleanInputBinding(cursorObject *cur)
{
  struct sqlda *da = cur->daIn;
  if (da && da->sqlvar) {
    int i;
    for (i=0; i<da->sqld; i++) {
      /* may not actually exist in some err cases */
      if ( da->sqlvar[i].sqldata) {
	if (da->sqlvar[i].sqltype == CLOCATORTYPE) {
	  loc_t *loc = (loc_t*) da->sqlvar[i].sqldata;
	  if (loc->loc_buffer) {
	    free(loc->loc_buffer);
	  }
	}
	free(da->sqlvar[i].sqldata);
      }
    }
    free(da->sqlvar);
    da->sqlvar = 0;
    da->sqld = 0;
  }
}

static void deleteInputBinding(cursorObject *cur)
{
  if (cur->daIn) {
    cleanInputBinding(cur);
    free(cur->daIn);
    cur->daIn = 0;
  }
}

/* XXX
*/
static void deleteOutputBinding(cursorObject *cur)
{
  struct sqlda *da = cur->daIn;
  if (cur->outputBuffer) {
    free(cur->outputBuffer);
    cur->outputBuffer = 0;
  }

  if (da && da->sqlvar) {
    int i;
    for (i=0; i<da->sqld; i++) {
      if (da->sqlvar[i].sqldata &&
	  (da->sqlvar[i].sqltype == CLOCATORTYPE)) {
	loc_t *loc = (loc_t*) da->sqlvar[i].sqldata;
	if (loc->loc_buffer) {
	  free(loc->loc_buffer);
	}
      }
    }
  }
}

/* Datastructure and methods for cursors.
*/
static void cursorDealloc(PyObject *self);
static PyObject * cursorGetAttr(PyObject *self, char *name);

static PyTypeObject Cursor_Type =
{
#ifdef WIN32 /* Whacky MS Compiler refuse to resolve. */
  PyObject_HEAD_INIT (0)
#else
  PyObject_HEAD_INIT (&PyType_Type)
#endif
  0,			/*ob_size */
  "ifxdbcur",		/*tp_name */
  sizeof(cursorObject),	/*tp_basicsize */
  0,			/*tp_itemsize */
  cursorDealloc,	/*tp_dealloc */
  0,			/*tp_print */
  cursorGetAttr,	/*tp_getattr */
  /* drop the rest */
};

static void cursorError(cursorObject *cur, const char *action)
{
  ifxdbPrintError(action);
}

static char
ifxdbCursorDoc[] =  "Create a new cursor object and associate it with db object";

static PyObject *ifxdbCursor(PyObject *self, PyObject *args)
{
  cursorObject *cur = PyObject_NEW(cursorObject, &Cursor_Type);
  if (cur) {
    cur->description = 0;
    cur->my_conx = self;	/* Reference to db object. */
    Py_INCREF(self); /* the cursors owns a reference to the connection */
    cur->open = 0;
    cur->daIn = newBinding();
    cur->daOut = 0;
    cur->originalType = 0;
    cur->inputBuffer = 0;
    cur->outputBuffer = 0;
    sprintf(cur->cursorName, "CUR%lX", (unsigned long) cur);
    sprintf(cur->queryName, "QRY%lX", (unsigned long) cur);
  }
  return (PyObject*) cur;
}

static void doCloseCursor(cursorObject *cur)
{
  if (cur->open) {
#ifdef STEP1
    connectionObject *conn = connection(cur->my_conx);
    setConnection(conn);
#endif
    setCursorName(cur);
    EXEC SQL CLOSE :cursorName ;
    EXEC SQL FREE :queryName ;
    EXEC SQL FREE :cursorName ;
    if (cur->daOut) {
      free(cur->daOut);
      cur->daOut = 0;
    }
    cur->open = 0;
  }
}

static void cursorDealloc(PyObject *self)
{
  cursorObject *cur = cursor(self);
  if (cur->description) {
    Py_DECREF(cur->description);
  }
  deleteInputBinding(cur);
  deleteOutputBinding(cur);
  if (cur->originalType) free(cur->originalType);
  Py_DECREF(cur->my_conx);
  PyMem_DEL(self);
}


static PyObject *ifxdbCurClose(PyObject *self, PyObject *args)
{
  doCloseCursor(cursor(self));
  Py_INCREF(Py_None);
  return Py_None;
}

/* End cursors === */

/* Datastructure and methods for connections.
*/
static void connectionDealloc(PyObject *self);
static PyObject * connectionGetAttr(PyObject *self, char *name);

static PyTypeObject Connection_Type =
{
#ifdef WIN32 /* Whacky MS Compiler refuse to resolve. */
  PyObject_HEAD_INIT (0)
#else
  PyObject_HEAD_INIT (&PyType_Type)
#endif
  0,				/*ob_size */
  "ifxdbconn",			/*tp_name */
  sizeof (connectionObject),	/*tp_basicsize */
  0,				/*tp_itemsize */
  connectionDealloc,		/*tp_dealloc */
  0,				/*tp_print */
  connectionGetAttr,		/*tp_getattr */
  /* drop the rest */
};

static void connectionError(connectionObject *conn, const char *action)
{
  ifxdbPrintError(action);
}

static void setConnectionName(connectionObject *conn)
{
  /* connectionName is global */
  connectionName = conn->name;
}

static void setConnection(connectionObject *conn)
{
  /* No need to swap connection if correctly set. */
  if ( connectionName != conn->name ) {
    setConnectionName(conn);
    EXEC SQL SET CONNECTION :connectionName;
    sqlCheck("SET-CONNECTION");
  }
}

/* End connections === */

static int unsuccessful()
{
  return SQLCODE;
}

/* Message generator to return human readable
   error message.
 */
static void ifxdbPrintError(const char *action)
{
  char message[512];
  EXEC SQL BEGIN DECLARE SECTION;
  char message1[255];
  char message2[255];
  int messlen1;
  int messlen2;
  EXEC SQL END DECLARE SECTION;

#ifdef EXTENDED_ERROR_HANDLING
  EXEC SQL BEGIN DECLARE SECTION;
  int exc_cnt;
  char sqlstate_2[6];
  EXEC SQL END DECLARE SECTION;
#endif /* EXTENDED_ERROR_HANDLING */

#ifdef EXTENDED_ERROR_HANDLING
  EXEC SQL get diagnostics :exc_cnt = NUMBER ;
#endif /* EXTENDED_ERROR_HANDLING */

  EXEC SQL get diagnostics  exception 1
      :message1 = MESSAGE_TEXT, :messlen1 = MESSAGE_LENGTH;

#ifdef EXTENDED_ERROR_HANDLING
  if (exc_cnt > 1) {
    EXEC SQL get diagnostics  exception 2
	:sqlstate_2 = RETURNED_SQLSTATE, :message2 = MESSAGE_TEXT, :messlen2 = MESSAGE_LENGTH;
  }
#endif /* EXTENDED_ERROR_HANDLING */

  /* In some cases, message1 is null and messlen1 undefined... */
  if ( messlen1 > 0 && messlen1 <= 255 )
    message1[messlen1-1] = 0;
  else
    strcpy(message1, "<NULL diagnostic message 1>");

#ifdef EXTENDED_ERROR_HANDLING
  /* In some cases, message1 is null and messlen1 undefined... */
  if ( messlen2 > 0 && messlen2 <= 255 )
    message2[messlen2-1] = 0;
  else
    strcpy(message2, "<NULL diagnostic message 2>");
#endif /* EXTENDED_ERROR_HANDLING */

#ifdef EXTENDED_ERROR_HANDLING
  if ( messlen2 > 0 ) {
      sprintf(message,
	      "Error %d performing %s: %s (%s:%s)",
	      SQLCODE, action, message1, sqlstate_2, message2);
  } else {
      sprintf(message,
	      "Error %d performing %s: %s",
	      SQLCODE, action, message1);
  }
#else
  sprintf(message,
	  "Error %d performing %s: %s",
	  SQLCODE, action, message1);
#endif /* EXTENDED_ERROR_HANDLING */
	  
  PyErr_SetString(ifxdbError, message);
}

static int sqlCheck(const char *action)
{
  if (unsuccessful()) {
    ifxdbPrintError(action);
    return 1;
  }
  return 0; 
}
    
/* End error reporter */

/* Begin section for transaction management.
*/

#ifdef STEP2
/* Experimental code.
   Idea is to allow for user to run without transactions at will.
*/
static PyObject *ifxdbBegin(PyObject *self, PyObject *args)
{
  connectionObject *conn = connection(self);
  if (conn->has_commit) {
    setConnection(conn);
    EXEC SQL BEGIN WORK;

    if (unsuccessful()) {
      connectionError(connection(self), "COMMIT");
      return 0;
    }
  /* success */
  Py_INCREF(Py_None);
  return Py_None;
}
#endif

static PyObject *ifxdbClose(PyObject *self, PyObject *args)
{
  connectionObject *conn = connection(self);
  if (conn->has_commit) {
    setConnection(conn);
    EXEC SQL COMMIT WORK;

    if (unsuccessful()) {
      connectionError(connection(self), "COMMIT");
      return 0;
    }

    /*
    EXEC SQL DISCONNECT :connectionName ;
    */

  }
  /* success */
  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *ifxdbRollback(PyObject *, PyObject *);
static PyObject *ifxdbCommit(PyObject *, PyObject *);

static PyMethodDef connectionMethods[] = {
  { "cursor", ifxdbCursor, 1 } ,
#ifdef STEP2
  { "begin", ifxdbBegin, 1 } ,
#endif
  { "commit", ifxdbCommit, 1 } ,
  { "rollback", ifxdbRollback, 1 } ,
#ifdef STEP2
  { "execute", ifxdbExec, 1} ,
  { "fetchone", ifxdbFetchOne, 1} ,
  { "fetchmany", ifxdbFetchMany, 1} ,
  { "fetchall", ifxdbFetchAll, 1} ,
  { "setinputsizes", ifxdbSetInputSizes, 1} ,
  { "setoutputsize", ifxdbSetOutputSize, 1} ,
#endif
  { "close", ifxdbClose, 1 } ,
  {0,     0}        /* Sentinel */
};

static PyObject *connectionGetAttr(PyObject *self,
			    char *name)
{
  if (!strcmp(name, "error")) {
    Py_INCREF(ifxdbError);
    return ifxdbError;
  }
  return Py_FindMethod (connectionMethods, self, name);
}

static void connectionDealloc(PyObject *self)
{
  connectionObject *conn = connection(self);
  setConnection(conn);
  EXEC SQL DISCONNECT :connectionName ;

  PyMem_DEL(self);
}

/* If this connection has transactions, commit and
   restart transaction.
*/
static PyObject *ifxdbCommit(PyObject *self, PyObject *args)
{
  connectionObject *conn = connection(self);
  if (conn->has_commit) {
    setConnection(conn);
    EXEC SQL COMMIT WORK;

    if (unsuccessful()) {
      connectionError(connection(self), "COMMIT");
      return 0;
    }
    else {
      EXEC SQL BEGIN WORK;
      if (unsuccessful()) {
	connectionError(connection(self), "BEGIN");
	return 0;
      }
    }
  }
  /* success */
  Py_INCREF(Py_None);
  return Py_None;
}

/*
 Pretty much the same as COMMIT operation.
 We ROLLBACK and restart transaction but this has only meaning if there
 is support for transactions.
 */
static PyObject *ifxdbRollback(PyObject *self, PyObject *args)
{
  connectionObject *conn = connection(self);
  if (conn->has_commit) {
    setConnection(conn);
    EXEC SQL ROLLBACK WORK;

    if (unsuccessful()) {
      connectionError(connection(self), "ROLLBACK");
      return 0;
    }
    else {
      EXEC SQL BEGIN WORK;
      if (unsuccessful()) {
	connectionError(connection(self), "BEGIN");
	return 0;
      }
    }
  Py_INCREF(Py_None);
  return Py_None;
  }
}

/* End sections transaction support === */

/* Begin section for parser of sql statements w.r.t.
   dynamic variable binding.
*/
typedef struct {
  const char *ptr;
  int parmCount;
  int parmIdx;
  int isParm;
  char state;
  char prev;
} parseContext;

static void initParseContext(parseContext *ct, const char *c)
{
  ct->state = 0;
  ct->ptr = c;
  ct->parmCount = 0;
}

static char doParse(parseContext *ct) 
{
  ct->isParm = 0;
  if (ct->state == *ct->ptr) {
    ct->state = 0;
  }
  else if (ct->state == 0){
    if ((*ct->ptr == '\'') || (*ct->ptr == '"')) {
      ct->state = *ct->ptr;
    }
    else if (*ct->ptr == '?') {
      ct->parmIdx = ct->parmCount;
      ct->parmCount++;
      ct->isParm = 1;
    }
    else if ((*ct->ptr == ':') && !isalnum(ct->prev)) {
      const char *m = ct->ptr + 1;
      int n = 0;
      while (isdigit(*m)) {
	n *= 10;
	n += *m - '0';
	m++;
      }
      if (n) {
	ct->parmIdx = n-1;
	ct->parmCount++;
	ct->ptr = m;
	ct->isParm = 1;
	ct->prev = '0';
	return '?';
      }
    }
  }
  ct->prev = *ct->ptr;
  return *ct->ptr++;
}


static int countVars(const char *in)
{
  parseContext ct;
  initParseContext(&ct, in);
  while (doParse(&ct)) {
  }
  return ct.parmCount;
}

static int ibindRaw(struct sqlvar_struct *var, PyObject *item)
{
  PyObject *nitem = dbiValue(item);
  PyObject *sitem = PyObject_Str(nitem);
  int n = PyObject_Length(sitem);
  loc_t *loc = (loc_t*) malloc(sizeof(loc_t)) ;
  loc->loc_loctype = LOCMEMORY;
  loc->loc_buffer = malloc(n);
  loc->loc_bufsize = n;
  loc->loc_size = n;
  loc->loc_oflags = 0;
  loc->loc_mflags = 0;
  memcpy(loc->loc_buffer,
	 PyString_AsString(sitem),
	 n);
    
  var->sqldata = (char *) loc;
  var->sqllen = sizeof(loc_t);
  var->sqltype = CLOCATORTYPE;
  Py_DECREF(sitem);
  return 1;
}

static int ibindDate(struct sqlvar_struct *var, PyObject *item)
{
  PyObject *sitem = PyNumber_Long(dbiValue(item));
  if (sitem) {
    long n = PyLong_AsLong(sitem);
    dtime_t *dt = (dtime_t*) malloc(sizeof(dtime_t)) ;
    char buf[20];
    strftime(buf, sizeof(buf), "%Y-%m-%d %H:%M:%S", localtime(&n));
    dt->dt_qual = TU_DTENCODE(TU_YEAR, TU_SECOND);
    dtcvasc(buf, dt);
    var->sqldata = (char *) dt;
    var->sqllen = sizeof(dtime_t);
    var->sqltype = CDTIMETYPE;
    Py_DECREF(sitem);
  }
  else {
    PyErr_SetString(ifxdbError, "type error in date object");
    return 0;
  }
  return 1;
}

static int ibindString(struct sqlvar_struct *var, PyObject *item)
{
  PyObject *sitem = PyObject_Str(item);
  const char *val = PyString_AsString(sitem);
  int n = strlen(val);
  var->sqltype = CSTRINGTYPE;
  var->sqldata = malloc(n+1);
  var->sqllen = n;
  strcpy(var->sqldata, val);
  if (PyLong_Check(item)) {
    var->sqldata[n-1] = 0; /* erase 'L' suffix */
  }
  Py_DECREF(sitem);
  return 1;
}

typedef int (*ibindFptr)(struct sqlvar_struct *, PyObject*);

ibindFptr ibindFcn(PyObject* item)
{
  if (dbiIsRaw(item)) {
    return ibindRaw;
  }
  else if(dbiIsDate(item)) {
    return ibindDate;
  }
  else {
    return ibindString;
  }
}

static void allocSlots(struct sqlda *da, int n_slots)
{
  size_t sz = n_slots * sizeof(struct sqlvar_struct);
  da->sqld = n_slots;
  da->sqlvar = malloc(sz);
  memset(da->sqlvar, 0, sz);
}

static int bindInput(cursorObject *cur,
		      char *out, const char *in,
		      PyObject *vars)
{
  int n_slots = countVars(in);
  if (n_slots) {
    struct sqlvar_struct *var;
    int n_vars = vars ? PyObject_Length(vars) : 0;
    parseContext ctx;
    
    initParseContext(&ctx, in);
    allocSlots(cur->daIn, n_slots);
    var = cur->daIn->sqlvar;
    while (*out++ = doParse(&ctx)) {
      if (ctx.isParm) {
	if (ctx.parmIdx < n_vars) {
	  PyObject *item = PySequence_GetItem(vars, ctx.parmIdx);
	  int success = (*ibindFcn(item))(var++, item);

	  Py_DECREF(item);	/* PySequence_GetItem increments it */
	  if (!success) {
	    return 0;
	  }
	}
	else {
	  PyErr_SetString(ifxdbError, " too few actual parameters");
	  return 0;
	}
      }
    }
  }
  else {
    strcpy(out, in);
  }
  return 1;
}

/* NOT IN USE ->
static void adjustType(struct sqlvar_struct *var)
{
  switch (var->sqltype & SQLTYPE) {
  case SQLCHAR:
  case SQLVCHAR:
  case SQLNCHAR:
  case SQLNVCHAR:
    var->sqltype = CSTRINGTYPE;
    var->sqllen++;
    break;
  case SQLSMINT:
  case SQLINT:
  case SQLFLOAT:
  case SQLSMFLOAT:
  case SQLDECIMAL:
  case SQLSERIAL:
  case SQLMONEY:
    var->sqltype = CDOUBLETYPE;
    var->sqllen = SIZDTIME;
    break;
  case SQLDATE:
#ifdef DATEFIX
    var->sqltype = CDATETYPE;
    var->sqllen++;
    break;
#endif
  case SQLDTIME:
    var->sqltype = CDTIMETYPE;
    var->sqllen++;
    break;
  case SQLBYTES:
  case SQLTEXT:
    var->sqltype = CLOCATORTYPE;
    break;
  }
}
<- NOT IN USE */

static PyObject *typeOf(int type)
{
  switch(type & SQLTYPE){
  case SQLBYTES:
  case SQLTEXT:
    return DbiRaw;
  case SQLDATE:
#ifdef DATEFIX
    return DbiNumber;
#endif
  case SQLDTIME:
    return DbiDate;
  case SQLFLOAT:
  case SQLSMFLOAT:
  case SQLDECIMAL:
  case SQLINT:
  case SQLSMINT:
  case SQLSERIAL:
  case SQLMONEY:
    return DbiNumber;
  default:
    return DbiString;
  }
}

static void bindOutput(cursorObject *cur)
{
  char * bufp;
  int pos;
  int count = 0;
  struct sqlvar_struct *var;

  if (cur->originalType)
	free(cur->originalType);
  cur->originalType = malloc(cur->daOut->sqld * sizeof(int));
  for (pos = 0, var = cur->daOut->sqlvar;
       pos < cur->daOut->sqld;
       pos++, var++) {
    PyObject *new_tuple = Py_BuildValue("(sOiiiii)",
					var->sqlname, 
					typeOf(var->sqltype),
					var->sqllen,
					var->sqllen,
					0, 0, !(var->sqltype & SQLNONULL));
    PyList_Append(cur->description, new_tuple);
    Py_DECREF(new_tuple);

    cur->originalType[pos] = var->sqltype;

    switch(var->sqltype & SQLTYPE){
    case SQLBYTES:
    case SQLTEXT:
      var->sqllen  = sizeof(loc_t);
      var->sqltype = CLOCATORTYPE;
      break;
    case SQLDATE:
    case SQLDTIME:
    case SQLFLOAT:
    case SQLSMFLOAT:
    case SQLDECIMAL:
    case SQLINT:
    case SQLSMINT:
    case SQLSERIAL:
    case SQLMONEY:
      var->sqllen = 20; /* big enough */
      /* fall through */
    default:
      var->sqltype = CCHARTYPE;
      var->sqllen = rtypmsize(var->sqltype, var->sqllen);
      break;
      
    }
    var->sqllen = rtypmsize(var->sqltype, var->sqllen);
    count = rtypalign(count, var->sqltype) + var->sqllen;
    var->sqlind = 0;
  }

  bufp = cur->outputBuffer = malloc(count);

  
  for (pos = 0, var = cur->daOut->sqlvar;
       pos < cur->daOut->sqld;
       pos++, var++) {
    bufp = (char *) rtypalign( (int) bufp, var->sqltype);
    
    if (var->sqltype == CLOCATORTYPE) {
      loc_t *loc = (loc_t*) bufp;
      loc->loc_loctype = LOCMEMORY;
      loc->loc_bufsize = -1;
      loc->loc_oflags = 0;
      loc->loc_mflags = 0;
    }
    var->sqldata = bufp;
    bufp += var->sqllen;
  }
}

#define returnOnError(action) if (sqlCheck(action)) return 0

static PyObject *ifxdbCurExec(PyObject *self, PyObject *args)
{
  cursorObject *cur = cursor(self);
  struct sqlda *tdaIn = cur->daIn;
  struct sqlda *tdaOut =  cur->daOut ;
  const char *sql;
  EXEC SQL BEGIN DECLARE SECTION;
  char newSql[2000];
  EXEC SQL END DECLARE SECTION;
  
  PyObject *inputvars = 0;
  if (!PyArg_ParseTuple(args, "s|O", &sql, &inputvars))
    return 0;

  /* clean up */
  cleanInputBinding(cur);
  deleteOutputBinding(cur);
  doCloseCursor(cur);
  if (cur->description) {
    Py_DECREF(cur->description);
  }

  cur->description = PyList_New(0);

  if (!bindInput(cur, newSql, sql, inputvars)) {
    return 0;
  }
    
  setCursorName(cur);
#ifdef STEP1
  /* Make sure we talk to the right database. */
  setConnection(connection(cur->my_conx));
#endif
  cur->open = 1;
  EXEC SQL PREPARE :queryName FROM :newSql;
  returnOnError("PREPARE");
  EXEC SQL DESCRIBE :queryName INTO tdaOut;
  if (SQLCODE == 0) {
    cur->daOut = tdaOut;
    EXEC SQL DECLARE :cursorName CURSOR FOR :queryName;
    returnOnError("DECLARE");

    bindOutput(cur);
    EXEC SQL OPEN :cursorName USING DESCRIPTOR tdaIn;
    returnOnError("OPEN");
    Py_INCREF(Py_None);
    return Py_None;
  }
  else {
    EXEC SQL EXECUTE :queryName USING DESCRIPTOR tdaIn;
    if (unsuccessful()) {
      cursorError(cur, "EXEC");
    }
    else {
      return Py_BuildValue("i", sqlca.sqlerrd[2]); /* number of row */
    }
  }
  /* error return */
  return 0;
}

/* Routines for manipulations of datetime
*/
static int convertToInt(dtime_t *d, const char *fmt)
{
  char buf[20];
  int x = dttofmtasc(d, buf, 20, (char *)fmt);
  return atoi(buf);
}

static time_t convertDtToUnix(dtime_t *d)
{
  struct tm gt;
    
  gt.tm_isdst = -1;
  
  gt.tm_year = convertToInt(d, "%Y") - 1900;
  gt.tm_mon = convertToInt(d, "%m") - 1;   /* month */
  gt.tm_mday = convertToInt(d, "%d");   /* day */
  gt.tm_hour = convertToInt(d, "%H");   /* hour */
  gt.tm_min = convertToInt(d, "%M");   /* minute */
  gt.tm_sec = convertToInt(d, "%S");   /* second */
  return mktime(&gt);
}

static time_t convertAscToUnix(const char *d)
{
  struct tm gt;
  sscanf(d, "%04d-%02d-%02d %02d:%02d:%02d",
	 &gt.tm_year,
	 &gt.tm_mon,
	 &gt.tm_mday,
	 &gt.tm_hour,
	 &gt.tm_min,
	 &gt.tm_sec);
  gt.tm_year -= 1900;
  gt.tm_mon -= 1;
  gt.tm_isdst = -1;

  return mktime(&gt);
}

/* End datetime === */

static PyObject *doCopy(/* const */ void *data, int type)
{
  switch(type){
  case SQLDATE:
#ifdef DATEFIX
/*    return dbiMakeDate(PyInt_FromLong(data)); */
    return PyInt_FromLong(*(int*)data); /* Informix internally stores DATE as signed long */
#endif
  case SQLDTIME:
    return dbiMakeDate(PyInt_FromLong(convertAscToUnix((char*)data)));
  case SQLCHAR:
  case SQLVCHAR:
  case SQLNCHAR:
  case SQLNVCHAR:
  {
      /* NOTE: we must axe trailing spaces in Informix (boggle) */
      size_t len = strlen((char*)data);
      char * p = (char*)data + len - 1;

      while ( len > 1 && *p == ' ' )
      {
	  *p-- = '\0';
	  --len;
      }
      return Py_BuildValue("s", (char*)data);
  }
  case SQLFLOAT:
  case SQLSMFLOAT:
  case SQLDECIMAL:
  case SQLINT:
  case SQLSMINT:
  case SQLSERIAL:
  case SQLMONEY:
    if (strlen(data) < 10) {
      return PyInt_FromLong(atol(data));
    }
    else {
      return PyLong_FromString(data, 0, 10);
    }
  case SQLBYTES:
  case SQLTEXT:
    return dbiMakeRaw
      (PyString_FromStringAndSize(((loc_t*)data)->loc_buffer,
				  ((loc_t*)data)->loc_size));
  }
  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *processOutput(cursorObject *cur)
{
  PyObject *row = PyTuple_New(cur->daOut->sqld);
  int pos;
  struct sqlvar_struct *var;

#ifdef AIX
  /* ### deal with some wacky bug in AIX's mktime() */
  printf("");
#endif

  for (pos = 0, var = cur->daOut->sqlvar;
       pos < cur->daOut->sqld;
       pos++, var++) {

    PyObject *v;
    if (var->sqlind) {
      v = Py_None;
      Py_INCREF(v);
    }
    else {
      v = doCopy(var->sqldata, cur->originalType[pos]);
    }
      
    PyTuple_SET_ITEM(row, pos, v);
  }
  return row;
}



static PyObject *ifxdbCurFetchOne(PyObject *self, PyObject *args)
{
  cursorObject *cur = cursor(self);
  struct sqlda *tdaOut = cur->daOut;
  setCursorName(cur);
#ifdef STEP1
  /* Make sure we talk to the right database. */
  setConnection(connection(cur->my_conx));
#endif
  EXEC SQL FETCH :cursorName using descriptor tdaOut;
  if (!strncmp(SQLSTATE, "02", 2)) {
    Py_INCREF(Py_None);
    return Py_None;
  }
  else if (strncmp(SQLSTATE, "00", 2)) {
    cursorError(cur, "FETCH");
    return 0;
  }
  return processOutput(cur);
}

static PyObject *ifxdbFetchCounted(PyObject *self, int count)
{
  PyObject *list = PyList_New(0);

  while ( count-- > 0 )
  {
      PyObject *entry = ifxdbCurFetchOne(self, 0);

      if ( entry == NULL )
      {
	  Py_DECREF(list);
	  return NULL;
      }
      if ( entry == Py_None )
      {
	  Py_DECREF(entry);
	  break;
      }

      if ( PyList_Append(list, entry) == -1 )
      {
	  Py_DECREF(entry);
	  Py_DECREF(list);
	  return NULL;
      }

      Py_DECREF(entry);
  }

  return list;
}

static PyObject *ifxdbCurFetchMany(PyObject *self, PyObject *args)
{
  int n_rows = 1;

  if (!PyArg_ParseTuple(args, "|i", &n_rows))
      return NULL;

  return ifxdbFetchCounted(self, n_rows);
}

static PyObject *ifxdbCurFetchAll(PyObject *self, PyObject *args)
{
  return ifxdbFetchCounted(self, 65536);
}

static PyObject *ifxdbCurSetInputSizes(PyObject *self, PyObject *args)
{
  Py_INCREF(Py_None);
  return Py_None;
}
static PyObject *ifxdbCurSetOutputSize(PyObject *self, PyObject *args)
{
  Py_INCREF(Py_None);
  return Py_None;
}


static PyMethodDef cursorMethods[] = {
  { "close", ifxdbCurClose, 1} ,
  { "execute", ifxdbCurExec, 1} ,
  { "fetchone", ifxdbCurFetchOne, 1} ,
  { "fetchmany", ifxdbCurFetchMany, 1} ,
  { "fetchall", ifxdbCurFetchAll, 1} ,
  { "setinputsizes", ifxdbCurSetInputSizes, 1} ,
  { "setoutputsize", ifxdbCurSetOutputSize, 1} ,
  {0,     0}        /* Sentinel */
};

static PyObject *cursorGetAttr(PyObject *self,
			     char *name)
{
  if (!strcmp(name, "description")) {
    if (cursor(self)->description) {
      Py_INCREF(cursor(self)->description);
      return cursor(self)->description;
    } else {
      Py_INCREF(Py_None);
      return Py_None;
    }
  }
  if (!strcmp(name, "error")) {
    Py_INCREF(ifxdbError);
    return ifxdbError;
  }
  return Py_FindMethod (cursorMethods, self, name);
}

static PyObject *ifxdbLogon(PyObject *self, PyObject *args)
{
  const char *connectionString;
  EXEC SQL BEGIN DECLARE SECTION;
  char *cString;
  EXEC SQL END DECLARE SECTION;
  
  connectionObject *conn = 0;
  if (PyArg_ParseTuple(args, "s", &connectionString)) {
    conn = PyObject_NEW (connectionObject, &Connection_Type);
    if (conn) {
      sprintf(conn->name, "CONN%lX", (unsigned long) conn);
      setConnectionName(conn);
      cString = (char *)connectionString;
      EXEC SQL CONNECT TO :cString AS :connectionName WITH CONCURRENT TRANSACTION;
      if (unsuccessful()) {
	connectionError(conn, "LOGON");
	PyMem_DEL(conn);
	conn = 0;
      }
      else {
	conn->has_commit = (sqlca.sqlwarn.sqlwarn1 == 'W') ;
	if (conn->has_commit) {
	  EXEC SQL BEGIN WORK;
	  if (unsuccessful()) {
	    connectionError(connection(self), "BEGIN");
	    PyMem_DEL(conn);
	    conn = 0;
	  }
	}
      }
    }
  }
  return (PyObject*) conn;
}

static PyMethodDef globalMethods[] = {
  { "informixdb", ifxdbLogon, 1} ,
  {0,     0}        /* Sentinel */
};


/* void initinformixdb() */
void init_informixdb()
{
  extern void initdbi();
#ifdef WIN32
  PyObject *m;
  Cursor_Type.ob_type = &PyType_Type;
  Connection_Type.ob_type = &PyType_Type;
  m = Py_InitModule("_informixdb", globalMethods);
#else
  PyObject *m = Py_InitModule("_informixdb", globalMethods);
#endif
  ifxdbError = Py_BuildValue("s", "InformixdbError");
  PyDict_SetItemString (PyModule_GetDict (m), "error", ifxdbError);

  initdbi();
  /* PyImport_ImportModule("dbi"); */
}
