/* 
 * Copyright (C) by Michael Lausch, G.A.M.S. edv Dienstleistungen GmbH
 *
 * $Id$
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <Python.h>
#include <cli0core.h>
#include <cli0ext1.h>


static char* SQLFunction = "";


#define MAX_NUM_PRECISION 15
/* Define max length of char string representation of */
/* number as: = max(precision) + leading sign + E +   */
/* exp sign + max exp length                          */
/* =  15 + 1 + 1 + 1 + 2                              */
/* =  15 + 5                                          */

__inline__ int max(int a, int b)
{
  a > b ? a : b;
}

#define MAX_NUM_STRING_SIZE (MAX_NUM_PRECISION + 5)

static UDWORD
display_size(SWORD coltype, UWORD collen, UCHAR* colname)
{
  switch (coltype)
    {
      
    case SQL_CHAR:
    case SQL_VARCHAR:
      return(max(collen, strlen(colname)));
      
    case SQL_SMALLINT:
      return(max(6, strlen(colname)));
      
    case SQL_INTEGER:
      return(max(11, strlen(colname)));
      
    case SQL_DECIMAL:
    case SQL_NUMERIC:
    case SQL_REAL:
    case SQL_FLOAT:
    case SQL_DOUBLE:
      return(max(MAX_NUM_STRING_SIZE, strlen(colname)));
      
      /* Note that this function only supports the */
      /* core data types. */
    default:
      return(0);
    }
}






static PyObject *SolidErrorObject;
static PyObject* SolidStringObject;
static PyObject* SolidRawObject;
static PyObject* SolidNumberObject;
static PyObject* SolidDateObject;
static PyObject* SolidRowidObject;

static int String_id = 1;
static int Raw_id    = 2;
static int Number_id = 3;
static int Date_id   = 4;
static int Row_id    = 5;


/* ----------------------------------------------------- */

/* Declarations for objects of type Solid Connection */

typedef struct {
  PyObject_HEAD;
  HENV     henv;
  HDBC     hdbc;
  /* XXXX Add your own stuff here */
} soldbobject;

staticforward PyTypeObject Soldbtype;

typedef struct {
  PyObject_HEAD;
  HSTMT        hstmt;
  PyObject*    cnx;
  void**       data;
  SWORD*       types;
  char*        stmt;
  int      arraysize;
}solcursorobject;

staticforward PyTypeObject SolCursorType;

static solcursorobject* newsolcursorobject();
static int soldb_close1(soldbobject*);




static char*
generate_Errmsg(soldbobject* sol, solcursorobject* solc, char* fname)
{
  UCHAR  sqlstate[SQL_MAX_MESSAGE_LENGTH];
  SDWORD nativeError;
  UCHAR  errMsg[SQL_MAX_MESSAGE_LENGTH];
  SWORD  actLength;
  char*  output;
  int    rc;
  
  rc = SQLError(sol->henv, sol->hdbc,
		(solc && solc->hstmt) ? solc->hstmt : SQL_NULL_HSTMT,
		sqlstate, &nativeError,
		errMsg, sizeof (errMsg), &actLength);
  
  
  asprintf(&output,
	   "Error in %s: \n\tsqlstate = <%s>\n\tnativeError = %d\n\terrMsg = <%s>\n",
	   fname, sqlstate, nativeError, errMsg);
  return output;
}


/* Start of module function */
/* ---------------------------------------------------------------- */

static soldbobject *
newsoldbobject()
{
  soldbobject *self;
	
  self = PyObject_NEW(soldbobject, &Soldbtype);
  if (self == NULL)
    return NULL;
  /* XXXX Add your own initializers here */
  self->henv = 0;
  self->hdbc = 0;
  return self;
}


static void
soldb_dealloc(soldbobject* self)
{
  soldb_close1(self);
  PyMem_DEL(self);
}


static char soliddb_soliddb__doc[] =
"Creates a new connection to the Solid DB Server. The arguments are "
"CONNECTION which spcifies the protocol, machine and port (TCP/IP "
"localhost 1313, USER which is the username, and PASSWD which is the passwod";

static PyObject *
soliddb_soliddb(self, args)
	PyObject *self;	/* Not used */
	PyObject *args;
{
  char* server = 0;
  char* uid    = 0;
  char* pwd    = 0;
  int rc;
  
  soldbobject* solid = newsoldbobject();
  
  if (!PyArg_ParseTuple(args, "sss", &server, &uid, &pwd))
    return NULL;
  rc = SQLAllocEnv(&solid->henv);
  if (rc != SQL_SUCCESS)
    {
      PyErr_SetString(SolidErrorObject, "SQLAlloc Env failed");
      return NULL;
    }
  rc = SQLAllocConnect(solid->henv, &solid->hdbc);
  switch (rc)
    {
    case SQL_ERROR:
      PyErr_SetString(SolidErrorObject, "SQLAllocConnect Error");
      return NULL;
    case SQL_INVALID_HANDLE:
      PyErr_SetString(SolidErrorObject, "SQLAllocConnect called with invalid handle");
      return NULL;
    }
  rc = SQLConnect(solid->hdbc, server, SQL_NTS, uid, SQL_NTS, pwd,
		  SQL_NTS);
  if (rc != SQL_SUCCESS && rc != SQL_SUCCESS_WITH_INFO)
    {

      char* errmsg = generate_Errmsg(solid, 0, "SQLConnect");
      PyErr_SetString(SolidErrorObject, errmsg);
      free(errmsg);
      return NULL;
    }
  rc = SQLSetConnectOption(solid->hdbc, SQL_AUTOCOMMIT, SQL_AUTOCOMMIT_OFF);
  if (!(rc == SQL_SUCCESS || rc == SQL_SUCCESS_WITH_INFO))
    {
      char* str = generate_Errmsg(solid, 0, "SQLSetConnectionOption(SQL_AUTOCOMMIT_OFF)");
      PyErr_SetString(SolidErrorObject, str);
      free(str);
      return NULL;
    }

  rc = SQLSetConnectOption(solid->hdbc, SQL_TXN_ISOLATION, SQL_TXN_SERIALIZABLE);
  if (!(rc == SQL_SUCCESS || rc == SQL_SUCCESS_WITH_INFO))
    {
      char* str = generate_Errmsg(solid, 0, "SQLSetConnectionOption( SQL_TXN_SERIALIZABLE");
      PyErr_SetString(SolidErrorObject, str);
      free(str); 
      return NULL;
    }

  return (PyObject*)solid;
}


static struct PyMethodDef soliddb_methods[] = {
  {"soliddb",	(PyCFunction)soliddb_soliddb,	METH_VARARGS,	soliddb_soliddb__doc},
  {NULL,		NULL}		/* sentinel */
};

/* End of module functions */
/* --------------------------------------- */


/* Start of database (connection) functions */
/* ---------------------------------------- */


static int
soldb_close1(soldbobject* sol)
{
  int rc;
  if (sol->hdbc)
    {
      rc = SQLDisconnect(sol->hdbc);
      if (!(rc == SQL_SUCCESS || SQL_SUCCESS_WITH_INFO))
	{
	  SQLFunction = "SQLDisconnect";
	  return -1;
	}
      rc = SQLFreeConnect(sol->hdbc);
      if (!(rc == SQL_SUCCESS || SQL_SUCCESS_WITH_INFO))
	{
	  SQLFunction = "SQLFreeConnect";
	  return -1;
	}
    }
  if (sol->henv)
    {
      rc = SQLFreeEnv(sol->henv);
      if (!(rc == SQL_SUCCESS || SQL_SUCCESS_WITH_INFO))
	{
	  SQLFunction = "SQLFreeEnv";
	  return -1;
	}
    }
  return 0;
}

  

static char soldb_close_doc[] =
"This function closes the connection to the DB server and frees all "
"allocated memory. If a transaction is in progress a SolidError "
"exception is raised";

static PyObject *
soldb_close(PyObject* self, PyObject* args)
{
  soldbobject* sol = (soldbobject*)self;
  
  if (!PyArg_ParseTuple(args, ""))
    {
      PyErr_SetString(PyExc_SyntaxError, "usage: close()");
      return NULL;
    }
  if (soldb_close1(sol) < 0)
    {
      char* error = generate_Errmsg(sol, 0, "SQLDisconnect");
      PyErr_SetString(SolidErrorObject, error);
      free(error);
      return NULL;
    }
  Py_INCREF(Py_None);
  return Py_None;
}

static char soldb_commit_doc[] =
"Commits a running transaction. Call this after every SQL statement if "
"you are not using transactions, or at the end of a successfull "
"transaction.";

static PyObject*
soldb_commit(PyObject* self, PyObject* args)
{
  soldbobject* sol = (soldbobject*)self;
  int rc;
  
  if (!PyArg_ParseTuple(args, ""))
    {
      PyErr_SetString(PyExc_SyntaxError, "usage: commit()");
      return NULL;
    }
  rc = SQLTransact(sol->henv, sol->hdbc, SQL_COMMIT);
  if (!(rc == SQL_SUCCESS || SQL_SUCCESS_WITH_INFO))
    {
      char* error = generate_Errmsg(sol, 0, "SQLTransact");
      PyErr_SetString(SolidErrorObject, error);
      free(error);
      return NULL;
    }
  Py_INCREF(Py_None);
  return Py_None;
}

static char soldb_rollback_doc[] =
"Aborts a running transaction. Call this at the end of an "
"unsuccessfull transaction." ;

static PyObject*
soldb_rollback(PyObject* self, PyObject* args)
{
  soldbobject* sol = (soldbobject*)self;
  int rc;
  
  if (!PyArg_ParseTuple(args, ""))
    {
      PyErr_SetString(PyExc_SyntaxError, "usage: rollback()");
      return NULL;
    }
  rc = SQLTransact(sol->henv, sol->hdbc, SQL_ROLLBACK);
  if (!(rc == SQL_SUCCESS || SQL_SUCCESS_WITH_INFO))
    {
      char* error = generate_Errmsg(sol, 0, "SQLTransact");
      PyErr_SetString(SolidErrorObject, error);
      free(error);
      return NULL;
    }
  Py_INCREF(Py_None);
  return Py_None;
}


static char soldb_cursor__doc[] =
"This creates a cursor of the database. Cursor are used to execute SQL "
"statements and to retrieve the data if available";

static PyObject*
soldb_cursor(PyObject* self, PyObject* args)
{
  solcursorobject* cursor = 0;
  soldbobject* sol = (soldbobject*)self;
  
  if (!PyArg_ParseTuple(args, ""))
    {
      PyErr_SetString(PyExc_SyntaxError, "usage: rollback()");
      return NULL;
    }
  cursor = newsolcursorobject();
  Py_INCREF(self);
  cursor->cnx = self;
  return (PyObject*)cursor;
}

static char soldb_callproc__doc[] =
"Thus function calls a built-in SQL procedure. Currently not "
"implemented. Use the execute function to call SQL Functions.";

static PyObject*
soldb_callproc(PyObject* self, PyObject* args)
{
  PyErr_SetString(SolidErrorObject, "Not yet implemented");
  return NULL;
}

static struct PyMethodDef solidobject_methods[] = {
  {"close",	  (PyCFunction)soldb_close,	METH_VARARGS,	soldb_close_doc},
  {"commit",      (PyCFunction)soldb_commit,    METH_VARARGS,   soldb_commit_doc},
  {"rollback",    (PyCFunction)soldb_rollback,  METH_VARARGS,   soldb_rollback_doc},
  {"cursor",      (PyCFunction)soldb_cursor,    METH_VARARGS,   soldb_cursor__doc},
  {"callproc",    (PyCFunction)soldb_callproc,  METH_VARARGS,   soldb_callproc__doc},
  {NULL,		NULL}		/* sentinel */
};



static PyObject *
soldb_getattr(soldbobject* self, char* name)
{
	/* XXXX Add your own getattr code here */
	return Py_FindMethod(solidobject_methods, (PyObject *)self, name);
}

static char Soldbtype__doc__[] = 
"A connection to the Solid DB Server."
;

static PyTypeObject Soldbtype = {
	PyObject_HEAD_INIT(&PyType_Type)
	0,				/*ob_size*/
	"Solid",			/*tp_name*/
	sizeof(soldbobject),		/*tp_basicsize*/
	0,				/*tp_itemsize*/
	/* methods */
	(destructor)soldb_dealloc,	/*tp_dealloc*/
	(printfunc)0,		/*tp_print*/
	(getattrfunc)soldb_getattr,	/*tp_getattr*/
	(setattrfunc)0,	/*tp_setattr*/
	(cmpfunc)0,		/*tp_compare*/
	(reprfunc)0,		/*tp_repr*/
	0,			/*tp_as_number*/
	0,		/*tp_as_sequence*/
	0,		/*tp_as_mapping*/
	(hashfunc)0,		/*tp_hash*/
	(ternaryfunc)0,		/*tp_call*/
	(reprfunc)0,		/*tp_str*/

	/* Space for future expansion */
	0L,0L,0L,0L,
	Soldbtype__doc__ /* Documentation string */
};


/* Start of code for Solid Connection Objects */
/* --------------------------------------------------------- */



/* End of code for Solid Connection Objects */
/* ---------------------------------------- */




/*---------------------------------------- */
/*
 * Cursor Part
 *
 */

static char solcursor_arraysize__doc[] =
"Sets the count of rows to be fetched with a fetchmany call. The "
"parameter SIZE determines the count. If SIZE is less/equal 0, the "
"count is set to 1";

static PyObject*
solcursor_arraysize(PyObject* self, PyObject* args)
{
  solcursorobject* solc = (solcursorobject*)self;
  int newsize = -1;
  
  if (!PyArg_ParseTuple(args, "i", &newsize))
    {
      PyErr_SetString(PyExc_SyntaxError, "arraysize(newsize)");
      return NULL;
    }

  if (newsize <= 0)
    {
      newsize = 1;
    }
  
  Py_INCREF(Py_None);
  return Py_None;
}

static char solcursor_description__doc[] =
"This function ";

static PyObject*
solcursor_description(solcursorobject* solc)
{
  int              rc;
  SWORD            nresultcols;
  int              i;
  PyObject*        retval;
  UCHAR            colname[32];
  SWORD            coltype;
  SWORD            colnamelen;
  SWORD            nullable;
  UDWORD           collen;
  SWORD            scale;
  
  if (!solc->hstmt)
    {
      Py_INCREF(Py_None);
      return Py_None;
    }
  rc = SQLNumResultCols(solc->hstmt, &nresultcols);
  if (!(rc == SQL_SUCCESS || rc == SQL_SUCCESS_WITH_INFO))
    {
      char* errmsg = generate_Errmsg((soldbobject*)solc->cnx, 0, "SQLNumResultCols");
      PyErr_SetString(SolidErrorObject, errmsg);
      return NULL;
    }
  if (nresultcols == 0)
    {
      Py_INCREF(Py_None);
      return Py_None;
    }
  retval = PyTuple_New(nresultcols);
  for (i = 0; i < nresultcols; i++)
    {
      PyObject* tuple;
      tuple = PyTuple_New(7);
      rc = SQLDescribeCol(solc->hstmt, i + 1, colname,
			  sizeof(colname), &colnamelen,
			  &coltype, &collen, &scale,
			  &nullable);

      PyTuple_SetItem(tuple, 0, PyString_FromStringAndSize(colname,
							   colnamelen));
      PyTuple_SetItem(tuple, 2, PyInt_FromLong(display_size(coltype, collen, colname)));
      Py_INCREF(Py_None);
      PyTuple_SetItem(tuple, 3, Py_None);
      PyTuple_SetItem(tuple, 4, PyInt_FromLong(collen));
      PyTuple_SetItem(tuple, 5, PyInt_FromLong(scale));
      if (nullable)
	PyTuple_SetItem(tuple, 6, PyInt_FromLong(1));
      else
	{
	  Py_INCREF(Py_None);
	  PyTuple_SetItem(tuple, 6, Py_None);
	}
      switch (coltype)
	{
	case SQL_BIGINT:
	case SQL_SMALLINT:
	case SQL_TINYINT:                       
	case SQL_INTEGER:
	case SQL_REAL:                          
	case SQL_DOUBLE:                        
	case SQL_FLOAT:
	  Py_INCREF(SolidNumberObject);
	  PyTuple_SetItem(tuple, 1, SolidNumberObject);
	  break;
	case SQL_CHAR:
	case SQL_VARCHAR:
	case SQL_LONGVARCHAR:
	  Py_INCREF(SolidStringObject);
	  PyTuple_SetItem(tuple, 1, SolidStringObject);
	  break;
	case SQL_DATE:
	case SQL_TIME:
	case SQL_TIMESTAMP:
	  Py_INCREF(SolidDateObject);
	  PyTuple_SetItem(tuple, 1, SolidDateObject);
	  break;
	case SQL_BINARY:
	case SQL_BIT:
	case SQL_LONGVARBINARY:
	case SQL_VARBINARY:
	  Py_INCREF(SolidRawObject);
	  PyTuple_SetItem(tuple, 1, SolidRawObject);
	  break;
	default:
	  {
	    char* bfr;
	    asprintf(&bfr, "Unknown columntype %d\n", coltype);
	    PyErr_SetString(SolidErrorObject, bfr);
	    free(bfr);
	    return NULL;
	  }
	}
      PyTuple_SetItem(retval, i, tuple);
    }
  return retval;
}

static int
solcursor_close1(solcursorobject* solc)
{
  int rc;

  if (solc->hstmt)
    {
      rc = SQLCancel(solc->hstmt);
      if (!(rc == SQL_SUCCESS || rc == SQL_SUCCESS_WITH_INFO))
	{
	  char* errmsg = generate_Errmsg((soldbobject*)solc->cnx, 0, "SQLCancel");
	  PyErr_SetString(SolidErrorObject, errmsg);
	  return -1;
	}
      rc = SQLFreeStmt(solc->hstmt, SQL_DROP);
      if (!(rc == SQL_SUCCESS || rc == SQL_SUCCESS_WITH_INFO))
	{
	  char* errmsg = generate_Errmsg((soldbobject*)solc->cnx, 0, "SQLFreeStmt");
	  PyErr_SetString(SolidErrorObject, errmsg);
	  return -1;
	}
      free(solc->stmt);
      solc->hstmt = 0;
    }
  if (solc->stmt)
    {
      free(solc->stmt);
      solc->stmt = 0;
    }
  return 0;
}

static char solcursor_close__doc[] =
"";

static PyObject*
solcursor_close(PyObject* self, PyObject* args)
{
  solcursorobject* solc = (solcursorobject*)self;
  int rc;
  
  if (!PyArg_ParseTuple(args, ""))
    {
      PyErr_SetString(PyExc_SyntaxError, "close()");
      return NULL;
    }
  if (!solc->hstmt)
    {
      Py_INCREF(Py_None);
      return Py_None;
    }
  rc = solcursor_close1(solc);
  if (rc < 0)
    {
      return NULL;
    }
  Py_INCREF(Py_None);
  return Py_None;
}

static void*
convert(PyObject* value, SWORD sqlType, UDWORD* length, SWORD* cType)
{

  void* data;
  
  switch (sqlType)
    {
    case SQL_BIGINT:
    case SQL_SMALLINT:                      
    case SQL_TINYINT:                       
    case SQL_INTEGER:
      if (value == Py_None)
	{
	  *cType = SQL_C_DEFAULT;
	  *length = SQL_NULL_DATA;
	  data = (void*)0xdeadbeaf;
	  return data;
	}
      *cType = SQL_C_LONG;
      *length = 0;
      data = malloc(sizeof(long));
      if (PyInt_Check(value))
	{
	  *(long*)data = PyInt_AsLong(value);
	  return data;
	}
      else if (PyLong_Check(value))
	{
	  *(long*)data = PyLong_AsLong(value);
	  return data;
	}
      else if (PyFloat_Check(value))
	{
	  *(long*)data = (long)PyFloat_AsDouble(value);
	  return data;
	}
      else if (PyString_Check(value))
	{
	  *length = PyString_Size(value);
	  realloc(data, *length);
	  memcpy(data, PyString_AsString(value), *length);
	  return data;
	}
      else
	{
	  char* bfr;
	  PyObject* s = PyObject_Repr(value);
	  
	  asprintf(&bfr, "%s:%d Unknown Conversion fro SQL_INTEGER from Python value %s",
		   __FILE__, __LINE__, PyString_AsString(s));
	  PyErr_SetString(SolidErrorObject, bfr);
	  free(bfr);
	  SQLFunction = "convert()";
	  return 0;
	}
      break;

    case SQL_FLOAT:
    case SQL_REAL:
      if (value == Py_None)
	{
	  *cType = SQL_C_DEFAULT;
	  *length = SQL_NULL_DATA;
	  data = (void*)0xdeadbeaf;
	  return data;
	}
      *cType = SQL_C_DOUBLE;
      *length = 0;
      data = malloc(sizeof(double));
      if (PyInt_Check(value))
	{
	  *(double*)data = (double) PyInt_AsLong(value);
	  return data;
	}
      else if (PyLong_Check(value))
	{
	  *(double*)data = (double)PyLong_AsLong(value);
	  return data;
	}
      else if (PyFloat_Check(value))
	{
	  *(double*)data = PyFloat_AsDouble(value);
	  return data;
	}
      else if (PyString_Check(value))
	{
	  *cType = SQL_C_CHAR;
	  *length = PyString_Size(value);
	  realloc(data, (*length) + 1);
	  memcpy(data, PyString_AsString(value), *length);
	  return data;
	}
      else
	{
	  char* bfr;
	  PyObject* s = PyObject_Repr(value);
	  asprintf(&bfr, "%s:%d Unknown Conversion fro SQL_INTEGER from Python value %s",
		   __FILE__, __LINE__, PyString_AsString(s));
	  PyErr_SetString(SolidErrorObject, bfr);
	  free(bfr);
	  SQLFunction = "convert()";
	  return 0;
	}
      break;
      
    case SQL_CHAR:
    case SQL_VARCHAR:
    case SQL_LONGVARCHAR:
    case SQL_LONGVARBINARY:
    case SQL_VARBINARY:
      if (value == Py_None)
	{
	  *cType = SQL_C_DEFAULT;
	  *length = SQL_NULL_DATA;
	  data = (void*)0xdeadbeaf;
	  return data;
	}
      *cType = SQL_BINARY;
      if (value == Py_None)
	{
	  *length = SQL_NULL_DATA;
	  data = (void*)0xdeadbeaf;
	  return data;
	}
      if (PyString_Check(value))
	{
	  *length = PyString_Size(value);
	  data = malloc(PyString_Size(value));
	  memcpy(data, PyString_AsString(value), *length);
	  return data;
	}
      else
	{
	  char* bfr;
	  PyObject* s = PyObject_Repr(value);
	  
	  asprintf(&bfr, "%s:%d Unknown Conversion fro SQL_INTEGER from Python value %s",
		   __FILE__, __LINE__, PyString_AsString(s));
	  PyErr_SetString(SolidErrorObject, bfr);
	  SQLFunction = "convert()";
	  free(bfr);
	  return 0;
	}
      break;
    case SQL_DATE:
    case SQL_TIME:
    case SQL_TIMESTAMP:
      fprintf(stderr, "Unknown conversion from SQL_DATE/TIME/TIMESTAMP from python vvalue");
      PyObject_Print(value, stderr, Py_PRINT_RAW);
      return 0;
      break;
    case SQL_BINARY:
    case SQL_BIT:
      fprintf(stderr, "Unknown conversion from SQL_BINARY/BIT from python vvalue");
      PyObject_Print(value, stderr, Py_PRINT_RAW);
      return 0;
      break;
    default:
      fprintf(stderr, "Unknown SQL type %d encountered\n", sqlType);
      return 0;
      break;
    }
}

	    
  

static int
executeSqlStmt(solcursorobject* solc, int nparams, PyObject* tuple)
{
  int       rc;
  int       idx;
  SWORD     sqlType;
  SWORD     cType;
  UDWORD    precision;
  SWORD     scale;
  SWORD     nullable;
  void**    data;
  PyObject* pyValue;
  int       i;
  SDWORD*   dataSize;
  void*     ptr;

  data = (void**)malloc(sizeof (void*) * nparams);
  dataSize = malloc(sizeof (SDWORD) * nparams);
  
  memset(data, nparams * sizeof (void*), '\0');
  for (i = 0; i < nparams; i++)
    {
      rc = SQLDescribeParam(solc->hstmt,
			     i + 1,
			     &sqlType,
			     &precision,
			     &scale,
   			     &nullable);
      pyValue = PyTuple_GetItem(tuple, i);
      ptr = convert(pyValue, sqlType, &dataSize[i], &cType);
      
      if (!ptr)
	return -100;
      data[i] = ptr;
      rc = SQLBindParameter(solc->hstmt,
			    i + 1,
			    SQL_PARAM_INPUT,
			    cType,
			    sqlType,
			    0,
			    0,
			    ptr,
			    0,
			    &dataSize[i]);
      if (!(rc == SQL_SUCCESS || rc == SQL_SUCCESS_WITH_INFO))
	{
	  SQLFunction = "SQLBindParameter";
	  return -1;
	}
    }
  rc = SQLExecute(solc->hstmt);
  SQLFunction = "SQLExecute";
  for (i = 0; i < nparams; i++)
    {
      free(data[i]);
      free(data);
    }
  free(dataSize);
  return rc;
}
			    

static char solcursor_execute__doc[] =
"";

static PyObject*
solcursor_execute(PyObject* self, PyObject* args)
{
  
  solcursorobject* solc = (solcursorobject*) self;
  char* stmt = 0;
  PyObject* params = 0;
  SWORD     nparams = 0;
  int rc;
  char      cmd[128];
  
  if (!PyArg_ParseTuple(args, "s|O", &stmt, &params))
    {
      PyErr_SetString(PyExc_SyntaxError, "execute(stmt[, params])");
      return NULL;
    }
  
  sscanf(stmt, "%127s", cmd);
  /*
    Cannot do any optimization on select statements. Reopeneing or
    reposinioning the cursor is not part of core ODBC but in a Cursor
    library which is only available from Microsoft :-(.
  */
  if ((strcasecmp(cmd, "select") == 0) || solc->stmt && strcmp(solc->stmt, stmt))
    {
      if (solcursor_close1(solc) < 0)
	{
	  return NULL;
	}
    }
  if (!solc->hstmt)
    {
      rc = SQLAllocStmt(((soldbobject*)solc->cnx)->hdbc, &solc->hstmt);
      if (!(rc == SQL_SUCCESS || rc == SQL_SUCCESS_WITH_INFO))
	{
	  char* errmsg = generate_Errmsg((soldbobject*)solc->cnx, 0, "SQLAllocStmt");
	  PyErr_SetString (SolidErrorObject, errmsg);
	  return NULL;
	}
      rc = SQLPrepare(solc->hstmt, stmt, strlen(stmt));
      if (!(rc == SQL_SUCCESS || rc == SQL_SUCCESS_WITH_INFO))
	{
	  char *errmsg = generate_Errmsg((soldbobject*)solc->cnx, solc, "SQLPrepare");
	  PyErr_SetString(SolidErrorObject, errmsg);
	  return NULL;
	}
      solc->stmt = strdup(stmt);
    }
  
  rc = SQLNumParams(solc->hstmt, &nparams);
  if (!(rc == SQL_SUCCESS || rc == SQL_SUCCESS_WITH_INFO))
    {
      	  char *errmsg = generate_Errmsg((soldbobject*)solc->cnx, solc, "SQLNumParams");
	  PyErr_SetString(SolidErrorObject, errmsg);
	  return NULL;
    }
  if (nparams && !params)
    {
      PyErr_SetString(SolidErrorObject, "Parameters required by SQL statement, but none passed\n");
      return NULL;
    }
  if (params && !nparams)
    {
      PyErr_SetString(SolidErrorObject, "Parameters passed not used by SQL statement\n");
    }
  if (params && PyList_Check(params))
    {
      /* okay we got a list. Execute the statment with bindings from each list element */
      PyObject* tuple;
      int       idx;
      int       count;

      count = PyList_Size(params);
      for (idx = 0; idx < count; idx++)
	{
	  rc = executeSqlStmt(solc, nparams, PyList_GetItem(params, idx));
	  if (!(rc == SQL_SUCCESS || rc == SQL_SUCCESS_WITH_INFO))
	    return NULL;
	}
    }
  else if (params)
    {
      rc = executeSqlStmt(solc, nparams, params);
    }
  else
    {
      rc = SQLExecute(solc->hstmt);
      SQLFunction = "SQLExecute";
    }
  if (rc == -100)
    return NULL;
  if (!(rc == SQL_SUCCESS || rc == SQL_SUCCESS_WITH_INFO))
    {
      char *errmsg = generate_Errmsg((soldbobject*)solc->cnx, solc, SQLFunction);
      PyErr_SetString(SolidErrorObject, errmsg);
      return NULL;
    }
  Py_INCREF(Py_None);
  return Py_None;
}

static char solcursor_fetchone__doc[] =
"";

#define MAXCOLS 1500

static PyObject*
solcursor_fetchone1(solcursorobject* solc)
{
  int rc;
  void* data[MAXCOLS];
  SWORD types[MAXCOLS];
  
  int     i;
  HENV    henv;
  HDBC    hdbc;
  UCHAR   errmsg[256];
  UCHAR   colname[32];
  SWORD   coltype;
  SWORD   colnamelen;
  SWORD   nullable;
  UDWORD  collen[MAXCOLS];
  SWORD   scale;
  SDWORD  outlen[MAXCOLS];
  SWORD   nresultcols;
  SDWORD  rowcount;
  PyObject* tuple;
  PyObject* retval;
  PyObject* v;

  
  rc = SQLNumResultCols(solc->hstmt, &nresultcols);
  if (! (rc == SQL_SUCCESS || SQL_SUCCESS_WITH_INFO))
    {
      char *errmsg = generate_Errmsg((soldbobject*)solc->cnx, solc, "SQLNumResultCols");
      PyErr_SetString(SolidErrorObject, errmsg);
      free(errmsg);
      return NULL;
    }
  
  if (nresultcols == 0)
    {
      PyErr_SetString(SolidErrorObject, "cursor does not contain any columns");
    }
  
  for (i = 0; i < nresultcols; i++)
    {
      SQLDescribeCol(solc->hstmt, i + 1, colname,
		     sizeof (colname), &colnamelen,
		     &coltype, &collen[i], &scale,
		     &nullable);
      types[i] = coltype;
      switch (coltype)
	{
	case SQL_BIGINT:
	case SQL_SMALLINT:                      
	case SQL_TINYINT:                       
	case SQL_INTEGER:
	  data[i] = malloc(sizeof(long));
	  rc = SQLBindCol(solc->hstmt, i + 1, SQL_C_LONG,
		     data[i], sizeof(long),&outlen[i]);
	  break;
	case SQL_CHAR:
	case SQL_VARCHAR:                       
	case SQL_VARBINARY:
	  data[i] = malloc(collen[i]);
	  rc = SQLBindCol(solc->hstmt, i + 1, SQL_C_BINARY,
		     data[i], collen[i],&outlen[i]);
	  break;
 	case SQL_LONGVARBINARY:                 
	case SQL_LONGVARCHAR:                   
	  data[i] = 0;
	  rc = SQLBindCol(solc->hstmt, i + 1, SQL_C_BINARY,
			  data[i], collen[i], &outlen[i]);
	  break;
	case SQL_REAL:                          
	case SQL_DOUBLE:                        
	case SQL_FLOAT:                         
	  data[i] = malloc(sizeof(double));
	  rc = SQLBindCol(solc->hstmt, i + 1, SQL_C_DOUBLE,
		     data[i], sizeof(double),&outlen[i]);
	  break;
	case SQL_DATE:
	  data[i] = malloc(sizeof (DATE_STRUCT ));
	  rc = SQLBindCol(solc->hstmt, i + 1, SQL_C_DATE,
		     data[i], 11,&outlen[i]);
	  break;
	case SQL_TIME:                         
	case SQL_TIMESTAMP:                     
	  data[i] = malloc(sizeof(long));
	  SQLBindCol(solc->hstmt, i + 1, SQL_C_LONG,
		     data[i], sizeof(long),&outlen[i]);
	  break;
	case SQL_BINARY:                  
	  fprintf(stderr,"Don't know how to handle SQL_BINARY now\n");
	  break;
	case SQL_BIT:                           
	  fprintf(stderr,"Don't know how to handle SQL_BIT now\n");
	  break;
	case SQL_DECIMAL:                       
	  fprintf(stderr,"Don't know how to handle SQL_DECIMAL now\n");
	  break;
	case SQL_NUMERIC:                       
	  fprintf(stderr,"Don't know how to handle SQL_NUMERIC now\n");
	  break;
	default:
	  fprintf(stderr,"Unknown SQL Type[%d] in SqlBindCol received\n",coltype); 
	  break;
	}
    }
  rc = SQLFetch(solc->hstmt);
  if (rc == SQL_NO_DATA_FOUND)
    {
      for (i = 0; i < nresultcols; i++)
	{
	  free(data[i]);
	}
      Py_INCREF(Py_None);
      return Py_None;
    }
  if (!(rc == SQL_SUCCESS || rc == SQL_SUCCESS_WITH_INFO))
    {
      char* error = generate_Errmsg((soldbobject*)solc->cnx, 0, "SQLTransact");
      PyErr_SetString(SolidErrorObject, error);
      free(error);
      for (i = 0; i < nresultcols; i++)
	{
	  free(data[i]);
	}
      return NULL;
    }
  tuple = PyTuple_New(nresultcols);
  for (i = 0; i < nresultcols; i++)
    {
      if (outlen[i] == SQL_NULL_DATA)
	{
	  Py_INCREF(Py_None);
	  PyTuple_SetItem(tuple, i, Py_None);
	}
      else
	{
	  switch (types[i])
	    {
	    case SQL_BIGINT:
	    case SQL_SMALLINT:                      
	    case SQL_TINYINT:                       
	    case SQL_INTEGER:
	      v = PyInt_FromLong(*(long*)data[i]);
	      PyTuple_SetItem(tuple, i, v);
	      free(data[i]);
	      break;
	    case SQL_CHAR:
	    case SQL_VARCHAR:
	    case SQL_VARBINARY:
	      {
		v = PyString_FromStringAndSize(data[i], outlen[i]);
		PyTuple_SetItem(tuple, i, v);
		free(data[i]);
		break;
	      }
	    case SQL_LONGVARBINARY:                 
	    case SQL_LONGVARCHAR:                   
	      {
		SDWORD avail;
		int    bfrlen = 1024;
 		int    done = 0;
		int    offset = 0;
		int    length = bfrlen;
		
		data[i] = malloc(bfrlen);
		
		while (!done)
		  {
		    rc = SQLGetData(solc->hstmt, i + 1, SQL_C_BINARY, data[i] + offset, bfrlen, &avail);
		    if (rc == SQL_SUCCESS)
		      {
			length = offset + avail;
			done = 1;
		      }
		    else if (rc == SQL_SUCCESS_WITH_INFO)
		      {
			/* realloc the whole space */
			data[i] = realloc(data[i], avail);
			/* advance pointer to the end of the previously fetched data */
			offset = bfrlen;
			/* the length of the remaining data */
			bfrlen = avail - bfrlen;
			/* the total length is in available */
			length = avail; 
		      }
		    else
		      {
			char* error = generate_Errmsg((soldbobject*)solc->cnx, 0, "SQLGetData");
			PyErr_SetString(SolidErrorObject, error);
			free(error);
			for (; i < nresultcols; i++)
			  {
			    free(data[i]);
			  }
			Py_DECREF(tuple);
			return NULL;
		      }
		  }
		v = PyString_FromStringAndSize(data[i], length);
		PyTuple_SetItem(tuple, i, v);
		free(data[i]);
	      }
	    break;
	    case SQL_REAL:                          
	    case SQL_DOUBLE:                        
	    case SQL_FLOAT:
	      v = PyFloat_FromDouble(*(double*)data[i]);
	      PyTuple_SetItem(tuple, i, v);
 	      free(data[i]);
	      break;
	    case SQL_DATE:
	      {
		char s[20];
		DATE_STRUCT* d = data[i];
		sprintf(s, "%04d-%02d-%02d", d->year, d->month, d->day);
		v = PyString_FromString(s);
		PyTuple_SetItem(tuple, i, v);
		free(data[i]);
	      }
	    break;
	    case SQL_TIME:                         
	    case SQL_TIMESTAMP:                     
	      v = PyInt_FromLong(*(long*)data[i]);
	      PyTuple_SetItem(tuple, i, v);
	      free(data[i]);
	      break;
	    break;
	    case SQL_BINARY:
	      fprintf(stderr, "Don't know how to handle SQL_BINARY now\n");
	      break;
	    case SQL_BIT:                           
	      fprintf(stderr,"Don't know how to handle SQL_BIT now\n");
	      break;
	    case SQL_DECIMAL:                       
	      fprintf(stderr,"Don't know how to handle SQL_DECIMAL now\n");
	      break;
	    break;
	    case SQL_NUMERIC:                       
	      fprintf(stderr,"Don't know how to handle SQL_NUMERIC now\n");
	      break;
	    default:
	      fprintf(stderr,"Unknown SQL Type[%d] in SqlBindCol received\n",coltype); 
	      break;
	    }
	}
    }
  return tuple;
}  

static PyObject*
solcursor_fetchone(PyObject* self, PyObject* args)
{
  solcursorobject* solc = (solcursorobject*) self;
  int rc;

  if (!PyArg_ParseTuple(args, ""))
    {
      PyErr_SetString(PyExc_SyntaxError, "fetchone()");
      return NULL;
    }
  if (!solc->hstmt)
    {
      PyErr_SetString(SolidErrorObject, "fetchone() called without execute()");
      return NULL;
    }
  return solcursor_fetchone1(solc);
}

static char solcursor_fetchmany__doc[] =
"";

static PyObject*
solcursor_fetchmany(PyObject* self, PyObject* args)
{
  solcursorobject* solc = (solcursorobject*)self;
  int              rc;
  int              nrows = 0;
  int              i;
  PyObject*        tuple = 0;
  PyObject*        retval = 0;
  
  if (!PyArg_ParseTuple(args,"|i", &nrows))
    {
      PyErr_SetString(PyExc_SyntaxError, "fetchmany([size])");
      return NULL;
    }
  if (nrows == 0)
    {
      nrows = solc->arraysize;
    }
  retval = PyList_New(0);
  
  for (i = 0; i < nrows; i++)
    {
      tuple = solcursor_fetchone1(solc);
      if (tuple == NULL)
	return NULL;
      if (tuple == Py_None)
	return retval;
      PyList_Append(retval, tuple);
    }
  return retval;
}

static char solcursor_fetchall__doc[] =
"";

static PyObject*
solcursor_fetchall(PyObject* self, PyObject* args)
{

  PyObject* retval = 0;
  PyObject* tuple  = 0;
  solcursorobject* solc = (solcursorobject*) self;

  if (!PyArg_ParseTuple(args,""))
    {
      PyErr_SetString(PyExc_SyntaxError, "fetchall()");
      return NULL;
    }
  retval = PyList_New(0);
  while(1)
    {
      tuple = solcursor_fetchone1(solc);
      Py_DECREF(retval);
      if (tuple == NULL)
	return NULL;
      if (tuple == Py_None)
	return retval;
      PyList_Append(retval, tuple);
    }
  return retval;
}


static char solcursor_setoutputsize__doc[] =
"";

static PyObject*
solcursor_setoutputsize(PyObject* self, PyObject* args)
{
  PyErr_SetString(SolidErrorObject, "setoutputsize() c7urrently not implemented");
  return NULL;
}


static struct PyMethodDef solcursor_methods[] = {
  {"arraysize",    (PyCFunction) solcursor_arraysize, METH_VARARGS,  solcursor_arraysize__doc},
  {"close",        (PyCFunction) solcursor_close,       METH_VARARGS, solcursor_close__doc},
  {"execute",      (PyCFunction) solcursor_execute,     METH_VARARGS, solcursor_execute__doc},
  {"fetchone",     (PyCFunction) solcursor_fetchone,    METH_VARARGS, solcursor_fetchone__doc},
  {"fetchmany",    (PyCFunction) solcursor_fetchmany,   METH_VARARGS, solcursor_fetchmany__doc},
  {"fetchall",     (PyCFunction) solcursor_fetchall,    METH_VARARGS, solcursor_fetchall__doc},
  {"setoutputsize",(PyCFunction) solcursor_setoutputsize, METH_VARARGS, solcursor_setoutputsize__doc}
};



static solcursorobject*
newsolcursorobject()
{
  solcursorobject* self;
  self = PyObject_NEW(solcursorobject, &SolCursorType);
  if (self == 0)
    return NULL;
  self->cnx = 0;
  self->hstmt = 0;
  self->data = 0;
  self->types = 0;
  self->stmt = 0;
  self->arraysize = 1;
  return self;
}

static void
solcursor_dealloc(solcursorobject* self)
{
  solcursor_close1(self);
  Py_DECREF(self->cnx);
  PyMem_DEL(self);
}


static PyObject*
solcursor_getattr(solcursorobject* self, char* name)
{
  if (!strcmp(name, "description"))
    return solcursor_description(self);
  return Py_FindMethod(solcursor_methods, (PyObject*)self, name);
}

static char SolCursor__doc[] =
"";


static PyTypeObject SolCursorType = {
  PyObject_HEAD_INIT(&PyType_Type)
  0,				/*ob_size*/
  "Cursor",			/*tp_name*/
  sizeof(solcursorobject),		/*tp_basicsize*/
  0,				/*tp_itemsize*/
  /* methods */
  (destructor)solcursor_dealloc,	/*tp_dealloc*/
  (printfunc)0,		/*tp_print*/
  (getattrfunc)solcursor_getattr,	/*tp_getattr*/
  (setattrfunc)0,	/*tp_setattr*/
  (cmpfunc)0,		/*tp_compare*/
  (reprfunc)0,		/*tp_repr*/
  0,			/*tp_as_number*/
  0,		/*tp_as_sequence*/
  0,		/*tp_as_mapping*/
  (hashfunc)0,		/*tp_hash*/
  (ternaryfunc)0,		/*tp_call*/
  (reprfunc)0,		/*tp_str*/
  
  /* Space for future expansion */
  0L,0L,0L,0L,
  SolCursor__doc /* Documentation string */
};




/* Initialization function for the module (*must* be called initsoliddb) */

static char solid_module_documentation[] = 
""
;

void
initsoliddb()
{
  PyObject *m, *d;

  /* Create the module and add the functions */
  m = Py_InitModule4("soliddb", soliddb_methods,
		     solid_module_documentation,
		     (PyObject*)NULL,PYTHON_API_VERSION);
  
  /* Add some symbolic constants to the module */
  d = PyModule_GetDict(m);
  SolidErrorObject = PyString_FromString("soliddb.error");
  SolidStringObject = PyInt_FromLong(1);
  SolidRawObject = PyInt_FromLong(2);
  SolidNumberObject = PyInt_FromLong(3);
  SolidDateObject = PyInt_FromLong(4);
  SolidRowidObject = PyInt_FromLong(5);

  PyDict_SetItemString(d, "error", SolidErrorObject);
  PyDict_SetItemString(d, "STRING", SolidStringObject);
  PyDict_SetItemString(d, "RAW", SolidRawObject);
  PyDict_SetItemString(d, "NUMBER", SolidNumberObject);
  PyDict_SetItemString(d, "DATE", SolidDateObject);
  PyDict_SetItemString(d, "ROWID", SolidRowidObject);
  
  /* XXXX Add constants here */
  
  /* Check for errors */
  if (PyErr_Occurred())
    Py_FatalError("can't initialize module solid");
}

/*
 * $Log$
 */
