/* 
 *
 * PyGres95, version 1.0b
 * A Python interface for Postgres95 database.
 * Written by Pascal Andre, andre@chimay.via.ecp.fr.
 * Copyright (c) 1995, Pascal Andre (andre@via.ecp.fr).
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose, without fee, and without a written agreement
 * is hereby granted, provided that the above copyright notice and this
 * paragraph and the following two paragraphs appear in all copies or in any 
 * new file that contains a substantial portion of this file.
 *
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT, 
 * SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES, INCLUDING LOST PROFITS, 
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE 
 * AUTHOR HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * THE AUTHOR SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE 
 * AUTHOR HAS NO OBLIGATIONS TO PROVIDE MAINTENANCE, SUPPORT, UPDATES, 
 * ENHANCEMENTS, OR MODIFICATIONS.
 *
 */

#include <allobjects.h>
#include <abstract.h>
#include <libpq-fe.h>
#include <libpq/libpq-fs.h>
#include <stdio.h>

static object *PG95Error;

/* taken from fileobject.c */
#define BUF(v) GETSTRINGVALUE((stringobject *)v)

#define CHECK_OBJECT      1
#define CHECK_RESULT      2
#define CHECK_LARGE       4          
#define CHECK_OPEN        8
#define CHECK_CLOSE       16
#define CHECK_LO          CHECK_LARGE | CHECK_OBJECT

#define MAX_BUFFER_SIZE   8192           /* maximum transaction size */

#ifndef NO_DIRECT
#define DIRECT_ACCESS     1              /* enables direct access functions */
#endif /* NO_DIRECT */

#ifndef NO_LARGE
#define LARGE_OBJECTS     1              /* enables large objects support */
#endif /* NO_LARGE */

#ifndef NO_DEF_VAR
#define DEFAULT_VARS      1              /* enables default variables use */ 
#endif /* NO_DEF_VAR */

/* --------------------------------------------------------------------- */

/* MODULE GLOBAL VARIABLES */

#ifdef DEFAULT_VARS

object *pg95_default_host;                    /* default database host */
object *pg95_default_base;                    /* default database name */
object *pg95_default_opt;                     /* default connection options */
object *pg95_default_tty;                     /* default debug tty */
object *pg95_default_port;                    /* default connection port */

#endif /* DEFAULT_VARS */

/* --------------------------------------------------------------------- */

/* OBJECTS DECLARATION */

/* pg95 connection object */

typedef struct {
  OB_HEAD
  int valid;              /* validity flag */
  PGresult *last_result;  /* last result content */
  PGconn *cnx;            /* PostGres connection handle */
} pg95object;

staticforward typeobject Pg95type;

#define is_pg95object(v) ((v)->ob_type == &Pg95type)

#ifdef LARGE_OBJECTS 
/* pg95 large object */

typedef struct {
  OB_HEAD
  pg95object *pgcnx;
  Oid lo_oid;
  int lo_fd;
} pg95largeobject;

staticforward typeobject Pg95largeType;

#define is_pg95largeobject(v) ((v)->ob_type == &Pg95largeType)
#endif /* LARGE_OBJECTS */

/* --------------------------------------------------------------------- */

/* INTERNAL FUNCTIONS */

/* validity check (connection object) */
static int 
check_up(self, level)
pg95object *self;
int level;
{
  if (level & CHECK_OBJECT)
    if (!self->valid) {
      err_setstr(PG95Error, "object is not valid (bad connection).");
      return 0;
    }
  if (level & CHECK_RESULT)
    if (!self->last_result) {
      err_setstr(PG95Error, "last result is not valid.");
      return 0;
    }
  return 1;
};

#ifdef LARGE_OBJECTS
/* validity check (large object) */
static int 
check_lo(self, level)
pg95largeobject *self;
int level;
{
  if (level & CHECK_OBJECT)
    if (self->pgcnx->valid) {
      err_setstr(PG95Error, "pg connection is not valid.");
      return 0;
    }
  if (level & CHECK_LARGE)
    if (!self->lo_oid) {
      err_setstr(PG95Error, "object is not valid (null oid).");
      return 0;
    }
  if (level & CHECK_OPEN)
    if (self->lo_fd<0) {
      err_setstr(IOError, "object is not opened.");
      return 0;
    }
  if (level & CHECK_CLOSE)
    if (self->lo_fd>=0) {
      err_setstr(IOError, "object is already opened.");
      return 0;
    }
  return 1;
}
#endif /* LARGE_OBJECTS */

/* --------------------------------------------------------------------- */

#ifdef LARGE_OBJECTS
/* PG95 CONNECTION OBJECT IMPLEMENTATION */

/* pg95largeobject initialisation (from pg95object) */

/* creates large object */
static object *
pg95_locreate(self, args)
pg95object *self;
object *args;
{
  int mode;
  pg95largeobject *npglo;

  /* checks validity */
  if (!check_up(self, CHECK_OBJECT))
    return NULL;

  /* gets arguments */
  if (!newgetargs(args, "i", &mode)) {
    err_setstr(TypeError, "locreate(mode), with mode (integer).");
    return NULL;
  }

  npglo = NEWOBJ(pg95largeobject, &Pg95largeType);
  if (npglo == NULL)
    return NULL;
  npglo->pgcnx = self;
  XINCREF(self);
  npglo->lo_fd = -1;
  npglo->lo_oid = lo_creat(self->cnx, mode);

  /* checks result validity */
  if (npglo->lo_oid == 0) {
    err_setstr(PG95Error, "can't create large object.");
    XDECREF(npglo);
    return NULL;
  };

  return (object *)npglo;
}

/* init from already known oid */
static object *
pg95_getlo(self, args)
pg95object *self;
object *args;
{
  int lo_oid;
  pg95largeobject *npglo;

  /* checks validity */
  if (!check_up(self, CHECK_OBJECT))
    return NULL;

  /* gets arguments */
  if (!newgetargs(args, "i", &lo_oid)) {
    err_setstr(TypeError, "loopen(oid), with oid (integer).");
    return NULL;
  }
  if (!lo_oid) {
    err_setstr(ValueError, "the object oid can't be null.");
    return NULL;
  }

  /* creates object */
  npglo = NEWOBJ(pg95largeobject, &Pg95largeType);
  if (npglo == NULL)
    return NULL;
  npglo->pgcnx = self;
  XINCREF(self);
  npglo->lo_fd = -1;
  npglo->lo_oid = lo_oid;

  return (object *)npglo;
}

/* import unix file */
static object *
pg95_loimport(self, args)
pg95object *self;
object *args;
{
  char *name;
  pg95largeobject *npglo;

  /* checks validity */
  if (!check_up(self, CHECK_OBJECT))
    return NULL;

  /* gets arguments */
  if (!newgetargs(args, "s", &name)) {
    err_setstr(TypeError, "loimport(name), with name (string).");
    return NULL;
  }

  npglo = NEWOBJ(pg95largeobject, &Pg95largeType);
  if (npglo == NULL)
    return NULL;
  npglo->pgcnx = self;
  XINCREF(self);
  npglo->lo_fd = -1;
  npglo->lo_oid = lo_import(self->cnx, name);

  /* checks result validity */
  if (npglo->lo_oid == 0) {
    err_setstr(PG95Error, "can't create large object.");
    XDECREF(npglo);
    return NULL;
  };

  return (object *)npglo;
}

/* pg95largeobject methods */

/* destructor */
static void
pg95large_dealloc(self)
pg95largeobject *self;
{
  if (self->lo_fd>=0)
    if (self->pgcnx->valid == 1)
      lo_close(self->pgcnx->cnx, self->lo_fd);
  XDECREF(self->pgcnx);
  DEL(self);
}

/* opens large object */
static object *
pg95large_open(self, args)
pg95largeobject *self;
object *args;
{
  int mode, fd;

  /* check validity */
  if (!check_lo(self, CHECK_LO | CHECK_CLOSE))
    return NULL;

  /* gets arguments */
  if (!newgetargs(args, "i", &mode)) {
    err_setstr(TypeError, "open(mode), with mode(integer).");
    return NULL;
  }

  /* opens large object */
  fd = lo_open(self->pgcnx->cnx, self->lo_oid, mode);
  if (fd<0) {
    err_setstr(IOError, "can't open large object.");
    return NULL;
  }
  self->lo_fd = fd;

  /* no error : returns None */
  INCREF(None);
  return None;
}

/* close large object */
static object *
pg95large_close(self, args)
pg95largeobject *self;
object *args;
{
  /* checks args */
  if (!newgetargs(args, "")) {
    err_setstr(SyntaxError, "method close() takes no parameters.");
    return NULL;
  }

  /* checks validity */
  if (!check_lo(self, CHECK_LO | CHECK_OPEN))
    return NULL;

  /* closes large object */
  if (lo_close(self->pgcnx->cnx, self->lo_fd)) {
    err_setstr(IOError, "error while closing large object fd.");
    return NULL;
  }
  self->lo_fd = -1;

  /* no error : returns None */
  INCREF(None);
  return None;
}

/* reads from large object */
static object *
pg95large_read(self, args)
pg95largeobject *self;
object *args;
{
  int size;
  object *buffer;

  /* checks validity */
  if (!check_lo(self, CHECK_LO | CHECK_OPEN))
    return NULL;

  /* gets arguments */
  if (!newgetargs(args, "i", &size)) {
    err_setstr(TypeError, "read(size), wih size (integer).");
    return NULL;
  }
  if (size <= 0) {
    err_setstr(ValueError, "size must be positive.");
    return NULL;
  }

  /* allocate buffer and runs read */
  buffer = newsizedstringobject((char *)NULL, size);
  size = lo_read(self->pgcnx->cnx, self->lo_fd, (char *)BUF(buffer), size);
  if (size < 0) {
    err_setstr(IOError, "error while reading.");
    XDECREF(buffer);
    return NULL;
  }

  /* resize buffer and returns it */
  resizestring(&buffer, size);
  return buffer;
}

/* write to large object */
static object *
pg95large_write(self, args)
pg95largeobject *self;
object *args;
{
  object *buffer;
  int size;

  /* checks validity */
  if (!check_lo(self, CHECK_LO | CHECK_OPEN))
    return NULL;

  /* gets arguments */
  if (!newgetargs(args, "s", &buffer)) {
    err_setstr(TypeError, "write(buffer), with buffer (sized string).");
    return NULL;
  }

  /* sends query */
  size = lo_write(self->pgcnx->cnx, self->lo_fd, (char *)BUF(buffer), 
		  getstringsize(buffer));
  if (size < getstringsize(buffer)) {
    err_setstr(IOError, "buffer truncated during write.");
    return NULL;
  }

  /* no error : returns None */
  INCREF(None);
  return None;
}

/* go to position in large object */
static object *
pg95large_lseek(self, args)
pg95largeobject *self;
object *args;
{
  int ret, offset, whence;

  /* checks validity */
  if (!check_lo(self, CHECK_LO | CHECK_OPEN))
    return NULL;

  /* gets arguments */
  if (!newgetargs(args, "ii", offset, whence)) {
    err_setstr(TypeError, "lseek(offset, whence), with offset and whence "
	       "(integers).");
    return NULL;
  }

  /* sends query */
  ret = lo_lseek(self->pgcnx->cnx, self->lo_fd, offset, whence);
  if (ret==-1) {
    err_setstr(IOError, "error while moving cursor.");
    return NULL;
  }

  /* returns position */
  return newintobject(ret);
};

/* gets large object size */
static object *
pg95large_size(self, args)
pg95largeobject *self;
object *args;
{
  int start, end;

  /* checks args */
  if (!newgetargs(args, "")) {
    err_setstr(SyntaxError, "method size() takes no parameters.");
    return NULL;
  }

  /* checks validity */
  if (!check_lo(self, CHECK_LO | CHECK_OPEN))
    return NULL;

  /* gets current position */
  start = lo_tell(self->pgcnx->cnx, self->lo_fd);
  if (start == -1) {
    err_setstr(IOError, "error while getting current position.");
    return NULL;
  }

  /* gets end position */
  end = lo_lseek(self->pgcnx->cnx, self->lo_fd, 0, SEEK_END);
  if (end == -1) {
    err_setstr(IOError, "error while getting end position.");
    return NULL;
  }

  /* move back to start position */
  start = lo_lseek(self->pgcnx->cnx, self->lo_fd, start, SEEK_SET);
  if (start == -1) {
    err_setstr(IOError, "error while moving back to first position.");
    return NULL;
  }

  /* returns size */
  return newintobject(end);
}

/* gets large object cursor position */
static object *
pg95large_tell(self, args)
pg95largeobject *self;
object *args;
{
  int start;

  /* checks args */
  if (!newgetargs(args, "")) {
    err_setstr(SyntaxError, "method tell() takes no parameters.");
    return NULL;
  }

  /* checks validity */
  if (!check_lo(self, CHECK_LO | CHECK_OPEN))
    return NULL;

  /* gets current position */
  start = lo_tell(self->pgcnx->cnx, self->lo_fd);
  if (start == -1) {
    err_setstr(IOError, "error while getting position.");
    return NULL;
  }

  /* returns size */
  return newintobject(start);
}

/* exports large object as unix file */
static object *
pg95large_export(self, args)
pg95largeobject *self;
object *args;
{
  char *name;

  /* checks validity */
  if (!check_lo(self, CHECK_LO | CHECK_CLOSE)) 
    return NULL;

  /* gets arguments */
  if (!newgetargs(args, "s", &name)) {
    err_setstr(TypeError, "export(filename), with filename (string).");
    return NULL;
  }
  
  /* runs command */
  if (!lo_export(self->pgcnx->cnx, self->lo_oid, name)) {
    err_setstr(IOError, "error while exporting large object.");
    return NULL;
  }

  INCREF(None);
  return None;
}

/* deletes a large object */
static object *
pg95large_unlink(self, args)
pg95largeobject *self;
object *args;
{
  /* checks args */
  if (!newgetargs(args, "")) {
    err_setstr(SyntaxError, "method unlink() takes no parameters.");
    return NULL;
  }

  /* checks validity */
  if (!check_lo(self, CHECK_LO | CHECK_CLOSE))
    return NULL;

  /* deletes the object, invalidate it on success */
  if (!lo_unlink(self->pgcnx->cnx, self->lo_oid)) {
    err_setstr(IOError, "error while unlinking large object");
    return NULL;
  }
  self->lo_oid = 0;

  INCREF(None);
  return None;
}

/* large object methods */
static struct methodlist pg95large_methods[] = {
  {"open", (method)pg95large_open, 1},          /* opens large object*/
  {"close", (method)pg95large_close, 1},        /* closes large object */
  {"read", (method)pg95large_read, 1},          /* reads from large object */
  {"write", (method)pg95large_write, 1},        /* writes to large object */
  {"seek", (method)pg95large_lseek, 1},         /* seeks position */
  {"size", (method)pg95large_size, 1},          /* gives object size */
  {"tell", (method)pg95large_tell, 1},          /* gives position in lobj */
  {"export", (method)pg95large_export, 1},      /* exports to unix file */
  {"unlink", (method)pg95large_unlink, 1},      /* deletes a large object */
  {NULL, NULL}		                        /* sentinel */
};

/* get attribute */
static object *
pg95large_getattr(self, name)
pg95largeobject *self;
char *name;
{
  /* list postgres95 large object fields */
  
  /* associated pg95 connection object */
  if (!strcmp(name, "pg95cnx"))
    if (check_lo(self, CHECK_LO)) {
      INCREF(self->pgcnx);
      return (object *)(self->pgcnx);
    }
    else {
      INCREF(None);
      return None;
    }

  /* large object oid */
  if (!strcmp(name, "oid"))
    if (check_lo(self, CHECK_LO)) {
      return newintobject(self->lo_oid);
    }
    else {
      INCREF(None);
      return None;
    }

  /* error (status) message */
  if (!strcmp(name, "error"))
    if (check_up(self, CHECK_OBJECT)) {
      char *error=PQerrorMessage(self->pgcnx->cnx);
      return newstringobject(error);
    }
    else {
      INCREF(None);
      return None;
    }

  /* attributes list */
  if (!strcmp(name, "__members__")) {
    object *list = newlistobject(3);
    if (list) {
      setlistitem(list, 0, newstringobject("oid"));
      setlistitem(list, 1, newstringobject("pg95cnx"));
      setlistitem(list, 2, newstringobject("error"));
    }
    return list;
  }

  return findmethod(pg95large_methods, (object *)self, name);
}

/* object type definition */
staticforward typeobject Pg95largeType = {
  OB_HEAD_INIT(&Typetype)
  0,			            /*ob_size*/
  "pg95large",	         	    /*tp_name*/
  sizeof(pg95largeobject),	    /*tp_basicsize*/
  0,			            /*tp_itemsize*/
  /* methods */
  (destructor)pg95large_dealloc,    /*tp_dealloc*/
  0,			            /*tp_print*/
  (getattrfunc)pg95large_getattr,   /*tp_getattr*/
  0,                                /*tp_setattr*/
  0,			            /*tp_compare*/
  0,			            /*tp_repr*/
  0,			            /*tp_as_number*/
  0,			            /*tp_as_sequence*/
  0,			            /*tp_as_mapping*/
  0,			            /*tp_hash*/
};

#endif /* LARGE_OBJECTS */

/* --------------------------------------------------------------------- */

/* PG95 CONNECTION OBJECT IMPLEMENTATION */

/* pg95object initialisation (from module) */

static object *
pg95connect(self, args, dict)
pg95object *self;
object *args;
object *dict;
{
  char *pghost, *pgopt, *pgtty, *pgdbname;
  int pgport;
  char port_buffer[20];
  object *temp;
  pg95object *npgobj;
  PGconn *test_cnx;

  pghost=pgopt=pgtty=pgdbname=NULL;
  pgport=-1;

  /* parses standard arguments */
  if (!newgetargs(args, "z|lzzz", &pghost, &pgport, &pgopt, &pgtty, 
		  &pgdbname)) {
    err_clear();
    if (!newgetargs(args, "")) {
      err_setstr(TypeError, "connect(host, port, opt, tty, dbname), "
		 "with args (opt., strings or None).");
      return NULL;
    }
  }

  /* looks for keywords arguments */
  if (PyMapping_Check(dict)) {
    /* server host */
    if (PyMapping_HasKeyString(dict, "host")) {
      temp = PyMapping_GetItemString(dict, "host");
      if (temp) {
	if (pghost) {
	  err_setstr(SyntaxError, "Duplicate argument definition.");
	  return NULL;
	}
	if (!is_stringobject(temp)) {
	  err_setstr(TypeError, "'Host' argument must be a string.");
	  return NULL;
	}
	pghost=getstringvalue(temp);
      }
    }

    /* server port */
    if (PyMapping_HasKeyString(dict, "port")) {
      temp = PyMapping_GetItemString(dict, "port");
      if (temp) {
	if (pgport != -1) {
	  err_setstr(SyntaxError, "Duplicate argument definition.");
	  return NULL;
	}
	if (!is_intobject(temp)) {
	  err_setstr(TypeError, "'Port' argument must be an integer.");
	  return NULL;
	}
	pgport = getintvalue(temp);
      }
    }

    /* connection options */
    if (PyMapping_HasKeyString(dict, "opt")) {
      temp = PyMapping_GetItemString(dict, "opt");
      if (temp) {
	if (pgtty) {
	  err_setstr(SyntaxError, "Duplicate argument definition.");
	  return NULL;
	}
	if (!is_stringobject(temp)) {
	  err_setstr(TypeError, "'opt' argument must be a string.");
	  return NULL;
	}
	pgopt = getstringvalue(temp);
      }
    }

    /* debug terminal */
    if (PyMapping_HasKeyString(dict, "tty")) {
      temp = PyMapping_GetItemString(dict, "tty");
      if (temp) {
	if (pgtty) {
	  err_setstr(SyntaxError, "Duplicate argument definition.");
	  return NULL;
	}
	if (!is_stringobject(temp)) {
	  err_setstr(TypeError, "'tty' argument must be a string.");
	  return NULL;
	}
	pgtty=getstringvalue(temp);
      }
    }

    /* database name */
    if (PyMapping_HasKeyString(dict, "dbname")) {
      temp = PyMapping_GetItemString(dict, "dbname");
      if (temp) {
	if (pgdbname) {
	  err_setstr(SyntaxError, "Duplicate argument definition.");
	  return NULL;
	}
	if (!is_stringobject(temp)) {
	  err_setstr(TypeError, "'dbname' argument must be a string.");
	  return NULL;
	}
	pgdbname=getstringvalue(temp);
      }
    }
  }

#ifdef DEFAULT_VARS
  /* handles defaults variables (for unintialised vars) */
  if ((!pghost) && (pg95_default_host!=None))
    pghost=getstringvalue(pg95_default_host);
  if ((pgport == -1) && (pg95_default_port!=None))
    pgport=getintvalue(pg95_default_port);
  if ((!pgopt) && (pg95_default_opt!=None))
    pgopt=getstringvalue(pg95_default_opt);
  if ((!pgtty) && (pg95_default_tty!=None))
    pgtty=getstringvalue(pg95_default_tty);
  if ((!pgdbname) && (pg95_default_base!=None)) 
    pgdbname=getstringvalue(pg95_default_base);
#endif /* DEFAULT_VARS */

  npgobj = NEWOBJ(pg95object, &Pg95type);
  if (npgobj == NULL)
    return NULL;
  npgobj->valid = 0;
  npgobj->last_result = NULL;

  if (pgport!=-1) {
    bzero(port_buffer, sizeof(port_buffer));
    sprintf(port_buffer, "%d", pgport);
    npgobj->cnx=PQsetdb(pghost, port_buffer, pgopt, pgtty, pgdbname);
  }
  else  npgobj->cnx=PQsetdb(pghost, NULL, pgopt, pgtty, pgdbname);

  if (PQstatus(npgobj->cnx) == CONNECTION_BAD) {
    char *error = PQerrorMessage(npgobj->cnx);
    err_setstr(PG95Error, error);
    XDECREF(npgobj);
    return NULL;
  };

  npgobj->valid = 1;
  return (object *)npgobj;
};

/* pg95object methods */

/* destructor */
static void
pg95_dealloc(self)
pg95object *self;
{
  PQfinish(self->cnx);
  DEL(self);
}

/* resets connection */
static object *pg95_reset(self, args)
pg95object *self;
object *args;
{
  /* checks object validity */
  if (!check_up(self, CHECK_OBJECT))
    return NULL;

  /* checks args */
  if (!newgetargs(args, "")) {
    err_setstr(SyntaxError, "method reset() takes no parameters.");
    return NULL;
  }

  /* resets the connection */
  PQreset(self->cnx);
  INCREF(None);
  return None;
};

/* list fields names from last result */
static object *
pg95_listfields(self, args)
pg95object *self;
object *args;
{
  int i, n;
  char *name;
  object *fieldstuple, *str;

  /* checks validity */
  if (!check_up(self, CHECK_OBJECT | CHECK_RESULT))
    return NULL;

  /* checks args */
  if (!newgetargs(args, "")) {
    err_setstr(SyntaxError, "method listfields() takes no parameters.");
    return NULL;
  }

  /* builds tuple */
  n = PQnfields(self->last_result);
  fieldstuple = newtupleobject(n);
  for (i=0; i<n; i++) {
    name = PQfname(self->last_result, i);
    str = newstringobject(name);
    settupleitem(fieldstuple, i, str);
  }
  return fieldstuple;
};

/* get field name from last result */
static object *
pg95_fieldname(self, args)
pg95object *self;
object *args;
{
  int i;
  char *name;
  object *str;

  /* checks validity */
  if (!check_up(self, CHECK_OBJECT | CHECK_RESULT))
    return NULL;

  /* gets args */
  if (!newgetargs(args, "i", &i)) {
    err_setstr(TypeError, "fieldname(number), with number(integer).");
    return NULL;
  }

  /* checks number validity */
  if (i >= PQnfields(self->last_result)) {
    err_setstr(ValueError, "invalid field number.");
    return NULL;
  }

  /* gets fields name and builds object */
  name = PQfname(self->last_result, i);
  str = newstringobject(name);
  return str;
}

/* gets fields number from name in last result */
static object *
pg95_fieldnum(self, args)
pg95object *self;
object *args;
{
  char *name;
  int num;

  /* checks validity */
  if (!check_up(self, CHECK_OBJECT | CHECK_RESULT))
    return NULL;

  /* gets args */
  if (!newgetargs(args, "s", &name)) {
    err_setstr(TypeError, "fieldnum(name), with name (string).");
    return NULL;
  }

  /* gets field number */
  num = PQfnumber(self->last_result, name);
  if (num == -1) {
    err_setstr(ValueError, "Unknown field.");
    return NULL;
  }

  return newintobject(num);
};

/* retrieves last result */
static object *
pg95_getresult(self, args)
pg95object *self;
object *args;
{
  object *rowtuple, *reslist, *str;
  int i, j, m, n;

  /* checks validity */
  if (!check_up(self, CHECK_OBJECT | CHECK_RESULT))
    return NULL;

  /* checks args (args == NULL for an internal call)*/
  if ((args != NULL) && (!newgetargs(args, ""))) {
    err_setstr(SyntaxError, "method getresult() takes no parameters.");
    return NULL;
  } 

  /* stores result in tuple */
  reslist = newlistobject(0);
  m = PQntuples(self->last_result);
  n = PQnfields(self->last_result);
  for (i=0; i<m; i++) {
    rowtuple = newtupleobject(n);
    for (j=0; j<n; j++) {
      char *value = PQgetvalue(self->last_result, i, j);
      str = newstringobject(value);
      settupleitem(rowtuple, j, str);
    }
    addlistitem(reslist, rowtuple);
    XDECREF(rowtuple);
  };

  /* returns list */
  return reslist;
};

/* getq asynchronous notify */
static object *
pg95_getnotify(self, args)
pg95object *self;
object *args;
{
  PGnotify *notify;
  PGresult *result;
  object *notify_result, *temp;

  /* checks object validity */
  if (!check_up(self, CHECK_OBJECT))
    return NULL;

  /* checks args */
  if (!newgetargs(args, "")) {
    err_setstr(SyntaxError, "method getnotify() takes no parameters.");
    return NULL;
  }

  /* gets notify and builds result */
  /* notifies only come back as result of a query, so I send an empty query */
  result = PQexec(self->cnx, " ");        
  notify = PQnotifies(self->cnx);
  if (notify) {
    notify_result = newtupleobject(2);
    temp = newstringobject(notify->relname);
    settupleitem(notify_result, 0, temp);
    temp = newintobject(notify->be_pid);
    settupleitem(notify_result, 1, temp);
    free(notify);
  }
  else {
    INCREF(None);
    notify_result = None;
  }
  PQclear(result);

  /* returns result */
  return notify_result;
}

/* database query */
static object *
pg95_query(self, args)
pg95object *self;
object *args;
{
  char *query;
  PGresult *result;
  object *rowtuple, *reslist, *str;
  int i, j, m, n, status;

  /* get query args */
  if (!newgetargs(args, "s", &query)) {
    err_setstr(TypeError, "query(sql), with sql (string).");
    return NULL;
  }

  /* checks object validity */
  if (!check_up(self, CHECK_OBJECT))
    return NULL;

  /* gets result */
  result = PQexec(self->cnx, query);
  if (self->last_result) {
    PQclear(self->last_result);
    self->last_result=NULL;
  };

  /* checks result validity */
  if (!result) {
    err_setstr(ValueError, PQerrorMessage(self->cnx));
    return NULL;
  }

  /* checks result status */
  status = PQresultStatus(result);
  if (status != PGRES_TUPLES_OK) {
    PQclear(result);
    switch (status) {
    case PGRES_EMPTY_QUERY:
      err_setstr(ValueError, "empty query.");
      break;
    case PGRES_BAD_RESPONSE:
      err_setstr(PG95Error, "unexpected responsed received from server.");
      break;
    case PGRES_FATAL_ERROR:
      err_setstr(PG95Error, "server fatal error. Please report to your "
		 "db administrator.");
      break;
    case PGRES_NONFATAL_ERROR:
      err_setstr(PG95Error, "server (non fatal) error.");
      break;
    case PGRES_COMMAND_OK:                  /* no data will be received */
    case PGRES_COPY_OUT:
    case PGRES_COPY_IN:
      INCREF(None);
      return None;
    default:
      err_setstr(PG95Error, "internal error : unknown result status.");
      break;
    }
    return NULL;                            /* error detected on query */
  };

  /* stores result and returns list */
  self->last_result=result;
  return (pg95_getresult(self, NULL));
}

/* direct acces function : putline */
static object *
pg95_putline(self, args)
pg95object *self;
object *args;
{
  char *line;

  /* checks object validity */
  if (!check_up(self, CHECK_OBJECT))
    return NULL;

  /* reads args */
  if (!newgetargs(args, "s", &line)) {
    err_setstr(TypeError, "putline(line), with line (string).");
    return NULL;
  }

  /* sends line to backend */
  PQputline(self->cnx, line);
  INCREF(None);
  return None;
}

/* direct access function : getline */
static object *
pg95_getline(self, args)
pg95object *self;
object *args;
{
  char *line;
  object *str;
  int ret;

  /* checks object validity */
  if (!check_up(self, CHECK_OBJECT))
    return NULL;

  /* checks args */
  if (!newgetargs(args, "")) {
    err_setstr(SyntaxError, "method getline() takes no parameters.");
    return NULL;
  }

  /* allocate buffer */
  line = (char *)malloc(MAX_BUFFER_SIZE);
  if (!line) {
    err_setstr(MemoryError, "can't allocate getline buffer");
    return NULL;
  }

  /* gets line */
  switch (PQgetline(self->cnx, line, MAX_BUFFER_SIZE)) {
  case 0:
    str = newstringobject(line);
    break;
  case 1:
    err_setstr(MemoryError, "buffer overflow");
    str = NULL;
    break;
  case EOF:
    INCREF(None);
    str = None;
    break;
  }
  free(line);
  return str;
} 

/* direct access function : end copy */
static object *
pg95_endcopy(self, args)
pg95object *self;
object *args;
{
  /* checks object validity */
  if (!check_up(self, CHECK_OBJECT))
    return NULL;

  /* checks args */
  if (!newgetargs(args, "")) {
    err_setstr(SyntaxError, "method endcopy() takes no parameters.");
    return NULL;
  }

  /* ends direct copy */
  PQendcopy(self->cnx);
  INCREF(None);
  return None;
}

/* insert table */
static object *
pg95_inserttable(self, args)
pg95object *self;
object *args;
{
  PGresult *result;
  char *table, *buffer, *temp;
  char temp_buffer[256];
  object *list, *sublist, *item;
  object *(*getitem)(object *, int);
  object *(*getsubitem)(object *, int);
  int i, j;

  /* checks object validity */
  if (!check_up(self, CHECK_OBJECT))
    return NULL;

  /* gets arguments */
  if (!newgetargs(args, "sO:filter", &table, &list)) {
    err_setstr(TypeError, "tableinsert(table, content), with table (string) "
	       "and content (list).");
    return NULL;
  }

  /* checks list type */
  if (is_tupleobject(list)) getitem=gettupleitem;
  else if (is_listobject(list)) getitem=getlistitem;
  else {
    err_setstr(TypeError, "second arg must be some kind of array.");
    return NULL;
  }

  /* checks sublists type */
  for(i=0; sublist = getitem(list, i); i++)
    if (!is_tupleobject(sublist) && !is_listobject(sublist)) {
      err_setstr(TypeError, "second arg must contain some kind of arrays.");
      return NULL;
    }

  /* allocate buffer */
  if (!(buffer = (char *)malloc(MAX_BUFFER_SIZE))) {
    err_setstr(MemoryError, "can't allocate insert buffer.");
    return NULL;
  }

  /* starts query */
  sprintf(buffer, "copy %s from stdin", table);
  result = PQexec(self->cnx, buffer);
  if (!result){
    free(buffer);
    err_setstr(ValueError, PQerrorMessage(self->cnx));
    return NULL;
  }
  PQclear(result);

  /* feeds table */
  for (i=0; sublist = getitem(list, i); i++) {
    if (is_tupleobject(sublist)) getsubitem=gettupleitem;
    else getsubitem=getlistitem;

    /* builds insert line */
    buffer[0]=0;
    for (j=0; item = getsubitem(sublist, j); j++) {
      /* converts item to string */
      if (is_stringobject(item))
	newgetargs(item, "s", &temp);
      else if (is_intobject(item)) {
	int k;
	newgetargs(item, "i", &k);
	sprintf(temp_buffer, "%d", k);
	temp=temp_buffer;
      }
      else if (is_longobject(item)) {
	long k;
	newgetargs(item, "l", &k);
	sprintf(temp_buffer, "%ld", k);
	temp=temp_buffer;
      }
      else if (is_floatobject(item)) {
	double k;
	newgetargs(item, "d", &k);
	sprintf(temp_buffer, "%g", k);
	temp=temp_buffer;
      }
      else {
	free(buffer);
	err_setstr(ValueError, "items must be strings, integers, "
		   "longs or double (real).");
	return NULL;
      }

      /* concats buffer */
      if (strlen(buffer))
	strncat(buffer, "\t", MAX_BUFFER_SIZE-strlen(buffer));
      strncat(buffer, temp, MAX_BUFFER_SIZE-strlen(buffer));
    }
    strncat(buffer, "\n", MAX_BUFFER_SIZE-strlen(buffer));

    /* sends data */
    PQputline(self->cnx, buffer);
  }

  /* ends query */
  PQputline(self->cnx, ".\n");
  PQendcopy(self->cnx);
  free(buffer);

  /* no error : returns nothing */
  INCREF(None);
  return None;
}

/* connection object methods */
static struct methodlist pg95obj_methods[] = {
  {"query", (method)pg95_query, 1},             /* query method */
  {"reset", (method)pg95_reset, 1},             /* connection reset */
  {"getresult", (method)pg95_getresult, 1},     /* get last result */
  {"fieldname", (method)pg95_fieldname, 1},     /* get field name */
  {"fieldnum", (method)pg95_fieldnum, 1},       /* get field number */
  {"listfields", (method)pg95_listfields, 1},   /* list fields names */
  {"getnotify", (method)pg95_getnotify, 1},     /* checks for a notify */
  {"inserttable", (method)pg95_inserttable, 1}, /* table insert */
#ifdef DIRECT_ACCESS
  {"putline", (method)pg95_putline, 1},         /* direct access : putline */
  {"getline", (method)pg95_getline, 1},         /* direct access : getline */
  {"endcopy", (method)pg95_endcopy, 1},         /* direct access : endcopy */
#endif /* DIRECT_ACCESS */
#ifdef LARGE_OBJECTS
  {"locreate", (method)pg95_locreate, 1},       /* creates large object */
  {"getlo", (method)pg95_getlo, 1},             /* get large object from oid */
  {"loimport", (method)pg95_loimport, 1},       /* imports lo from unix file */
#endif /* LARGE_OBJECTS */
  {NULL, NULL}		                     /* sentinel */
};

/* get attribute */
static object *
pg95_getattr(self, name)
pg95object *self;
char *name;
{
  /* list postgres95 connection fields */
  
  /* postmaster host */
  if (!strcmp(name, "host"))
    if (check_up(self, CHECK_OBJECT)) {
      char *host=PQhost(self->cnx);
      return newstringobject(host);
    }
    else {
      INCREF(None);
      return None;
    }

  /* postmaster port */
  if (!strcmp(name, "port"))
    if (check_up(self, CHECK_OBJECT)) {
      char *port=PQport(self->cnx);
      return newintobject(atol(port));
    }
    else {
      INCREF(None);
      return None;
    }

  /* selected database */
  if (!strcmp(name, "db"))
    if (check_up(self, CHECK_OBJECT)) {
      char *db=PQdb(self->cnx);
      return newstringobject(db);
    }
    else {
      INCREF(None);
      return None;
    }

  /* selected options */
  if (!strcmp(name, "options"))
    if (check_up(self, CHECK_OBJECT)) {
      char *options=PQoptions(self->cnx);
      return newstringobject(options);
    }
    else {
      INCREF(None);
      return None;
    }

  /* selected postgres tty */
  if (!strcmp(name, "tty"))
    if (check_up(self, CHECK_OBJECT)) {
      char *tty=PQtty(self->cnx);
      return newstringobject(tty);
    }
    else {
      INCREF(None);
      return None;
    }

  /* error (status) message */
  if (!strcmp(name, "error"))
    if (check_up(self, CHECK_OBJECT)) {
      char *error=PQerrorMessage(self->cnx);
      return newstringobject(error);
    }
    else {
      INCREF(None);
      return None;
    }

  /* connection status : 1 - OK, 0 - BAD */
  if (!strcmp(name, "status"))
    if (check_up(self, CHECK_OBJECT)) {
      if (PQstatus(self->cnx) == CONNECTION_OK)
	return newintobject(1);
      return newintobject(0);
    }
    else {
      INCREF(None);
      return None;
    }

  /* provided user name */
  if (!strcmp(name, "user"))
    if (check_up(self, CHECK_OBJECT)) {
      char *user=fe_getauthname("<unknown user>");
      return newstringobject(user);
    }
    else {
      INCREF(None);
      return None;
    }

  /* attributes list */
  if (!strcmp(name, "__members__")) {
    object *list = newlistobject(8);
    if (list) {
      setlistitem(list, 0, newstringobject("host"));
      setlistitem(list, 1, newstringobject("port"));
      setlistitem(list, 2, newstringobject("db"));
      setlistitem(list, 3, newstringobject("options"));
      setlistitem(list, 4, newstringobject("tty"));
      setlistitem(list, 5, newstringobject("error"));
      setlistitem(list, 6, newstringobject("status"));
      setlistitem(list, 7, newstringobject("user"));
    }
    return list;
  }
  return findmethod(pg95obj_methods, (object *)self, name);
}

/* object type definition */
staticforward typeobject Pg95type = {
  OB_HEAD_INIT(&Typetype)
  0,			            /*ob_size*/
  "pg95object",			    /*tp_name*/
  sizeof(pg95object),	            /*tp_basicsize*/
  0,			            /*tp_itemsize*/
  /* methods */
  (destructor)pg95_dealloc,         /*tp_dealloc*/
  0,			            /*tp_print*/
  (getattrfunc)pg95_getattr,        /*tp_getattr*/
  0,                                /*tp_setattr*/
  0,			            /*tp_compare*/
  0,			            /*tp_repr*/
  0,			            /*tp_as_number*/
  0,			            /*tp_as_sequence*/
  0,			            /*tp_as_mapping*/
  0,			            /*tp_hash*/
};

/* --------------------------------------------------------------------- */

/* MODULE FUNCTIONS */

#ifdef DEFAULT_VARS

/* gets default host */
object *
pg95getdefhost(self, args) 
object *self, *args;
{
  /* checks args */
  if (!newgetargs(args, "")) {
    err_setstr(SyntaxError, "method get_defhost() takes no parameter.");
    return NULL;
  }

  XINCREF(pg95_default_host);
  return pg95_default_host;
}

/* sets default host */
object *
pg95setdefhost(self, args) 
object *self, *args;
{
  char *temp = NULL;
  object *old;

  /* gets arguments */
  if (!newgetargs(args, "z", &temp)) {
    err_setstr(TypeError, "set_defhost(name), with name (string/None).");
    return NULL;
  }

  /* adjusts value */
  old = pg95_default_host;
  if (temp) pg95_default_host = newstringobject(temp);
  else {
    INCREF(None);
    pg95_default_host = None;
  }

  return old;
}

/* gets default base */
object *
pg95getdefbase(self, args) 
object *self, *args;
{
  /* checks args */
  if (!newgetargs(args, "")) {
    err_setstr(SyntaxError, "method get_defbase() takes no parameter.");
    return NULL;
  }

  XINCREF(pg95_default_base);
  return pg95_default_base;
}

/* sets default base */
object *
pg95setdefbase(self, args) 
object *self, *args;
{
  char *temp = NULL;
  object *old;

  /* gets arguments */
  if (!newgetargs(args, "z", &temp)) {
    err_setstr(TypeError, "set_defbase(name), with name (string/None).");
    return NULL;
  }

  /* adjusts value */
  old = pg95_default_base;
  if (temp) pg95_default_base = newstringobject(temp);
  else {
    INCREF(None);
    pg95_default_base = None;
  }

  return old;
}

/* gets default options */
object *
pg95getdefopt(self, args) 
object *self, *args;
{
  /* checks args */
  if (!newgetargs(args, "")) {
    err_setstr(SyntaxError, "method get_defopt() takes no parameter.");
    return NULL;
  }

  XINCREF(pg95_default_opt);
  return pg95_default_opt;
}

/* sets default opt */
object *
pg95setdefopt(self, args) 
object *self, *args;
{
  char *temp = NULL;
  object *old;

  /* gets arguments */
  if (!newgetargs(args, "z", &temp)) {
    err_setstr(TypeError, "set_defopt(name), with name (string/None).");
    return NULL;
  }

  /* adjusts value */
  old = pg95_default_opt;
  if (temp) pg95_default_opt = newstringobject(temp);
  else {
    INCREF(None);
    pg95_default_opt = None;
  }

  return old;
}

/* gets default tty */
object *
pg95getdeftty(self, args) 
object *self, *args;
{
  /* checks args */
  if (!newgetargs(args, "")) {
    err_setstr(SyntaxError, "method get_deftty() takes no parameter.");
    return NULL;
  }

  XINCREF(pg95_default_tty);
  return pg95_default_tty;
}

/* sets default tty */
object *
pg95setdeftty(self, args) 
object *self, *args;
{
  char *temp=NULL;
  object *old;

  /* gets arguments */
  if (!newgetargs(args, "z", &temp)) {
    err_setstr(TypeError, "set_deftty(name), with name (string/None).");
    return NULL;
  }

  /* adjusts value */
  old = pg95_default_tty;
  if (temp) pg95_default_tty = newstringobject(temp);
  else {
    INCREF(None);
    pg95_default_tty = None;
  }

  return old;
}

/* gets default port */
object *
pg95getdefport(self, args) 
object *self, *args;
{
  char *temp;

  /* checks args */
  if (!newgetargs(args, "")) {
    err_setstr(SyntaxError, "method get_defport() takes no parameter.");
    return NULL;
  }

  XINCREF(pg95_default_port);
  return pg95_default_port;
}

/* sets default port */
object *
pg95setdefport(self, args) 
object *self, *args;
{
  long int port=-2;
  char buffer[64], *temp;
  object *old;

  /* gets arguments */
  if ((!newgetargs(args, "l", &port)) || (port <-1)) {
    err_setstr(TypeError, "set_defport(port), with port "
	       "(positive integer/-1).");
    return NULL;
  }

  /* adjusts value */
  old = pg95_default_port;
  if (port!=-1) pg95_default_port = newlongobject(port);
  else {
    INCREF(None);
    pg95_default_port = None;
  }

  return old;
}

#endif /* DEFAULT_VARS */

/* List of functions defined in the module */

static struct methodlist pg95_methods[] = {
  {"connect", pg95connect, 3},        /* connect to a postgres database */
#ifdef DEFAULT_VARS
  {"get_defhost", pg95getdefhost, 1}, /* gets default host */
  {"set_defhost", pg95setdefhost, 1}, /* sets default host */
  {"get_defbase", pg95getdefbase, 1}, /* gets default base */
  {"set_defbase", pg95setdefbase, 1}, /* sets default base */
  {"get_defopt", pg95getdefopt, 1},   /* gets default options */
  {"set_defopt", pg95setdefopt, 1},   /* sets default options */
  {"get_deftty", pg95getdeftty, 1},   /* gets default debug tty */
  {"set_deftty", pg95setdeftty, 1},   /* sets default debug tty */
  {"get_defport", pg95getdefport, 1}, /* gets default port */
  {"set_defport", pg95setdefport, 1}, /* sets default port */
#endif /* DEFAULT_VARS */
  {NULL, NULL}		              /* sentinel */
};

/* Initialization function for the module */
void
initpg95()
{
  object *m, *d, *t;

  /* Create the module and add the functions */
  m = initmodule("pg95", pg95_methods);

  /* Add some symbolic constants to the module */
  d = getmoduledict(m);
  PG95Error = newstringobject("pg95.error");
  dictinsert(d, "error", PG95Error);

  /* create mode for large objects */
  t = newintobject(INV_READ);
  dictinsert(d, "INV_READ", t);
  t = newintobject(INV_WRITE);
  dictinsert(d, "INV_WRITE", t);
  t = newintobject(INV_ARCHIVE);
  dictinsert(d, "INV_ARCHIVE", t);

  /* position flags for lo_lseek */
  t = newintobject(SEEK_SET);
  dictinsert(d, "SEEK_SET", t);
  t = newintobject(SEEK_CUR);
  dictinsert(d, "SEEK_CUR", t);
  t = newintobject(SEEK_END);
  dictinsert(d, "SEEK_END", t);

#ifdef DEFAULT_VARS
  /* prepares default values */
  INCREF(None);
  pg95_default_host = None;
  INCREF(None);
  pg95_default_base = None;
  INCREF(None);
  pg95_default_opt = None;
  INCREF(None);
  pg95_default_port = None;
  INCREF(None);
  pg95_default_tty = None;
#endif /* DEFAULT_VARS */

  /* Check for errors */
  if (err_occurred()) fatal("can't initialize module pg95");
};
