
/************************************************************************
*
* These routines implement a sorted list of tasks. They are designed
* to be called from C or Fortran.
*
* Memory operations:
* TL_INIT( M ) ==> dynamically allocates M pointers (approx. 4*M bytes)
* TL_CREATE( K ) ==> dynamically allocates approx. 32*K bytes
* TL_DESTROY( TLID ) ==> frees the space allocated by a corresponding
*                     TL_CREATE call
* TL_FREE() ==> frees the space allocated by a corresponding TL_INIT call.
*
************************************************************************/

#include "array.h"
#include "decide.h"
#include "task.h"

/* Globals */
static Array_t **g_TaskList = NULL;
static int g_NumTaskLists = 0;

#ifdef NoChange
#define tl_init_ tl_init
#define tl_create_ tl_create
#define tl_add_ tl_add
#define tl_get_ tl_get
#define tl_get2_ tl_get2
#define tl_size_ tl_size
#define tl_free_ tl_free
#define tl_destroy_ tl_destroy
#define doswitch_ doswitch
#endif 

#ifdef NDEBUG
#define assert(_expr) ((void)0)
#else
extern void myassert();
#define assert(_expr) \
   (void)((_expr) || (myassert(#_expr, __FILE__, __LINE__),0))
#endif


void PrintTask(Task *t) 
{
	if (t) {
		fprintf(stderr, "Task {\n");
		fprintf(stderr, "\t key = %f\n", t->key);
		fprintf(stderr, "\t ilo = %d\n", t->ilow);
		fprintf(stderr, "\t ihigh = %d\n", t->ihigh);
		fprintf(stderr, "\t param1 = %f\n", t->param1);
		fprintf(stderr, "\t param2 = %f\n", t->param2);
		fprintf(stderr, "}\n");
	}
}

void PrintTaskList(Array_t *a)
{
	int i, n = ArraySize(a);
	Task t;
	fprintf(stderr, ">>>>>> Printing task list of %d elements:\n", n);
	for (i=1; i<=n; i++) {
		fprintf(stderr, "Task # %d:\n", i);
		ArrayGet(a, i, &t);
		PrintTask(&t);
	}
	fprintf(stderr, "<<<<<<< Done.\n");
}

int tl_init_(maxLists)
int *maxLists;
/* Creates the list of lists, with space for 'maxLists' lists. */
{
	int i;
	if (g_TaskList || *maxLists <= 0) {
		/* Error */
		return 0;
	}
	g_NumTaskLists = *maxLists;
	g_TaskList = (Array_t **) malloc(g_NumTaskLists * sizeof(Array_t *));
	if (!g_TaskList)
		return 0;
	for (i = 0; i < g_NumTaskLists; i++)
		g_TaskList[i] = NULL;
	return 1;
}
		
int tl_create_(size)
int *size;
/* Create a task list with space for up to 'size' tasks. Returns a handle. */
{
	int i;

	assert(g_TaskList);
	assert(*size > 0);

	if (!g_TaskList || *size <= 0) {
		return -1;
	}

	/* Find an unused Task List */
	for (i = 0; i < g_NumTaskLists; i++) {
		if (g_TaskList[i] == NULL)
			break;
	} 
	if (i == g_NumTaskLists) { 
		/* Too many task lists! */
		return -1;
	}
	g_TaskList[i] = ArrayCreate(sizeof(Task), *size);
	return i;
}

int tl_add_(id, ilow, ihigh, param1, param2)
int *id;
int *ilow, *ihigh;
double *param1, *param2;
/* Adds a task to the specified Task List */
{
	Task newTask;
 	int retVal;

	assert(g_TaskList);
	assert(*id >= 0 && *id < g_NumTaskLists);
	assert(g_TaskList[*id]);
	if (!g_TaskList || *id < 0 || *id > g_NumTaskLists 
              || !g_TaskList[*id]) {
		/* Parameter error */
		return 0;
	}
	newTask.key = (double) *ihigh - *ilow + 1;
	newTask.ilow = *ilow; newTask.ihigh = *ihigh;
	newTask.param1 = *param1; newTask.param2 = *param2;
#ifdef TASK_DEBUG
	fprintf(stderr, "Adding task:\n");
	PrintTask(&newTask);
#endif
	retVal = ArrayInsert(g_TaskList[*id], &newTask);
#ifdef TASK_DEBUG2
	PrintTaskList(g_TaskList[*id]);
#endif
      	return retVal;
}

int tl_size_(id)
int *id;
/* Returns the number of tasks in the specified task list */
{
        assert(g_TaskList);
        assert(*id >= 0 && *id < g_NumTaskLists);
        assert(g_TaskList[*id]);
        if (!g_TaskList || *id < 0 || *id > g_NumTaskLists 
               || !g_TaskList[*id]) {
                /* Parameter error */
                return 0;
        }
	return ArraySize(g_TaskList[*id]);
}


int doswitch_(id, nprocs)
int *id;
int *nprocs;
/* Returns 1 if it is cheaper to operate in with the selected
   list in task parallel mode; otherwise 0. */
{
	return DecideSwitch(*nprocs, ArraySize(g_TaskList[*id]),
		(Task *) g_TaskList[*id]->array);
}

int tl_get_(id, ilow, ihigh, param1, param2)
int *id;
int *ilow, *ihigh;
double *param1, *param2;
/* Gets the next largest task from the specified task list */
{
	Task nextTask;

        assert(g_TaskList);
        assert(*id >= 0 && *id < g_NumTaskLists);
        assert(g_TaskList[*id]);
        if (!g_TaskList || *id < 0 || *id > g_NumTaskLists 
              || !g_TaskList[*id]) {
                /* Parameter error */
                return 0;
        }

	if (!ArrayDelete(g_TaskList[*id], &nextTask))
		return 0;

#ifdef TASK_DEBUG
	fprintf(stderr, "Getting task: (tl_get)\n");
	PrintTask(&nextTask);
#endif
	*ilow = nextTask.ilow; *ihigh = nextTask.ihigh;
	*param1 = nextTask.param1;  *param2 = nextTask.param2;
	return 1;
}


int tl_get2_(id, k, ilow, ihigh, param1, param2)
int *id;
int *k;
int *ilow, *ihigh;
double *param1, *param2;
/* Gets the k'th task from the Task List */
{
	Task nextTask;

        assert(g_TaskList);
        assert(*id >= 0 && *id < g_NumTaskLists);
        assert(g_TaskList[*id]);
	assert(*k > 0);
        if (!g_TaskList || *id < 0 || *id > g_NumTaskLists 
             || !g_TaskList[*id] || *k <= 0) {
                /* Parameter error */
                return 0;
        }

	if (!ArrayGet(g_TaskList[*id], *k, &nextTask))
		return 0;

#ifdef TASK_DEBUG
	fprintf(stderr, "Getting task (tl_get2):\n");
	PrintTask(&nextTask);
#endif
	*ilow = nextTask.ilow; *ihigh = nextTask.ihigh;
	*param1 = nextTask.param1;  *param2 = nextTask.param2;
	return 1;
}

int tl_destroy_(id)
int *id;
/* Destroys the specified task list */
{
        assert(g_TaskList);
        assert(*id >= 0 && *id < g_NumTaskLists);
        assert(g_TaskList[*id]);
        if (!g_TaskList || *id < 0 || *id > g_NumTaskLists
             || !g_TaskList[*id]) {
                /* Parameter error */
                return 0;
        }

	ArrayDestroy(g_TaskList[*id]);
	g_TaskList[*id] = NULL;
	return 1;
}

int tl_free_()
/* Destroys all task lists and frees the list of lists. */
{
	int i;
	assert(g_TaskList);
	for (i = 0; i < g_NumTaskLists; i++) {
		assert(g_TaskList[i] == NULL);
		if (g_TaskList[i])
			tl_destroy_(i);
	}
	free(g_TaskList);
	g_TaskList = NULL;
    	return 1;
}
