        SUBROUTINE PDHALFP( WANTT, WANTZ, N, A, IA, JA, ILO,
     $                      N2, DESCA, Z, IZ, JZ, DESCZ, X,
     $                      RHP, MAXITER, SCALING, K, ERROR, NITER,
     $                      WORK, LWORK, IWORK, LIWORK, INFO )
*     ..
*     .. Scalar Arguments ..
      INTEGER            N, IA, JA, ILO, N2, IZ, JZ, MAXITER, K,
     $                   LWORK, LIWORK, NITER, INFO
      CHARACTER          SCALING
      DOUBLE PRECISION   X, ERROR
      LOGICAL            WANTT, WANTZ, RHP
*     ..
*     .. Array Arguments ..
      INTEGER            DESCA( * ), DESCZ( * ), IWORK( LIWORK )
      DOUBLE PRECISION   A( * ), Z( * ), WORK( LWORK )
*     ..
*
*  Purpose
*  =======
*
*  PDHALFP computes A', the block upper triangularization of the input
*  matrix A as shown below.
*
*                             k  n-k
*          A' = Q^T*A*Q =  [ A11 A12 ] k
*                          [  0  A22 ] n-k
*
*  On return A is overwritten with A'.
*
*  If RHP is .TRUE. then each eigenvalue x + i*y of
*  A11 satisfies  x > X  (i.e. is in the right halfplane at x = X),
*  whilst each eigenvalue x + i*y of A22 satisfies  x < X (i.e.
*  is in the left halfplane at x = X).
*  If RHP is .FALSE. the roles of A11 and A22 are reversed.
*
*  This routine is designed to be called repeatedly to block upper 
*  triangularize a matrix. The original/root matrix being triangularized
*  is A(IA:IA+N-1, JA:JA+N-1); on each call to this routine, the sub-block
*  A(IA+ILO-1:IA+ILO+N2-2, JA+ILO-1:JA+ILO-N2-2) is deflated to 2x2
*  block upper triangular form as described above and parts
*  of the original matrix are updated to reflect the application of
*  the sub-block's orthogonal matrix.
*  That is, if we have already deflated the root matrix twice, it may resemble:
*	A_2 = [ A11 A12 A13 ]
*             [  0  A22 A23 ]
*             [  0   0  A33 ]
*  along with the accumulation of orthogonal transformations applied so far:
*	Z_2 = [ Z11 Z12 Z13 ]
*  If the task is to block triangularize A22, then we have
*       Q^T*A22*Q = T = 2x2 block triangular form.
*  We accumulate the orthogonal transformation:
*     	Z_3 = [ Z11 Z12*Q Z13 ]
*  and update the off-diagonal blocks of the root matrix:
*      	A_3 = [ A11 A12*Q   A13   ]
*	      [  0   T    Q^T*A23 ]
*             [  0   0      A33   ]
*  so that the Schur decomposition relationship (Z_3^T * A_3 * Z_3)
*  still holds.
*
*  Notes
*  =====
*
*  Note that since the matrix may split at any point, we cannot
*  know in advance exactly how much memory will be required.
*  Thus a call with LWORK == -1 returns a reasonable (if imprecise) 
*  estimate.
*
*  A description vector is associated with each 2D block-cyclicly dis-
*  tributed matrix.  This vector stores the information required to
*  establish the mapping between a matrix entry and its corresponding
*  process and memory location.
*
*  In the following comments, the character _ should be read as
*  "of the distributed matrix".  Let A be a generic term for any 2D
*  block cyclicly distributed matrix.  Its description vector is DESCA:
*
*  NOTATION        STORED IN      EXPLANATION
*  --------------- -------------- --------------------------------------
*  DT_A   (global) DESCA( DT_ )   The descriptor type.
*  CTXT_A (global) DESCA( CTXT_ ) The BLACS context handle, indicating
*                                 the BLACS process grid A is distribu-
*                                 ted over. The context itself is glo-
*                                 bal, but the handle (the integer
*                                 value) may vary.
*  M_A    (global) DESCA( M_ )    The number of rows in the distributed
*                                 matrix A.
*  N_A    (global) DESCA( N_ )    The number of columns in the distri-
*                                 buted matrix A.
*  MB_A   (global) DESCA( MB_ )   The blocking factor used to distribute
*                                 the rest M_A-IMB_A rows of A.
*  NB_A   (global) DESCA( NB_ )   The blocking factor used to distribute
*                                 the rest N_A-INB_A columns of A.
*  RSRC_A (global) DESCA( RSRC_ ) The process row over which the first
*                                 row of the matrix A is distributed.
*  CSRC_A (global) DESCA( CSRC_ ) The process column over which the
*                                 first column of A is distributed.
*  LLD_A  (local)  DESCA( LLD_ )  The leading dimension of the local
*                                 array storing the local blocks of the
*                                 distributed matrix A.
*                                 LLD_A >= MAX(1,LOCp(M_A)).
*
*  Let K be the number of rows or columns of a distributed matrix,
*  and assume that its process grid has dimension p x q.
*  LOCp( K ) denotes the number of elements of K that a process
*  would receive if K were distributed over the p processes of its
*  process column.
*  Similarly, LOCq( K ) denotes the number of elements of K that a
*  process would receive if K were distributed over the q processes of
*  its process row.
*  The values of LOCp() and LOCq() may be determined via a call to the
*  ScaLAPACK tool function, NUMROC:
*          LOCp( M ) = NUMROC( M, MB_A, MYROW, RSRC_A, NPROW ),
*          LOCq( N ) = NUMROC( N, NB_A, MYCOL, CSRC_A, NPCOL ).
*
*
*  Arguments
*  =========
*
*  WANTT   (global input) LOGICAL
*          = .TRUE. : the full Schur form T is required;
*          = .FALSE.: only eigenvalues are required.
*
*  WANTZ   (global input) LOGICAL
*          = .TRUE. : the matrix of Schur vectors Z is required;
*          = .FALSE.: Schur vectors are not required.
*
*  N       (global input) INTEGER
*          The number of rows and columns of the matrix A.
*
*  A       (local input/output) DOUBLE PRECISION
*          An array of dimension (LLD_A, LOCq(JA+DESCA(N_)-1)) containing the
*          local pieces of the distributed matrix A.
*          On entry, A(IA:IA+N-1, JA:JA+N-1) is the original/root matrix 
*          whose spectrum is to be split, and
*          A(IA+ILO-1:IA+ILO+N-2, JA+ILO-1:JA+ILO+N-2) contains sub( A ), 
*          the current diagonal sub-block to be deflated.
*          On exit, sub( A ) contains the deflated matrix, where the leading
*          principal submatrix A(IA+ILO-1:IA+ILO+K-2, JA+ILO-1:JA+ILO+K-2)
*          contains all K eigenvalues of the original matrix sub( A )
*          in the open half plane (right or left - as indicated by RHP)
*          at x = X. A(IA+ILO+K-1:IA+N, JA+ILO+K-1:JA+N) contains the
*          remaining eigenvalues.
*
*  IA      (global input) INTEGER
*          A's global column index. Points to the beginning of
*          the root submatrix.
*          That is, if the user calls PDHALFP repeatedly on sub-blocks to
*          upper triangularize the matrix, then IA must be the starting 
*          row index of the original/root matrix.
*
*  JA      (global input) INTEGER
*          A's global column index. Points to the beginning of
*          the root submatrix.
*          That is, if the user calls PDHALFP repeatedly on sub-blocks to
*          upper triangularize the matrix, then JA must be the starting 
*          column index of the original/root matrix.
*
*  ILO     (global input) INTEGER
*          Sub-index of A. IA+ILO-1 points to the beginning of the N2 x N2
*          sub matrix of A to be deflated.
*
*  N2      (global input) INTEGER
*          The size of the sub matrix to be deflated.
*
*  DESCA   (global and local input) INTEGER array of dimension DLEN_.
*          The array descriptor for the distributed matrix A.
*
*  Z       (local input/output) DOUBLE PRECISION 
*          An array of dimension (LLD_Z, LOCq(JA+DESCZ(N_)-1)) 
*          If WANTZ is .FALSE. Z is not referenced.
*          If WANTZ is .TRUE., on entrance Z(IZ:IZ+N,JZ:JZ+N) must be 
*          either 
*          (1) the accumulated orthogonal transformations from previous 
*              calls to PDHALFP; or 
*          (2) the identity.
*          On exit Z will have been multiplied by the orthogonal matrix
*          from the deflation process.
*
*  IZ      (global input) INTEGER
*          Z's global row index. 
*
*  JZ      (global input) INTEGER
*          Z's global column index. 
*
*  DESCZ   (global and local input) INTEGER array of dimension DLEN_.
*          The array descriptor for the distributed matrix Z.
*
*  X       (global input) DOUBLE PRECISION
*          The point on the real axis at which to split the spectrum.
*
*  RHP     (global input) LOGICAL
*          Indicates which halfplane to map to the upper block of the
*          matrix after deflation. 
*          = .TRUE. : Put the eigenvalues in the right halfplane at X in the
*                     upper block of the matrix. On exit sub(A)(1:K, 1:K)
*                     contains the eigenvalues in the RHP, 
*                     sub(A)(K+1:N, K+1:N) contains those in the LHP.
*          = .FALSE.: Put the eigenvalues in the left halfplane at X in the
*                     upper block of the matrix. On exit sub(A)(1:K, 1:K)
*                     contains the eigenvalues in the LHP, 
*                     sub(A)(K+1:N, K+1:N) contains those in the RHP.
*
*  MAXITER (global input) INTEGER
*          The maximam number of iterations to take when computing 
*          the sign function.
*
*  SCALING (global input) CHARACTER
*          Indicates the scaling scheme to use when computing the sign function.
*          'N' ==> No scaling
*          'R' ==> Robert's scaling
*
*  K       (global output) INTEGER
*          On exit, K is the number of eigenvalues of sub(A) in the
*          selected half-plane at x = X (see variable RHP).
*
*  ERROR   (global output) DOUBLE PRECISION
*          A posteriori measure of error in the decomposition process.
*          See PDSGNDFL for more details.
*
*  ITER    (global output) INTEGER
*          On exit, ITER is the number of iterations taken when computing
*          the sign function.
*
*  WORK    (local workspace) DOUBLE PRECISION array, dimension (LWORK)
*          On exit, if INFO = 0, WORK(1) returns the optimal and
*          minimal LWORK.
*
*  LWORK   (local input) INTEGER
*          The dimension of the array WORK.  LWORK >= 
*
*  IWORK   (local workspace) INTEGER array, dimension
*
*  INFO    (global output) INTEGER
*          = 0: successful exit
*          < 0: if INFO = -i, the i-th argument had an illegal value.
*          = 1: Error in computing the sign function.
*          = 2: Error in the splitting process
*
*  =====================================================================
*
*  Further Details
*  ===============
*
*     If      f(x + i*y) = (x - INTERCEPT) + i*y
*
*     then the function Real(f(z)) maps the halfplane at INTERCEPT
*     to the halfplane at 0.
*     Thus given a matrix A, we compute f(A) = (A - INTERCEPT*I)
*     compute sign(f(A)), and decompose A to produce A' above.
*     A11 will have eigenvalues of f(A) in the right/left halfplane at 0,
*     which are the eigenvalues of A in the right/left halfplane at
*     INTERCEPT.
*
*  =====================================================================
*
*  Working Note:
*  ============
*
*  This routine is still in the prototype stage and much of the debugging
*  code has been left in. We have tried to isolate these blocks of code
*  and identify them wherever possible.
*
*  =====================================================================
*
*     .. Parameters ..
      INTEGER            CSRC_, CTXT_, DLEN_, DT_,
     $                   LLD_, MB_, M_, NB_, N_, RSRC_
      PARAMETER          ( DLEN_ = 9, DT_ = 1,
     $                     CTXT_ = 2, M_ = 3, N_ = 4, MB_ = 5, NB_ = 6,
     $                     RSRC_ = 7, CSRC_ = 8, LLD_ = 9 )
      DOUBLE PRECISION   ZERO
      PARAMETER          ( ZERO = 0.0D+0 )
*     ..
*     .. Local Scalars ..
      INTEGER            ICTXT, NPROW, NPCOL, MYROW, MYCOL, LWMIN,
     $                   MP, NQ, IPB, IPW, IAROW, IACOL, INFO2,
     $                   NB, MB, IROFF, ICOFF, IA2, JA2, IB, JB
*     ..
*     .. Local arrays ..
      INTEGER            DESCB( DLEN_ )
*     ..
*     .. External Subroutines ..
      EXTERNAL           PDGESGN, PDSGNDFL, PDAJDIAG, PDLACPY,
     $                   BLACS_GRIDINFO, PXERBLA,
     $                   DESCSET, PDSDCUPD, print_desc
*     ..
*     .. External Functions ..
      INTEGER            NUMROC, INDXG2P
      DOUBLE PRECISION   PDLANGE
      LOGICAL            LSAME
      EXTERNAL           NUMROC, LSAME, INDXG2P, PDLANGE
*     ..
*     .. Intrinsic Functions ..
      INTRINSIC          DBLE, MAX, MIN, MOD
***
***   Debugging variables/functions/subroutines
***
      logical            debug, printit
      double precision   dummy

*     ..
*     .. Executable Statements ..
*
*     Get grid details
*
      ICTXT = DESCA( CTXT_ )
      MB = DESCA( MB_ )
      NB = DESCA( NB_ )
      CALL BLACS_GRIDINFO( ICTXT, NPROW, NPCOL, MYROW, MYCOL )
***
***   Debugging - intialization
***
      debug = .false.
      printit = (debug .and. myrow.eq.0 .and. mycol.eq.0)
*
*     Test input
*
      INFO = 0
      IF( NPROW.EQ.-1 ) THEN
         INFO = -(900+CTXT_)
      ELSE
         IF ( N .LT. 0 ) THEN
            INFO = -1
         ELSE IF ( IA .LE. 0 ) THEN
            INFO = -5
         ELSE IF ( JA .LE. 0) THEN
            INFO = -6
         ELSE IF ( ILO .LE. 0 ) THEN
            INFO = -7
         ELSE IF ( N2 .LT. 0 .OR. N2 .GT. N ) THEN
            INFO = -8
         ELSE IF ( WANTZ .AND. IZ .LE. 0 ) THEN
            INFO = -11
         ELSE IF ( WANTZ .AND. JZ .LE. 0 ) THEN
            INFO = -12
         ELSE IF ( MAXITER .LE. 0 ) THEN
            INFO = -16
         ELSE IF ( .NOT. (LSAME(SCALING, 'ROBERTS') .OR.
     $                 LSAME(SCALING, 'NONE')) ) THEN
            INFO = -17
         ELSE
*
*           Compute required workspace
*           Note that since the matrix may split at any point, we cannot
*           know in advance exactly how much memory will be required.
*
*           Copy of sub( A )
*
            IA2 = IA + ILO - 1
            JA2 = JA + ILO - 1
            IROFF = MOD( IA2 - 1, MB )
            ICOFF = MOD( JA2 - 1, NB )
            IAROW = INDXG2P( IA2, MB, MYROW, DESCA( RSRC_ ), NPROW )
            IACOL = INDXG2P( JA2, NB, MYCOL, DESCA( CSRC_ ), NPCOL )
            MP = NUMROC( N2 + IROFF, MB, MYROW, IAROW, NPROW )
            NQ = NUMROC( N2 + ICOFF, NB, MYCOL, IACOL, NPCOL )
            LWMIN = MP * NQ
*
*           Add max of calls to PDGESGN, PDSGNDFL
*
            CALL PDGESGN( N2, WORK, 1, 1, DESCA,
     $              MAXITER, SCALING, NITER, WORK,
     $              -1, IWORK, LIWORK, INFO2 )
            CALL PDSGNDFL( WANTT, WANTZ, N, N2, K, A, IA, JA,
     $              1, DESCA, WORK,
     $              1, 1, DESCA, Z, IZ, JZ, DESCZ,
     $              RHP, ERROR, WORK( 2 ), -1,
     $              IWORK, INFO2 )
            IF ( WORK( 1 ) .GT. WORK( 2 ) ) THEN
               LWMIN = LWMIN + ANINT( WORK( 1 ) )
            ELSE
               LWMIN = LWMIN + ANINT( WORK( 2 ) )
            END IF
*
            IF ( LWORK .EQ. -1 ) THEN
               if (printit) print *, 'PDHALFP: Min LWORK = ', 
     $                               LWMIN
               WORK( 1 ) = DBLE( LWMIN )
               RETURN
            ELSE IF ( LWORK .LT. LWMIN ) THEN
               if (printit) print *, 'PDHALFP: Min Work, LWORK', 
     $                               lwmin, lwork
               INFO = -22
            END IF

         END IF
      END IF

      IF ( INFO .NE. 0 ) THEN
         CALL PXERBLA( ICTXT, 'PDHALFP', -INFO )
         RETURN
      END IF

      K = 0
*
*     Quick return if possible
*
      IF ( N .EQ. 0 ) 
     $   RETURN
*
*     Copy sub(A) to a new N2 x N2 matrix and compute the sign function
*
      IA2 = IA + ILO - 1
      JA2 = JA + ILO - 1
      IROFF = MOD( IA2 - 1, MB )
      ICOFF = MOD( JA2 - 1, NB )
      IAROW = INDXG2P( IA2, MB, MYROW, DESCA( RSRC_ ), NPROW )
      IACOL = INDXG2P( JA2, NB, MYCOL, DESCA( CSRC_ ), NPCOL )
      MP = NUMROC( N2 + IROFF, MB, MYROW, IAROW, NPROW )
      NQ = NUMROC( N2 + ICOFF, NB, MYCOL, IACOL, NPCOL )
      IPB = 1
      IPW = IPB + MP * NQ
      IB = IROFF + 1
      JB = ICOFF + 1
      CALL DESCSET( DESCB, N2 + IROFF, N2 + ICOFF, MB, NB,
     $              IAROW, IACOL, ICTXT, MAX( 1, MP ) )
      CALL PDLACPY( 'FULL', N2, N2, A, IA2, JA2, DESCA, 
     $              WORK( IPB ), IB, JB, DESCB )
*
*     Shift spectrum if necessary
*
      IF ( X .NE. ZERO ) THEN
         CALL PDAJDIAG( 'ADD', N2, WORK( IPB ), IB, JB, DESCB, -X )
      END IF
*
*     Compute the sign function
*
      CALL PDGESGN( N2, WORK( IPB ), IB, JB, DESCB,
     $              MAXITER, SCALING, NITER, WORK( IPW ),
     $              LWORK - IPW + 1, IWORK, LIWORK, INFO2 )
      IF ( INFO2 .NE. 0 ) THEN
         IF (printit) print *, '<PDHALFP> Info of PDGESGN = ', 
     $                         INFO2
         INFO = 1
         RETURN
      END IF
*
*     Deflate to block 2x2, grouping according to the desired 
*     halfplane and updating the orthogonal matrix.
*
      CALL PDSGNDFL( WANTT, WANTZ, N, N2, K, A, IA, JA,
     $              ILO, DESCA, WORK( IPB ), 
     $              IB, JB, DESCB, Z, IZ, JZ, DESCZ,
     $              RHP, ERROR, WORK( IPW ), LWORK - IPW + 1,
     $              IWORK, INFO2 )

      IF ( INFO2 .NE. 0 ) THEN
         if (printit) print *, '<PDHALFP> Info of PDSGNDFL = ', 
     $                         info2
         INFO = 2
         RETURN
      END IF
*
*     End of PDHALFP
*
      RETURN
      END
