/* lstsq.c: 
C   forms least squares fit of straight line to data received so far.
C        input data in pairs (n,tn). graph is drawn of tn (ordinate)
C        against n (abscissa).  best straight line fitted described
C        by the output parameters rinf and xn12
C        k - =0 to initialize, =1 to fit line
C        xn - vector length
C        tn - time per vector operation of above length
C        rinf - inverse slope of best line using all data received
C        xni2 - negative intercept of best line on n-axis
C        per - percentage error
 */

#include <math.h>

static double si, sti, sti2, sni, sni2, stini, xb, yb;

lstsq(int k, double xn, double tn, double *rinf, double *xn12, double *per)
{
    int mode, kbpnt, nsi;
    double ti, rmse, denom, arg, xni;

    mode = 0;	/* mode=0  for least squares fit of all data */
	/* =1 least squres forced through the kbpntth data point received */
    kbpnt = 1;
    /* values at the kbpnt point are stored in (xb,yb) in double precision */

    if ( k == 0 ){
	si = 0.0;
	sti = 0.0;
	sti2 = 0.0;
	sni = 0.0;
	sni2 = 0.0;
	stini = 0.0;
	*per = 0.0;

	return;
    }

    /* Take new data and update sums */
    xni = xn;
    ti = tn;
    si = si + 1;
    nsi = si;
    sti = sti + ti;
    sti2 = sti2 + ti * ti;
    sni = sni + xni;
    sni2 = sni2 + xni * xni;
    stini = stini + ti * xni;

    if ( nsi == kbpnt ){
	xb = xni;	/* Save values of (xb,yb) */
	yb = ti;
    }

    if ( nsi < 2 ){
	*rinf = 0.0;
	*xn12 = 0.0;
	rmse = 0.0;
    }
    else{	/* Calculate new rinf and n1/2 */
	denom = si * stini - sti * sni;
	if ( denom == 0.0 ){
	    *rinf = 0.0;
	    *xn12 = 0.0;
	    rmse = 0.0;
	}
	else{
	    if ( mode == 0 ){
		*rinf = (si * sni2 - sni * sni) / denom;
		*xn12 = (sti * sni2 - stini * sni) / denom;
	    }
	    else if ( mode == 1 ){
		*rinf = (sni2 - 2.0 * xb * sni + xb * xb * si) /
			(stini - xb * sti - yb * sni + yb * xb * si);
		*xn12 = yb * (*rinf) - xb;
	    }

	    arg = (sti2 - (stini + (*xn12) * sti) / (*rinf)) / si;
	    if ( arg >= 0 )
		rmse = sqrt(arg);	/*!org: dsqrt()*/
	    else
		rmse = -sqrt(-arg);
	}
    }

    /* Update percentage errors */
    if ( nsi > 2 ){
	*per = 100.0 * rmse / ti;
    }
}
