/* ---------------------------------------------------------------------
*
*  -- PBLAS routine (version 1.0) --
*     University of Tennessee, Knoxville, Oak Ridge National Laboratory,
*     and University of California, Berkeley.
*     March 17, 1995
*
*  ---------------------------------------------------------------------
*/
/*
*  Include files
*/
#include "pblas.h"

void pdsymm_( side, uplo, m, n, alpha, A, ia, ja, desc_A, B, ib, jb,
              desc_B, beta, C, ic, jc, desc_C )
/*
*  .. Scalar Arguments ..
*/
   F_CHAR      side, uplo;
   int         * ia, * ib, * ic, * ja, * jb, * jc, * m, * n;
   double      * alpha, * beta;
/* ..
*  .. Array Arguments ..
*/
   int         desc_A[], desc_B[], desc_C[];
   double      A[], B[], C[];
{
/*
*  Purpose
*  =======
*
*  PDSYMM  performs one of the distributed matrix-matrix operations
*
*     sub( C ) := alpha*sub( A )*sub( B ) + beta*sub( C ),
*
*  or
*
*     sub( C ) := alpha*sub( B )*sub( A ) + beta*sub( C ),
*
*  where sub( C ) denotes C(IC:IC+M-1,JC:JC+N-1),
*
*        sub( A ) denotes A(IA:IA+M-1,JA:JA+M-1)  if SIDE = 'L',
*                         A(IA:IA+N-1,JA:JA+N-1)  if SIDE = 'R',
*
*        sub( B ) denotes B(IB:IB+M-1,JB:JB+N-1).
*
*  Alpha and beta are scalars, sub( A ) is a symmetric distributed
*  matrix and  sub( B ) and sub( C ) are M-by-N distributed matrices.
*
*  Notes
*  =====
*
*  A description vector is associated with each 2D block-cyclicly dis-
*  tributed matrix.  This vector stores the information required to
*  establish the mapping between a matrix entry and its corresponding
*  process and memory location.
*
*  In the following comments, the character _ should be read as
*  "of the distributed matrix".  Let A be a generic term for any 2D
*  block cyclicly distributed matrix.  Its description vector is DESC_A:
*
*  NOTATION        STORED IN     EXPLANATION
*  --------------- ------------- ---------------------------------------
*  M_A    (global) desc_A[M_]    The number of rows in the distributed
*                                matrix.
*  N_A    (global) desc_A[N_]    The number of columns in the distribu-
*                                ted matrix.
*  MB_A   (global) desc_A[MB_]   The blocking factor used to distribute
*                                the rows of the matrix.
*  NB_A   (global) desc_A[NB_]   The blocking factor used to distribute
*                                the columns of the matrix.
*  RSRC_A (global) desc_A[RSRC_] The process row over which the first
*                                row of the matrix is distributed.
*  CSRC_A (global) desc_A[CSRC_] The process column over which the first
*                                column of the matrix is distributed.
*  CTXT_A (global) desc_A[CTXT_] The BLACS context handle, indicating
*                                the global context of the operation on
*                                the matrix.
*                                The context itself is global.
*  LLD_A  (local)  desc_A[LLD_]  The leading dimension of the local
*                                array storing the local blocks of the
*                                distributed matrix A.
*                                LLD_A >= MAX(1,LOCp(M_A)).
*
*  Let K be the number of rows or columns of a distributed matrix,
*  and assume that its process grid has dimension p x q.
*  LOCp( K ) denotes the number of elements of K that a process
*  would receive if K were distributed over the p processes of its
*  process column.
*  Similarly, LOCq( K ) denotes the number of elements of K that a
*  process would receive if K were distributed over the q processes of
*  its process row.
*  The values of LOCp() and LOCq() may be determined via a call to the
*  ScaLAPACK tool function, NUMROC.
*          LOCp( M ) = NUMROC( M, MB_A, MYROW, RSRC_A, NPROW ),
*          LOCq( N ) = NUMROC( N, NB_A, MYCOL, CSRC_A, NPCOL ).
*
*  The symmetric distributed matrix sub( A ) must be distributed
*  according to a square block cyclic decomposition, i.e MB_A = NB_A.
*  If SIDE = 'Left', the distributed matrix sub( A ) is of order NA = M,
*  and NA = N if SIDE = 'Right'. If NA+MOD(IA-1,MB_A) > MB_A or
*  NA+MOD(JA-1,NB_A) > NB_A, then sub( A ) is not just contained into a
*  block, in which case it is required that IA-1 (resp. JA-1) is a
*  multiple of MB_A (resp. NB_A).
*
*  If SIDE = 'L', the row process having the first entries of sub( A )
*  must also own the first entries of sub( B ) and sub( C ).  The
*  column block sizes of B and C should be equal i.e NB_B = NB_C.
*  If sub( A ) is not just contained into a block, IB-1 (resp. IC-1)
*  must be a multiple of MB_B (resp. MB_C). The row block sizes of B
*  (and C) must be equal to the column block size of A, i.e
*  MB_B = MB_C = NB_A.
*
*  If SIDE = 'R', the column process having the first entries of
*  sub( A ) must also own the first entries of sub( B ) and sub( C ).
*  The row block sizes of B and C should be equal i.e MB_B = MB_C.
*  If sub( A ) is not just contained into a block, JB-1 (resp. JC-1)
*  must be a multiple of NB_B (resp. NB_C). The column block sizes of
*  B (and C) must be equal to the row block size of A, i.e
*  NB_B = NB_C = MB_A.
*
*  Parameters
*  ==========
*
*  SIDE    (global input) pointer to CHARACTER
*          On entry, SIDE specifies whether the symmetric distributed
*          matrix  sub( A ) appears on the left or right in the
*          operation as follows:
*
*          SIDE = 'L' or 'l'
*             sub( C ) := alpha*sub( A )*sub( B ) + beta*sub( C ),
*
*          SIDE = 'R' or 'r'
*             sub( C ) := alpha*sub( B )*sub( A ) + beta*sub( C ),
*
*  UPLO    (global input) CHARACTER
*          On  entry, UPLO specifies whether the upper or lower trian-
*          gular part of the symmetric distributed matrix sub( A ) is
*          to be referenced as follows:
*
*          UPLO = 'U' or 'u' Only the upper triangular part of the
*                            symmetric distributed matrix is to be
*                            referenced.
*
*          UPLO = 'L' or 'l' Only the lower triangular part of the
*                            symmetric distributed matrix is to be
*                            referenced.
*
*  M       (global input) pointer to INTEGER
*          The number of rows to be operated on i.e the number of rows
*          of the distributed submatrix sub( C ). M >= 0.
*
*  N       (global input) pointer to INTEGER
*          The number of columns to be operated on i.e the number of
*          columns of the distributed submatrix sub( C ). N >= 0.
*
*  ALPHA   (global input) pointer to DOUBLE PRECISION
*          On entry, ALPHA specifies the scalar alpha.
*
*  A       (local input) DOUBLE PRECISION pointer into the local memory
*          to an array of dimension (LLD_A, LOCq(JA+NA-1). Before entry
*          this array contains the local pieces of the symmetric
*          distributed matrix sub( A ), such that when  UPLO = 'U' or
*          'u', the NA-by-NA upper triangular part of the distributed
*          matrix sub( A ) must contain the upper triangular part of
*          the symmetric distributed matrix and the strictly lower
*          triangular part of sub( A ) is not referenced,  and when
*          UPLO = 'L' or 'l', the NA-by-NA lower triangular part of
*          the distributed matrix sub( A ) must contain the lower
*          triangular part of the symmetric distributed matrix and
*          the strictly lower triangular part of sub( A ) is not
*          referenced.
*
*  IA      (global input) pointer to INTEGER
*          The global row index of the submatrix of the distributed
*          matrix A to operate on.
*
*  JA      (global input) pointer to INTEGER
*          The global column index of the submatrix of the distributed
*          matrix A to operate on.
*
*  DESCA   (global and local input) INTEGER array of dimension 8.
*          The array descriptor of the distributed matrix A.
*
*  B       (local input) DOUBLE PRECISION pointer into the local memory
*          to an array of dimension (LLD_B, LOCq(JB+N-1)). Before
*          entry, this array contains the local pieces of the distribu-
*          ted matrix sub( B ).
*
*  IB      (global input) pointer to INTEGER
*          The global row index of the submatrix of the distributed
*          matrix B to operate on.
*
*  JB      (global input) pointer to INTEGER
*          The global column index of the submatrix of the distributed
*          matrix B to operate on.
*
*  DESCB   (global and local input) INTEGER array of dimension 8.
*          The array descriptor of the distributed matrix B.
*
*  BETA    (global input) DOUBLE PRECISION
*          On entry,  BETA  specifies the scalar  beta.  When  BETA  is
*          supplied as zero then sub( C ) need not be set on input.
*
*  C       (local input/local output) DOUBLE PRECISION pointer into the
*          local memory to an array of dimension (LLD_C, LOCq(JC+N-1)).
*          Before entry, this array contains the local pieces of the
*          distributed matrix sub( C ). On exit, sub( C ) is overwritten
*          by the M-by-N updated distributed matrix.
*
*  IC      (global input) pointer to INTEGER
*          The global row index of the submatrix of the distributed
*          matrix C to operate on.
*
*  JC      (global input) pointer to INTEGER
*          The global column index of the submatrix of the distributed
*          matrix C to operate on.
*
*  DESCC   (global and local input) INTEGER array of dimension 8.
*          The array descriptor of the distributed matrix C.
*
*  =====================================================================
*
*  .. Local Scalars ..
*/
   char        * ctop, matblk, * rtop, SideA, UploA;
   int         block, i, iacol, iarow, ibcol, iblk, ibpos, ibrow, iccol,
               icoffa, icoffb, icoffc, icpos, icrow, ictxt, iia, iib,
               iic, in, info, iroffa, iroffb, iroffc, iszcmp, j, jblk,
               jja, jjb, jjc, jn, lcm, lcmp, lcmq, loop, lside, mone=-1,
               mq0, mullen, mycol, myrow, nca, ncb, ncc, np0, nprow,
               npcol, nra, nrb, nrc, szcmp, tmp0, tmp1, tmp2, wksz;
/* ..
*  .. PBLAS Buffer ..
*/
   double      * buff;
/* ..
*  .. External Functions ..
*/
   void        blacs_gridinfo_();
   void        pberror_();
   void        pbchkmat();
   char        * getpbbuf();
   char        * ptop();
   F_VOID_FCT  pbdsymm_();
   F_INTG_FCT  ilcm_();
/* ..
*  .. Executable Statements ..
*
*  Get grid parameters
*/
   ictxt = desc_A[CTXT_];
   blacs_gridinfo_( &ictxt, &nprow, &npcol, &myrow, &mycol );
/*
*  Test the input parameters
*/
   info = 0;
   if( nprow == -1 )
      info = -907;
   else
   {
      UploA = Mupcase( F2C_CHAR( uplo )[0] );
      SideA = Mupcase( F2C_CHAR( side )[0] );
      lside = ( SideA == 'L' );
      iroffa = (*ia-1) % desc_A[MB_];
      icoffa = (*ja-1) % desc_A[NB_];
      iroffb = (*ib-1) % desc_B[MB_];
      icoffb = (*jb-1) % desc_B[NB_];
      iroffc = (*ic-1) % desc_C[MB_];
      icoffc = (*jc-1) % desc_C[NB_];
      if( lside )
      {
         pbchkmat( *m, 3, *m, 3, *ia, *ja, desc_A, 9, &iia, &jja,
                   &iarow, &iacol, nprow, npcol, myrow, mycol,
                   &nra, &nca, &info );
         pbchkmat( *m, 3, *n, 4, *ib, *jb, desc_B, 13, &iib, &jjb,
                   &ibrow, &ibcol, nprow, npcol, myrow, mycol,
                   &nrb, &ncb, &info );
         pbchkmat( *m, 3, *n, 4, *ic, *jc, desc_C, 18, &iic, &jjc,
                   &icrow, &iccol, nprow, npcol, myrow, mycol,
                   &nrc, &ncc, &info );
         block = ( ( ( (*m) + iroffa ) <= desc_A[MB_] ) &&
                   ( ( (*m) + icoffa ) <= desc_A[NB_] ) &&
                   ( ibcol == iccol ) );
         loop = ( ( ( (*n) + icoffb ) > desc_B[NB_] ) ||
                  ( ( (*n) + icoffc ) > desc_C[NB_] ) );
      }
      else
      {
         pbchkmat( *n, 4, *n, 4, *ia, *ja, desc_A, 9, &iia, &jja,
                   &iarow, &iacol, nprow, npcol, myrow, mycol,
                   &nra, &nca, &info );
         pbchkmat( *m, 3, *n, 4, *ib, *jb, desc_B, 13, &iib, &jjb,
                   &ibrow, &ibcol, nprow, npcol, myrow, mycol,
                   &nrb, &ncb, &info );
         pbchkmat( *m, 3, *n, 4, *ic, *jc, desc_C, 18, &iic, &jjc,
                   &icrow, &iccol, nprow, npcol, myrow, mycol,
                   &nrc, &ncc, &info );
         block = ( ( ( (*n) + iroffa ) <= desc_A[MB_] ) &&
                   ( ( (*n) + icoffa ) <= desc_A[NB_] ) &&
                   ( ibrow == icrow ) );
         loop = ( ( ( (*m) + iroffb ) > desc_B[MB_] ) ||
                  ( ( (*m) + iroffc ) > desc_C[MB_] ) );
      }
      if( info == 0 )
      {
         if( ( SideA != 'L' ) && ( SideA != 'R' ) )
            info = -1;
         else if( ( UploA != 'U' ) && ( UploA != 'L' ) )
            info = -2;
         else if( desc_A[MB_] != desc_A[NB_] )
            info = -904;
         if( block )
         {
            matblk = 'B';
            if( lside )
            {
               if( nprow != 1 )
               {
                  if( (*m)+iroffb > desc_B[MB_] )
                     info = -11;
                  else if( (*m)+iroffc > desc_C[MB_] )
                     info = -16;
               }
            }
            else
            {
               if( npcol != 1 )
               {
                  if( (*n)+icoffb > desc_B[NB_] )
                     info = -12;
                  else if( (*n)+icoffc > desc_C[NB_] )
                     info = -17;
               }
            }
         }
         else
         {
            matblk = 'M';
            if( iroffa != 0 )
               info = -7;
            else if( icoffa != 0 )
               info = -8;
            if( lside )
            {
               if( iroffb != 0 )
                  info = -11;
               else if( desc_B[MB_] != desc_A[NB_] )
                  info = -1303;
               else if( iroffc != 0 )
                  info = -16;
               else if( desc_C[MB_] != desc_A[MB_] )
                  info = -1803;
            }
            else
            {
               if( icoffb != 0 )
                  info = -12;
               else if( desc_B[NB_] != desc_A[NB_] )
                  info = -1304;
               else if( icoffc != 0 )
                  info = -17;
               else if( desc_C[NB_] != desc_A[NB_] )
                  info = -1804;
            }
         }
         if( lside )
         {
            if( ibrow != iarow )
               info = -11;
            else if( icrow != iarow )
               info = -16;
            else if( loop && ( icoffb != icoffc ) )
               info = -17;
            else if( loop && ( desc_B[NB_] != desc_C[NB_] ) )
               info = -1804;
         }
         else
         {
            if( ibcol != iacol )
               info = -12;
            else if( loop && ( iroffb != iroffc ) )
               info = -16;
            else if( iccol != iacol )
               info = -17;
            else if( loop && ( desc_B[MB_] != desc_C[MB_] ) )
               info = -1803;
         }
      }
      if( ictxt != desc_B[CTXT_] )
         info = -1307;
      else if( ictxt != desc_C[CTXT_] )
         info = -1807;
   }
   if( info )
   {
      pberror_( &ictxt, "PDSYMM", &info );
      return;
   }
/*
*  Quick return if possible.
*/
   if( ( *m == 0 ) || ( *n == 0 ) ||
       ( ( *alpha == ZERO ) && ( *beta == ONE ) ) )
      return;
/*
*  Figure out the arguments to be passed to pbdsymm
*/
   mullen = MULLENFAC * desc_A[MB_];
   if( block )
   {
       if( lside )
       {
          ibpos = ibcol;
          icpos = iccol;
          wksz = (*m) * (*m);
       }
       else
       {
          ibpos = ibrow;
          icpos = icrow;
          wksz = (*n) * (*n);
       }
   }
   else
   {
       lcm = ilcm_( &nprow, &npcol );
       lcmp = lcm / nprow;
       lcmq = lcm / npcol;
       tmp0 = lcmq * desc_A[MB_];
       iszcmp = CEIL( mullen, tmp0 );
       szcmp = iszcmp * iszcmp * lcmq * desc_A[MB_] * lcmp * desc_A[MB_];
       if( lside )
       {
           ibpos = ibcol;
           icpos = iccol;
           tmp1 = (*m) / desc_C[MB_];
           mq0 = MYROC0( tmp1, *m, desc_C[MB_], npcol );
           tmp2 = mq0 / desc_C[MB_];
           tmp0 = desc_B[NB_] * MYROC0( tmp2, mq0, desc_C[MB_], lcmq );
           tmp2 = CEIL( *m, desc_A[MB_] );
           tmp2 = MIN( lcmq, tmp2 );
           tmp0 = tmp0 * tmp2;
           wksz = ( 2 * desc_B[NB_] *
                    ( mq0 + MYROC0( tmp1, *m, desc_C[MB_], nprow ) ) ) +
                  MAX( szcmp, tmp0 );
       }
       else
       {
           ibpos = ibrow;
           icpos = icrow;
           tmp1 = (*n) / desc_C[NB_];
           np0 = MYROC0( tmp1, *n, desc_C[NB_], nprow );
           tmp2 = np0 / desc_C[NB_];
           tmp0 = desc_B[MB_] * MYROC0( tmp2, np0, desc_C[NB_], lcmp );
           tmp2 = CEIL( *n, desc_A[NB_] );
           tmp2 = MIN( lcmp, tmp2 );
           tmp0 = tmp0 * tmp2;
           wksz = ( 2 * desc_B[MB_] *
                    ( np0 + MYROC0( tmp1, *n, desc_C[NB_], npcol ) ) ) +
                  MAX( szcmp, tmp0 );
       }
   }
   buff = (double *)getpbbuf( "PDSYMM", wksz*sizeof(double) );
/*
*  Call PB-BLAS routine
*/
   if( block )
   {
      if( lside )
      {
         rtop = ptop( BROADCAST, ROW, TOPGET );
         j = CEIL( (*jb), desc_B[NB_] ) * desc_B[NB_];
         jn = (*jb)+(*n)-1;
         jn = MIN( j, jn );
                                     /* Handle first block separately */
         jblk = jn-(*jb)+1;

         pbdsymm_( &ictxt, C2F_CHAR( &matblk ), side, uplo, m, &jblk,
                   &desc_B[NB_], alpha, &A[iia-1+(jja-1)*desc_A[LLD_]],
                   &desc_A[LLD_], &B[iib-1+(jjb-1)*desc_B[LLD_]],
                   &desc_B[LLD_], beta, &C[iic-1+(jjc-1)*desc_C[LLD_]],
                   &desc_C[LLD_], &iarow, &iacol, &ibpos, &icpos,
                   C2F_CHAR( rtop ), C2F_CHAR( NO ), C2F_CHAR( NO ),
                   &mullen, buff );
         if( mycol == ibpos )
         {
            jjb += jblk;
            jjb = MIN( jjb, ncb );
         }
         ibpos = (ibpos+1) % npcol;
         if( mycol == icpos )
         {
            jjc += jblk;
            jjc = MIN( jjc, ncc );
         }
         icpos = (icpos+1) % npcol;
         jblk = (*n) - jblk;
         pbdsymm_( &ictxt, C2F_CHAR( &matblk ), side, uplo, m, &jblk,
                   &desc_B[NB_], alpha, buff, m,
                   &B[iib-1+(jjb-1)*desc_B[LLD_]], &desc_B[LLD_], beta,
                   &C[iic-1+(jjc-1)*desc_C[LLD_]], &desc_C[LLD_],
                   &iarow, &mone, &ibpos, &icpos, C2F_CHAR( rtop ),
                   C2F_CHAR( YES ), C2F_CHAR( NO ), &mullen, buff );
      }
      else
      {
         ctop = ptop( BROADCAST, COLUMN, TOPGET );
         i = CEIL( (*ib), desc_B[MB_] ) * desc_B[MB_];
         in = (*ib)+(*m)-1;
         in = MIN( i, in );
                                     /* Handle first block separately */
         iblk = in-(*ib)+1;
         pbdsymm_( &ictxt, C2F_CHAR( &matblk ), side, uplo, &iblk, n,
                   &desc_B[MB_], alpha, &A[iia-1+(jja-1)*desc_A[LLD_]],
                   &desc_A[LLD_], &B[iib-1+(jjb-1)*desc_B[LLD_]],
                   &desc_B[LLD_], beta, &C[iic-1+(jjc-1)*desc_C[LLD_]],
                   &desc_C[LLD_], &iarow, &iacol, &ibpos, &icpos,
                   C2F_CHAR( ctop ), C2F_CHAR( NO ), C2F_CHAR( NO ),
                   &mullen, buff );
         if( myrow == ibpos )
         {
            iib += iblk;
            iib = MIN( iib, nrb );
         }
         ibpos = (ibpos+1) % nprow;
         if( myrow == icpos )
         {
            iic += iblk;
            iic = MIN( iic, nrc );
         }
         icpos = (icpos+1) % nprow;
         iblk = (*m) - iblk;
         pbdsymm_( &ictxt, C2F_CHAR( &matblk ), side, uplo, &iblk, n,
                   &desc_B[MB_], alpha, buff, n,
                   &B[iib-1+(jjb-1)*desc_B[LLD_]], &desc_B[LLD_], beta,
                   &C[iic-1+(jjc-1)*desc_C[LLD_]], &desc_C[LLD_],
                   &mone, &iacol, &ibpos, &icpos, C2F_CHAR( ctop ),
                   C2F_CHAR( YES ), C2F_CHAR( NO ), &mullen, buff );
      }
   }
   else
   {
      if( lside )
      {
         j = CEIL( (*jb), desc_B[NB_] ) * desc_B[NB_];
         jn = (*jb)+(*n)-1;
         jn = MIN( j, jn );
                                     /* Handle first block separately */
         jblk = jn-(*jb)+1;
         pbdsymm_( &ictxt, C2F_CHAR( &matblk ), side, uplo, m, &jblk,
                   &desc_A[MB_], alpha, &A[iia-1+(jja-1)*desc_A[LLD_]],
                   &desc_A[LLD_], &B[iib-1+(jjb-1)*desc_B[LLD_]],
                   &desc_B[LLD_], beta, &C[iic-1+(jjc-1)*desc_C[LLD_]],
                   &desc_C[LLD_], &iarow, &iacol, &ibpos, &icpos,
                   C2F_CHAR( TOPDEF ), C2F_CHAR( NO ), C2F_CHAR( NO ),
                   &mullen, buff );
         if( mycol == ibpos )
         {
            jjb += jblk;
            jjb = MIN( jjb, ncb );
         }
         if( mycol == icpos )
         {
            jjc += jblk;
            jjc = MIN( jjc, ncc );
         }
         ibpos = (ibpos+1) % npcol;
         icpos = (icpos+1) % npcol;
                              /* loop over remaining block of columns */
         tmp0 = (*jb)+(*n)-1;
         for( j=jn+1; j<=tmp0; j+=desc_B[NB_] )
         {
            jblk = (*n)-j+(*jb);
            jblk = MIN( desc_B[NB_], jblk );
            pbdsymm_( &ictxt, C2F_CHAR( &matblk ), side, uplo, m, &jblk,
                      &desc_A[MB_], alpha, &A[iia-1+(jja-1)*desc_A[LLD_]],
                      &desc_A[LLD_], &B[iib-1+(jjb-1)*desc_B[LLD_]],
                      &desc_B[LLD_], beta, &C[iic-1+(jjc-1)*desc_C[LLD_]],
                      &desc_C[LLD_], &iarow, &iacol, &ibpos, &icpos,
                      C2F_CHAR( TOPDEF ), C2F_CHAR( NO ), C2F_CHAR( NO ),
                      &mullen, buff );
            if( mycol == ibpos )
            {
               jjb += jblk;
               jjb = MIN( jjb, ncb );
            }
            if( mycol == icpos )
            {
               jjc += jblk;
               jjc = MIN( jjc, ncc );
            }
            ibpos = (ibpos+1) % npcol;
            icpos = (icpos+1) % npcol;
         }
      }
      else
      {
         i = CEIL( (*ib), desc_B[MB_] ) * desc_B[MB_];
         in = (*ib)+(*m)-1;
         in = MIN( i, in );
                                     /* Handle first block separately */
         iblk = in-(*ib)+1;
         pbdsymm_( &ictxt, C2F_CHAR( &matblk ), side, uplo, &iblk, n,
                   &desc_A[MB_], alpha, &A[iia-1+(jja-1)*desc_A[LLD_]],
                   &desc_A[LLD_], &B[iib-1+(jjb-1)*desc_B[LLD_]],
                   &desc_B[LLD_], beta, &C[iic-1+(jjc-1)*desc_C[LLD_]],
                   &desc_C[LLD_], &iarow, &iacol, &ibpos, &icpos,
                   C2F_CHAR( TOPDEF ), C2F_CHAR( NO ), C2F_CHAR( NO ),
                   &mullen, buff );
         if( myrow == ibpos )
         {
            iib += iblk;
            iib = MIN( iib, nrb );
         }
         if( myrow == icpos )
         {
            iic += iblk;
            iic = MIN( iic, nrc );
         }
         ibpos = (ibpos+1) % nprow;
         icpos = (icpos+1) % nprow;
                                 /* loop over remaining block of rows */
         tmp0 = (*ib)+(*m)-1;
         for( i=in+1; i<=tmp0; i+=desc_B[MB_] )
         {
             iblk = (*m)-i+(*ib);
             iblk = MIN( desc_B[MB_], iblk );
             pbdsymm_( &ictxt, C2F_CHAR( &matblk ), side, uplo, &iblk, n,
                       &desc_A[MB_], alpha, &A[iia-1+(jja-1)*desc_A[LLD_]],
                       &desc_A[LLD_], &B[iib-1+(jjb-1)*desc_B[LLD_]],
                       &desc_B[LLD_], beta, &C[iic-1+(jjc-1)*desc_C[LLD_]],
                       &desc_C[LLD_], &iarow, &iacol, &ibpos, &icpos,
                       C2F_CHAR( TOPDEF ), C2F_CHAR( NO ), C2F_CHAR( NO ),
                       &mullen, buff );
             if( myrow == ibpos )
             {
                iib += iblk;
                iib = MIN( iib, nrb );
             }
             if( myrow == icpos )
             {
                iic += iblk;
                iic = MIN( iic, nrc );
             }
             ibpos = (ibpos+1) % nprow;
             icpos = (icpos+1) % nprow;
         }
      }
   }
}
