/* 
 * This utility configures a NPB to be built for a specific number
 * of nodes and a specific class. It creates a file "npbparams.h" 
 * in the source directory. This file keeps state information about 
 * which size of benchmark is currently being built (so that nothing
 * if unnecessarily rebuilt) and defines (through PARAMETER statements)
 * the number of nodes and class for which a benchmark is being built. 

 * The utility takes 3 arguments: 
 *       setparams benchmark-name nprocs class
 *    benchmark-name is "sp", "bt", etc
 *    nprocs is the number of processors to run on
 *    class is the size of the benchmark
 * These parameters are checked for the current benchmark. If they
 * are invalid, this program prints a message and aborts. 
 * If the parameters are ok, the current npbsize.h (actually just
 * the first line) is read in. If the new parameters are the same as 
 * the old, nothing is done, but an exit code is returned to force the
 * user to specify (otherwise the make procedure succeeds but builds a
 * binary of the wrong name).  Otherwise the file is rewritten. 
 * Errors write a message (to stdout) and abort. 
 * 
 * This program makes use of two extra benchmark "classes"
 * class "X" means an invalid specification. It is returned if
 * there is an error parsing the config file. 
 * class "U" is an external specification meaning "unknown class"
 * 
 * Unfortunately everything has to be case sensitive. This is
 * because we can always convert lower to upper or v.v. but
 * can't feed this information back to the makefile, so typing
 * make CLASS=a and make CLASS=A will produce different binaries.
 *
 * 
 */

#include <sys/types.h>
#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <time.h>

/*
 * This is the master version number for this set of 
 * NPB benchmarks. It is in an obscure place to people
 * won't accidentally change it. 
 */

#define VERSION "2.1"

#define FILENAME "npbparams.h"
#define DESC_LINE "c NPROCS = %d CLASS = %c\n"
#define FINDENT  "        "
#define CONTINUE "     > "

void get_info(char *argv[], int *typep, int *nprocsp, char *classp);
void check_info(int type, int nprocs, char class);
void read_info(int *nprocsp, char *classp);
void write_info(int type, int nprocs, char class);
void write_sp_info(FILE *fp, int nprocs, char class);
void write_bt_info(FILE *fp, int nprocs, char class);
void write_lu_info(FILE *fp, int nprocs, char class);
void write_mg_info(FILE *fp, int nprocs, char class);
void write_ft_info(FILE *fp, int nprocs, char class);
void write_compiler_info(FILE *fp);
void check_line(char *line, char *label, char *val);
void put_string(FILE *fp, char *name, char *val);
int isqrt(int i);
int ilog2(int i);
int ipow2(int i);

enum benchmark_types {SP, BT, LU, MG, FT};

main(int argc, char *argv[])
{
  int nprocs, nprocs_old, type;
  char class, class_old;
  
  if (argc != 4) {
    printf("Usage: %s benchmark-name nprocs class\n", argv[0]);
    exit(1);
  }

  /* Get command line arguments. Make sure they're ok. */
  get_info(argv, &type, &nprocs, &class);
  if (class != 'U') {
    printf("setparams: For benchmark %s: number of processors = %d class = %c\n", 
	   argv[1], nprocs, class);
    check_info(type, nprocs, class);
  }

  /* Get old information. */
  read_info(&nprocs_old, &class_old);
  if (class != 'U') {
    if (class_old != 'X') {
      printf("setparams:     old settings: number of processors = %d class = %c\n", 
	     nprocs_old, class_old);
    }
  } else {
    printf("setparams:\n\
  *********************************************************************\n\
  * You must specify NPROCS and CLASS to build this benchmark         *\n\
  * For example, to build a class A benchmark for 4 processors, type  *\n\
  *       make {benchmark-name} NPROCS=4 CLASS=A                      *\n\
  *********************************************************************\n\n"); 

    if (class_old != 'X') {
      printf("setparams: Previous settings were CLASS=%c NPROCS=%d\n", class_old, nprocs_old);
    }
    exit(1); /* exit on class==U */
  }

  /* Write out new information if it's different. */
  if (nprocs != nprocs_old || class != class_old) {
    printf("setparams: Writing %s\n", FILENAME);
    write_info(type, nprocs, class);
  } else {
    printf("setparams: Settings unchanged. %s unmodified\n", FILENAME);
  }

  exit(0);
}


/*
 *  get_info(): Get parameters from command line 
 */

void get_info(char *argv[], int *typep, int *nprocsp, char *classp) 
{

  *nprocsp = atoi(argv[2]);

  *classp = *argv[3];

  if      (!strcasecmp(argv[1], "sp")) *typep = SP;
  else if (!strcasecmp(argv[1], "bt")) *typep = BT;
  else if (!strcasecmp(argv[1], "ft")) *typep = FT;
  else if (!strcasecmp(argv[1], "lu")) *typep = LU;
  else if (!strcasecmp(argv[1], "mg")) *typep = MG;
  else {
    printf("setparams: Error: unknown benchmark type %s\n", argv[1]);
    exit(1);
  }
}

/*
 *  check_info(): Make sure command line data is ok for this benchmark 
 */

void check_info(int type, int nprocs, char class) 
{
  int rootprocs, logprocs, tmplog; 

  /* check number of processors */
  if (nprocs <= 0) {
    printf("setparams: Number of processors must be greater than zero\n");
    exit(1);
  }
  switch(type) {

  case SP:
  case BT:
    rootprocs = isqrt(nprocs);
    if (rootprocs < 0) {
      printf("setparams: Number of processors %d must be a square (1,4,9,...) for this benchmark", 
              nprocs);
      exit(1);
    }
    if (class == 'S' && nprocs > 16) {
      printf("setparams: BT and SP sample sizes cannot be run on more\n");
      printf("           than 16 processors because the cell size would be too small.\n");
      exit(1);
    }
    break;

  case FT:
  case MG:
  case LU:
    logprocs = ilog2(nprocs);
    if (logprocs < 0) {
      printf("setparams: Number of processors must be a power of two (1,2,4,...) for this benchmark\n");
      exit(1);
    }

    break;

  default:
    /* never should have gotten this far with a bad name */
    printf("setparams: (Internal Error) Benchmark type %d unknown to this program\n", type); 
    exit(1);
  }

  /* check class */
  if (class != 'S' && 
      class != 'A' && 
      class != 'B' && 
      class != 'R' && 
      class != 'C') {
    printf("setparams: Unknown benchmark class %c\n", class); 
    printf("setparams: Allowed classes are \"S\", \"A\", \"B\" and \"C\"\n");
    exit(1);
  }

}


/* 
 * read_info(): Read previous information from file. 
 *              Not an error if file doesn't exist, because this
 *              may be the first time we're running. 
 *              Assumes the first line of the file is in a special
 *              format that we understand (since we wrote it). 
 */

void read_info(int *nprocsp, char *classp)
{
  int nread, gotem = 0;
  char line[200];
  FILE *fp;
  fp = fopen(FILENAME, "r");
  if (fp == NULL) {
    printf("setparams: INFO: configuration file %s does not exist (yet)\n", FILENAME);
    goto abort;
  }
  
  /* first line of file contains info */
  nread = fscanf(fp, DESC_LINE, nprocsp, classp);
  if (nread != 2) {
    printf("setparams: Error parsing config file %s. Ignoring previous settings\n", FILENAME);
    goto abort;
  }
  *classp = *classp;
  fclose(fp);

  return;

 abort:
  *nprocsp = -1;
  *classp = 'X';
  return;
}


/* 
 * write_info(): Write new information to config file. 
 *               First line is in a special format so we can read
 *               it in again. Then comes a warning. The rest is all
 *               specific to a particular benchmark. 
 */

void write_info(int type, int nprocs, char class) 
{
  FILE *fp;
  fp = fopen(FILENAME, "w");
  if (fp == NULL) {
    printf("setparams: Can't open file %d for writing\n", FILENAME);
    exit(1);
  }

  /* Write out the header */
  fprintf(fp, DESC_LINE, nprocs, class);

  /* Print out a warning so bozos don't mess with the file */
  fprintf(fp, "\
c  \n\
c  \n\
c  This file is generated automatically by the setparams utility.\n\
c  It sets the number of processors and the class of the NPB\n\
c  in this directory. Do not modify it by hand.\n\
c  \n");

  /* Now do benchmark-specific stuff */
  switch(type) {
  case SP:
    write_sp_info(fp, nprocs, class);
    break;
  case BT:
    write_bt_info(fp, nprocs, class);
    break;
  case LU:
    write_lu_info(fp, nprocs, class);
    break;
  case MG:
    write_mg_info(fp, nprocs, class);
    break;
  case FT:
    write_ft_info(fp, nprocs, class);
    break;
  default:
    printf("setparams: (Internal error): Unknown benchmark type %d\n", type);
    exit(1);
  }
  write_compiler_info(fp);
  fclose(fp);
  return;
}


/* 
 * write_sp_info(): Write SP specific info to config file
 */

void write_sp_info(FILE *fp, int nprocs, char class) 
{
  int maxcells, problem_size, niter;
  char *dt;
  maxcells = isqrt(nprocs);
  if      (class == 'S') { problem_size = 12;  dt = "0.015d0";   niter = 100; }
  else if (class == 'A') { problem_size = 64;  dt = "0.0015d0";  niter = 400; }
  else if (class == 'B') { problem_size = 102; dt = "0.001d0";   niter = 400; }
  else if (class == 'C') { problem_size = 162; dt = "0.00067d0"; niter = 400; }
  else {
    printf("setparams: Internal error: invalid class %c\n", class);
    exit(1);
  }
  fprintf(fp, "%sinteger maxcells, problem_size, niter_default\n", FINDENT);
  fprintf(fp, "%sparameter (maxcells=%d, problem_size=%d, niter_default=%d)\n", 
	       FINDENT, maxcells, problem_size, niter);
  fprintf(fp, "%sdouble precision dt_default\n", FINDENT);
  fprintf(fp, "%sparameter (dt_default = %s)\n", FINDENT, dt);
}
  
/* 
 * write_bt_info(): Write BT specific info to config file
 */

void write_bt_info(FILE *fp, int nprocs, char class) 
{
  int maxcells, problem_size, niter;
  char *dt;
  maxcells = isqrt(nprocs);
  if      (class == 'S') { problem_size = 12;  dt = "0.010d0";   niter = 60; }
  else if (class == 'A') { problem_size = 64;  dt = "0.0008d0";  niter = 200; }
  else if (class == 'B') { problem_size = 102; dt = "0.0003d0";  niter = 200; }
  else if (class == 'C') { problem_size = 162; dt = "0.0001d0";  niter = 200; }
  else {
    printf("setparams: Internal error: invalid class %c\n", class);
    exit(1);
  }
  fprintf(fp, "%sinteger maxcells, problem_size, niter_default\n", FINDENT);
  fprintf(fp, "%sparameter (maxcells=%d, problem_size=%d, niter_default=%d)\n", 
	       FINDENT, maxcells, problem_size, niter);
  fprintf(fp, "%sdouble precision dt_default\n", FINDENT);
  fprintf(fp, "%sparameter (dt_default = %s)\n", FINDENT, dt);
}
  


/* 
 * write_lu_info(): Write SP specific info to config file
 */

void write_lu_info(FILE *fp, int nprocs, char class) 
{
  int isiz1, isiz2, itmax, inorm, problem_size;
  int xdiv, ydiv; /* number of cells in x and y direction */
  char *dt_default;

  if      (class == 'S') { problem_size = 12;  dt_default = "0.5d0"; itmax = 50; }
  else if (class == 'A') { problem_size = 64;  dt_default = "2.0d0"; itmax = 250; }
  else if (class == 'B') { problem_size = 102; dt_default = "2.0d0"; itmax = 250; }
  else if (class == 'C') { problem_size = 162; dt_default = "2.0d0"; itmax = 250; }
  else {
    printf("setparams: Internal error: invalid class %c\n", class);
    exit(1);
  }
  inorm = itmax;
  xdiv = ydiv = ilog2(nprocs)/2;
  if (xdiv+ydiv != ilog2(nprocs)) xdiv += 1;
  xdiv = ipow2(xdiv); ydiv = ipow2(ydiv);
  isiz1 = problem_size/xdiv; if (isiz1*xdiv < problem_size) isiz1++;
  isiz2 = problem_size/ydiv; if (isiz2*ydiv < problem_size) isiz2++;
  

  fprintf(fp, "\nc number of nodes for which this version is compiled\n");
  fprintf(fp, "%sinteger nnodes_compiled\n", FINDENT);
  fprintf(fp, "%sparameter (nnodes_compiled = %d)\n", FINDENT, nprocs);

  fprintf(fp, "\nc full problem size\n");
  fprintf(fp, "%sinteger isiz01, isiz02, isiz03\n", FINDENT);
  fprintf(fp, "%sparameter (isiz01=%d, isiz02=%d, isiz03=%d)\n", 
	  FINDENT, problem_size, problem_size, problem_size);

  fprintf(fp, "\nc sub-domain array size\n");
  fprintf(fp, "%sinteger isiz1, isiz2, isiz3\n", FINDENT);
  fprintf(fp, "%sparameter (isiz1=%d, isiz2=%d, isiz3=isiz03)\n", 
	       FINDENT, isiz1, isiz2);

  fprintf(fp, "\nc number of iterations and how often to print the norm\n");
  fprintf(fp, "%sinteger itmax_default, inorm_default\n", FINDENT);
  fprintf(fp, "%sparameter (itmax_default=%d, inorm_default=%d)\n", 
	  FINDENT, itmax, inorm);

  fprintf(fp, "%sdouble precision dt_default\n", FINDENT);
  fprintf(fp, "%sparameter (dt_default = %s)\n", FINDENT, dt_default);
  
}

/* 
 * write_mg_info(): Write MG specific info to config file
 */

void write_mg_info(FILE *fp, int nprocs, char class) 
{
  int problem_size, nit, log2_size, log2_nprocs, lt_default, lm;
  int ndim1, ndim2, ndim3;
  if      (class == 'S') { problem_size = 32; nit = 4; }
  else if (class == 'A') { problem_size = 256; nit = 4; }
  else if (class == 'B') { problem_size = 256; nit = 20; }
  else if (class == 'C') { problem_size = 512; nit = 20; }
  else {
    printf("setparams: Internal error: invalid class type %c\n", class);
    exit(1);
  }
  log2_size = ilog2(problem_size);
  log2_nprocs = ilog2(nprocs);
  /* lt is log of largest total dimension */
  lt_default = log2_size;
  /* log of log of maximum dimension on a node */
  lm = log2_size - log2_nprocs/3;
  ndim1 = lm;
  ndim3 = log2_size - (log2_nprocs+2)/3;
  ndim2 = log2_size - (log2_nprocs+1)/3;

  fprintf(fp, "%sinteger nprocs_compiled\n", FINDENT);
  fprintf(fp, "%sparameter (nprocs_compiled = %d)\n", FINDENT, nprocs);
  fprintf(fp, "%sinteger nx_default, ny_default, nz_default\n", FINDENT);
  fprintf(fp, "%sparameter (nx_default=%d, ny_default=%d, nz_default=%d)\n", 
	  FINDENT, problem_size, problem_size, problem_size);
  fprintf(fp, "%sinteger nit_default, lm, lt_default\n", FINDENT);
  fprintf(fp, "%sparameter (nit_default=%d, lm = %d, lt_default=%d)\n", 
	  FINDENT, nit, lm, lt_default);
  fprintf(fp, "%sinteger debug_default\n", FINDENT);
  fprintf(fp, "%sparameter (debug_default=%d)\n", FINDENT, 0);
  fprintf(fp, "%sinteger ndim1, ndim2, ndim3\n", FINDENT);
  fprintf(fp, "%sparameter (ndim1 = %d, ndim2 = %d, ndim3 = %d)\n", 
	  FINDENT, ndim1, ndim2, ndim3);
}

/* 
 * write_ft_info(): Write FT specific info to config file
 */

void write_ft_info(FILE *fp, int nprocs, char class) 
{
  /* easiest way (given the way the benchmark is written)
   * is to specify log of number of grid points in each
   * direction m1, m2, m3. nt is the number of iterations
   */
  int m1, m2, m3, mx, nt;
  if      (class == 'S') { m1 = 6; m2 = 6; m3 = 6; nt = 6; }
  else if (class == 'A') { m1 = 8; m2 = 8; m3 = 7; nt = 6; }
  else if (class == 'B') { m1 = 9; m2 = 8; m3 = 8; nt = 20; }
  else if (class == 'C') { m1 = 9; m2 = 9; m3 = 9; nt = 20; }
  else {
    printf("setparams: Internal error: invalid class type %c\n", class);
    exit(1);
  }
  /* mx = max(m1, m2, m3) */
  if (m1 > m2) mx = m1; else mx = m2;
  if (m3 > mx) mx = m3;
  /* number of processors given by "npm" */

  fprintf(fp, "%sinteger m1, m2, m3, mx, npm, nt\n", FINDENT);
  fprintf(fp, "%sparameter (m1=%d, m2=%d, m3=%d)\n", 
          FINDENT, m1, m2, m3);
  fprintf(fp, "%sparameter (mx=%d, npm=%d, nt=%d)\n", 
          FINDENT, mx, nprocs, nt);
}

/* 
 * This is a gross hack to allow the benchmarks to 
 * print out how they were compiled. Various other ways
 * of doing this have been tried and they all fail on
 * some machine - due to a broken "make" program, or
 * F77 limitations, of whatever. Hopefully this will
 * always work because it uses very portable C. Unfortunately
 * it relies on parsing the make.def file - YUK. 
 * If your machine doesn't have <string.h> or <ctype.h>, happy hacking!
 * 
 */

#define LL 400
#include <stdio.h>
#define DEFFILE "../config/make.def"
#define DEFAULT_MESSAGE "(none)"
void write_compiler_info(FILE *fp)
{
  FILE *deffile;
  char line[LL];
  char f77[LL], load[LL], mpi_lib[LL], mpi_inc[LL], fflags[LL], ldflags[LL];
  char compiletime[LL];
  struct tm *tmp;
  time_t t;
  deffile = fopen(DEFFILE, "r");
  if (deffile == NULL) {
    printf("\n\
setparams: File %s doesn't exist. To build the NAS benchmarks\n\
           you need to create is according to the instructions\n\
           in the README in the main directory and comments in \n\
           the file config/make.def.template\n", DEFFILE);
    exit(1);
  }
  strcpy(f77, DEFAULT_MESSAGE);
  strcpy(load, DEFAULT_MESSAGE);
  strcpy(mpi_lib, DEFAULT_MESSAGE);
  strcpy(mpi_inc, DEFAULT_MESSAGE);
  strcpy(fflags, DEFAULT_MESSAGE);
  strcpy(ldflags, DEFAULT_MESSAGE);

  while (fgets(line, LL, deffile) != NULL) {
    if (*line == '#') continue;
    /* yes, this is inefficient. but it's simple! */
    check_line(line, "F77_MPI", f77);
    check_line(line, "F77LOADER", load);
    check_line(line, "MPI_LIBS", mpi_lib);
    check_line(line, "FMPI_INC", mpi_inc);
    check_line(line, "FFLAGS", fflags);
    check_line(line, "F77LOADFLAGS", ldflags);
  }

  
  (void) time(&t);
  tmp = localtime(&t);
  (void) strftime(compiletime, (size_t)LL, "%d %b %Y", tmp);

  put_string(fp, "compiletime", compiletime);
  put_string(fp, "npbversion", VERSION);
  put_string(fp, "cs1", f77);
  put_string(fp, "cs2", load);
  put_string(fp, "cs3", mpi_lib);
  put_string(fp, "cs4", mpi_inc);
  put_string(fp, "cs5", fflags);
  put_string(fp, "cs6", ldflags);

}

void check_line(char *line, char *label, char *val)
{
  char *original_line;
  original_line = line;
  /* compare beginning of line and label */
  while (*label != '\0' && *line == *label) {
    line++; label++; 
  }
  /* if *label is not EOS, we must have had a mismatch */
  if (*label != '\0') return;
  /* if *line is not a space, actual label is longer than test label */
  if (!isspace(*line)) return ;
  /* skip over white space */
  while (isspace(*line)) line++;
  /* next char should be '=' */
  if (*line != '=') return;
  /* skip over white space */
  while (isspace(*++line));
  /* if EOS, nothing was specified */
  if (*line == '\0') return;
  /* finally we've come to the value */
  strcpy(val, line);
  /* chop off the newline at the end */
  val[strlen(val)-1] = '\0';
  if (val[strlen(val) - 1] == '\\') {
    printf("\n\
setparams: Error in file make.def. Because of the way in which\n\
           command line arguments are incorporated into the\n\
           executable benchmark, you can't have any continued\n\
           lines in the file make.def, that is, lines ending\n\
           with the character \"\\\". Although it may be ugly, \n\
           you should be able to reformat without continuation\n\
           lines. The offending line is\n\
  %s\n", original_line);
    exit(1);
  }
}

#define MAXL 46
void put_string(FILE *fp, char *name, char *val)
{
  int len;
  len = strlen(val);
  if (len > MAXL) {
    val[MAXL] = '\0';
    val[MAXL-1] = '.';
    val[MAXL-2] = '.';
    val[MAXL-3] = '.';
    len = MAXL;
  }
  fprintf(fp, "%scharacter*%d %s\n", FINDENT, len, name);
  fprintf(fp, "%sparameter (%s=\'%s\')\n", FINDENT, name, val);
}



#if 0

/* this version allows arbitrarily long lines but 
 * some compilers don't like that and they're rarely
 * useful 
 */

#define LINELEN 65
void put_string(FILE *fp, char *name, char *val)
{
  int len, nlines, pos, i;
  char line[100];
  len = strlen(val);
  nlines = len/LINELEN;
  if (nlines*LINELEN < len) nlines++;
  fprintf(fp, "%scharacter*%d %s\n", FINDENT, nlines*LINELEN, name);
  fprintf(fp, "%sparameter (%s = \n", FINDENT, name);
  for (i = 0; i < nlines; i++) {
    pos = i*LINELEN;
    if (i == 0) fprintf(fp, "%s\'", CONTINUE);
    else        fprintf(fp, "%s", CONTINUE);
    /* number should be same as LINELEN */
    fprintf(fp, "%.65s", val+pos);
    if (i == nlines-1) fprintf(fp, "\')\n");
    else             fprintf(fp, "\n");
  }
}

#endif


/* integer square root. Return error if argument isn't
 * a perfect square or is less than or equal to zero 
 */

int isqrt(int i)
{
  int root, square;
  if (i <= 0) return(-1);
  square = 0;
  for (root = 1; square <= i; root++) {
    square = root*root;
    if (square == i) return(root);
  }
  return(-1);
}
  

/* integer log base two. Return error is argument isn't
 * a power of two or is less than or equal to zero 
 */

int ilog2(int i)
{
  int log2;
  int exp2 = 1;
  if (i <= 0) return(-1);

  for (log2 = 0; log2 < 20; log2++) {
    if (exp2 == i) return(log2);
    exp2 *= 2;
  }
  return(-1);
}

int ipow2(int i)
{
  int pow2 = 1;
  if (i < 0) return(-1);
  if (i == 0) return(1);
  while(i--) pow2 *= 2;
  return(pow2);
}
 

