C#######################################################################
C PSTSWM Version 4.0 (12/1/94)                                         #
C  (Stripped down PVM-only version (4/13/95), for use in ParkBench     #
C   benchmark suite)                                                   #
C  A message-passing benchmark code and parallel algorithm testbed     #
C  that solves the nonlinear shallow water equations using the spectral#
C  transform method.                                                   #
C Written by:                                                          #
C  Patrick Worley of Oak Ridge National Laboratory                     #
C  Ian Foster of Argonne National Laboratory                           #
C Based on the sequential code STSWM 2.0 by James Hack and Ruediger    #
C  Jakob of the National Center for Atmospheric Research.              #
C Research and development funded by the Computer Hardware, Advanced   #
C  Mathematics, and Model Physics (CHAMMP) program of the U.S.         #
C  Department of Energy.                                               # 
C                                                                      #
C Questions and comments should be directed to worley@msr.epm.ornl.gov #
C Please notify and acknowledge the authors in any research or         #
C publications utilizing PSTSWM or any part of the code.               #
C                                                                      #
C NOTICE: Neither the institutions nor the authors make any            #
C representations about the suitability of this software for any       #
C purpose. This software is provided "as is", without express or       #
C implied warranty.                                                    #
C#######################################################################
C include precision declaration definitions                            #
#include "precision.i"
C#######################################################################
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C timing.f                                                             C
C                                                                      C
C The following routines are used to collect (or enable collection of) C
C performance data:                                                    C
C                                                                      C
C TINIT   -    used to ready the code for tracing and timing.          C
C TSTART  -    used to mark the beginning of a timestep calculattion.  C
C TSTOP   -    used to mark the end of a timestep calculation.         C
C TOSTART  -   used to mark the beginning of an overhead measurement.  C
C TOSTOP   -   used to mark the end of an overhead measurement.        C
C TEXIT   -    used to finish the tracing and timing calculation.      C
C TOUTPUT -    used to output the timing data.                         C
C                                                                      C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE TINIT
C
C This subroutine prepares the code for tracing and timing, if
C requested. It eliminates paging anomalies on the Paragon, turns off 
C PICL error checking, open and initializes trace files and tracing
C parameters, and synchronizes the processes (and their local clocks).
C
C called by: PSTSWM
C calls: PICL routines
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C     
C---- Parameters -------------------------------------------------------
C
#     include "params.i"
C     
C---- Common Blocks ----------------------------------------------------
C     
C machine architecture information
#     include "machine.i"
C problem resolution information
#     include "problem.i"
C parallel algorithm information
#     include "algorithm.i"
C constants & timesteps
#     include "consts.i"
C domain decomposition information
#     include "physical.i"
C mountain data
#     include "finit.i"
C time dependent fields
#     include "tdvars.i"
C performance measurement parameters and variables
#     include "perf.i"
C workspace
#     include "wrkspc.i"
C     
C---- Local Variables -------------------------------------------------
C     
C vector used to record problem and algorithm parameters in the trace 
C file
      INTEGER PVECTOR(19)
C
C---- External Functions -----------------------------------------------
C
C PICL clock
      EXTERNAL CLOCK0F
      DOUBLE PRECISION CLOCK0F
C     
C---- Executable Statements --------------------------------------------       
C     
      IF (TIMING) THEN
C
C       Eliminate timing anomalies due to virtual memory on the PARAGON:
C       hide paging start-up on the PARAGON by touching (most of) the
C       code and data storage once initially.
        NSTEP = 1
        CALL STEP(DT, WS(IALP), WS(IDALP), WS(IWTS), WS(IWTACSJ), 
     &            WS(IANNP1), WS(IA2NNP1),
     &            WS(ITRIGS), WS(IMOUNT), WS(ITOPOSC), 
     &            WS(IDIV), WS(IZETA), WS(IPHI), 
     &            WS(IUCOS), WS(IVCOS), WS(IWS1), WS(IWS2), WS(IWS3),
     &            WS(IDIVSC), WS(IZETASC), WS(IPHISC))
C
C       Reinitialize data.
        CALL TDVMOD(1, PHIBAR, -1, WS(IMOUNT), WS(ID), WS(IZ), WS(IH), 
     &              WS(IU), WS(IV), WS(IDSC), WS(IZSC), WS(IPSC), 
     &              WS(IDIV), WS(IZETA), WS(IPHI), WS(IUCOS), WS(IVCOS),
     &              WS(IDIVSC), WS(IZETASC), WS(IPHISC))
        NSTEP = 0
C
C       Initialize maximum/minimum execution times per timestep.
        TMAX = 0.0
        TMIN = 1000000.0
C
C       Initialize subroutine overhead estimate.
        TOVER = 0.0
C
      ENDIF
C
      IF (TRACING) THEN
C
C       Enable tracing and save problem and algorithm parameters in
C       the trace file .
        CALL CHECK0F(0)
        IF (TRACEFILE) CALL TRACEFILESF(TMPNAME, PERMNAME, VERBOSE)
        CALL TRACESTATISTICSF(610, 0, 0, 0, 0, 0)
        CALL TRACELEVELF(10, 10, 10)
        CALL TRACENODEF(TRSIZE, 0, 1)
        IF (ME .EQ. 0) THEN
C
          PVECTOR(1) = MM
          PVECTOR(2) = NN
          PVECTOR(3) = KK
          PVECTOR(4) = NLAT
          PVECTOR(5) = NLON
          PVECTOR(6) = NVER
          PVECTOR(7) = NALP
          PVECTOR(8) = (TAUE*3600.0)/DT
          PVECTOR(9) = ICOND
          IF (SITS) THEN
            PVECTOR(10) = 1
          ELSE
            PVECTOR(10) = 0
          ENDIF
          IF (FORCED) THEN
            PVECTOR(11) = 1
          ELSE
            PVECTOR(11) = 0
          ENDIF
          IF (MOMENT) THEN
            PVECTOR(12) = 1
          ELSE
            PVECTOR(12) = 0
          ENDIF
          CALL TRACEDATAF(7, 0, 'L', 12, PVECTOR)
C
          PVECTOR(1)  = NPROCS
          PVECTOR(2)  = NPLON_P
          PVECTOR(3)  = NPLAT_P
          PVECTOR(4)  = FTOPT
          PVECTOR(5)  = LTOPT
          PVECTOR(6)  = COMMFFT
          PVECTOR(7)  = COMMIFT
          PVECTOR(8)  = COMMFLT
          PVECTOR(9)  = COMMILT
          PVECTOR(10) = BUFSFFT
          PVECTOR(11) = BUFSIFT
          PVECTOR(12) = BUFSFLT
          PVECTOR(13) = BUFSILT
          PVECTOR(14) = PROTFFT
          PVECTOR(15) = PROTIFT
          PVECTOR(16) = PROTFLT
          PVECTOR(17) = PROTILT
          PVECTOR(18) = SUMOPT
          PVECTOR(19) = EXCHSIZE
          CALL TRACEDATAF(7, 1, 'L', 19, PVECTOR)
C
        ENDIF
C
        CALL TRACELEVELF(0, 0, 0)
C
      ELSEIF (TIMING) THEN
C
C       Synchronize before starting timing.
        CALL CHECK0F(0)
        CALL CLOCKSYNC0F()
C
      ENDIF
C
C     Start timing.
      IF (TIMING) TBEGIN = CLOCK0F()
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE TSTART
C
C This subroutine marks the beginning of a timestep calculation, both 
C for tracing and for timing.
C
C called by: PSTSWM
C calls: PICL routines
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C     
C---- Parameters -------------------------------------------------------
C
#     include "params.i"
C     
C---- Common Blocks ----------------------------------------------------
C     
C constants and timesteps
#     include "consts.i"
C performance measurement parameters and variables
#     include "perf.i"
C
C---- External Functions -----------------------------------------------
C
C PICL clock
      EXTERNAL CLOCK0F
      DOUBLE PRECISION CLOCK0F
C     
C---- Executable Statements --------------------------------------------       
C
      IF (TRACING) THEN
C
C       Turn on/off tracing.
        IF (NSTEP .EQ. TRSTART) CALL TRACELEVELF(TL1, TL2, TL3)
        IF (NSTEP .EQ. TRSTOP) CALL TRACELEVELF(0, 0, 0)
C
C       Mark beginning of next timestep calculation.
        CALL TRACEEVENTF('entry', 10+NSTEP, 0, 0)
        CALL TRACEEVENTF('entry', 0, 1, NSTEP)
C
      ENDIF
C
      IF (TIMING) TIN = CLOCK0F()
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE TSTOP
C
C This subroutine marks the end of a timestep calculation, both for
C tracing and for timing. 
C
C called by: PSTSWM
C calls: PICL routines
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C     
C---- Parameters -------------------------------------------------------
C
#     include "params.i"
C     
C---- Common Blocks ----------------------------------------------------
C     
C constants & timesteps
#     include "consts.i"
C performance measurement parameters and variables
#     include "perf.i"
C
C---- External Functions -----------------------------------------------
C
C PICL clock
      EXTERNAL CLOCK0F
      DOUBLE PRECISION CLOCK0F
C
C---- Local Variables -------------------------------------------------
C
C elapsed time for previous timestep calculation
      DOUBLE PRECISION TSTEP
C     
C---- Executable Statements --------------------------------------------       
C
      IF (TIMING) THEN
C
C       Calculate elapsed time.
        TSTEP = CLOCK0F() - TIN
C
C       Since first step is really two steps, halve the time.
        IF (NSTEP .EQ. 1) TSTEP = TSTEP/2
C
C       Save min and max.
        IF (TMIN .GT. TSTEP) TMIN = TSTEP
        IF (TMAX .LT. TSTEP) TMAX = TSTEP
C
      ENDIF
C
      IF (TRACING) THEN
        CALL TRACEEVENTF('exit', 0, 1, NSTEP)
        CALL TRACEEVENTF('exit', 10+NSTEP, 0, 0)
      ENDIF
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE TOSTART
C
C This subroutine marks the beginning of an overhead measurement.
C
C called by: (unused at present)
C calls: PICL routines
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C     
C---- Common Blocks ----------------------------------------------------
C     
C performance measurement parameters and variables
#     include "perf.i"
C
C---- External Functions -----------------------------------------------
C
C PICL clock
      EXTERNAL CLOCK0F
      DOUBLE PRECISION CLOCK0F
C     
C---- Executable Statements --------------------------------------------       
C
      TOVER = TOVER - CLOCK0F()
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE TOSTOP
C
C This subroutine marks the end of an overhead measurement.
C
C called by: (unused at present)
C calls: PICL routines
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C     
C---- Common Blocks ----------------------------------------------------
C     
C performance measurement parameters and variables
#     include "perf.i"
C
C---- External Functions -----------------------------------------------
C
C PICL clock
      EXTERNAL CLOCK0F
      DOUBLE PRECISION CLOCK0F
C     
C---- Executable Statements --------------------------------------------       
C
      TOVER = TOVER + CLOCK0F()
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE TEXIT
C
C This subroutine completes the timing and tracing of PSTSWM.
C
C called by: PSTSWM
C calls: PICL routines
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C     
C---- Parameters -------------------------------------------------------
C
#     include "params.i"
C     
C---- Common Blocks ----------------------------------------------------
C     
C performance measurement parameters and variables
#     include "perf.i"
C
C---- External Functions -----------------------------------------------
C
C PICL clock
      EXTERNAL CLOCK0F
      DOUBLE PRECISION CLOCK0F
C     
C---- Executable Statements --------------------------------------------       
C
C     Finish timing and tracing.
      IF (TIMING)  TTOTAL = CLOCK0F() - TBEGIN
      IF (TRACING) CALL TRACEEXITF()
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE TOUTPUT
C
C This subroutine outputs the timing information.
C
C called by: PSTSWM
C calls: PICL routines
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C     
C---- Parameters -------------------------------------------------------
C
#     include "params.i"
C     
C---- Common Blocks ----------------------------------------------------
C     
C machine architecture information
#     include "machine.i"
C problem resolution information
#     include "problem.i"
C parallel algorithm information
#     include "algorithm.i"
C constants & timesteps
#     include "consts.i"
C domain decomposition information
#     include "physical.i"
C performance measurement parameters and variables
#     include "perf.i"
C workspace
#     include "wrkspc.i"
C     
C------Local Variables ------------------------------------------------
C     
c base for message types used in the output of the timing information
      INTEGER BASE
C message buffer
      real*8 TIMES(3)
C switch indicating whether the file where the timing information is to 
C saved has been created yet (0: yes, 1: no)
      LOGICAL NEWFILE
C
C---- External Functions ----------------------------------------------
C
C PICL clock
      EXTERNAL CLOCK0F
      DOUBLE PRECISION CLOCK0F
C offset for message types
      EXTERNAL MSGBASE
      INTEGER MSGBASE
C     
C---- Executable Statements --------------------------------------------       
C
C     Calculate min, max, and total times over all processors.
      IF (TIMING) THEN
C
C       Calculate minimum per processor execution time for a single time
C       step.
        BASE = MSGBASE()
        CALL GMIN0F(TMIN, 1, 5, BASE, 0)
C
C       Calculate maximum per processor execution time for a single time
C       step, for the total time, and for the overhead.
        BASE = MSGBASE()
        TIMES(1) = TMAX
        TIMES(2) = TTOTAL
        TIMES(3) = TOVER
        CALL GMAX0F(TIMES, 3, 5, BASE, 0)
C
        IF (ME .EQ. 0) THEN
C         Write out timing data 
C
          TMAX   = TIMES(1)
          TTOTAL = TIMES(2)
          TOVER  = TIMES(3)
C
          NEWFILE = .TRUE.
          OPEN(8, ERR=996, FILE=TIMEFILE, STATUS='old')
          NEWFILE = .FALSE.
 996      IF (NEWFILE) THEN
C           This is a new file, so write the header line.
            OPEN(8, FILE=TIMEFILE)
            REWIND 8
            WRITE(8,997) 
 997        FORMAT('   total     ','     min     ','     max     ',
     &             '  overhead   ',
     &             ' nstep', ' spectral space ', ' physical space ', 
     &             ' problem ',' processors  ',' map ',
     &             '      parallel ft       ', 
     &             '      parallel lt       ', ' sum ',' exchsize ',
     &             ' workspace ')
          ELSE
C           This is an existing file, so find the end of file.
            REWIND 8
            DO WHILE (.TRUE.) 
              READ(8,995,END=998)
 995          FORMAT(/)
            ENDDO
 998        BACKSPACE 8
          ENDIF
C
          WRITE(8,999) TTOTAL, TMIN, TMAX, TOVER, NSTEP, 
     &                 MM, NN, KK, NLON, NLAT, NVER, 
     &                 ICOND, SITS, FORCED, MOMENT, 
     &                 NPLON_P, NPLAT_P, MESHOPT,
     &                 FTOPT, COMMFFT, COMMIFT, BUFSFFT, BUFSIFT, 
     &                 PROTFFT, PROTIFT,
     &                 LTOPT, COMMFLT, COMMILT, BUFSFLT, BUFSILT,
     &                 PROTFLT, PROTILT, SUMOPT, EXCHSIZE, IWS4
 999      FORMAT(E12.5, 1X, E12.5, 1X, E12.5, 1X, E12.5, 1X, I5, 2X,
     &          '(',I3,',',I3,',',I3,')',2X,
     &          '(',I4,',',I4,',',I3,')',1X,
     &          '(',I1,',',L1,',',L1,',',L1,')',1X,
     &          '(',I4,',',I4,')',2X,I2,2X,
     &          '(',I1,',',I2,',',I2,',',I4,',',I4,',',I1,',',I1,')',1X,
     &          '(',I1,',',I2,',',I2,',',I4,',',I4,',',I1,',',I1,')',3X,
     &           I1,2X,I8,2X,I9,/)
          CLOSE(8)
        ENDIF
      ENDIF
C
      IF (TRACING) THEN
C
C       Update clock normalization parameters and resynchronize before
C       outputing trace date
        CALL CLOCKSYNC0F()
C
C       Write out trace data one processor at a time.
        CALL SETDATA0F('CHARACTER')
        BASE = MSGBASE()
        IF (ME .NE. 0) THEN
          CALL RECV0F(TIMES, 1, BASE+ME-1)
        ENDIF
C
        CALL TRACEFLUSHF()
C
        IF (ME .NE. NPROCS-1) THEN
          CALL SEND0F(TIMES, 1, BASE+ME, ME+1)
        ENDIF
C
      ENDIF
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
