/*
C#######################################################################
C PSTSWM Version 4.0 (12/1/94)                                         #
C  (Stripped down PVM-only version (4/13/95), for use in ParkBench     #
C   benchmark suite)                                                   #
C  A message-passing benchmark code and parallel algorithm testbed     #
C  that solves the nonlinear shallow water equations using the spectral#
C  transform method.                                                   #
C Written by:                                                          #
C  Patrick Worley of Oak Ridge National Laboratory                     #
C  Ian Foster of Argonne National Laboratory                           #
C Based on the sequential code STSWM 2.0 by James Hack and Ruediger    #
C  Jakob of the National Center for Atmospheric Research.              #
C Research and development funded by the Computer Hardware, Advanced   #
C  Mathematics, and Model Physics (CHAMMP) program of the U.S.         #
C  Department of Energy.                                               # 
C                                                                      #
C Questions and comments should be directed to worley@msr.epm.ornl.gov #
C Please notify and acknowledge the authors in any research or         #
C publications utilizing PSTSWM or any part of the code.               #
C                                                                      #
C NOTICE: Neither the institutions nor the authors make any            #
C representations about the suitability of this software for any       #
C purpose. This software is provided "as is", without express or       #
C implied warranty.                                                    #
C#######################################################################
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C pvmpicl.c                                                            C
C                                                                      C
C The following support routines are used by picl.F to implement       C
C PICL 2.0 commands using PVM commands:                                C
C                                                                      C
C open0     - entry points to envopen                                  C
C envopen   - spawn processes, initialize TIDS array, and create       C
C              pstswm group                                            C
C gettids   - return TIDS array                                        C
C                                                                      C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
*/

#define MAXPROCS 1024
#define HOST_INIT 999999994
#include <stdio.h>
#include "pvm3.h"

int envnpa;
int envtids[MAXPROCS];

/*CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC*/
/* These subroutines are entry points into envopen, which spawns      */
/* PSTSWM processes, initialize the TIDS array, and creates the pstswm*/
/* group.                                                             */
/*                                                                    */
/* called by: OPEN0F                                                  */
/*                                                                    */
void open0(numprocs, me, host)
int *numprocs, *me, *host;
{
  envOPEN(numprocs, me, host);
}

void OPEN0(numprocs, me, host)
int *numprocs, *me, *host;
{
  envOPEN(numprocs, me, host);
}

void open0_(numprocs, me, host)
int *numprocs, *me, *host;
{
  envOPEN(numprocs, me, host);
}

void _open0(numprocs, me, host)
int *numprocs, *me, *host;
{
  envOPEN(numprocs, me, host);
}

/* open communication channel at node */
envOPEN(numprocs, me, host)
int *numprocs, *me, *host;
{
  int j, mypid, parenttid, spawn;
  int nhosts, formats, ntasks, inum, info;
  char *s, *getenv();
  char file[100];
  FILE *inputfp;

  /* initialize host-independent info */
  mypid = pvm_mytid();

  /* get parent tid */
  parenttid = pvm_parent();

  /* determine whether need to spawn processes, */
  s = getenv("PVM_EXPORT");
  if (s == NULL) spawn = 1;
    else{
    if (s[0] != 'N') spawn = 1;
      else spawn = 0;
    };

  /* open communication channel */
  if (spawn == 1){
    /* "node 0" */

    /* define this process to be node 0 */
    envtids[0] = mypid;
    *me = 0;

    /* create and join pstswm group */
    inum = pvm_joingroup("pstswm");
    if (inum != *me){ 
      printf("error in node %d joining pstswm group: %d\n", *me, inum);
      fflush(stdout);
      pvm_exit();
      exit(1);
      };

    /* tell other tasks that they were spawned by a node task, */
    /* and that stderr and stdout messages should be sent to stderr or */
    /* stdout for this process. */
    putenv("PVM_EXPORT=N");
    pvm_catchout(stdout);

    /* get number of instances to run: */
    /* first, look for an input file */
    inputfp = fopen("pvm3/pstswm_input", "r");
    if ((inputfp == NULL) && (parenttid >= 0)){
      printf(
"File ~/pvm3/pstswm_input needed to run PSTSWM from PVM console - exiting.\n");
      fflush(stdout);
      pvm_exit();
      exit(1);
      };

    if (inputfp != NULL){

      /* if there, read from input file */
      printf(
        "Reading number of processes to spawn from pvm3/pstswm_input file\n");
      fflush(stdout);
      fscanf(inputfp,"%d",&envnpa);

      if ((envnpa <= 0) || (envnpa > MAXPROCS)){
        printf(
         "pstswm_input does not contain a legal number of processes to spawn");
        printf(" - exiting\n");
        fflush(stdout);
        pvm_exit();
        exit(1);
        };

      }
      else{

      /* otherwise, read from stdin */
      envnpa = 0;
      while (envnpa <= 0){
        printf("Enter number of processes:\n");
        fflush(stdout);
        scanf("%d",&envnpa);
        if ((envnpa <= 0) || (envnpa > MAXPROCS)){
          printf(
            "%d processes can not be spawned. Please enter a legal value.\n",
            envnpa);
          fflush(stdout);
          envnpa = 0;
          };
        };

      };

    /* spawn tasks */
    if (envnpa > 1){

      /* determine number of hosts */
      info = pvm_config(&nhosts, &formats, NULL);
      if (info < 0){
        pvm_perror(NULL);
        printf("pvm_config error %d - exiting\n",info);
        fflush(stdout);
        pvm_exit();
        exit(1);
        };

      if (inputfp != NULL){
        /* get name from input file */
        printf(
         "Reading name of current executable from pvm3/pstswm_input file\n");
        fflush(stdout);
        fscanf(inputfp,"%s",file);
        }
        else{
        /* get name from stdin */
        printf("Enter name of current executable:\n");
        fflush(stdout);
        scanf("%s",file);
        };

      if (nhosts > 1)
        ntasks = pvm_spawn(file, (char**)0, 33, ".", envnpa-1, &envtids[1]); 
        else
        ntasks = pvm_spawn(file, (char**)0, 0, "", envnpa-1, &envtids[1]); 

      if (ntasks != envnpa-1){
        if (ntasks < 0) ntasks = 1;
          else ntasks = ntasks + 1;
        j=1;
        while ((envtids[j] >= 0) && (j < envnpa-1)) j++;
        printf(
"only %d instances of %s spawned (%d requested): tid[%d]=%d - exiting\n",
         ntasks, file, envnpa, j, envtids[j]);
        fflush(stdout);
        pvm_exit();
        exit(1);
        };

      /* broadcast tid array and other information to all processes */
      pvm_initsend(PvmDataDefault);
      pvm_pkint(&envnpa, 1, 1);
      pvm_pkint(envtids, envnpa, 1);
      for (j=1; j<envnpa; j++){
        pvm_send(envtids[j], HOST_INIT);
        pvm_recv(envtids[j], HOST_INIT);
        };

      };

    }
    else{

    /* get envnpa and envtids info */
    pvm_recv(parenttid, HOST_INIT);
    pvm_upkint(&envnpa, 1, 1);
    pvm_upkint(envtids, envnpa, 1);

    /* calculate me */  
    for (j=0; j<envnpa; j++) 
      if (mypid == envtids[j]){
        *me = j;
        break;
        };

    /* join the pstswm group */
    inum = pvm_joingroup("pstswm");
    if (inum != *me){
      printf("error in node %d joining pstswm group: %d\n", *me, inum);
      fflush(stdout);
      pvm_exit();
      exit(1);
      };

    /* tell node 0 that have joined pstswm */
    pvm_initsend(PvmDataDefault);
    pvm_send(parenttid, HOST_INIT);

    };

  /* return envnpa */
  *numprocs = envnpa;

  /* set host */
  *host = 32767;

}

/*CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC*/
/* These subroutines return the TIDS array values.                    */
/*                                                                    */
/* called by: COMMINIT, OPEN0F                                        */
/*                                                                    */
void gettids(tids)
int *tids;
{
  int i;

  for (i=0; i<envnpa; i++)
    tids[i] = envtids[i];
}

void GETTIDS(tids)
int *tids;
{
  int i;

  for (i=0; i<envnpa; i++)
    tids[i] = envtids[i];
}

void gettids_(tids)
int *tids;
{
  int i;

  for (i=0; i<envnpa; i++)
    tids[i] = envtids[i];
}

void _gettids(tids)
int *tids;
{
  int i;

  for (i=0; i<envnpa; i++)
    tids[i] = envtids[i];
}

