C#######################################################################
C PSTSWM Version 4.0 (12/1/94)                                         #
C  (Stripped down PVM-only version (4/13/95), for use in ParkBench     #
C   benchmark suite)                                                   #
C  A message-passing benchmark code and parallel algorithm testbed     #
C  that solves the nonlinear shallow water equations using the spectral#
C  transform method.                                                   #
C Written by:                                                          #
C  Patrick Worley of Oak Ridge National Laboratory                     #
C  Ian Foster of Argonne National Laboratory                           #
C Based on the sequential code STSWM 2.0 by James Hack and Ruediger    #
C  Jakob of the National Center for Atmospheric Research.              #
C Research and development funded by the Computer Hardware, Advanced   #
C  Mathematics, and Model Physics (CHAMMP) program of the U.S.         #
C  Department of Energy.                                               # 
C                                                                      #
C Questions and comments should be directed to worley@msr.epm.ornl.gov #
C Please notify and acknowledge the authors in any research or         #
C publications utilizing PSTSWM or any part of the code.               #
C                                                                      #
C NOTICE: Neither the institutions nor the authors make any            #
C representations about the suitability of this software for any       #
C purpose. This software is provided "as is", without express or       #
C implied warranty.                                                    #
C#######################################################################
C include precision declaration definitions                            #
#include "precision.i"
C#######################################################################
      SUBROUTINE ANLYTC(TIME, CASE, PHICON, UCON, MOUNT, UIC12, VIC12,
     &                  PIC12, DIC12, EIC12, DICLL, EICLL, PICLL, UICLL,
     &                  VICLL)
C
C This procedure returns the analytic solution for the test cases:
C  case 1: advection equation for solid body flow
C  case 2: solid body rotation steady state flow
C  case 3: jetstream steady state flow
C  case 4: forced low in jetstream
C  case 5: zonal flow over isolated mountain
C  case 6: Rossby-Haurwitz wave
C It is called for the initialization of the prognostic and analytic 
C fields and during error analysis. The results are returned in the 
C arrays UICLL, VICLL, PICLL, DICLL, and EICLL. The routine uses many 
C values from the common blocks /CONST/ and /FINIT/ which were computed
C in routines INPUT and INIT. 
C                                                                              
C called by: ERRANL, PSTSWM
C calls: GLAT_P, GLON_P, ROTATE
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Parameters -------------------------------------------------------
C
#     include "params.i"
C
C---- Common Blocks ----------------------------------------------------
C
C machine architecture information
#     include "machine.i"
C domain decomposition information
#     include "physical.i"
C constants & timesteps
#     include "consts.i"
C initial conditions
#     include "finit.i"
C
C---- Arguments --------------------------------------------------------
C
C     Input
C
C model time in seconds
      REAL TIME
C test case number
      INTEGER CASE
C balanced PHI (PHICON) and UCON (used for 
C geopotential that balances steady zonal flow)
      REAL PHICON(NLLAT_P)
      REAL UCON(NLLAT_P)
C mountain height
      REAL MOUNT(NLLON_P,NLLAT_P)
C initial U, V wind
      REAL UIC12(NLLON_P,NLLAT_P)
      REAL VIC12(NLLON_P,NLLAT_P)
C initial height
      REAL PIC12(NLLON_P,NLLAT_P)
C initial divergence and vorticity
      REAL DIC12(NLLON_P,NLLAT_P)
      REAL EIC12(NLLON_P,NLLAT_P)
C
C     Output
C
C divergence on grid
      REAL DICLL(NLLON_P,NLLAT_P)
C vorticity on grid
      REAL EICLL(NLLON_P,NLLAT_P)
C height on grid
      REAL PICLL(NLLON_P,NLLAT_P)
C eastward wind velocity on grid
      REAL UICLL(NLLON_P,NLLAT_P)
C northward wind velocity on grid
      REAL VICLL(NLLON_P,NLLAT_P)
C
C---- Local Variables --------------------------------------------------
C
C "constant" variables: pi
      REAL PI
C test case 1 temporaries
      REAL RADIUS, DIST
C longitudes and longitude loop index
      REAL RLONA, RRLONA, RLON
      INTEGER I
C latitudes and latitude loop index
      REAL RLATA, RRLATA, RLAT
      INTEGER J
C test case 4 temporaries (see code for description)
      REAL TMSHFT, AI, A2I, SNJ, CSJ, SRCSJ, TMPRY, TMPRY2, DEN,
     &     AACSJI, CORR, BIGUBR, DBUB, C, PSIB, DCDM, DCDL, D2CDM,
     &     D2CDL, TMP1, TMP2, DKDM, DKDL, D2KDM, D2KDL, DLON, SINT,
     &     COST
C test case 6 temporaries (see code for description)
      REAL PHIA, PHIB, PHIC
C
C---- External Functions -----------------------------------------------
C
C latitude / longitude grid coordinate functions
      EXTERNAL GLAT_P, GLON_P
      REAL GLAT_P, GLON_P
C
C---- Statement Functions ----------------------------------------------
C
C zonal flow functions
#     include "bubfnc.i"
C
C---- Executable Statements --------------------------------------------
C
C     Calculate "constants"
      PI = 4.0D0*ATAN(1.0D0)
C
      IF (CASE .EQ. 1) THEN
C       ## initial condition 1 ##
C       Copy data for U, V, ZETA, DIV since steady flow field.
C       Compute location of advected height.
C
        RLATA = RLAT0
        RLONA = RLON0 + SU0*COS(RLAT0)*TIME/A
        CALL ROTATE(RLONA, RLATA, -ALPHA, RRLONA, RRLATA)
C
C       size of feature
        RADIUS = A/3.0
C
        DO J=1,NLLAT_P
C         latitude = RLAT = GLAT_P(J)
          RLAT = GLAT_P(J)
          DO I=1,NLLON_P
C           longitude = RLON = GLON_P(I) 
            RLON = GLON_P(I)
            DICLL(I,J) = DIC12(I,J)
            EICLL(I,J) = EIC12(I,J)
            UICLL(I,J) = UIC12(I,J)
            VICLL(I,J) = VIC12(I,J)
C
C           construct advected low
            DIST = A*ACOS(SIN(RRLATA)*SIN(RLAT) + COS(RRLATA)
     &          *COS(RLAT)*COS(RLON-RRLONA))
            IF (DIST .LE. RADIUS) THEN
               PICLL(I,J) = (PHI0/2.0)
     &                      *(1.0 + COS(PI*DIST/RADIUS))
            ELSE
               PICLL(I,J) = 0.0
            ENDIF
C
          ENDDO
        ENDDO
C
      ELSEIF ((CASE .EQ. 2) .OR. (CASE .EQ. 3)) THEN
C       ## initial conditions 2 and 3 ##
C       Copy initial data, since steady state solution.
C
        DO J=1,NLLAT_P
          DO I=1,NLLON_P  
            DICLL(I,J) = DIC12(I,J)
            EICLL(I,J) = EIC12(I,J)
            UICLL(I,J) = UIC12(I,J)
            VICLL(I,J) = VIC12(I,J)
            PICLL(I,J) = PIC12(I,J)
          ENDDO
        ENDDO
C
      ELSEIF (CASE .EQ. 4) THEN
C       ## condition 4 ##
C       Calculate analytic solution to forced nonlinear problem
C
C       longitudinal change of low in basic flow
        TMSHFT =  SU0*TIME/A
C
        DO J=1,NLLAT_P
C         latitude = RLAT = GLAT_P(J)
          RLAT = GLAT_P(J)
C
C         temporary variables independent of longitude
          AI     = 1.0/A  
          A2I    = 1.0/(A*A) 
          SNJ    = SIN(RLAT)
          CSJ    = COS(RLAT)*COS(RLAT)
          SRCSJ  = COS(RLAT)
          TMPRY  = TAN(RLAT)
          TMPRY2 = TMPRY*TMPRY
          DEN    = 1.0/COS(RLAT)
          AACSJI = 1.0/(A*A*CSJ)
          CORR   = 2.0*OMEGA*SNJ
C
C         nonlinear steady zonal flow
          BIGUBR = UCON(J)*SRCSJ
          DBUB   = DBUBF(SIN(RLAT),COS(RLAT))
C
          DO I=1,NLLON_P  
C           longitude = RLON = GLON_P(I)
            RLON = GLON_P(I)
C
C           compute location of translating low
            C      = SIN(RLAT0)*SNJ 
     &             + COS(RLAT0)*SRCSJ*COS(RLON-TMSHFT-RLON0)
            PSIB   = ALFA*EXP(-SIGMA*((1.0-C)/(1.0+C)))
C
C           compute partial derivatives of C
            DCDM   = SIN(RLAT0) - COS(RLON-TMSHFT-RLON0)
     &             * COS(RLAT0)*TMPRY
            DCDL   = -COS(RLAT0)*SRCSJ*SIN(RLON-TMSHFT-RLON0)
            D2CDM  = -COS(RLAT0)*COS(RLON-TMSHFT-RLON0)
     &             * (1.0 + TMPRY2)/SRCSJ
            D2CDL  = -COS(RLAT0)*SRCSJ*COS(RLON-TMSHFT-RLON0)
C
C           compute partial derivatives of PSI BAR
            TMP1   = 2.0*SIGMA*PSIB/((1.0 + C)**2)
            TMP2   = (SIGMA - (1.0 + C))/((1.0 + C)**2)
            DKDM   = TMP1*DCDM
            DKDL   = TMP1*DCDL
            D2KDM  = TMP1*(D2CDM + 2.0*(DCDM**2)*TMP2)
            D2KDL  = TMP1*(D2CDL + 2.0*(DCDL**2)*TMP2)
C
C           analytic solutions
            DICLL(I,J) = 0.0
            EICLL(I,J) = D2KDL*AACSJI + CORR - DBUB*AI
     &                 + (CSJ*D2KDM - 2.0*SNJ*DKDM)*A2I 
            UICLL(I,J) = BIGUBR*DEN - SRCSJ*AI*DKDM 
            VICLL(I,J) = (DKDL*AI)*DEN 
            PICLL(I,J) = PHICON(J)+CORR*PSIB/GRAV
          ENDDO
        ENDDO
C
      ELSEIF ((CASE .EQ. 5) .AND. (TIME .EQ. 0.0)) THEN
C       ## initial condition 5 ##
C       zonal flow over isolated mountain
C
C       copy initial data 
        DO J=1,NLLAT_P
          DO I=1,NLLON_P
            DICLL(I,J) = DIC12(I,J)
            EICLL(I,J) = EIC12(I,J)
            UICLL(I,J) = UIC12(I,J)
            VICLL(I,J) = VIC12(I,J)
            PICLL(I,J) = PIC12(I,J) 
          ENDDO
        ENDDO
C
      ELSEIF ((CASE .EQ. 6) .AND. (TIME .EQ. 0.0)) THEN
C       ## initial condition 6 ##
C       Rossby-Haurwitz wave
C
C       longitudinal change of feature
        DLON = (R*(3+R)*OMG - 2.0*OMEGA)/((1+R)*(2+R))*TIME
C
        DO J=1,NLLAT_P
C         latitude = RLAT = GLAT_P(J)
          RLAT = GLAT_P(J)
C
C         Compute latitude-dependent factors for geopotential
C         PHIA, PHIB and PHIC
          SINT = SIN(RLAT)
          COST = COS(RLAT)
          PHIA = 0.5*OMG*(2.0*OMEGA+OMG)*COST*COST
     &         + 0.25*K*K*COST**(2*R)
     &         * ((R+1)*COST*COST+(2*R*R-R-2) 
     &           - 2.0*R*R/(COST*COST))
          PHIB = (2.0*(OMEGA+OMG)*K)/((R+1)*(R+2))
     &         * COST**R*((R*R+2*R+2)-(R+1)**2*COST*COST)
          PHIC = 0.25*K*K*COST**(2*R)
     &         * ((R+1)*COST*COST-(R+2))
C
          DO I=1,NLLON_P
C           longitude = RLON = GLON_P(I)
            RLON = GLON_P(I)
            DICLL(I,J) = 0.0
            EICLL(I,J) = 2.0*(OMG+OMEGA)*SINT - (1+R)*(2+R)*SINT
     &                 * K*COST**R*COS(R*(RLON-DLON))
            UICLL(I,J) = A*OMG*COST + A*K*COST**(R-1)
     &                 * (R*SINT*SINT-COST*COST)*COS(R*(RLON-DLON))
            VICLL(I,J) = -A*K*R*COST**(R-1)*SINT
     &                 * SIN(R*(RLON-DLON))
            PICLL(I,J) = PHI0 + (A*A*(PHIA+PHIB
     &                 * COS(R*(RLON-DLON))+PHIC
     &                 * COS(2*R*(RLON-DLON))))/GRAV
          ENDDO
        ENDDO
C
      ELSE
C
        IF (ME .EQ. 0) WRITE (0,300) CASE
  300   FORMAT (/,' PSTSWM: FATAL ERROR IN ANLYTC: ',/,
     &          ' NO ANALYTIC SOLUTION PROVIDED FOR TEST CASE ',I2,/)
        STOP
C
      ENDIF
C
      RETURN
      END

