C#######################################################################
C PSTSWM Version 4.0 (12/1/94)                                         #
C  A message-passing benchmark code and parallel algorithm testbed     #
C  that solves the nonlinear shallow water equations using the spectral#
C  transform method.                                                   #
C Written by:                                                          #
C  Patrick Worley of Oak Ridge National Laboratory                     #
C  Ian Foster of Argonne National Laboratory                           #
C Based on the sequential code STSWM 2.0 by James Hack and Ruediger    #
C  Jakob of the National Center for Atmospheric Research.              #
C Research and development funded by the Computer Hardware, Advanced   #
C  Mathematics, and Model Physics (CHAMMP) program of the U.S.         #
C  Department of Energy.                                               # 
C                                                                      #
C Questions and comments should be directed to worley@msr.epm.ornl.gov #
C Please notify and acknowledge the authors in any research or         #
C publications utilizing PSTSWM or any part of the code.               #
C                                                                      #
C NOTICE: Neither the institutions nor the authors make any            #
C representations about the suitability of this software for any       #
C purpose. This software is provided "as is", without express or       #
C implied warranty.                                                    #
C#######################################################################
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C sendrecv.f                                                           C
C                                                                      C
C The following routines implement the following low level             C
C interprocessor communication commands used in PSTSWM using           C
C MPI interprocessor communication commands:                           C
C                                                                      C
C SENDRECV - used in algorithms where sends and receives come in       C
C            logical pairs, for example, when shifting data around a   C
C            ring.                                                     C
C SRBEGIN  - used when initiating a SENDRECV                           C
C SREND    - used when completing a SENDRECV initiated by              C
C            SRBEGIN                                                   C
C SR1      - first of three routines which implement SENDRECV          C
C SRSAFE   - optional companion to SR1 that allows SAFEFORCE           C
C            handshaking to be postponed                               C
C SR2      - second of three routines which implement SENDRECV         C
C SR3      - third of three routines which implement SENDRECV          C
C                                                                      C
C These routines all have the following communication options:         C
C  if (COMMOPT .EQ. 0) simple                                          C
C  if (COMMOPT .EQ. 1) ordered                                         C
C They also have the following communication protocol options:         C
C  if (PROTOPT .EQ. 1, 3, .OR. 5)     nonblocking send                 C
C  if (PROTOPT .EQ. 2, 3, 4, .OR. 5)  nonblocking receive              C
C  if (PROTOPT .EQ. 4 .OR. 5)         forcetype                        C
C  if (PROTOPT .EQ. 6 .AND. COMMOPT .EQ. 0) native                     C
C  if (PROTOPT .EQ. 6 .AND. COMMOPT .EQ. 1) synchronous                C
C                                                                      C
C SRBEGIN/SREND and SR1(/SRSAFE)/SR2/SR3 represent different ways of   C
C partitioning the basic SENDRECV routine                              C
C                                                                      C
C Use of SRBEGIN/SREND and SR1/SR2/SR3 allows receives used in         C
C SENDRECV to be posted ahead of time, and sends and/or receives       C
C to be completed just before they are needed.                         C
C                                                                      C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE SENDRECV(COMMOPT, PROTOPT, ORDER, ME, MTAG,
     &                    SNDDST, SNDLTH, SNDMSG, RCVSRC, RCVLTH,
     &                    RCVMSG)
C
C This subroutine sends the message in the SNDMSG buffer to SNDDST,
C and receives a message into RCVMSG. 
C
C Communication options (COMMOPT) for SENDRECV include:
C  if (COMMOPT .EQ. 0) simple SENDRECV: send/recv
C  if (COMMOPT .EQ. 1) ordered SENDRECV: [send/recv]|[recv/send]
C Communication protocol options (PROTOPT) for SENDRECV include:
C  if (PROTOPT .EQ. 1, 3, .OR. 5)     nonblocking send   
C  if (PROTOPT .EQ. 2, 3, 4, .OR. 5)  nonblocking receive
C  if (PROTOPT .EQ. 4 .OR. 5)         forcetype          
C  if (PROTOPT .EQ. 6 .AND. COMMOPT .EQ. 0) native
C  if (PROTOPT .EQ. 6 .AND. COMMOPT .EQ. 1) synchronous  
C
C called by: RING1, SRTRNS1
C calls: MPI routines
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Parameters -------------------------------------------------------
C
C mpi implementation-specific parameters
#     include "mpif.h"
C
C forcetype message tag offset
      INTEGER FORCETYPE
      PARAMETER (FORCETYPE = 15380)
C
C---- Common Blocks ----------------------------------------------------
C
C picl/mpi interface variables
#     include "picl.i"
C
C---- Arguments --------------------------------------------------------
C
C     Input 
C
C Communication algorithm option
      INTEGER COMMOPT
C communication protocol option 
      INTEGER PROTOPT
C order of send/recv
      INTEGER ORDER
C processor id
      INTEGER ME
C message tag offset 
C (MTAG+NPROCS should be < FORCETYPE to guarantee correctness)
      INTEGER MTAG
C message destination
      INTEGER SNDDST
C length of message to be sent
      INTEGER SNDLTH
C message to be sent
      INTEGER SNDMSG(*)
C source of message to be received
      INTEGER RCVSRC
C length of message to be received
      INTEGER RCVLTH
C
C     Output
C
C message received
      INTEGER RCVMSG(*)
C
C---- Local Variables --------------------------------------------------
C
C message request IDs for nonblocking communication calls
      INTEGER RCVID, SNDID
C error return variables
      INTEGER IERR0, IERR1, IERR2, IERR3, IERR4
C array for status values
      INTEGER STATUS(MPI_STATUS_SIZE)
C
C---- Data Statements -------------------------------------------------
C
      DATA IERR0 /0/, IERR1 /0/, IERR2 /0/, IERR3 /0/, IERR4 /0/
C
C---- Executable Statements --------------------------------------------
C
C     Choose communication algorithm.
      IF (COMMOPT .EQ. 0) THEN
C***********************************************************************
C       simple SENDRECV: send/recv
C***********************************************************************
C
C       Choose communication protocol.
        IF (PROTOPT .LE. 1) THEN
C
          IF (PROTOPT .EQ. 0) THEN
C           simple SENDRECV
            CALL MPI_BSEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                     SNDDST, MTAG, COMM, IERR1)
            DO WHILE (IERR1 .GT. 0)
              CALL MPI_ERROR_CLASS(IERR1, IERR2, IERR0)
              IF (IERR2 .EQ. MPI_ERR_BUFFER) THEN
                CALL MPI_BSEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                         SNDDST, MTAG, COMM, IERR1)
              ELSE
                IERR1 = -IERR1
              ENDIF
            ENDDO
            CALL MPI_RECV(RCVMSG, RCVLTH/DATALTH, DATATYPE,
     &                     RCVSRC, MTAG, COMM, STATUS, IERR2)
          ELSE
C           Do not block for send, increasing odds that receive will
C           be posted before the message arrives.
            CALL MPI_ISEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                     SNDDST, MTAG, COMM, SNDID, IERR1)
            CALL MPI_RECV(RCVMSG, RCVLTH/DATALTH, DATATYPE,
     &                     RCVSRC, MTAG, COMM, STATUS, IERR2)
            CALL MPI_WAIT(SNDID, STATUS, IERR1)
          ENDIF
C
        ELSEIF (PROTOPT .LE. 3) THEN
C    
C         IF (PROTOPT .EQ. 2) THEN
C           Post receive before send, increasing odds that receive will 
C           be posted before message arrives.
            CALL MPI_IRECV(RCVMSG, RCVLTH/DATALTH, DATATYPE,
     &                     RCVSRC, MTAG, COMM, RCVID, IERR2)
            CALL MPI_SEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                     SNDDST, MTAG, COMM, IERR1)
            CALL MPI_WAIT(RCVID, STATUS, IERR2)
C         ELSE
C           This case is "included" for completeness. There is no
C           advantage to using sendbegin/end here, so code is commented
C           out. There is an advantage to this case in SRBEGIN, for
C           example. 
C           CALL MPI_IRECV(RCVMSG, RCVLTH/DATALTH, DATATYPE,
C    &                     RCVSRC, MTAG, COMM, RCVID, IERR2)
C           CALL MPI_ISEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
C    &                     SNDDST, MTAG, COMM, SNDID, IERR1)
C           CALL MPI_WAIT(RCVID, STATUS, IERR2)
C           CALL MPI_WAIT(SNDID, STATUS, IERR1)
C         ENDIF
C
        ELSEIF (PROTOPT .LE. 5) THEN
C    
C         IF (PROTOPT .EQ. 4) THEN
C           Post receive before send to allow use of forcetypes.
            CALL MPI_IRECV(RCVMSG, RCVLTH/DATALTH, DATATYPE,
     &                     RCVSRC, MTAG+FORCETYPE, COMM, RCVID, IERR2)
            CALL MPI_BSEND(RCVMSG, 0, DATATYPE, 
     &                     RCVSRC, MTAG, COMM, IERR3)
            DO WHILE (IERR3 .GT. 0)
              CALL MPI_ERROR_CLASS(IERR3, IERR4, IERR0)
              IF (IERR4 .EQ. MPI_ERR_BUFFER) THEN
                CALL MPI_BSEND(RCVMSG, 0, DATATYPE, 
     &                         RCVSRC, MTAG, COMM, IERR3)
              ELSE
                IERR3 = -IERR3
              ENDIF
            ENDDO
            CALL MPI_RECV(SNDMSG, 0, DATATYPE,
     &                     SNDDST, MTAG, COMM, STATUS, IERR4)
            CALL MPI_RSEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                     SNDDST, MTAG+FORCETYPE, COMM, IERR1)
            CALL MPI_WAIT(RCVID, STATUS, IERR2)
C         ELSE
C           This case is "included" for completeness. There is no
C           advantage to using sendbegin/end here, so code is commented
C           out. There is an advantage to this case in SRBEGIN, for
C           example. 
C           CALL MPI_IRECV(RCVMSG, RCVLTH/DATALTH, DATATYPE,
C    &                      RCVSRC, MTAG+FORCETYPE, COMM, RCVID, IERR2)
C           CALL MPI_BSEND(RCVMSG, 0, DATATYPE, 
C    &                      RCVSRC, MTAG, COMM, IERR3)
C           DO WHILE (IERR3 .GT. 0)
C             CALL MPI_ERROR_CLASS(IERR3, IERR4, IERR0)
C             IF (IERR4 .EQ. MPI_ERR_BUFFER) THEN
C               CALL MPI_BSEND(RCVMSG, 0, DATATYPE, 
C    &                         RCVSRC, MTAG, COMM, IERR3)
C             ELSE
C               IERR3 = -IERR3
C             ENDIF
C           ENDDO
C           CALL MPI_RECV(SNDMSG, 0, DATATYPE,
C    &                      SNDDST, MTAG, COMM, STATUS, IERR4)
C           CALL MPI_IRSEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
C    &                      SNDDST, MTAG+FORCETYPE, COMM, SNDID, IERR1)
C           CALL MPI_WAIT(RCVID, STATUS, IERR2)
C           CALL MPI_WAIT(SNDID, STATUS, IERR1)
C         ENDIF
C
        ELSEIF (PROTOPT .EQ. 6) THEN
C
C         native SENDRECV 
          CALL MPI_SENDRECV(SNDMSG, SNDLTH/DATALTH, DATATYPE, SNDDST, 
     &                      MTAG, RCVMSG, RCVLTH/DATALTH, DATATYPE,
     &                      RCVSRC, MTAG, COMM, STATUS, IERR1)
C
        ELSE
C
          WRITE (0,901) COMMOPT, PROTOPT
  901     FORMAT(/,' PSTSWM: FATAL ERROR IN SUBROUTINE SENDRECV:',/,
     &             ' UNKNOWN COMMUNICATION PROTOCOL SPECIFIED',/, 
     &             ' COMMOPT = ',I3, ' PROTOPT = ',I3)
          STOP                                                   
C
        ENDIF
C
      ELSEIF (COMMOPT .EQ. 1) THEN
C***********************************************************************
C       ordered SENDRECV:
C       IF (ORDER .EQ. 1) send/recv
C       IF (ORDER .NE. 1) recv/send
C       (Deadlock avoidance is the responsibility of the calling
C        program.)
C***********************************************************************
C
C       Choose communication protocol.
        IF (PROTOPT .LE. 1) THEN
C
          IF (PROTOPT .EQ. 0) THEN
C
C           ordered SENDRECV
            IF (ORDER .EQ. 1) THEN
              CALL MPI_SEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                      SNDDST, MTAG, COMM, IERR1)
              CALL MPI_RECV(RCVMSG, RCVLTH/DATALTH, DATATYPE,
     &                      RCVSRC, MTAG, COMM, STATUS, IERR2)
            ELSE
              CALL MPI_RECV(RCVMSG, RCVLTH/DATALTH, DATATYPE,
     &                       RCVSRC, MTAG, COMM, STATUS, IERR2)
              CALL MPI_SEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                      SNDDST, MTAG, COMM, IERR1)
            ENDIF
C
          ELSE
C
C           Do not block for initial send, increasing odds that receive 
C           will be posted before message arrives.
            IF (ORDER .EQ. 1) THEN
              CALL MPI_ISEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                       SNDDST, MTAG, COMM, SNDID, IERR1)
              CALL MPI_RECV(RCVMSG, RCVLTH/DATALTH, DATATYPE,
     &                       RCVSRC, MTAG, COMM, STATUS, IERR2)
              CALL MPI_WAIT(SNDID, STATUS, IERR1)
            ELSE
              CALL MPI_RECV(RCVMSG, RCVLTH/DATALTH, DATATYPE,
     &                      RCVSRC, MTAG, COMM, STATUS, IERR2)
              CALL MPI_SEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                      SNDDST, MTAG, COMM, IERR1)
            ENDIF
C
          ENDIF
C
        ELSEIF (PROTOPT .LE. 3) THEN
C
C         IF (PROTOPT .EQ. 2) THEN
C
C           Post receive before initial send, increasing odds that
C           receive will be posted before message arrives.
            IF (ORDER .EQ. 1) THEN
              CALL MPI_IRECV(RCVMSG, RCVLTH/DATALTH, DATATYPE,
     &                       RCVSRC, MTAG, COMM, RCVID, IERR2)
              CALL MPI_SEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                       SNDDST, MTAG, COMM, IERR1)
              CALL MPI_WAIT(RCVID, STATUS, IERR2)
            ELSE
              CALL MPI_RECV(RCVMSG, RCVLTH/DATALTH, DATATYPE,
     &                      RCVSRC, MTAG, COMM, STATUS, IERR2)
              CALL MPI_SEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                      SNDDST, MTAG, COMM, IERR1)
            ENDIF
C
C         ELSE
C
C           This case is "included" for completeness. There is no
C           advantage to using sendbegin/end here, so code is commented
C           out. There is an advantage to this case in SRBEGIN, for
C           example. 
C           IF (ORDER .EQ. 1) THEN
C             CALL MPI_IRECV(RCVMSG, RCVLTH/DATALTH, DATATYPE,
C    &                       RCVSRC, MTAG, COMM, RCVID, IERR2)
C             CALL MPI_ISEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
C    &                       SNDDST, MTAG, COMM, SNDID, IERR1)
C             CALL MPI_WAIT(RCVID, STATUS, IERR2)
C             CALL MPI_WAIT(SNDID, STATUS, IERR1)
C           ELSE
C             CALL MPI_RECV(RCVMSG, RCVLTH/DATALTH, DATATYPE,
C    &                      RCVSRC, MTAG, COMM, STATUS, IERR2)
C             CALL MPI_SEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
C    &                      SNDDST, MTAG, COMM, IERR1)
C           ENDIF
C
C         ENDIF
C
        ELSEIF (PROTOPT .LE. 5) THEN
C
C         IF (PROTOPT .EQ. 4) THEN
C
C           Post receive before send to allow use of forcetypes.
            IF (ORDER .EQ. 1) THEN
              CALL MPI_IRECV(RCVMSG, RCVLTH/DATALTH, DATATYPE,
     &                       RCVSRC, MTAG+FORCETYPE, COMM, RCVID, IERR2)
              CALL MPI_RECV(SNDMSG, 0, DATATYPE,
     &                       SNDDST, MTAG, COMM, STATUS, IERR4)
              CALL MPI_RSEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                       SNDDST, MTAG+FORCETYPE, COMM, IERR1)
              CALL MPI_SEND(RCVMSG, 0, DATATYPE, 
     &                       RCVSRC, MTAG, COMM, IERR3)
              CALL MPI_WAIT(RCVID, STATUS, IERR2)
            ELSE
              CALL MPI_IRECV(RCVMSG, RCVLTH/DATALTH, DATATYPE,
     &                       RCVSRC, MTAG+FORCETYPE, COMM, RCVID, IERR2)
              CALL MPI_SEND(RCVMSG, 0, DATATYPE, 
     &                       RCVSRC, MTAG, COMM, IERR3)
              CALL MPI_WAIT(RCVID, STATUS, IERR2)
              CALL MPI_RECV(SNDMSG, 0, DATATYPE,
     &                       SNDDST, MTAG, COMM, STATUS, IERR4)
              CALL MPI_RSEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                       SNDDST, MTAG+FORCETYPE, COMM, IERR1)
            ENDIF
C
C         ELSE
C
C           This case is "included" for completeness. There is no
C           advantage to using sendbegin/end here, so code is commented
C           out. There is an advantage to this case in SRBEGIN, for
C           example. 
C           IF (ORDER .EQ. 1) THEN
C             CALL MPI_IRECV(RCVMSG, RCVLTH/DATALTH, DATATYPE,
C    &                       RCVSRC, MTAG+FORCETYPE, COMM, RCVID, IERR2)
C             CALL MPI_RECV(SNDMSG, 0, DATATYPE,
C    &                        SNDDST, MTAG, COMM, STATUS, IERR4)
C             CALL MPI_IRSEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
C    &                       SNDDST, MTAG+FORCETYPE, COMM, SNDID, IERR1)
C             CALL MPI_SEND(RCVMSG, 0, DATATYPE, 
C    &                        RCVSRC, MTAG, COMM, IERR3)
C             CALL MPI_WAIT(RCVID, STATUS, IERR2)
C             CALL MPI_WAIT(SNDID, STATUS, IERR1)
C           ELSE
C             CALL MPI_IRECV(RCVMSG, RCVLTH/DATALTH, DATATYPE,
C    &                       RCVSRC, MTAG+FORCETYPE, COMM, RCVID, IERR2)
C             CALL MPI_SEND(RCVMSG, 0, DATATYPE, 
C    &                       RCVSRC, MTAG, COMM, IERR3)
C             CALL MPI_WAIT(RCVID, STATUS, IERR2)
C             CALL MPI_RECV(SNDMSG, 0, DATATYPE,
C    &                       SNDDST, MTAG, COMM, STATUS, IERR4)
C             CALL MPI_RSEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
C    &                       SNDDST, MTAG+FORCETYPE, COMM, IERR1)
C           ENDIF
C
C         ENDIF
C
        ELSEIF (PROTOPT .EQ. 6) THEN
C
C         synchronous ordered SENDRECV 
          IF (ORDER .EQ. 1) THEN
            CALL MPI_RECV(SNDMSG, 0, DATATYPE,
     &                    SNDDST, MTAG, COMM, STATUS, IERR4)
            CALL MPI_SEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                    SNDDST, MTAG, COMM, IERR1)
            IF (SNDDST .NE. RCVSRC)
     &        CALL MPI_SEND(RCVMSG, 0, DATATYPE, 
     &                      RCVSRC, MTAG, COMM, IERR3)
            CALL MPI_RECV(RCVMSG, RCVLTH/DATALTH, DATATYPE,
     &                    RCVSRC, MTAG, COMM, STATUS, IERR2)
          ELSE
            CALL MPI_SEND(RCVMSG, 0, DATATYPE, 
     &                    RCVSRC, MTAG, COMM, IERR3)
            CALL MPI_RECV(RCVMSG, RCVLTH/DATALTH, DATATYPE,
     &                    RCVSRC, MTAG, COMM, STATUS, IERR2)
            IF (SNDDST .NE. RCVSRC)
     &        CALL MPI_RECV(SNDMSG, 0, DATATYPE,
     &                      SNDDST, MTAG, COMM, STATUS, IERR4)
            CALL MPI_SEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                    SNDDST, MTAG, COMM, IERR1)
          ENDIF
C
        ELSE
C
C         protocol error
          WRITE (0,901) COMMOPT, PROTOPT
          STOP                                                   
C
        ENDIF
C
      ELSE
C***********************************************************************
C       undefined SENDRECV option
C***********************************************************************
C
        WRITE (0,900) COMMOPT
  900   FORMAT(/,' PSTSWM: FATAL ERROR IN SUBROUTINE SENDRECV:',/,
     &           ' UNKNOWN COMMUNICATION OPTION SPECIFIED',/, 
     &           ' COMMOPT = ',I3)                                 
        STOP                                                   
C
      ENDIF
C
      IF ((IERR1 .NE. 0) .OR. (IERR2 .NE. 0) .OR. 
     &    (IERR3 .NE. 0) .OR. (IERR4 .NE. 0)) THEN
         WRITE (0,902) IERR1, IERR2, IERR3, IERR4
  902    FORMAT(/,' PSTSWM: FATAL ERROR IN SUBROUTINE SENDRECV:',/,
     &           ' NONZERO MPI ROUTINE ERROR CODE RETURNED',/, 
     &           ' IERR1 = ',I3,' IERR2 = ',I3,/,
     &           ' IERR3 = ',I3,' IERR4 = ',I3)
        STOP
      ENDIF
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE SRBEGIN(COMMOPT, PROTOPT, ORDER, ME, MTAG, 
     &                   SNDDST, SNDLTH, SNDMSG, RCVSRC, RCVLTH, 
     &                   RCVMSG)
C
C This subroutine begins a SENDRECV operation that will be completed by
C a SREND. It initiates the sending of the message in the SNDMSG 
C buffer to SNDDST, and the receiving of a message into RCVMSG. 
C Everything is completed except the sendend in the nonblocking
C send option, and the recv or recvend in the delayed-recv option.
C
C Communication options (COMMOPT) for SRBEGIN/END include:
C  IF (COMMOPT .EQ. 0) simple SENDRECV: send/recv
C  IF (COMMOPT .EQ. 1) ordered SENDRECV: [send/recv]|[recv/send]
C  IF (COMMOPT .EQ. 2) delayed-recv SENDRECV: send ... recv
C Communication protocol options (PROTOPT) for SRBEGIN/END include:
C  IF (PROTOPT .EQ. 1, 3, .OR. 5)     nonblocking send   
C  IF (PROTOPT .EQ. 2, 3, 4, .OR. 5)  nonblocking receive
C  IF (PROTOPT .EQ. 4 .OR. 5)         forcetype          
C  if (PROTOPT .EQ. 6 .AND. COMMOPT .EQ. 0) native
C  IF (PROTOPT .EQ. 6 .AND. COMMOPT .EQ. 1) synchronous  
C
C called by: SHIFTCAST, SHIFTSUM
C calls: ADDRECV, ADDSEND, MPI routines
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Parameters -------------------------------------------------------
C
C mpi implementation-specific parameters
#     include "mpif.h"
C
C forcetype message tag offset
      INTEGER FORCETYPE
      PARAMETER (FORCETYPE = 15380)
C
C---- Common Blocks ----------------------------------------------------
C
C picl/mpi interface variables
#     include "picl.i"
C
C---- Arguments --------------------------------------------------------
C
C     Input 
C
C communication algorithm option
      INTEGER COMMOPT
C communication protocol option 
      INTEGER PROTOPT
C order of send/recv
      INTEGER ORDER
C processor id
      INTEGER ME
C message tag offset 
C (MTAG+NPROCS should be < FORCETYPE to guarantee correctness)
      INTEGER MTAG
C message destination
      INTEGER SNDDST
C length of message to be sent
      INTEGER SNDLTH
C message to be sent
      INTEGER SNDMSG(*)
C source of message to be received
      INTEGER RCVSRC
C length of message to be received
      INTEGER RCVLTH
C
C     Output
C
C message received
      INTEGER RCVMSG(*)
C
C---- Local Variables --------------------------------------------------
C
C message request IDs for nonblocking communication calls
      INTEGER RCVID, SNDID
C error return variables
      INTEGER IERR0, IERR1, IERR2, IERR3, IERR4
C array for status values
      INTEGER STATUS(MPI_STATUS_SIZE)
C
C---- Data Statements -------------------------------------------------
C
      DATA IERR0 /0/, IERR1 /0/, IERR2 /0/, IERR3 /0/, IERR4 /0/
C
C---- Executable Statements --------------------------------------------
C
C     Choose communication algorithm.
      IF (COMMOPT .EQ. 0) THEN
C***********************************************************************
C       simple SENDRECV: send/recv
C***********************************************************************
C
C       Choose communication protocol.
        IF (PROTOPT .LE. 1) THEN
C
          IF (PROTOPT .EQ. 0) THEN
C           simple SENDRECV
            CALL MPI_BSEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                     SNDDST, MTAG, COMM, IERR1)
            DO WHILE (IERR1 .GT. 0)
              CALL MPI_ERROR_CLASS(IERR1, IERR2, IERR0)
              IF (IERR2 .EQ. MPI_ERR_BUFFER) THEN
                CALL MPI_BSEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                         SNDDST, MTAG, COMM, IERR1)
              ELSE
                IERR1 = -IERR1
              ENDIF
            ENDDO
            CALL MPI_RECV(RCVMSG, RCVLTH/DATALTH, DATATYPE,
     &                     RCVSRC, MTAG, COMM, STATUS, IERR2)
          ELSE
C           Do not block for send, increasing odds that receive will
C           be posted before message arrives
            CALL MPI_ISEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                     SNDDST, MTAG, COMM, SNDID, IERR1)
            CALL MPI_RECV(RCVMSG, RCVLTH/DATALTH, DATATYPE,
     &                     RCVSRC, MTAG, COMM, STATUS, IERR2)
            CALL ADDSEND(SNDID, MTAG, SNDDST)
          ENDIF
C
        ELSEIF (PROTOPT .LE. 3) THEN
C    
          IF (PROTOPT .EQ. 2) THEN
C           Post receive before send, increasing odds that receive will
C           be posted before message arrives.
            CALL MPI_IRECV(RCVMSG, RCVLTH/DATALTH, DATATYPE,
     &                     RCVSRC, MTAG, COMM, RCVID, IERR2)
            CALL MPI_SEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                     SNDDST, MTAG, COMM, IERR1)
            CALL MPI_WAIT(RCVID, STATUS, IERR2)
          ELSE
C           Also do not block for send, enabling overlap of 
C           communication with computation.
            CALL MPI_IRECV(RCVMSG, RCVLTH/DATALTH, DATATYPE,
     &                     RCVSRC, MTAG, COMM, RCVID, IERR2)
            CALL MPI_ISEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                     SNDDST, MTAG, COMM, SNDID, IERR1)
            CALL ADDSEND(SNDID, MTAG, SNDDST)
            CALL MPI_WAIT(RCVID, STATUS, IERR2)
          ENDIF
C
        ELSEIF (PROTOPT .LE. 5) THEN
C    
          IF (PROTOPT .EQ. 4) THEN
C           Post receive before send to allow use of forcetypes.
            CALL MPI_IRECV(RCVMSG, RCVLTH/DATALTH, DATATYPE,
     &                     RCVSRC, MTAG+FORCETYPE, COMM, RCVID, IERR2)
            CALL MPI_BSEND(RCVMSG, 0, DATATYPE, 
     &                     RCVSRC, MTAG, COMM, IERR3)
            DO WHILE (IERR3 .GT. 0)
              CALL MPI_ERROR_CLASS(IERR3, IERR4, IERR0)
              IF (IERR4 .EQ. MPI_ERR_BUFFER) THEN
                CALL MPI_BSEND(RCVMSG, 0, DATATYPE, 
     &                         RCVSRC, MTAG, COMM, IERR3)
              ELSE
                IERR3 = -IERR3
              ENDIF
            ENDDO
            CALL MPI_RECV(SNDMSG, 0, DATATYPE,
     &                     SNDDST, MTAG, COMM, STATUS, IERR4)
            CALL MPI_RSEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                     SNDDST, MTAG+FORCETYPE, COMM, IERR1)
            CALL MPI_WAIT(RCVID, STATUS, IERR2)
          ELSE
C           Also do not block for send, enabling overlap of 
C           communication with computation.
            CALL MPI_IRECV(RCVMSG, RCVLTH/DATALTH, DATATYPE,
     &                      RCVSRC, MTAG+FORCETYPE, COMM, RCVID, IERR2)
            CALL MPI_BSEND(RCVMSG, 0, DATATYPE, 
     &                     RCVSRC, MTAG, COMM, IERR3)
            DO WHILE (IERR3 .GT. 0)
              CALL MPI_ERROR_CLASS(IERR3, IERR4, IERR0)
              IF (IERR4 .EQ. MPI_ERR_BUFFER) THEN
                CALL MPI_BSEND(RCVMSG, 0, DATATYPE, 
     &                         RCVSRC, MTAG, COMM, IERR3)
              ELSE
                IERR3 = -IERR3
              ENDIF
            ENDDO
            CALL MPI_RECV(SNDMSG, 0, DATATYPE,
     &                      SNDDST, MTAG, COMM, STATUS, IERR4)
            CALL MPI_IRSEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                      SNDDST, MTAG+FORCETYPE, COMM, SNDID, IERR1)
            CALL ADDSEND(SNDID, MTAG+FORCETYPE, SNDDST)
            CALL MPI_WAIT(RCVID, STATUS, IERR2)
          ENDIF
C
        ELSEIF (PROTOPT .EQ. 6) THEN
C
C         native SENDRECV 
          CALL MPI_SENDRECV(SNDMSG, SNDLTH/DATALTH, DATATYPE, SNDDST, 
     &                      MTAG, RCVMSG, RCVLTH/DATALTH, DATATYPE,
     &                      RCVSRC, MTAG, COMM, STATUS, IERR1)
C
        ELSE
C
          WRITE (0,901) COMMOPT, PROTOPT
  901     FORMAT(/,' PSTSWM: FATAL ERROR IN SUBROUTINE SRBEGIN:',/,
     &             ' UNKNOWN COMMUNICATION PROTOCOL SPECIFIED',/, 
     &             ' COMMOPT = ',I3, ' PROTOPT = ',I3)
          STOP                                                   
C
        ENDIF
C
      ELSEIF (COMMOPT .EQ. 1) THEN
C***********************************************************************
C       ordered SENDRECV:
C       IF (ORDER .EQ. 1) send/recv
C       IF (ORDER .NE. 1) recv/send
C       (Deadlock avoidance is the responsibility of the calling
C        program.)
C***********************************************************************
C
C       Choose communication protocol.
        IF (PROTOPT .LE. 1) THEN
C
          IF (PROTOPT .EQ. 0) THEN
C
C           ordered SENDRECV
            IF (ORDER .EQ. 1) THEN
              CALL MPI_SEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                      SNDDST, MTAG, COMM, IERR1)
              CALL MPI_RECV(RCVMSG, RCVLTH/DATALTH, DATATYPE,
     &                      RCVSRC, MTAG, COMM, STATUS, IERR2)
            ELSE
              CALL MPI_RECV(RCVMSG, RCVLTH/DATALTH, DATATYPE,
     &                       RCVSRC, MTAG, COMM, STATUS, IERR2)
              CALL MPI_SEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                      SNDDST, MTAG, COMM, IERR1)
            ENDIF
C
          ELSE
C
C           Do not block for initial send, increasing odds that receive 
C           will be posted before message arrives.
            IF (ORDER .EQ. 1) THEN
              CALL MPI_ISEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                       SNDDST, MTAG, COMM, SNDID, IERR1)
              CALL MPI_RECV(RCVMSG, RCVLTH/DATALTH, DATATYPE,
     &                       RCVSRC, MTAG, COMM, STATUS, IERR2)
              CALL ADDSEND(SNDID, MTAG, SNDDST)
            ELSE
              CALL MPI_RECV(RCVMSG, RCVLTH/DATALTH, DATATYPE,
     &                      RCVSRC, MTAG, COMM, STATUS, IERR2)
              CALL MPI_ISEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                       SNDDST, MTAG, COMM, SNDID, IERR1)
              CALL ADDSEND(SNDID, MTAG, SNDDST)
            ENDIF
C
          ENDIF
C
        ELSEIF (PROTOPT .LE. 3) THEN
C
          IF (PROTOPT .EQ. 2) THEN
C
C           Post receive before initial send, increasing odds that 
C           receive will be posted before message arrives.
            IF (ORDER .EQ. 1) THEN
              CALL MPI_IRECV(RCVMSG, RCVLTH/DATALTH, DATATYPE,
     &                       RCVSRC, MTAG, COMM, RCVID, IERR2)
              CALL MPI_SEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                       SNDDST, MTAG, COMM, IERR1)
              CALL MPI_WAIT(RCVID, STATUS, IERR2)
            ELSE
              CALL MPI_RECV(RCVMSG, RCVLTH/DATALTH, DATATYPE,
     &                      RCVSRC, MTAG, COMM, STATUS, IERR2)
              CALL MPI_SEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                      SNDDST, MTAG, COMM, IERR1)
            ENDIF
C
          ELSE
C
C           Also do not block for send, enabling overlap of 
C           communication with computation.
            IF (ORDER .EQ. 1) THEN
              CALL MPI_IRECV(RCVMSG, RCVLTH/DATALTH, DATATYPE,
     &                       RCVSRC, MTAG, COMM, RCVID, IERR2)
              CALL MPI_ISEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                       SNDDST, MTAG, COMM, SNDID, IERR1)
              CALL ADDSEND(SNDID, MTAG, SNDDST)
              CALL MPI_WAIT(RCVID, STATUS, IERR2)
            ELSE
              CALL MPI_RECV(RCVMSG, RCVLTH/DATALTH, DATATYPE,
     &                       RCVSRC, MTAG, COMM, STATUS, IERR2)
              CALL MPI_ISEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                       SNDDST, MTAG, COMM, SNDID, IERR1)
              CALL ADDSEND(SNDID, MTAG, SNDDST)
            ENDIF
C
          ENDIF
C
        ELSEIF (PROTOPT .LE. 5) THEN
C
          IF (PROTOPT .EQ. 4) THEN
C
C           Post receive before initial send, increasing odds that 
C           receive will be posted before message arrives.
            IF (ORDER .EQ. 1) THEN
              CALL MPI_IRECV(RCVMSG, RCVLTH/DATALTH, DATATYPE,
     &                       RCVSRC, MTAG+FORCETYPE, COMM, RCVID, IERR2)
              CALL MPI_RECV(SNDMSG, 0, DATATYPE,
     &                       SNDDST, MTAG, COMM, STATUS, IERR4)
              CALL MPI_RSEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                       SNDDST, MTAG+FORCETYPE, COMM, IERR1)
              CALL MPI_SEND(RCVMSG, 0, DATATYPE, 
     &                       RCVSRC, MTAG, COMM, IERR3)
              CALL MPI_WAIT(RCVID, STATUS, IERR2)
            ELSE
              CALL MPI_IRECV(RCVMSG, RCVLTH/DATALTH, DATATYPE,
     &                       RCVSRC, MTAG+FORCETYPE, COMM, RCVID, IERR2)
              CALL MPI_SEND(RCVMSG, 0, DATATYPE, 
     &                       RCVSRC, MTAG, COMM, IERR3)
              CALL MPI_WAIT(RCVID, STATUS, IERR2)
              CALL MPI_RECV(SNDMSG, 0, DATATYPE,
     &                       SNDDST, MTAG, COMM, STATUS, IERR4)
              CALL MPI_RSEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                       SNDDST, MTAG+FORCETYPE, COMM, IERR1)
            ENDIF
C
          ELSE
C
C           Also do not block for send, enabling overlap of 
C           communication with computation.
            IF (ORDER .EQ. 1) THEN
              CALL MPI_IRECV(RCVMSG, RCVLTH/DATALTH, DATATYPE,
     &                       RCVSRC, MTAG+FORCETYPE, COMM, RCVID, IERR2)
              CALL MPI_RECV(SNDMSG, 0, DATATYPE,
     &                        SNDDST, MTAG, COMM, STATUS, IERR4)
              CALL MPI_IRSEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                       SNDDST, MTAG+FORCETYPE, COMM, SNDID, IERR1)
              CALL MPI_SEND(RCVMSG, 0, DATATYPE, 
     &                        RCVSRC, MTAG, COMM, IERR3)
              CALL ADDSEND(SNDID, MTAG+FORCETYPE, SNDDST)
              CALL MPI_WAIT(RCVID, STATUS, IERR2)
            ELSE
              CALL MPI_IRECV(RCVMSG, RCVLTH/DATALTH, DATATYPE,
     &                       RCVSRC, MTAG+FORCETYPE, COMM, RCVID, IERR2)
              CALL MPI_SEND(RCVMSG, 0, DATATYPE, 
     &                        RCVSRC, MTAG, COMM, IERR3)
              CALL MPI_WAIT(RCVID, STATUS, IERR2)
              CALL MPI_RECV(SNDMSG, 0, DATATYPE,
     &                        SNDDST, MTAG, COMM, STATUS, IERR4)
              CALL MPI_IRSEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                       SNDDST, MTAG+FORCETYPE, COMM, SNDID, IERR1)
              CALL ADDSEND(SNDID, MTAG+FORCETYPE, SNDDST)
            ENDIF
C
          ENDIF
C
        ELSEIF (PROTOPT .EQ. 6) THEN
C
C         synchronous ordered SENDRECV 
          IF (ORDER .EQ. 1) THEN
            CALL MPI_RECV(SNDMSG, 0, DATATYPE,
     &                    SNDDST, MTAG, COMM, STATUS, IERR4)
            CALL MPI_SEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                    SNDDST, MTAG, COMM, IERR1)
            IF (SNDDST .NE. RCVSRC)
     &        CALL MPI_SEND(RCVMSG, 0, DATATYPE, 
     &                      RCVSRC, MTAG, COMM, IERR3)
            CALL MPI_RECV(RCVMSG, RCVLTH/DATALTH, DATATYPE,
     &                    RCVSRC, MTAG, COMM, STATUS, IERR2)
          ELSE
            CALL MPI_SEND(RCVMSG, 0, DATATYPE, 
     &                    RCVSRC, MTAG, COMM, IERR3)
            CALL MPI_RECV(RCVMSG, RCVLTH/DATALTH, DATATYPE,
     &                    RCVSRC, MTAG, COMM, STATUS, IERR2)
            IF (SNDDST .NE. RCVSRC)
     &        CALL MPI_RECV(SNDMSG, 0, DATATYPE,
     &                      SNDDST, MTAG, COMM, STATUS, IERR4)
            CALL MPI_SEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                    SNDDST, MTAG, COMM, IERR1)
          ENDIF
C
        ELSE
C
C         protocol error
          WRITE (0,901) COMMOPT, PROTOPT
          STOP                                                   
C
        ENDIF
C
      ELSEIF (COMMOPT .EQ. 2) THEN
C***********************************************************************
C       delayed-recv SENDRECV: send ... recv/send ... recv
C***********************************************************************
C
C       Choose communication protocol.
        IF (PROTOPT .LE. 1) THEN
C
          IF (PROTOPT .EQ. 0) THEN
C           delaying receive of SENDRECV
            CALL MPI_BSEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                     SNDDST, MTAG, COMM, IERR1)
            DO WHILE (IERR1 .GT. 0)
              CALL MPI_ERROR_CLASS(IERR1, IERR2, IERR0)
              IF (IERR2 .EQ. MPI_ERR_BUFFER) THEN
                CALL MPI_BSEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                         SNDDST, MTAG, COMM, IERR1)
              ELSE
                IERR1 = -IERR1
              ENDIF
            ENDDO
          ELSE
C           Do not block for send, enabling overlap of communication
C           with computation.
            CALL MPI_ISEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                     SNDDST, MTAG, COMM, SNDID, IERR1)
            CALL ADDSEND(SNDID, MTAG, SNDDST)
          ENDIF
C
        ELSEIF (PROTOPT .LE. 3) THEN
C    
          IF (PROTOPT .EQ. 2) THEN
C           Post receive before send, increasing odds that receive will
C           be posted before message arrives.
            CALL MPI_IRECV(RCVMSG, RCVLTH/DATALTH, DATATYPE,
     &                     RCVSRC, MTAG, COMM, RCVID, IERR2)
            CALL MPI_SEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                     SNDDST, MTAG, COMM, IERR1)
            CALL ADDRECV(RCVID, MTAG, RCVSRC)
          ELSE
C           Also do not block for send, enabling overlap of 
C           communication with computation.
            CALL MPI_IRECV(RCVMSG, RCVLTH/DATALTH, DATATYPE,
     &                     RCVSRC, MTAG, COMM, RCVID, IERR2)
            CALL MPI_ISEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                     SNDDST, MTAG, COMM, SNDID, IERR1)
            CALL ADDSEND(SNDID, MTAG, SNDDST)
            CALL ADDRECV(RCVID, MTAG, RCVSRC)
          ENDIF
C
        ELSEIF (PROTOPT .LE. 5) THEN
C    
          IF (PROTOPT .EQ. 4) THEN
C           Post receive before send to allow use of forcetypes.
            CALL MPI_IRECV(RCVMSG, RCVLTH/DATALTH, DATATYPE,
     &                     RCVSRC, MTAG+FORCETYPE, COMM, RCVID, IERR2)
            CALL MPI_BSEND(RCVMSG, 0, DATATYPE, 
     &                     RCVSRC, MTAG, COMM, IERR3)
            DO WHILE (IERR3 .GT. 0)
              CALL MPI_ERROR_CLASS(IERR3, IERR4, IERR0)
              IF (IERR4 .EQ. MPI_ERR_BUFFER) THEN
                CALL MPI_BSEND(RCVMSG, 0, DATATYPE, 
     &                         RCVSRC, MTAG, COMM, IERR3)
              ELSE
                IERR3 = -IERR3
              ENDIF
            ENDDO
            CALL MPI_RECV(SNDMSG, 0, DATATYPE,
     &                     SNDDST, MTAG, COMM, STATUS, IERR4)
            CALL MPI_RSEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                     SNDDST, MTAG+FORCETYPE, COMM, IERR1)
            CALL ADDRECV(RCVID, MTAG+FORCETYPE, RCVSRC)
          ELSE
C           Also do not block for send, enabling overlap of 
C           communication with computation.
            CALL MPI_IRECV(RCVMSG, RCVLTH/DATALTH, DATATYPE,
     &                      RCVSRC, MTAG+FORCETYPE, COMM, RCVID, IERR2)
            CALL MPI_BSEND(RCVMSG, 0, DATATYPE, 
     &                     RCVSRC, MTAG, COMM, IERR3)
            DO WHILE (IERR3 .GT. 0)
              CALL MPI_ERROR_CLASS(IERR3, IERR4, IERR0)
              IF (IERR4 .EQ. MPI_ERR_BUFFER) THEN
                CALL MPI_BSEND(RCVMSG, 0, DATATYPE, 
     &                         RCVSRC, MTAG, COMM, IERR3)
              ELSE
                IERR3 = -IERR3
              ENDIF
            ENDDO
            CALL MPI_RECV(SNDMSG, 0, DATATYPE,
     &                      SNDDST, MTAG, COMM, STATUS, IERR4)
            CALL MPI_IRSEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                      SNDDST, MTAG+FORCETYPE, COMM, SNDID, IERR1)
            CALL ADDSEND(SNDID, MTAG+FORCETYPE, SNDDST)
            CALL ADDRECV(RCVID, MTAG+FORCETYPE, RCVSRC)
          ENDIF
C
        ELSEIF (PROTOPT .EQ. 6) THEN
C
C         native SENDRECV 
          CALL MPI_SENDRECV(SNDMSG, SNDLTH/DATALTH, DATATYPE, SNDDST, 
     &                      MTAG, RCVMSG, RCVLTH/DATALTH, DATATYPE,
     &                      RCVSRC, MTAG, COMM, STATUS, IERR1)
C
        ELSE
C
C         protocol error
          WRITE (0,901) COMMOPT, PROTOPT
          STOP                                                   
C
        ENDIF
C
      ELSE
C***********************************************************************
C       undefined SENDRECV option
C***********************************************************************
C
        WRITE (0,900) COMMOPT
  900   FORMAT(/,' PSTSWM: FATAL ERROR IN SUBROUTINE SRBEGIN:',/,
     &           ' UNKNOWN COMMUNICATION OPTION SPECIFIED',/, 
     &           ' COMMOPT = ',I3)                                 
        STOP                                                   
C
      ENDIF
C
      IF ((IERR1 .NE. 0) .OR. (IERR2 .NE. 0) .OR. 
     &    (IERR3 .NE. 0) .OR. (IERR4 .NE. 0)) THEN
        WRITE (0,902) IERR1, IERR2, IERR3, IERR4
  902   FORMAT(/,' PSTSWM: FATAL ERROR IN SUBROUTINE SRBEGIN:',/,
     &           ' NONZERO MPI ROUTINE ERROR CODE RETURNED',/, 
     &           ' IERR1 = ',I3,' IERR2 = ',I3,/,
     &           ' IERR3 = ',I3,' IERR4 = ',I3)
        STOP
      ENDIF
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE SREND(COMMOPT, PROTOPT, ORDER, ME, MTAG, 
     &                 SNDDST, RCVSRC, RCVLTH, RCVMSG)
C
C This subroutine completes the operation started in SRBEGIN. It
C completes any outstanding send and receive requests.
C
C Communication options (COMMOPT) for SRBEGIN/END include:
C  IF (COMMOPT .EQ. 0) simple SENDRECV: send/recv
C  IF (COMMOPT .EQ. 1) ordered SENDRECV: [send/recv]|[recv/send]
C  IF (COMMOPT .EQ. 2) delayed-recv SENDRECV: send ... recv
C Communication protocol options (PROTOPT) for SRBEGIN/END include:
C  IF (PROTOPT .EQ. 1, 3, .OR. 5)     nonblocking send   
C  IF (PROTOPT .EQ. 2, 3, 4, .OR. 5)  nonblocking receive
C  IF (PROTOPT .EQ. 4 .OR. 5)         forcetype          
C  if (PROTOPT .EQ. 6 .AND. COMMOPT .EQ. 0) native
C  IF (PROTOPT .EQ. 6 .AND. COMMOPT .EQ. 1) synchronous  
C
C called by: SHIFTCAST, SHIFTSUM
C calls: RMVRECV, RMVSEND, MPI routines
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Parameters -------------------------------------------------------
C
C mpi implementation-specific parameters
#     include "mpif.h"
C
C forcetype message tag offset
      INTEGER FORCETYPE
      PARAMETER (FORCETYPE = 15380)
C
C---- Common Blocks ----------------------------------------------------
C
C picl/mpi interface variables
#     include "picl.i"
C
C---- Arguments --------------------------------------------------------
C
C     Input 
C
C communication algorithm option
      INTEGER COMMOPT
C communication protocol option 
      INTEGER PROTOPT
C order of send/recv
      INTEGER ORDER
C processor id
      INTEGER ME
C message tag offset 
C (MTAG+NPROCS should be < FORCETYPE to guarantee correctness)
      INTEGER MTAG
C message destination
      INTEGER SNDDST
C source of message to be received
      INTEGER RCVSRC
C length of message to be received
      INTEGER RCVLTH
C
C     Output
C
C message received
      INTEGER RCVMSG(*)
C
C---- Local Variables --------------------------------------------------
C
C message request IDs for nonblocking communication calls
      INTEGER RCVID, SNDID
C error return variables
      INTEGER IERR1, IERR2
C array for status values
      INTEGER STATUS(MPI_STATUS_SIZE)
C
C---- External Functions -----------------------------------------------
C
C return nonblocking communication request ids
      EXTERNAL RMVRECV, RMVSEND
      INTEGER RMVRECV, RMVSEND
C
C---- Data Statements -------------------------------------------------
C
      DATA IERR1 /0/, IERR2 /0/
C
C---- Executable Statements --------------------------------------------
C
C     Choose communication algorithm.
C
      IF (COMMOPT .EQ. 0) THEN
C***********************************************************************
C       simple SENDRECV: send/recv
C***********************************************************************
C
C       Choose communication protocol.
C
C       Complete send for nonblocking send protocols.
        IF ((PROTOPT .EQ. 1) .OR. (PROTOPT .EQ. 3)) THEN
          SNDID = RMVSEND(MTAG, SNDDST)
          CALL MPI_WAIT(SNDID, STATUS, IERR1)
        ELSEIF (PROTOPT .EQ. 5) THEN
          SNDID = RMVSEND(MTAG+FORCETYPE, SNDDST)
          CALL MPI_WAIT(SNDID, STATUS, IERR1)
        ELSEIF (PROTOPT .GT. 6) THEN
          WRITE (0,901) COMMOPT, PROTOPT
  901     FORMAT(/,' PSTSWM: FATAL ERROR IN SUBROUTINE SREND:',
     &           /,' UNKNOWN COMMUNICATION PROTOCOL SPECIFIED',/, 
     &             ' COMMOPT = ',I3, ' PROTOPT = ',I3)
          STOP                                                   
        ENDIF
C
      ELSEIF (COMMOPT .EQ. 1) THEN
C***********************************************************************
C       ordered SENDRECV:
C       IF (ORDER .EQ. 1) send/recv
C       IF (ORDER .NE. 1) recv/send
C       (Deadlock avoidance is the responsibility of the calling
C        program.)
C***********************************************************************
C
C       Choose communication protocol.
C
C       Complete send for nonblocking send protocols.
        IF ((PROTOPT .EQ. 1) .OR. (PROTOPT .EQ. 3)) THEN
          SNDID = RMVSEND(MTAG, SNDDST)
          CALL MPI_WAIT(SNDID, STATUS, IERR1)
        ELSEIF (PROTOPT .EQ. 5) THEN
          SNDID = RMVSEND(MTAG+FORCETYPE, SNDDST)
          CALL MPI_WAIT(SNDID, STATUS, IERR1)
        ELSEIF (PROTOPT .GT. 6) THEN
          WRITE (0,901) COMMOPT, PROTOPT
          STOP                                                   
        ENDIF
C
      ELSEIF (COMMOPT .EQ. 2) THEN
C***********************************************************************
C       delayed-recv SENDRECV: send ... recv/send ... recv
C***********************************************************************
C
C       Choose communication protocol.
        IF (PROTOPT .LE. 1) THEN
C
          IF (PROTOPT .EQ. 0) THEN
C           Delay receive of SENDRECV.
            CALL MPI_RECV(RCVMSG, RCVLTH/DATALTH, DATATYPE,
     &                    RCVSRC, MTAG, COMM, STATUS, IERR2)
          ELSE
C           Also complete send.
            CALL MPI_RECV(RCVMSG, RCVLTH/DATALTH, DATATYPE,
     &                     RCVSRC, MTAG, COMM, STATUS, IERR2)
            SNDID = RMVSEND(MTAG, SNDDST)
            CALL MPI_WAIT(SNDID, STATUS, IERR1)
          ENDIF
C
        ELSEIF (PROTOPT .LE. 3) THEN
C
          IF (PROTOPT .EQ. 2) THEN
C           Complete receive.
            RCVID = RMVRECV(MTAG, RCVSRC)
            CALL MPI_WAIT(RCVID, STATUS, IERR2)
          ELSE
C           Also complete send.
            RCVID = RMVRECV(MTAG, RCVSRC)
            SNDID = RMVSEND(MTAG, SNDDST)
            CALL MPI_WAIT(RCVID, STATUS, IERR2)
            CALL MPI_WAIT(SNDID, STATUS, IERR1)
          ENDIF
C
        ELSEIF (PROTOPT .LE. 5) THEN
C
          IF (PROTOPT .EQ. 4) THEN
C           Complete forcetype receive.
            RCVID = RMVRECV(MTAG+FORCETYPE, RCVSRC)
            CALL MPI_WAIT(RCVID, STATUS, IERR2)
          ELSE
C           Also complete forcetype send.
            RCVID = RMVRECV(MTAG+FORCETYPE, RCVSRC)
            SNDID = RMVSEND(MTAG+FORCETYPE, SNDDST)
            CALL MPI_WAIT(RCVID, STATUS, IERR2)
            CALL MPI_WAIT(SNDID, STATUS, IERR1)
          ENDIF
C
        ELSEIF (PROTOPT .GT. 6) THEN
C
C         protocol error
          WRITE (0,901) COMMOPT, PROTOPT
          STOP                                                   
C
        ENDIF
C
      ELSE
C***********************************************************************
C       undefined SENDRECV option
C***********************************************************************
C
        WRITE (0,900) COMMOPT
  900   FORMAT(/,' PSTSWM: FATAL ERROR IN SUBROUTINE SREND:',/,
     &           ' UNKNOWN COMMUNICATION OPTION SPECIFIED',/, 
     &           ' COMMOPT = ',I3)                                 
        STOP                                                   
C
      ENDIF
C
      IF ((IERR1 .NE. 0) .OR. (IERR2 .NE. 0)) THEN
        WRITE (0,902) IERR2
  902   FORMAT(/,' PSTSWM: FATAL ERROR IN SUBROUTINE SREND:',/,
     &           ' NONZERO MPI ROUTINE ERROR CODE RETURNED',/, 
     &           ' IERR2 = ',I3)
        STOP
      ENDIF
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE SR1(PROTOPT, SAFEFORCE, ME, MTAG, RCVSRC, 
     &               RCVLTH, RCVMSG)
C
C This subroutine begins a SENDRECV operation which will be completed 
C by SR2 and SR3. It posts a receive and sends handshaking messages 
C when forcetypes are used.
C
C Communication options (COMMOPT) for SR1/SR2/SR3 include:
C  IF (COMMOPT .EQ. 0) simple SENDRECV: send/recv
C  IF (COMMOPT .EQ. 1) ordered SENDRECV: 
C    IF (ORDER .EQ. 1) recv/send
C    IF (ORDER .NE. 1) recv/send
C  IF (COMMOPT .EQ. 2) delayed-recv SENDRECV: send ... recv
C Communication protocol options (PROTOPT) for SR1/SR2/SR3 include:
C  IF (PROTOPT .EQ. 1, 3, .OR. 5)     nonblocking send   
C  IF (PROTOPT .EQ. 2, 3, 4, .OR. 5)  nonblocking receive
C  IF (PROTOPT .EQ. 4 .OR. 5)         forcetype          
C  if (PROTOPT .EQ. 6 .AND. COMMOPT .EQ. 0) native
C  IF (PROTOPT .EQ. 6 .AND. COMMOPT .EQ. 1) synchronous  
C
C called by: RING2, SHIFTCAST, SHIFTSUM, SRTRNS2
C calls: ADDRECV, MPI routines
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Parameters -------------------------------------------------------
C
C mpi implementation-specific parameters
#     include "mpif.h"
C
C forcetype message tag offset
      INTEGER FORCETYPE
      PARAMETER (FORCETYPE = 15380)
C
C---- Common Blocks ----------------------------------------------------
C
C picl/mpi interface variables
#     include "picl.i"
C
C---- Arguments --------------------------------------------------------
C
C     Input 
C
C Communication protocol option 
      INTEGER PROTOPT
C Send handshaking message to guarantee correctness of using forcetype
C protocol? (not necessary if user has other guarantees)
      LOGICAL SAFEFORCE
C processor id
      INTEGER ME
C message tag offset 
C (MTAG+NPROCS should be < FORCETYPE to guarantee correctness)
      INTEGER MTAG
C source of message to be received
      INTEGER RCVSRC
C length of message to be received
      INTEGER RCVLTH
C
C     Output
C
C message received (eventually)
      INTEGER RCVMSG(*)
C
C---- Local Variables --------------------------------------------------
C
C message request IDs for nonblocking communication calls
      INTEGER RCVID
C error return variables
      INTEGER IERR0, IERR2, IERR3, IERR4
C
C---- Data Statements -------------------------------------------------
C
      DATA IERR0 /0/, IERR2 /0/, IERR3 /0/, IERR4 /0/
C
C---- Executable Statements --------------------------------------------
C
C     (All communication algorithm options share the same code.)
C
C     Choose communication protocol.
      IF (PROTOPT .LE. 1) THEN
C       This procotol does not use nonblocking receive.
      ELSEIF (PROTOPT .LE. 3) THEN
C       Post receive before send, increasing odds that receive will
C       be posted before message arrives.
        CALL MPI_IRECV(RCVMSG, RCVLTH/DATALTH, DATATYPE,
     &                 RCVSRC, MTAG, COMM, RCVID, IERR2)
        CALL ADDRECV(RCVID, MTAG, RCVSRC)
      ELSEIF (PROTOPT .LE. 5) THEN
C       Post receive before send to allow use of forcetypes.
        CALL MPI_IRECV(RCVMSG, RCVLTH/DATALTH, DATATYPE,
     &                 RCVSRC, MTAG+FORCETYPE, COMM, RCVID, IERR2)
        IF (SAFEFORCE) THEN
          CALL MPI_BSEND(RCVMSG, 0, DATATYPE, 
     &                   RCVSRC, MTAG, COMM, IERR3)
          DO WHILE (IERR3 .GT. 0)
            CALL MPI_ERROR_CLASS(IERR3, IERR4, IERR0)
            IF (IERR4 .EQ. MPI_ERR_BUFFER) THEN
              CALL MPI_BSEND(RCVMSG, 0, DATATYPE, 
     &                       RCVSRC, MTAG, COMM, IERR3)
              ELSE
                IERR3 = -IERR3
            ENDIF
          ENDDO
        ENDIF
        CALL ADDRECV(RCVID, MTAG+FORCETYPE, RCVSRC)
      ELSEIF (PROTOPT .GT. 6) THEN
        WRITE (0,901) PROTOPT
  901   FORMAT(/,' PSTSWM: FATAL ERROR IN SUBROUTINE SR1:',
     &         /,' UNKNOWN COMMUNICATION PROTOCOL SPECIFIED',/, 
     &           ' PROTOPT = ',I3)
        STOP                                                   
      ENDIF
C
      IF ((IERR2 .NE. 0) .OR. (IERR3 .NE. 0)) THEN
        WRITE (0,902) IERR2, IERR3
  902   FORMAT(/,' PSTSWM: FATAL ERROR IN SUBROUTINE SR1:',/,
     &           ' NONZERO MPI ROUTINE ERROR CODE RETURNED',/, 
     &           ' IERR2 = ',I3,' IERR3 = ',I3)
        STOP
      ENDIF
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE SRSAFE(PROTOPT, ME, MTAG, RCVSRC)
C
C This subroutine sends the SAFEFORCE message, completing a call to SR1
C in which SAFEFORCE was set to .FALSE. but a handshake is still needed.
C
C Communication options (COMMOPT) for SR1/SR2/SR3 include:
C  IF (COMMOPT .EQ. 0) simple SENDRECV: send/recv
C  IF (COMMOPT .EQ. 1) ordered SENDRECV: 
C    IF (ORDER .EQ. 1) recv/send
C    IF (ORDER .NE. 1) recv/send
C  IF (COMMOPT .EQ. 2) delayed-recv SENDRECV: send ... recv
C Communication protocol options (PROTOPT) for SR1/SR2/SR3 include:
C  IF (PROTOPT .EQ. 1, 3, .OR. 5)     nonblocking send   
C  IF (PROTOPT .EQ. 2, 3, 4, .OR. 5)  nonblocking receive
C  IF (PROTOPT .EQ. 4 .OR. 5)         forcetype          
C  if (PROTOPT .EQ. 6 .AND. COMMOPT .EQ. 0) native
C  IF (PROTOPT .EQ. 6 .AND. COMMOPT .EQ. 1) synchronous  
C
C called by: RING2, SHIFTCAST, SHIFTSUM, SRTRNS2
C calls: MPI routines
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Parameters -------------------------------------------------------
C
C mpi implementation-specific parameters
#     include "mpif.h"
C
C---- Common Blocks ----------------------------------------------------
C
C picl/mpi interface variables
#     include "picl.i"
C
C---- Arguments --------------------------------------------------------
C
C     Input 
C
C Communication protocol option 
      INTEGER PROTOPT
C processor id
      INTEGER ME
C message tag offset 
C (MTAG+NPROCS should be < FORCETYPE to guarantee correctness)
      INTEGER MTAG
C source of message to be received
      INTEGER RCVSRC
C
C---- Local Variables --------------------------------------------------
C
C error return variables
      INTEGER IERR0, IERR3, IERR4
C
C---- Data Statements -------------------------------------------------
C
      DATA IERR0 /0/, IERR3 /0/, IERR4 /0/
C
C---- Executable Statements --------------------------------------------
C
C     (All communication algorithm options share the same code.)
C
C     Choose communication protocol.
      IF (PROTOPT .LE. 3) THEN
C       These procotols do not use safeforce handshaking.
      ELSEIF (PROTOPT .LE. 5) THEN
C       Send safeforce handshake.
        CALL MPI_BSEND(IERR3, 0, DATATYPE, 
     &                 RCVSRC, MTAG, COMM, IERR3)
        DO WHILE (IERR3 .GT. 0)
          CALL MPI_ERROR_CLASS(IERR3, IERR4, IERR0)
          IF (IERR4 .EQ. MPI_ERR_BUFFER) THEN
            CALL MPI_BSEND(IERR3, 0, DATATYPE, 
     &                     RCVSRC, MTAG, COMM, IERR3)
          ELSE
            IERR3 = -IERR3
          ENDIF
        ENDDO
      ELSEIF (PROTOPT .GT. 6) THEN
        WRITE (0,901) PROTOPT
  901   FORMAT(/,' PSTSWM: FATAL ERROR IN SUBROUTINE SRSAFE:',
     &         /,' UNKNOWN COMMUNICATION PROTOCOL SPECIFIED',/, 
     &           ' PROTOPT = ',I3)
        STOP                                                   
      ENDIF
C
      IF (IERR3 .NE. 0) THEN
        WRITE (0,902) IERR3
  902   FORMAT(/,' PSTSWM: FATAL ERROR IN SUBROUTINE SRSAFE:',/,
     &           ' NONZERO MPI ROUTINE ERROR CODE RETURNED',/, 
     &           ' IERR3 = ',I3)
        STOP
      ENDIF
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE SR2(COMMOPT, PROTOPT, SAFEFORCE, ORDER, ME, 
     &               MTAG, SNDDST, SNDLTH, SNDMSG, RCVSRC, RCVLTH,
     &               RCVMSG)
C
C This subroutine continues the SENDRECV operation begun in SR1.
C It initiates the send and (sometimes) waits for the receive
C to complete.
C
C Communication options (COMMOPT) for SR1/SR2/SR3 include:
C  IF (COMMOPT .EQ. 0) simple SENDRECV: send/recv
C  IF (COMMOPT .EQ. 1) ordered SENDRECV: 
C    IF (ORDER .EQ. 1) send/recv
C    IF (ORDER .NE. 1) recv/send
C  IF (COMMOPT .EQ. 2) delayed-recv SENDRECV: send ... recv
C Communication protocol options (PROTOPT) for SR1/SR2/SR3 include:
C  IF (PROTOPT .EQ. 1, 3, .OR. 5)     nonblocking send   
C  IF (PROTOPT .EQ. 2, 3, 4, .OR. 5)  nonblocking receive
C  IF (PROTOPT .EQ. 4 .OR. 5)         forcetype          
C  if (PROTOPT .EQ. 6 .AND. COMMOPT .EQ. 0) native
C  IF (PROTOPT .EQ. 6 .AND. COMMOPT .EQ. 1) synchronous  
C
C called by: RING2, SHIFTCAST, SHIFTSUM, SRTRNS2
C calls: ADDSEND, RMVRECV, MPI routines
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Parameters -------------------------------------------------------
C
C mpi implementation-specific parameters
#     include "mpif.h"
C
C forcetype message tag offset
      INTEGER FORCETYPE
      PARAMETER (FORCETYPE = 15380)
C
C---- Common Blocks ----------------------------------------------------
C
C picl/mpi interface variables
#     include "picl.i"
C
C---- Arguments --------------------------------------------------------
C
C     Input 
C
C Communication algorithm option
      INTEGER COMMOPT
C communication protocol option 
      INTEGER PROTOPT
C Using handshaking message to guarantee correctness of using forcetype
C protocol? (not necessary if user has other guarantees)
      LOGICAL SAFEFORCE
C order of send/recv
      INTEGER ORDER
C processor id
      INTEGER ME
C message tag offset 
C (MTAG+NPROCS should be < FORCETYPE to guarantee correctness)
      INTEGER MTAG
C message destination
      INTEGER SNDDST
C length of message to be sent
      INTEGER SNDLTH
C message to be sent
      INTEGER SNDMSG(*)
C source of message to be received
      INTEGER RCVSRC
C length of message to be received
      INTEGER RCVLTH
C
C     Output
C
C message received
      INTEGER RCVMSG(*)
C
C---- Local Variables --------------------------------------------------
C
C message request IDs for nonblocking communication calls
      INTEGER RCVID, SNDID
C error return variables
      INTEGER IERR0, IERR1, IERR2, IERR3, IERR4
C array for status values
      INTEGER STATUS(MPI_STATUS_SIZE)
C
C---- External Functions -----------------------------------------------
C
C return nonblocking receive request ids
      EXTERNAL RMVRECV
      INTEGER RMVRECV
C
C---- Data Statements -------------------------------------------------
C
      DATA IERR0 /0/, IERR1 /0/, IERR2 /0/, IERR3 /0/, IERR4 /0/
C
C---- Executable Statements --------------------------------------------
C
C     Choose communication algorithm.
      IF (COMMOPT .EQ. 0) THEN
C***********************************************************************
C       simple SENDRECV: send/recv
C***********************************************************************
C
C       Choose communication protocol.
        IF (PROTOPT .LE. 1) THEN
C
          IF (PROTOPT .EQ. 0) THEN
C           simple SENDRECV
            CALL MPI_BSEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                     SNDDST, MTAG, COMM, IERR1)
            DO WHILE (IERR1 .GT. 0)
              CALL MPI_ERROR_CLASS(IERR1, IERR2, IERR0)
              IF (IERR2 .EQ. MPI_ERR_BUFFER) THEN
                CALL MPI_BSEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                         SNDDST, MTAG, COMM, IERR1)
              ELSE
                IERR1 = -IERR1
              ENDIF
            ENDDO
            CALL MPI_RECV(RCVMSG, RCVLTH/DATALTH, DATATYPE,
     &                     RCVSRC, MTAG, COMM, STATUS, IERR2)
          ELSE
C           Do not block for send, enabling overlap of communication
C           with computation.
            CALL MPI_ISEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                     SNDDST, MTAG, COMM, SNDID, IERR1)
            CALL MPI_RECV(RCVMSG, RCVLTH/DATALTH, DATATYPE,
     &                     RCVSRC, MTAG, COMM, STATUS, IERR2)
            CALL ADDSEND(SNDID, MTAG, SNDDST)
          ENDIF
C
        ELSEIF (PROTOPT .LE. 3) THEN
C
          IF (PROTOPT .EQ. 2) THEN
C           Complete outstanding receive.
            CALL MPI_SEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                     SNDDST, MTAG, COMM, IERR1)
            RCVID = RMVRECV(MTAG, RCVSRC)
            CALL MPI_WAIT(RCVID, STATUS, IERR2)
          ELSE
C           Do not block for send, enabling overlap of communication
C           with computation.
            CALL MPI_ISEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                     SNDDST, MTAG, COMM, SNDID, IERR1)
            CALL ADDSEND(SNDID, MTAG, SNDDST)
            RCVID = RMVRECV(MTAG, RCVSRC)
            CALL MPI_WAIT(RCVID, STATUS, IERR2)
          ENDIF
C
        ELSEIF (PROTOPT .LE. 5) THEN
C
          IF (PROTOPT .EQ. 4) THEN
C           Send and receive forcetype messages.
            IF (SAFEFORCE)
     &        CALL MPI_RECV(SNDMSG, 0, DATATYPE,
     &                      SNDDST, MTAG, COMM, STATUS, IERR4)
            CALL MPI_RSEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                      SNDDST, MTAG+FORCETYPE, COMM, IERR1)
            RCVID = RMVRECV(MTAG+FORCETYPE, RCVSRC)
            CALL MPI_WAIT(RCVID, STATUS, IERR2)
          ELSE
C           Do not block for send, enabling overlap of communication
C           with computation.
            IF (SAFEFORCE)
     &        CALL MPI_RECV(SNDMSG, 0, DATATYPE,
     &                      SNDDST, MTAG, COMM, STATUS, IERR4)
            CALL MPI_IRSEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                      SNDDST, MTAG+FORCETYPE, COMM, SNDID, IERR1)
            CALL ADDSEND(SNDID, MTAG+FORCETYPE, SNDDST)
            RCVID = RMVRECV(MTAG+FORCETYPE, RCVSRC)
            CALL MPI_WAIT(RCVID, STATUS, IERR2)
          ENDIF
C
        ELSEIF (PROTOPT .EQ. 6) THEN
C
C         native SENDRECV 
          CALL MPI_SENDRECV(SNDMSG, SNDLTH/DATALTH, DATATYPE, SNDDST, 
     &                      MTAG, RCVMSG, RCVLTH/DATALTH, DATATYPE,
     &                      RCVSRC, MTAG, COMM, STATUS, IERR1)
C
        ELSE
C
          WRITE (0,901) COMMOPT, PROTOPT
  901     FORMAT(/,' PSTSWM: FATAL ERROR IN SUBROUTINE SR2:',
     &           /,' UNKNOWN COMMUNICATION PROTOCOL SPECIFIED',/, 
     &             ' COMMOPT = ',I3, ' PROTOPT = ',I3)
          STOP                                                   
C
        ENDIF
C
      ELSEIF (COMMOPT .EQ. 1) THEN
C***********************************************************************
C       ordered SENDRECV:
C       IF (ORDER .EQ. 1) send/recv
C       IF (ORDER .NE. 1) recv/send
C       (Deadlock avoidance is the responsibility of the calling
C        program.)
C***********************************************************************
C
C       Choose communication protocol.
        IF (PROTOPT .LE. 1) THEN
C
          IF (PROTOPT .EQ. 0) THEN
C
C           ordered SENDRECV
            IF (ORDER .EQ. 1) THEN
              CALL MPI_SEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                      SNDDST, MTAG, COMM, IERR1)
              CALL MPI_RECV(RCVMSG, RCVLTH/DATALTH, DATATYPE,
     &                      RCVSRC, MTAG, COMM, STATUS, IERR2)
            ELSE
              CALL MPI_RECV(RCVMSG, RCVLTH/DATALTH, DATATYPE,
     &                       RCVSRC, MTAG, COMM, STATUS, IERR2)
              CALL MPI_SEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                      SNDDST, MTAG, COMM, IERR1)
            ENDIF
C
          ELSE
C
C           Do not block for send, enabling overlap of communication
C           with computation.
            IF (ORDER .EQ. 1) THEN
              CALL MPI_ISEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                       SNDDST, MTAG, COMM, SNDID, IERR1)
              CALL MPI_RECV(RCVMSG, RCVLTH/DATALTH, DATATYPE,
     &                       RCVSRC, MTAG, COMM, STATUS, IERR2)
              CALL ADDSEND(SNDID, MTAG, SNDDST)
            ELSE
              CALL MPI_RECV(RCVMSG, RCVLTH/DATALTH, DATATYPE,
     &                      RCVSRC, MTAG, COMM, STATUS, IERR2)
              CALL MPI_ISEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                       SNDDST, MTAG, COMM, SNDID, IERR1)
              CALL ADDSEND(SNDID, MTAG, SNDDST)
            ENDIF
C
          ENDIF
C
        ELSEIF (PROTOPT .LE. 3) THEN
C
          IF (PROTOPT .EQ. 2) THEN
C
C           ordered SENDRECV
            IF (ORDER .EQ. 1) THEN
              CALL MPI_SEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                       SNDDST, MTAG, COMM, IERR1)
              RCVID = RMVRECV(MTAG, RCVSRC)
              CALL MPI_WAIT(RCVID, STATUS, IERR2)
            ELSE
              RCVID = RMVRECV(MTAG, RCVSRC)
              CALL MPI_WAIT(RCVID, STATUS, IERR2)
              CALL MPI_SEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                      SNDDST, MTAG, COMM, IERR1)
            ENDIF
C
          ELSE
C
C           Do not block for send, enabling overlap of communication
C           with computation.
            IF (ORDER .EQ. 1) THEN
              CALL MPI_ISEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                       SNDDST, MTAG, COMM, SNDID, IERR1)
              CALL ADDSEND(SNDID, MTAG, SNDDST)
              RCVID = RMVRECV(MTAG, RCVSRC)
              CALL MPI_WAIT(RCVID, STATUS, IERR2)
            ELSE
              RCVID = RMVRECV(MTAG, RCVSRC)
              CALL MPI_WAIT(RCVID, STATUS, IERR2)
              CALL MPI_ISEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                       SNDDST, MTAG, COMM, SNDID, IERR1)
              CALL ADDSEND(SNDID, MTAG, SNDDST)
            ENDIF
C
          ENDIF
C
        ELSEIF (PROTOPT .LE. 5) THEN
C
          IF (PROTOPT .EQ. 4) THEN
C
C           Send and receive forcetype messages.
            IF (ORDER .EQ. 1) THEN
              IF (SAFEFORCE)
     &          CALL MPI_RECV(SNDMSG, 0, DATATYPE,
     &                        SNDDST, MTAG, COMM, STATUS, IERR4)
              CALL MPI_RSEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                        SNDDST, MTAG+FORCETYPE, COMM, IERR1)
              RCVID = RMVRECV(MTAG+FORCETYPE, RCVSRC)
              CALL MPI_WAIT(RCVID, STATUS, IERR2)
            ELSE
              RCVID = RMVRECV(MTAG+FORCETYPE, RCVSRC)
              CALL MPI_WAIT(RCVID, STATUS, IERR2)
              IF (SAFEFORCE)
     &          CALL MPI_RECV(SNDMSG, 0, DATATYPE,
     &                        SNDDST, MTAG, COMM, STATUS, IERR4)
              CALL MPI_RSEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                        SNDDST, MTAG+FORCETYPE, COMM, IERR1)
            ENDIF
C
          ELSE
C
C           Do not block for send, enabling overlap of communication
C           with computation.
            IF (ORDER .EQ. 1) THEN
              IF (SAFEFORCE)
     &          CALL MPI_RECV(SNDMSG, 0, DATATYPE,
     &                        SNDDST, MTAG, COMM, STATUS, IERR4)
              CALL MPI_IRSEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                       SNDDST, MTAG+FORCETYPE, COMM, SNDID, IERR1)
              CALL ADDSEND(SNDID, MTAG+FORCETYPE, SNDDST)
              RCVID = RMVRECV(MTAG+FORCETYPE, RCVSRC)
              CALL MPI_WAIT(RCVID, STATUS, IERR2)
            ELSE
              RCVID = RMVRECV(MTAG+FORCETYPE, RCVSRC)
              CALL MPI_WAIT(RCVID, STATUS, IERR2)
              IF (SAFEFORCE) 
     &          CALL MPI_RECV(SNDMSG, 0, DATATYPE,
     &                        SNDDST, MTAG, COMM, STATUS, IERR4)
              CALL MPI_IRSEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                       SNDDST, MTAG+FORCETYPE, COMM, SNDID, IERR1)
              CALL ADDSEND(SNDID, MTAG+FORCETYPE, SNDDST)
            ENDIF
C
          ENDIF
C
        ELSEIF (PROTOPT .EQ. 6) THEN
C
C         synchronous ordered SENDRECV 
          IF (ORDER .EQ. 1) THEN
            CALL MPI_RECV(SNDMSG, 0, DATATYPE,
     &                    SNDDST, MTAG, COMM, STATUS, IERR4)
            CALL MPI_SEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                    SNDDST, MTAG, COMM, IERR1)
            IF (SNDDST .NE. RCVSRC)
     &        CALL MPI_SEND(RCVMSG, 0, DATATYPE, 
     &                      RCVSRC, MTAG, COMM, IERR3)
            CALL MPI_RECV(RCVMSG, RCVLTH/DATALTH, DATATYPE,
     &                    RCVSRC, MTAG, COMM, STATUS, IERR2)
          ELSE
            CALL MPI_SEND(RCVMSG, 0, DATATYPE, 
     &                    RCVSRC, MTAG, COMM, IERR3)
            CALL MPI_RECV(RCVMSG, RCVLTH/DATALTH, DATATYPE,
     &                    RCVSRC, MTAG, COMM, STATUS, IERR2)
            IF (SNDDST .NE. RCVSRC)
     &        CALL MPI_RECV(SNDMSG, 0, DATATYPE,
     &                      SNDDST, MTAG, COMM, STATUS, IERR4)
            CALL MPI_SEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                    SNDDST, MTAG, COMM, IERR1)
          ENDIF
C
        ELSE
C
C         protocol error
          WRITE (0,901) COMMOPT, PROTOPT
          STOP                                                   
C
        ENDIF
C
      ELSEIF (COMMOPT .EQ. 2) THEN
C***********************************************************************
C       delayed-recv SENDRECV: send ... recv
C***********************************************************************
C
C       Choose communication protocol.
        IF (PROTOPT .LE. 3) THEN
C
          IF ((PROTOPT .EQ. 0) .OR. (PROTOPT .EQ. 2)) THEN
C           send of SENDRECV
            CALL MPI_BSEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                     SNDDST, MTAG, COMM, IERR1)
            DO WHILE (IERR1 .GT. 0)
              CALL MPI_ERROR_CLASS(IERR1, IERR2, IERR0)
              IF (IERR2 .EQ. MPI_ERR_BUFFER) THEN
                CALL MPI_BSEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                         SNDDST, MTAG, COMM, IERR1)
              ELSE
                IERR1 = -IERR1
              ENDIF
            ENDDO
          ELSE
C           Do not block for send, enabling overlap of communication
C           with computation.
            CALL MPI_ISEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                     SNDDST, MTAG, COMM, SNDID, IERR1)
            CALL ADDSEND(SNDID, MTAG, SNDDST)
          ENDIF
C
        ELSEIF (PROTOPT .LE. 5) THEN
C
          IF (PROTOPT .EQ. 4) THEN
C           Send forcetype message.
            IF (SAFEFORCE) 
     &        CALL MPI_RECV(SNDMSG, 0, DATATYPE,
     &                      SNDDST, MTAG, COMM, STATUS, IERR4)
            CALL MPI_RSEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                      SNDDST, MTAG+FORCETYPE, COMM, IERR1)
          ELSE
C           Do not block for send, enabling overlap of 
C           communication with computation,
            IF (SAFEFORCE) 
     &        CALL MPI_RECV(SNDMSG, 0, DATATYPE,
     &                      SNDDST, MTAG, COMM, STATUS, IERR4)
            CALL MPI_IRSEND(SNDMSG, SNDLTH/DATALTH, DATATYPE, 
     &                      SNDDST, MTAG+FORCETYPE, COMM, SNDID, IERR1)
            CALL ADDSEND(SNDID, MTAG+FORCETYPE, SNDDST)
          ENDIF
C
        ELSEIF (PROTOPT .EQ. 6) THEN
C
C         native SENDRECV 
          CALL MPI_SENDRECV(SNDMSG, SNDLTH/DATALTH, DATATYPE, SNDDST, 
     &                      MTAG, RCVMSG, RCVLTH/DATALTH, DATATYPE,
     &                      RCVSRC, MTAG, COMM, STATUS, IERR1)
C
        ELSE
C
C         protocol error
          WRITE (0,901) COMMOPT, PROTOPT
          STOP                                                   
C
        ENDIF
C
      ELSE
C***********************************************************************
C       undefined SENDRECV option
C***********************************************************************
C
        WRITE (0,900) COMMOPT
  900   FORMAT(/,' PSTSWM: FATAL ERROR IN SUBROUTINE SR2:',/,
     &           ' UNKNOWN COMMUNICATION OPTION SPECIFIED',/, 
     &           ' COMMOPT = ',I3)                                 
        STOP                                                   
C
      ENDIF
C
      IF ((IERR1 .NE. 0) .OR. (IERR2 .NE. 0) .OR. 
     &    (IERR3 .NE. 0) .OR. (IERR4 .NE. 0)) THEN
        WRITE (0,902) IERR1, IERR2, IERR3, IERR4
  902   FORMAT(/,' PSTSWM: FATAL ERROR IN SUBROUTINE SR2:',/,
     &           ' NONZERO MPI ROUTINE ERROR CODE RETURNED',/, 
     &           ' IERR1 = ',I3,' IERR2 = ',I3,/,
     &           ' IERR3 = ',I3,' IERR4 = ',I3)
        STOP
      ENDIF
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE SR3(COMMOPT, PROTOPT, ME, MTAG, SNDDST,
     &               RCVSRC, RCVLTH, RCVMSG)
C
C This subroutine completes the SENDRECV operation begun in SR1 and
C SR2. It waits until the message requested in SR1 has arrived and
C the send request in SR2 has completed.
C
C Communication options (COMMOPT) for SR1/SR2/SR3 include:
C  IF (COMMOPT .EQ. 0) simple SENDRECV: send/recv
C  IF (COMMOPT .EQ. 1) ordered SENDRECV: 
C    IF (ORDER .EQ. 1) send/recv
C    IF (ORDER .NE. 1) recv/send
C  IF (COMMOPT .EQ. 2) delayed-recv SENDRECV: send ... recv
C Communication protocol options (PROTOPT) for SR1/SR2/SR3 include:
C  IF (PROTOPT .EQ. 1, 3, .OR. 5)     nonblocking send   
C  IF (PROTOPT .EQ. 2, 3, 4, .OR. 5)  nonblocking receive
C  IF (PROTOPT .EQ. 4 .OR. 5)         forcetype          
C  if (PROTOPT .EQ. 6 .AND. COMMOPT .EQ. 0) native
C  IF (PROTOPT .EQ. 6 .AND. COMMOPT .EQ. 1) synchronous  
C
C called by: RING2, SHIFTCAST, SHIFTSUM, SRTRNS2
C calls: RMVRECV, RMVSEND, MPI routines
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Parameters -------------------------------------------------------
C
C mpi implementation-specific parameters
#     include "mpif.h"
C
C forcetype message tag offset
      INTEGER FORCETYPE
      PARAMETER (FORCETYPE = 15380)
C
C---- Common Blocks ----------------------------------------------------
C
C picl/mpi interface variables
#     include "picl.i"
C
C---- Arguments --------------------------------------------------------
C
C     Input 
C
C communication algorithm option
      INTEGER COMMOPT
C communication protocol option 
      INTEGER PROTOPT
C processor id
      INTEGER ME
C message tag offset 
C (MTAG+NPROCS should be < FORCETYPE to guarantee correctness)
      INTEGER MTAG
C message destination
      INTEGER SNDDST
C source of message to be received
      INTEGER RCVSRC
C length of message to be received
      INTEGER RCVLTH
C
C     Output
C
C message received
      INTEGER RCVMSG(*)
C
C---- Local Variables --------------------------------------------------
C
C message request IDs for nonblocking communication calls
      INTEGER RCVID, SNDID
C error return variables
      INTEGER IERR1, IERR2
C array for status values
      INTEGER STATUS(MPI_STATUS_SIZE)
C
C---- External Functions -----------------------------------------------
C
C return nonblocking communication request ids
      EXTERNAL RMVRECV, RMVSEND
      INTEGER RMVRECV, RMVSEND
C
C---- Data Statements -------------------------------------------------
C
      DATA IERR1 /0/, IERR2 /0/
C
C---- Executable Statements --------------------------------------------
C
C     Choose communication algorithm.
C
      IF (COMMOPT .EQ. 0) THEN
C***********************************************************************
C       simple SENDRECV: send/recv
C***********************************************************************
C
C       Choose communication protocol.
C
C       Complete send for nonblocking send protocols.
        IF ((PROTOPT .EQ. 1) .OR. (PROTOPT .EQ. 3)) THEN
          SNDID = RMVSEND(MTAG, SNDDST)
          CALL MPI_WAIT(SNDID, STATUS, IERR1)
        ELSEIF (PROTOPT .EQ. 5) THEN
          SNDID = RMVSEND(MTAG+FORCETYPE, SNDDST)
          CALL MPI_WAIT(SNDID, STATUS, IERR1)
        ELSEIF (PROTOPT .GT. 6) THEN
          WRITE (0,901) COMMOPT, PROTOPT
  901     FORMAT(/,' PSTSWM: FATAL ERROR IN SUBROUTINE SR3:',
     &           /,' UNKNOWN COMMUNICATION PROTOCOL SPECIFIED',/, 
     &             ' COMMOPT = ',I3, ' PROTOPT = ',I3)
          STOP                                                   
        ENDIF
C
      ELSEIF (COMMOPT .EQ. 1) THEN
C***********************************************************************
C       ordered SENDRECV:
C       IF (ORDER .EQ. 1) send/recv
C       IF (ORDER .NE. 1) recv/send
C       (Deadlock avoidance is the responsibility of the calling
C        program.)
C***********************************************************************
C
C       Choose communication protocol.
C
C       Complete send for nonblocking send protocols.
        IF ((PROTOPT .EQ. 1) .OR. (PROTOPT .EQ. 3)) THEN
          SNDID = RMVSEND(MTAG, SNDDST)
          CALL MPI_WAIT(SNDID, STATUS, IERR1)
        ELSEIF (PROTOPT .EQ. 5) THEN
          SNDID = RMVSEND(MTAG+FORCETYPE, SNDDST)
          CALL MPI_WAIT(SNDID, STATUS, IERR1)
        ELSEIF (PROTOPT .GT. 6) THEN
          WRITE (0,901) COMMOPT, PROTOPT
          STOP                                                   
        ENDIF
C
      ELSEIF (COMMOPT .EQ. 2) THEN
C***********************************************************************
C       delayed-recv SENDRECV: recvbegin ... send ... recvend
C***********************************************************************
C
C       Choose communication protocol.
        IF (PROTOPT .LE. 1) THEN
C
          IF (PROTOPT .EQ. 0) THEN
C           Receive message.
            CALL MPI_RECV(RCVMSG, RCVLTH/DATALTH, DATATYPE,
     &                    RCVSRC, MTAG, COMM, STATUS, IERR2)
          ELSE
C           Also complete send.
            CALL MPI_RECV(RCVMSG, RCVLTH/DATALTH, DATATYPE,
     &                    RCVSRC, MTAG, COMM, STATUS, IERR2)
            SNDID = RMVSEND(MTAG, SNDDST)
            CALL MPI_WAIT(SNDID, STATUS, IERR1)
          ENDIF
C
        ELSEIF (PROTOPT .LE. 3) THEN
C
          IF (PROTOPT .EQ. 2) THEN
C           Complete receive.
            RCVID = RMVRECV(MTAG, RCVSRC)
            CALL MPI_WAIT(RCVID, STATUS, IERR2)
          ELSE
C           Also complete send.
            RCVID = RMVRECV(MTAG, RCVSRC)
            SNDID = RMVSEND(MTAG, SNDDST)
            CALL MPI_WAIT(RCVID, STATUS, IERR2)
            CALL MPI_WAIT(SNDID, STATUS, IERR1)
          ENDIF
C
        ELSEIF (PROTOPT .LE. 5) THEN
C
          IF (PROTOPT .EQ. 4) THEN
C           Complete receive.
            RCVID = RMVRECV(MTAG+FORCETYPE, RCVSRC)
            CALL MPI_WAIT(RCVID, STATUS, IERR2)
          ELSE
C           Also complete send.
            RCVID = RMVRECV(MTAG+FORCETYPE, RCVSRC)
            SNDID = RMVSEND(MTAG+FORCETYPE, SNDDST)
            CALL MPI_WAIT(RCVID, STATUS, IERR2)
            CALL MPI_WAIT(SNDID, STATUS, IERR1)
          ENDIF
C
        ELSEIF (PROTOPT .GT. 6) THEN
C
C         protocol error
          WRITE (0,901) COMMOPT, PROTOPT
          STOP                                                   
C
        ENDIF
C
      ELSE
C***********************************************************************
C       undefined SENDRECV option
C***********************************************************************
C
        WRITE (0,900) COMMOPT
  900   FORMAT(/,' PSTSWM: FATAL ERROR IN SUBROUTINE SR3:',/,
     &           ' UNKNOWN COMMUNICATION OPTION SPECIFIED',/, 
     &           ' COMMOPT = ',I3)                                 
        STOP                                                   
C
      ENDIF
C
      IF ((IERR1 .NE. 0) .OR. (IERR2 .NE. 0)) THEN
        WRITE (0,902) IERR2
  902   FORMAT(/,' PSTSWM: FATAL ERROR IN SUBROUTINE SR3:',/,
     &           ' NONZERO MPI ROUTINE ERROR CODE RETURNED',/, 
     &           ' IERR2 = ',I3)
        STOP
      ENDIF
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
