C#######################################################################
C PSTSWM Version 4.0 (12/1/94)                                         #
C  A message-passing benchmark code and parallel algorithm testbed     #
C  that solves the nonlinear shallow water equations using the spectral#
C  transform method.                                                   #
C Written by:                                                          #
C  Patrick Worley of Oak Ridge National Laboratory                     #
C  Ian Foster of Argonne National Laboratory                           #
C Based on the sequential code STSWM 2.0 by James Hack and Ruediger    #
C  Jakob of the National Center for Atmospheric Research.              #
C Research and development funded by the Computer Hardware, Advanced   #
C  Mathematics, and Model Physics (CHAMMP) program of the U.S.         #
C  Department of Energy.                                               # 
C                                                                      #
C Questions and comments should be directed to worley@msr.epm.ornl.gov #
C Please notify and acknowledge the authors in any research or         #
C publications utilizing PSTSWM or any part of the code.               #
C                                                                      #
C NOTICE: Neither the institutions nor the authors make any            #
C representations about the suitability of this software for any       #
C purpose. This software is provided "as is", without express or       #
C implied warranty.                                                    #
C#######################################################################
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C picl.F                                                               C
C                                                                      C
C These routines implement the following PICL 2.0 commands using MPI   C
C commands:                                                            C
C                                                                      C
C BCAST0F   - broadcast                                                C
C CHECK0F   - enable/disable PICL-level error checking          (NOP)  C
C CLOCKSYNC0- synchronize clocks                                       C
C CLOCK0F   - double precision real time clock                         C
C CLOSE0F   - disable interprocess communication                       C
C GETDATA0F - query datatype of message being communicated             C
C GMAX0F    - global vector maximum                                    C
C GMIN0F    - global vector minimum                                    C
C GCMPI     - global integer vector maximum or minimum                 C
C GCMPR     - global real vector maximum or minimum                    C
C GCMPD     - global double precision vector maximum or minimum        C
C GRAY0F    - forward gray code mapping                                C
C OPEN0F    - enable interprocess communication                        C
C RECV0F    - receive a tagged message                          (NOP)  C
C SEND0F    - send a tagged message                             (NOP)  C
C SETDATA0F - set datatype of message being communicated               C
C TRACESTATISTICSF - enable collection of profile data          (NOP)  C
C TRACENODEF       - enable collection of trace data            (NOP)  C
C TRACEDATAF       - save data associated with specified event  (NOP)  C
C TRACEEVENTF      - mark occurrence of an event                (NOP)  C
C TRACEEXITF       - disable collection of trace data           (NOP)  C
C TRACEFILESF      - open files for saving trace data           (NOP)  C
C TRACEFLUSHF      - send trace data to disk                    (NOP)  C
C TRACELEVELSF     - specify what and how much trace data to collect   C
C                                                               (NOP)  C
C WHO0F     - number of processors, process id, host id query          C
C                                                                      C
C The PICL commands are used in the initialization, timing, and error  C
C analysis sections of the code. The intention is that the performance C
C sensitive code (in swap.F and sendrecv.F) be implemented directly    C
C using native commands. The routines in picl.F are then used to       C
C eliminate the remaining dependence on the PICL library. The reasons  C
C for this are twofold:                                                C
C a) On some machines, the swap and sendrecv routines can be           C
C    implemented much more efficiently by not using PICL message       C
C    passing.                                                          C
C b) Porting PSTSWM to a new platform need not depend on porting       C
C    the full PICL library first.                                      C
C Note that                                                            C
C  1) PICL provides a major subset of the MPI message passing          C
C     semantics, and no efficiency is given up by using PICL. These    C
C     routines are provided simply for consistency when comparing      C
C     timings with other platforms for which PICL is not efficient.    C
C  2) Note that the full PICL library must be used in order to trace   C
C     or profile the performance of the code.                          C
C  3) These routines do not represent safe implementations of the PICL C
C     commands outside of PSTSWM. The contexts in which the commands   C
C     are called are used to keep the code simple.                     C
C                                                                      C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE BCAST0F(BUF, BUFSIZE, MSGTAG, ROOT)
C
C This routine broadcasts a message buf of given length and type from
C the root to all processors.
C
C called by: ALGINP, GLOBALMAX, GLOBALMIN, INPUT, MESH_MAP_INIT,
C            RING_MAP_INIT, PRBINP, PSTSWM
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C     
C---- Parameters -------------------------------------------------------
C
C mpi implementation-specific parameters
#     include "mpif.h"
C
C---- Common Blocks ----------------------------------------------------
C
C definition for communicator and datatype for native MPI implementation
#     include "picl.i"
C
C---- Arguments --------------------------------------------------------
C
C     Input
C 
C NUMBER OF BYTES IN BUFFER
      INTEGER BUFSIZE
C COMMUNICATION BUFFER
      INTEGER BUF(1)
C MESSAGE TYPE
      INTEGER MSGTAG
C ROOT 
      INTEGER ROOT
C
C---- Local Variables --------------------------------------------------
C
C error return variable
      INTEGER IERR
C
C---- Executable Statements -------------------------------------------
C
#if defined(T3D)
      IF (DATATYPE .NE. MPI_BYTE) THEN
        CALL MPI_BCAST(BUF, BUFSIZE/DATALTH, DATATYPE, ROOT, 
     &                 MPI_COMM_WORLD, IERR)
      ENDIF
#else
      CALL MPI_BCAST(BUF, BUFSIZE/DATALTH, DATATYPE, ROOT, 
     &               MPI_COMM_WORLD, IERR)
#endif
C
      RETURN
C
      END
C                                                                      
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE CHECK0F(CHECK)
C
C This function enables or disables PICL error checking. In the native 
C MPI implementation of PSTSWM, it does nothing.
C
C called by: TINIT
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Arguments --------------------------------------------------------
C
C     Input
C 
C TURN CHECKING ON (1) OR OFF (0)
      INTEGER CHECK
C
C---- Executable Statements -------------------------------------------
C
      RETURN
C
      END
C                                                                      
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE CLOCKSYNC0F()
C
C This function synchronizes the processors. It does not attempt to
C synchronize the clocks, because event tracing is only supported in the
C PICL implementation.
C
C called by: TINIT
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C     
C---- Parameters -------------------------------------------------------
C
C mpi implementation-specific parameters
#     include "mpif.h"
C
C---- Common Blocks ----------------------------------------------------
C
C definition for communicator and datatype for native MPI implementation
#     include "picl.i"
C
C---- Local Variables --------------------------------------------------
C
C error return variable
      INTEGER IERR
C
C---- Executable Statements --------------------------------------------
C
      CALL MPI_BARRIER(MPI_COMM_WORLD, IERR)
C
      RETURN
C
      END
C                                                                      
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      DOUBLE PRECISION FUNCTION CLOCK0F()
C
C This function returns the real time clock as a double precision
C value in seconds.
C
C called by: TINIT, TSTART, TSTOP, TOSTART, TOSTOP, TEXIT, TOUTPUT
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- External Functions -----------------------------------------------
C
C MPI double precision clock routine
      EXTERNAL MPI_WTIME
      DOUBLE PRECISION MPI_WTIME
C
C---- Executable Statements --------------------------------------------
C
      CLOCK0F = MPI_WTIME()
C        
      RETURN
C
      END
C                                                                      
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE CLOSE0F()
C
C This function disables interprocess communication.
C
C called by: PSTSWM
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Local Variables --------------------------------------------------
C
C error return variable
      INTEGER IERR
C
C---- Executable Statements --------------------------------------------
C
      CALL MPI_FINALIZE(IERR)
C
      RETURN
C
      END
C                                                                      
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE GETDATA0F(SAVETYPE)
C
C This function returns the data type specified by the last call to
C SETDATA0F. 
C
C called by: MESH_MAP_INIT, RING_MAP_INIT
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Common Blocks ----------------------------------------------------
C
C definition for communicator and datatype for native MPI implementation
#     include "picl.i"
C
C---- Arguments --------------------------------------------------------
C
C     Output
C 
C Current data type
      CHARACTER*1 SAVETYPE
C
C----- Executable Statements -------------------------------------------
C
      SAVETYPE = PICLTYPE(1)
C
      RETURN
C
      END
C                                                                      
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE GMAX0F(DATA, NDATA, MSGTYPE, MSGTAG, ROOT)
C
C This routine computes a componentwise maximum of a vector over all
C processors. This implementation of the routine is sufficient for use 
C with the current implementation of PSTSWM, but has significant 
C limitations:
C 1) Only datatypes 2, 4, and 5 are needed in PSTSWM, and only
C    these are supported here. 
C 2) The PICL routine mallocs the necessary work space. To keep all of
C    these routines in "standard" FORTRAN, a fixed length work space is 
C    used that is sufficiently large for PSTSWM.  
C
C called by: GLOBALMAX, INPUT, TOUTPUT
C calls: GCMPI, GCMPR, GCMPD
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C     
C---- Parameters -------------------------------------------------------
C
C mpi implementation-specific parameters
#     include "mpif.h"
C
C---- Common Blocks ----------------------------------------------------
C
C definition for communicator and datatype for native MPI implementation
#     include "picl.i"
C
C---- Arguments --------------------------------------------------------
C
C     Input
C 
C NUMBER OF ITEMS IN DATA ARRAY
      INTEGER NDATA
C DATA BUFFER
      INTEGER DATA(*)
C DATA TYPE
C  1: INTEGER*2
C  2: INTEGER
C  3: INTEGER*4
C  4: REAL*4
C  5: REAL*8
      INTEGER MSGTYPE
C MESSAGE TYPE
      INTEGER MSGTAG
C ROOT 
      INTEGER ROOT
C
C---- Local Variables --------------------------------------------------
C
C work space
      INTEGER WSI(10)
      REAL*4 WSR(10)
      REAL*8 WSD(10)
C process id number
      INTEGER ME
C error return variable
      INTEGER IERR
C
C----- Executable Statements -------------------------------------------
C
      IF ((NDATA .GT. 0) .AND. (NDATA .LE. 10)) THEN
        IF (MSGTYPE .EQ. 2) THEN
          CALL GCMPI(+1, DATA, NDATA, WSI, MSGTAG, ROOT)
        ELSE
          IF (MSGTYPE .EQ. 4) THEN
            CALL GCMPR(+1, DATA, NDATA, WSR, MSGTAG, ROOT)
          ELSE
            IF (MSGTYPE .EQ. 5) THEN
              CALL GCMPD(+1, DATA, NDATA, WSD, MSGTAG, ROOT)
            ELSE
              CALL MPI_COMM_RANK(MPI_COMM_WORLD, ME, IERR)
              WRITE(0,901) ME, MSGTYPE
  901         FORMAT (/,' PSTSWM: FATAL ERROR IN SUBROUTINE GMAX0F: ',
     &                /,' BAD MSGTYPE FOR VERTEX PICL EMULATION MODE ',
     &                /,' (ONLY TYPES 2, 4, AND 5 SUPPORTED) ',
     &                /,' ME = ',I4,' MSGTYPE = ',I4)
              STOP
            ENDIF
          ENDIF
        ENDIF
      ELSE
        CALL MPI_COMM_RANK(MPI_COMM_WORLD, ME, IERR)
        WRITE(0,902) ME, NDATA
  902   FORMAT (/,' PSTSWM: FATAL ERROR IN SUBROUTINE GMAX0F: ',
     &         /,' BAD NUMBER OF ITEMS FOR VERTEX PICL EMULATION MODE ',
     &          /,' (AT MOST 10 ITEMS SUPPORTED) ',
     &          /,' ME = ',I4,' NDATA = ',I4)
        STOP
      ENDIF
      RETURN
C
      END
C                                                                      
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE GMIN0F(DATA, NDATA, MSGTYPE, MSGTAG, ROOT)
C
C This routine computes a componentwise minimum of a vector over all
C processors. This implementation of the routine is sufficient for use 
C with the current implementation of PSTSWM, but has significant 
C limitations:
C 1) Only datatypes 2, 4, and 5 are needed in PSTSWM, and only
C    these are supported here. 
C 2) The PICL routine mallocs the necessary work space. To keep all of
C    these routines in "standard" FORTRAN, a fixed length work space is 
C    used that is sufficiently large for PSTSWM.  
C
C called by: GLOBALMIN, TOUTPUT
C calls: GCMPI, GCMPR, GCMPD
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C     
C---- Parameters -------------------------------------------------------
C
C mpi implementation-specific parameters
#     include "mpif.h"
C
C---- Common Blocks ----------------------------------------------------
C
C definition for communicator and datatype for native MPI implementation
#     include "picl.i"
C
C---- Arguments --------------------------------------------------------
C
C     Input
C 
C NUMBER OF ITEMS IN DATA ARRAY
      INTEGER NDATA
C DATA BUFFER
      INTEGER DATA(*)
C DATA TYPE
C  1: INTEGER*2
C  2: INTEGER
C  3: INTEGER*4
C  4: REAL*4
C  5: REAL*8
      INTEGER MSGTYPE
C MESSAGE TYPE
      INTEGER MSGTAG
C ROOT 
      INTEGER ROOT
C
C---- Local Variables --------------------------------------------------
C
C work space
      INTEGER WSI(10)
      REAL*4 WSR(10)
      REAL*8 WSD(10)
C process, host, and other id numbers
      INTEGER ME
C error return variable
      INTEGER IERR
C
C----- Executable Statements -------------------------------------------
C
      IF ((NDATA .GT. 0) .AND. (NDATA .LE. 10)) THEN
        IF (MSGTYPE .EQ. 2) THEN
          CALL GCMPI(-1, DATA, NDATA, WSI, MSGTAG, ROOT)
        ELSE 
          IF (MSGTYPE .EQ. 4) THEN
            CALL GCMPR(-1, DATA, NDATA, WSR, MSGTAG, ROOT)
          ELSE
            IF (MSGTYPE .EQ. 5) THEN
              CALL GCMPD(-1, DATA, NDATA, WSD, MSGTAG, ROOT)
            ELSE
              CALL MPI_COMM_RANK(MPI_COMM_WORLD, ME, IERR)
              WRITE(0,901) ME, MSGTYPE
  901         FORMAT (/,' PSTSWM: FATAL ERROR IN SUBROUTINE GMIN0F: ',
     &                /,' BAD MSGTYPE FOR VERTEX PICL EMULATION MODE ',
     &                /,' (ONLY TYPES 2, 4, AND 5 SUPPORTED) ',
     &                /,' ME = ',I4,' MSGTYPE = ',I4)
              STOP
            ENDIF
          ENDIF
        ENDIF
      ELSE
        CALL MPI_COMM_RANK(MPI_COMM_WORLD, ME, IERR)
        WRITE(0,902) ME, NDATA
  902   FORMAT (/,' PSTSWM: FATAL ERROR IN SUBROUTINE GMIN0F: ',
     &         /,' BAD NUMBER OF ITEMS FOR VERTEX PICL EMULATION MODE ',
     &          /,' (AT MOST 10 ITEMS SUPPORTED) ',
     &          /,' ME = ',I4,' NDATA = ',I4)
        STOP
      ENDIF
      RETURN
C
      END
C                                                                      
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE GCMPI(SIGN, DATA, NDATA, WORK, MSGTAG, ROOT)
C
C This routine uses a brute force method to compute a componentwise
C maximum of an INTEGER vector over all processors. 
C
C called by: GMAX0F, GMIN0F
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C     
C---- Parameters -------------------------------------------------------
C
C mpi implementation-specific parameters
#     include "mpif.h"
C
C---- Common Blocks ----------------------------------------------------
C
C definition for communicator and datatype for native MPI implementation
#     include "picl.i"
C
C---- Arguments --------------------------------------------------------
C
C     Input
C 
C DIRECTION OF COMPARISON (SIGN = 1: MAX, SIGN=-1: MIN)
      INTEGER SIGN
C NUMBER OF ITEMS IN DATA ARRAY
      INTEGER NDATA
C DATA BUFFER
      INTEGER DATA(NDATA)
C WORK BUFFER
      INTEGER WORK(NDATA)
C MESSAGE TYPE
      INTEGER MSGTAG
C ROOT 
      INTEGER ROOT
C
C---- Local Variables --------------------------------------------------
C
C index variables
      INTEGER I, J
C processor id
      INTEGER ME
C number of processors
      INTEGER NPROCS
C array for status values
      INTEGER STATUS(MPI_STATUS_SIZE)
C error return variable
      INTEGER IERR
C
C----- Executable Statements -------------------------------------------
C
      CALL MPI_COMM_RANK(MPI_COMM_WORLD, ME, IERR)
      CALL MPI_COMM_SIZE(MPI_COMM_WORLD, NPROCS, IERR)
C
      IF (ME .EQ. ROOT) THEN
C
        DO I = 0, NPROCS-1
          IF (I .NE. ME) THEN
            CALL MPI_RECV(WORK, NDATA, MPI_INTEGER, I, MSGTAG, 
     &                    MPI_COMM_WORLD, STATUS, IERR)
            DO J=1,NDATA
              IF (SIGN*DATA(J) .LT. SIGN*WORK(J)) DATA(J) = WORK(J)
            ENDDO
          ENDIF
        ENDDO
C
      ELSE
C
        CALL MPI_SEND(DATA, NDATA, MPI_INTEGER, ROOT, MSGTAG, 
     &                MPI_COMM_WORLD, IERR)
C
      ENDIF
C
      RETURN
C
      END
C                                                                      
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE GCMPR(SIGN, DATA, NDATA, WORK, MSGTAG, ROOT)
C
C This routine uses a brute force method to compute a componentwise
C maximum of a REAL*4 vector over all processors. 
C
C called by: GMAX0F, GMIN0F
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C     
C---- Parameters -------------------------------------------------------
C
C mpi implementation-specific parameters
#     include "mpif.h"
C
C---- Common Blocks ----------------------------------------------------
C
C definition for communicator and datatype for native MPI implementation
#     include "picl.i"
C
C---- Arguments --------------------------------------------------------
C
C     Input
C 
C DIRECTION OF COMPARISON (SIGN = 1: MAX, SIGN=-1: MIN)
      INTEGER SIGN
C NUMBER OF ITEMS IN DATA ARRAY
      INTEGER NDATA
C DATA BUFFER
      REAL*4 DATA(NDATA)
C WORK BUFFER
      REAL*4 WORK(NDATA)
C MESSAGE TYPE
      INTEGER MSGTAG
C ROOT 
      INTEGER ROOT
C
C---- Local Variables --------------------------------------------------
C
C index variables
      INTEGER I, J
C processor id
      INTEGER ME
C number of processors
      INTEGER NPROCS
C array for status values
      INTEGER STATUS(MPI_STATUS_SIZE)
C error return variable
      INTEGER IERR
C
C----- Executable Statements -------------------------------------------
C
      CALL MPI_COMM_RANK(MPI_COMM_WORLD, ME, IERR)
      CALL MPI_COMM_SIZE(MPI_COMM_WORLD, NPROCS, IERR)
C
      IF (ME .EQ. ROOT) THEN
C
        DO I = 0, NPROCS-1
          IF (I .NE. ME) THEN
            CALL MPI_RECV(WORK, NDATA, MPI_REAL4, I, MSGTAG, 
     &                    MPI_COMM_WORLD, STATUS, IERR)
            DO J=1,NDATA
              IF (SIGN*DATA(J) .LT. SIGN*WORK(J)) DATA(J) = WORK(J)
            ENDDO
          ENDIF
        ENDDO
C
      ELSE
C
        CALL MPI_SEND(DATA, NDATA, MPI_REAL4, ROOT, MSGTAG, 
     &                MPI_COMM_WORLD, IERR)
C
      ENDIF
C
      RETURN
C
      END
C                                                                      
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE GCMPD(SIGN, DATA, NDATA, WORK, MSGTAG, ROOT)
C
C This routine uses a brute force method to compute a componentwise
C maximum of a REAL*8 vector over all processors. 
C
C called by: GMAX0F, GMIN0F
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C     
C---- Parameters -------------------------------------------------------
C
C mpi implementation-specific parameters
#     include "mpif.h"
C
C---- Common Blocks ----------------------------------------------------
C
C definition for communicator and datatype for native MPI implementation
#     include "picl.i"
C
C---- Arguments --------------------------------------------------------
C
C     Input
C 
C DIRECTION OF COMPARISON (SIGN = 1: MAX, SIGN=-1: MIN)
      INTEGER SIGN
C NUMBER OF ITEMS IN DATA ARRAY
      INTEGER NDATA
C DATA BUFFER
      REAL*8 DATA(NDATA)
C WORK BUFFER
      REAL*8 WORK(NDATA)
C MESSAGE TYPE
      INTEGER MSGTAG
C ROOT 
      INTEGER ROOT
C
C---- Local Variables --------------------------------------------------
C
C index variables
      INTEGER I, J
C processor id
      INTEGER ME
C number of processors
      INTEGER NPROCS
C array for status values
      INTEGER STATUS(MPI_STATUS_SIZE)
C error return variable
      INTEGER IERR
C
C----- Executable Statements -------------------------------------------
C
      CALL MPI_COMM_RANK(MPI_COMM_WORLD, ME, IERR)
      CALL MPI_COMM_SIZE(MPI_COMM_WORLD, NPROCS, IERR)
C
      IF (ME .EQ. ROOT) THEN
C
        DO I = 0, NPROCS-1
          IF (I .NE. ME) THEN
            CALL MPI_RECV(WORK, NDATA, MPI_REAL8, I, 
     &                    MSGTAG, MPI_COMM_WORLD, STATUS, IERR)
            DO J=1,NDATA
              IF (SIGN*DATA(J) .LT. SIGN*WORK(J)) DATA(J) = WORK(J)
            ENDDO
          ENDIF
        ENDDO
C
      ELSE
C
        CALL MPI_SEND(DATA, NDATA, MPI_REAL8, ROOT, MSGTAG, 
     &                MPI_COMM_WORLD, IERR)
C
      ENDIF
C
      RETURN
C
      END
C                                                                      
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      INTEGER FUNCTION GRAY0F(INDEX)
C
C This routine returns the binary-reflected Gray code for the input.
C
C called by: MESH_MAP_INIT, RING_MAP_INIT
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Arguments --------------------------------------------------------
C
C     Input
C 
C INDEX
      INTEGER INDEX
C
C---- External Functions ----------------------------------------------
C
C Exclusive OR
      INTEGER XOR
C
C----- Executable Statements -------------------------------------------
C
      GRAY0F = XOR(INDEX/2,INDEX)
      RETURN
C
      END
C                                                                      
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE OPEN0F(NPROCS,ME,HOST)
C
C This routine enables interprocess communication. It also returns the
C number of processors, local process id, and host process id.
C
C called by: PSTSWM
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Parameters -------------------------------------------------------
C
C mpi implementation-specific parameters
#     include "mpif.h"
C
C---- Common Blocks ----------------------------------------------------
C
C definition for communicator and datatype for native MPI implementation
#     include "picl.i"
C
C---- Arguments --------------------------------------------------------
C
C     Output
C 
C NUMBER OF PROCESSORS
      INTEGER NPROCS
C PROCESSOR ID
      INTEGER ME
C HOST ID
      INTEGER HOST
C
C---- Local Variables --------------------------------------------------
C
C error return variable
      INTEGER IERR
C
C----- Executable Statements -------------------------------------------
C
C     Initialize MPI system
      CALL MPI_INIT(IERR)
C
C     Generate new communicator to use in swap and sendrecv, disabling
C     MPI_ABORT for errors.
      CALL MPI_COMM_DUP(MPI_COMM_WORLD, COMM, IERR)
      CALL MPI_ERRHANDLER_SET(COMM, MPI_ERRORS_RETURN, IERR)
C
C     Initialize defaults
      DATATYPE = MPI_BYTE
      DATALTH  = 1
C
C     Determine number of processors and various ids
      CALL MPI_COMM_SIZE(MPI_COMM_WORLD, NPROCS, IERR)
      CALL MPI_COMM_RANK(MPI_COMM_WORLD, ME, IERR)
      HOST = 32767
C
      RETURN
C
      END
C                                                                      
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE RECV0F(BUF, BUFSIZE, TYPE)
C
C This routine blocks until a message with the indicated message tag
C is received. In the native MPI implementation of PSTSWM, it does 
C nothing. (It is used only in the handshaking when saving trace data.)
C
C called by: TOUTPUT
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Arguments --------------------------------------------------------
C
C     Input
C 
C NUMBER OF BYTES IN BUFFER
      INTEGER BUFSIZE
C COMMUNICATION BUFFER
      INTEGER BUF(1)
C MESSAGE TYPE
      INTEGER TYPE
C
C----- Executable Statements -------------------------------------------
C
      RETURN
C
      END
C                                                                      
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE SEND0F(BUF, BUFSIZE, MSGTAG, DEST)
C
C This routine sends a message with message tag TYPE to DEST.
C In the native MPI implementation of PSTSWM, it does nothing.
C (It is used only in the handshaking when saving trace data.)
C
C called by: TOUTPUT
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Arguments --------------------------------------------------------
C
C     Input
C 
C NUMBER OF BYTES IN BUFFER
      INTEGER BUFSIZE
C COMMUNICATION BUFFER
      INTEGER BUF(1)
C MESSAGE MSGTAG
      INTEGER MSGTAG
C MESSAGE DESTINATION
      INTEGER DEST
C
C----- Executable Statements -------------------------------------------
C
      RETURN
C
      END
C                                                                      
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE SETDATA0F(NEWTYPE)
C
C This function sets the data type for subsequent interprocess 
C communication calls. 
C
C called by: ALGINP, FLTSUM, ILTCAST, MESH_MAP_INIT, PRBINP,
C            PRFINP, PSTSWM, RFTLON, RING_MAP_INIT, TIMING
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Parameters -------------------------------------------------------
C
C mpi implementation-specific parameters
#     include "mpif.h"
C pstswm-specific parameters
#     include "params.i"
C
C---- Common Blocks ----------------------------------------------------
C
C definition for communicator and datatype for native MPI implementation
#     include "picl.i"
C
C---- Arguments --------------------------------------------------------
C
C     Input
C 
C Current data type
      CHARACTER*1 NEWTYPE(1)
C
C----- Executable Statements -------------------------------------------
C
      PICLTYPE(1) = NEWTYPE(1)
      IF ((PICLTYPE(1) .EQ. 'I') .OR. (PICLTYPE(1) .EQ. 'i')) THEN
        DATATYPE = MPI_INTEGER
        DATALTH  = IBYTES
      ELSEIF ((PICLTYPE(1) .EQ. 'R') .OR. (PICLTYPE(1) .EQ. 'r') .OR.
     &        (PICLTYPE(1) .EQ. 'F') .OR. (PICLTYPE(1) .EQ. 'f')) THEN
        DATATYPE = MPI_REAL
        DATALTH  = RBYTES
      ELSEIF ((PICLTYPE(1) .EQ. 'D') .OR. (PICLTYPE(1) .EQ. 'd')) THEN
        DATATYPE = MPI_DOUBLE_PRECISION
        DATALTH  = DBYTES
      ELSEIF ((PICLTYPE(1) .EQ. 'C') .OR. (PICLTYPE(1) .EQ. 'c')) THEN
        DATATYPE = MPI_BYTE
        DATALTH  = 1
      ENDIF
C
      RETURN
C
      END
C                                                                      
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE TRACESTATISTICSF(EVENTS, PICLTIME, PICLCNT, PICLVOL,
     &                           USERTIME, USERCNT)
C
C This routine enables the collection of profile data. In the native MPI
C implementation of PSTSWM, it does nothing.
C
C called by: TINIT
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Arguments --------------------------------------------------------
C
C     Input
C
C NUMBER OF EVENTS
      INTEGER EVENTS
C PROFILING LEVELS
      INTEGER PICLTIME, PICLCNT, PICLVOL, USERTIME, USERCNT
C
C----- Executable Statements -------------------------------------------
C
      RETURN
C
      END
C                                                                      
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE TRACENODEF(TRACESIZE, FLUSH, SYNC)
C
C This routine enables the collection of trace data. In the native MPI
C implementation of PSTSWM, it does nothing.
C
C called by: TINIT
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Arguments --------------------------------------------------------
C
C     Input
C
C TRACE BUFFER SIZE
      INTEGER TRACESIZE
C FLUSH OPTION
      INTEGER FLUSH
C SYNC OPTION
      INTEGER SYNC
C
C----- Executable Statements -------------------------------------------
C
      RETURN
C
      END
C                                                                      
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE TRACEDATAF(EVENTTYPE, DATAID, TYPE, SIZE, DATA)
C
C This routine saves data associated with a given user event. In the
C native MPI implementation of PSTSWM, it does nothing.
C
C called by: TINIT
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Arguments --------------------------------------------------------
C
C     Input
C
C EVENT TYPE
      INTEGER EVENTTYPE
C DATA ID
      INTEGER DATAID
C DATA TYPE
      CHARACTER*1 TYPE(1)
C DATA SIZE
      INTEGER SIZE
C DATA
      INTEGER DATA(*)
C
C----- Executable Statements -------------------------------------------
C
      RETURN
C
      END
C                                                                      
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE TRACEEVENTF(RECORDSTRING, EVENT, NPARAMS, PARAMS)
C
C This routine marks the occurrence of an event. In the native MPI
C implementation of PSTSWM, it does nothing.
C
C called by: ADVECT, EXPLIC, FLTSUM, ILTCAST, SIMPLIC, TRANSPOSE,
C            TSTART, TSTOP
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Arguments --------------------------------------------------------
C
C     Input
C
C RECORD STRING TYPE
      CHARACTER*1 RECORDSTRING(1)
C EVENT ID
      INTEGER EVENT
C NUMBER OF PARAMETERS
      INTEGER NPARAMS
C PARAMETERS
      INTEGER PARAMS(NPARAMS)
C
C----- Executable Statements -------------------------------------------
C
      RETURN
C
      END
C                                                                      
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE TRACEEXITF()
C
C This routine disables the collection of trace data. In the native MPI
C implementation of PSTSWM, it does nothing.
C
C called by: TEXIT
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C----- Executable Statements -------------------------------------------
C
      RETURN
C
      END
C                                                                      
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE TRACEFILESF(TEMPFILE, PERMFILE, VERBOSE)
C
C This routine opens disk files for saving trace data.
C In the native MPI implementation of PSTSWM, it does nothing.
C
C called by: TEXIT
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Arguments --------------------------------------------------------
C
C     Input
C
C NAME OF TEMPORARY FILE
      CHARACTER*1 TEMPFILE(1)
C NAME OF PERMANENT FILE
      CHARACTER*1 PERMFILE(1)
C TRACE MODE
      INTEGER VERBOSE
C
C----- Executable Statements -------------------------------------------
C
      RETURN
C
      END
C                                                                      
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE TRACEFLUSHF()
C
C This routine sends trace data to temporary or permanent disk storage.
C In the native MPI implementation of PSTSWM, it does nothing.
C
C called by: TOUTPUT
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C----- Executable Statements -------------------------------------------
C
      RETURN
C
      END
C                                                                      
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE TRACELEVELF(PICL, USER, TRACE)
C
C This routine specifes the types and amounts of trace data to collect.
C In the native MPI implementation of PSTSWM, it does nothing.
C
C called by: TINIT, TSTART
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Arguments --------------------------------------------------------
C
C     Input
C
C LEVEL OF PICL EVENT TRACING
      INTEGER PICL
C LEVEL OF USER EVENT TRACING
      INTEGER USER
C LEVEL OF TRACE EVENT TRACING
      INTEGER TRACE
C
C----- Executable Statements -------------------------------------------
C
      RETURN
C
      END
C                                                                      
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE WHO0F(NPROCS, ME, HOST)
C
C This routine returns the number of processors, local process id, and 
C host process id. 
C
C called by: MESH_MAP_INIT, RING_MAP_INIT
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C     
C---- Parameters -------------------------------------------------------
C
C mpi implementation-specific parameters
#     include "mpif.h"
C
C---- Common Blocks ----------------------------------------------------
C
C definition for communicator and datatype for native MPI implementation
#     include "picl.i"
C
C---- Arguments --------------------------------------------------------
C
C     Output
C 
C NUMBER OF PROCESSORS
      INTEGER NPROCS
C PROCESSOR ID
      INTEGER ME
C HOST ID
      INTEGER HOST
C
C---- Local Variables --------------------------------------------------
C
C error return variable
      INTEGER IERR
C
C----- Executable Statements -------------------------------------------
C
C     Determine number of processors and various ids
      CALL MPI_COMM_SIZE(MPI_COMM_WORLD, NPROCS, IERR)
      CALL MPI_COMM_RANK(MPI_COMM_WORLD, ME, IERR)
      HOST = 32767
C
      RETURN
C
      END
C                                                                      C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
