C#######################################################################
C PSTSWM Version 4.0 (12/1/94)                                         #
C  A message-passing benchmark code and parallel algorithm testbed     #
C  that solves the nonlinear shallow water equations using the spectral#
C  transform method.                                                   #
C Written by:                                                          #
C  Patrick Worley of Oak Ridge National Laboratory                     #
C  Ian Foster of Argonne National Laboratory                           #
C Based on the sequential code STSWM 2.0 by James Hack and Ruediger    #
C  Jakob of the National Center for Atmospheric Research.              #
C Research and development funded by the Computer Hardware, Advanced   #
C  Mathematics, and Model Physics (CHAMMP) program of the U.S.         #
C  Department of Energy.                                               # 
C                                                                      #
C Questions and comments should be directed to worley@msr.epm.ornl.gov #
C Please notify and acknowledge the authors in any research or         #
C publications utilizing PSTSWM or any part of the code.               #
C                                                                      #
C NOTICE: Neither the institutions nor the authors make any            #
C representations about the suitability of this software for any       #
C purpose. This software is provided "as is", without express or       #
C implied warranty.                                                    #
C#######################################################################
C include precision declaration definitions                            #
#include "precision.i"
C#######################################################################
      SUBROUTINE NRGTCS(DIV, ZETA, H, U, V, MOUNT, ICTR, WSA, WSB)
C                                                                              
C This routine computes a number of energetics parameters when invoked.
C
C called by: PSTSWM
C calls: GLAT_P, GLON_P, GRIDMAP_P, WEIGHT_P
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C     
C---- Parameters -------------------------------------------------------
C
#     include "params.i"
C     
C---- Common Blocks ----------------------------------------------------
C     
C machine architecture information
#     include "machine.i"
C problem resolution information
#     include "problem.i"
C domain decomposition definition variables
#     include "physical.i"
C constants
#     include "consts.i"
C
C---- Arguments --------------------------------------------------------
C
C     Input
C
C divergence field
      REAL DIV(NLLON_P,NLLAT_P)
C vorticity field
      REAL ZETA(NLLON_P,NLLAT_P)
C height field
      REAL H(NLLON_P,NLLAT_P)
C eastward wind
      REAL U(NLLON_P,NLLAT_P)
C northward wind
      REAL V(NLLON_P,NLLAT_P)
C mountain height
      REAL MOUNT(NLLON_P,NLLAT_P)
C array index for energetics data store
      INTEGER ICTR
C
C     Work Space
C
C work arrays
      REAL WSA(5,NLLON_P,NLLAT_P)
      REAL WSB(5,NLLON_P)
C
C---- Local Variables --------------------------------------------------
C
C "constant" variables: pi
      REAL PI
C storage for energetics parameters for up to NGRPHS entries:
C potential enstrophy, divergence, vorticity, total mass, total energy,
C model time, and gridpoint for height timeseries.
C (Retained for future (re)introduction of graphics option in 
C  PSTSWM.)
      REAL PE(NGRPHSX), AAM(NGRPHSX), APE(NGRPHSX), TMSS(NGRPHSX),
     &     TE(NGRPHSX), TME(NGRPHSX), HTS(NGRPHSX) 
      SAVE PE, AAM, APE, TMSS, TE, TME, HTS
C initial values, used to produce normalized graphs: potential
C enstrophy, total mass, and total energy 
      REAL PEN, TMSSN, TOTEN
      SAVE PEN, TMSSN, TOTEN
C temporary summation variables: potential enstrophy, divergence,
C vorticity,  total mass, total energy
      REAL TMP(5)
      INTEGER TPE, TAM, TAPE, TMASS, TOTE
      PARAMETER (TPE=1, TAM=2, TAPE=3, TMASS=4, TOTE=5)
C loop indices
      INTEGER I, J
C fluid height
      REAL HFLUID
C integration weight
      REAL WTS 
C point (PLON, PLAT) for height time series
      INTEGER PLON, PLAT
      SAVE PLON, PLAT
C (longitude, latitude) for closest gridpoint to (PLON, PLAT)
      REAL PLONG, PLATG
C closest gridpoint for height timeseries (in degrees east of Greenwich
C and north of equator) 
      INTEGER ILON, JLAT
C local coordinates for (ILON, JLAT, LVER)
      INTEGER ILLON, JLLAT, LLVER
      SAVE ILLON, JLLAT, LLVER
C
C---- External Functions -----------------------------------------------
C
C spatial latitude/longitude grid
      EXTERNAL GLAT_P, GLON_P
      REAL GLAT_P,GLON_P
C relative weights for physical grid
      EXTERNAL WEIGHT_P
      REAL WEIGHT_P
C
C---- Initialized Variables --------------------------------------------
C
C height timeseries for point (longitude,latitude)
C (near Boulder, Colorado)
      DATA PLON, PLAT / 255.0, 40.0 /
C
C---- Executable Statements --------------------------------------------
C
C     Calculate common constants.
      PI  = 4.0D0*ATAN(1.0D0)
C
C     energetics analysis on model grid
C
      IF (ME .EQ. 0) WRITE(6,199) NSTEP, TAU
  199   FORMAT (/, ' CONSERVATION ANALYSIS FOLLOWS FOR NSTEP = ', 
     &          I4,', TAU = ', 0PF6.2, ' HRS')
C
C     Initialize local summation variables for energy totals.
      TMP(TOTE)  = 0.0
      TMP(TPE)   = 0.0
      TMP(TAM)   = 0.0
      TMP(TAPE)  = 0.0
      TMP(TMASS) = 0.0
C
C     Compute gridpoint energetics measures.
      DO J=1,NLLAT_P                                  
C       weight independent of longitude
        WTS = WEIGHT_P(1,J)
        DO I=1,NLLON_P                                   
          IF (FTOPO) THEN
            HFLUID = H(I,J) - MOUNT(I,J)
            WSA(1,I,J) = 0.5*ZETA(I,J)**2/HFLUID*WTS
            WSA(4,I,J) = HFLUID*WTS
            WSA(5,I,J) = ((U(I,J)**2+V(I,J)**2)*HFLUID 
     &                  + (H(I,J)**2 - MOUNT(I,J)**2)*GRAV)/2.0*WTS
          ELSE
            HFLUID = H(I,J) 
            WSA(1,I,J) = 0.5*ZETA(I,J)**2/HFLUID*WTS
            WSA(4,I,J) = HFLUID*WTS
            WSA(5,I,J) = (U(I,J)**2+V(I,J)**2 + GRAV*HFLUID)
     &                * HFLUID/2.0*WTS
          ENDIF
          WSA(2,I,J) = DIV(I,J)*WTS
          WSA(3,I,J) = ZETA(I,J)*WTS
        ENDDO
      ENDDO
C
C     Finish conservation analysis by summing all of the 
C     contributions (using a processor-independent ordering).
      CALL PGRIDSUM(5, 5, NLLON_P, NLLAT_P, WSA, WSB, TMP)
C
C     Check for data array overflow.
      IF (ICTR .GE. NGRPHS) THEN
        WRITE (0,905) NGRPHS
  905   FORMAT(/,' PTSWM: FATAL ERROR IN SUBROUTINE NRGTCS:',/,
     &           ' STORAGE FOR CONSERVATION PLOTS INSUFFICIENT',/,
     &           ' INCREASE PARAMETER NGRPHS = ', I4)               
        STOP 
      ENDIF 
C
C     Save initial values for subsequent normalization.
      IF (ICTR .EQ. 1) THEN
C
        TOTEN = TMP(TOTE)
        PEN   = TMP(TPE)
        TMSSN = TMP(TMASS)
C
C       Find closest gridpoint for height time series and check for 
C       correctness.
C
        ILON = 1 + NINT(PLON/360.0*NLON)
        IF ((ILON .LT. 1) .OR. (ILON .GT. NLON)) THEN
          IF (ME .EQ. 0) WRITE(0, 906)
  906     FORMAT(/,' PTSWM: FATAL ERROR IN SUBROUTINE NRGTCS:',/,
     &           ' OUT OF BOUND LONGITUDE')
          STOP
        ENDIF
C
        JLAT = NINT((90.0-PLAT)/180.0*NLAT)
        IF ((JLAT .LT. 1) .OR. (JLAT .GT. NLAT)) THEN
          IF (ME .EQ. 0) WRITE(0, 907)
  907     FORMAT(/,' PTSWM: FATAL ERROR IN SUBROUTINE NRGTCS:',/,
     &           ' OUT OF BOUND LATITUDE')
          STOP
        ENDIF
C
        CALL GRIDMAP_P(ILON, JLAT, 1, ILLON, JLLAT, LLVER)
        IF (ILLON .NE. -1) THEN
C
          PLONG = GLON_P(ILLON)/PI*180.0
          IF (ABS(PLONG-PLON) .GT. 360.0/NLON) THEN
            WRITE(0,908)
  908       FORMAT(/,' PTSWM: FATAL ERROR IN SUBROUTINE NRGTCS:',/,
     &           ' ERROR IN GRIDPOINT CHOICE')
            STOP
          ENDIF
C
          PLATG = GLAT_P(JLLAT)/PI*180.0
          IF (ABS(PLATG-PLAT) .GT. 180.0/NLAT) THEN
            WRITE(0,908)
            STOP 
          ENDIF
C
          WRITE(6,909) PLATG, PLONG
  909     FORMAT (/, ' INITIAL DATA FOR NORMALIZATION',/,
     &               ' GRIDPOINT LATITUDE    = ',F8.2,/,
     &               ' GRIDPOINT LONGITUDE   = ',F8.2)
C
        ENDIF
C
      ENDIF
C
      IF (ILLON .NE. -1) THEN
C
        WRITE(6,986) TMP(TMASS),TMP(TOTE),TMP(TAPE),TMP(TAM),
     &               TMP(TPE),H(ILLON,JLLAT)
  986   FORMAT (/, 
     &            ' MASS                  = ',1PE16.9,/,
     &            ' TOTAL ENERGY          = ',1PE16.9,/,
     &            ' VORTICITY             = ',1PE16.9,/,
     &            ' DIVERGENCE            = ',1PE16.9,/,
     &            ' POTENTIAL ENSTROPHY   = ',1PE16.9,/,
     &            ' GRIDPOINT HEIGHT      = ',1PE16.9)
C
C       Store relative errors for later plotting.
C
C       total energy
        IF (TOTEN .NE. 0.0) THEN
          TE(ICTR) = (TMP(TOTE)-TOTEN)/TOTEN 
        ELSE
          TE(ICTR) = TMP(TOTE)
        ENDIF
C
C       potential enstrophy
        IF (PEN .NE. 0.0) THEN
          PE(ICTR) = (TMP(TPE)-PEN)/PEN
        ELSE
          PE(ICTR) = TMP(TPE)
        ENDIF
C
C       global vorticity (approximately zero)
        APE(ICTR) = TMP(TAPE)
C
C       global divergence (approximately zero)
        AAM(ICTR) = TMP(TAM)
C
C       total mass
        IF (TMSSN .NE. 0.0) THEN
          TMSS(ICTR) = (TMP(TMASS)-TMSSN)/TMSSN
        ELSE
          TMSS(ICTR) = TMP(TMASS)
        ENDIF
C
C       gridpoint for height time series
        HTS(ICTR) = H(ILLON,JLLAT)
C
C       save model time
        TME(ICTR) = TAU
C
      ENDIF
C                                                                   
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE GRIDMAP_P(I, J, L, LI, LJ, LL)
C
C This subroutine calculates the local coordinates (LI,LJ,LL) in the
C standard physical space partitioning corresponding to the global
C coordinates (I,J,L), where I is the longitude index, J is the latitude
C index, and L is the vertical index. If these coordinates are not
C "owned" by this processors, then LI, LJ, and LL are all set to -1.
C
C called by: NRGTCS
C calls:
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C     
C---- Parameters -------------------------------------------------------
C
#     include "params.i"
C     
C---- Common Blocks ----------------------------------------------------
C     
C domain decomposition definition variables
#     include "physical.i"
C
C---- Arguments --------------------------------------------------------
C
C     Input
C
C longitude index
      INTEGER I
C latitude index
      INTEGER J
C vertical index
      INTEGER L
C
C     Output
C
C local longitude index
      INTEGER LI
C local latitude index
      INTEGER LJ
C local vertical index
      INTEGER LL
C
C---- Local Variables --------------------------------------------------
C
C loop indices
      INTEGER II, JJ
C
C---- Executable Statements --------------------------------------------
C
C     Find longitude index.
      LI = -1
      DO II=1,NLLON_P
        IF (LONTRUE_P(II) .EQ. I) LI = II
      ENDDO
C
C     Find latitude index.
      LJ = -1
      DO JJ=1,NLLAT_P
        IF (LATTRUE_P(JJ) .EQ. J) LJ = JJ
      ENDDO
C
C     Find vertical index.
C     (Vertical is not partitioned in physical space partition.)
      LL = L
C
      IF ((LI .EQ. -1) .OR. (LJ .EQ. -1) .OR. (LL .EQ. -1)) THEN
        LI = -1
        LJ = -1
        LL = -1
      ENDIF
C
      RETURN
      END
