function [A, b, x, ProbInfo] = PRblur(varargin) 
% PRblur Generates data for use in image deblurring problems
%
% [A, b, x, ProbInfo] = PRblur
% [A, b, x, ProbInfo] = PRblur(n)
% [A, b, x, ProbInfo] = PRblur(n, options)
% [A, b, x, ProbInfo] = PRblur(options)
%
% This function can generate a variety of image deblurring test problems.
% Specific test problems can be generated by the PRblur___ functions.
%
% Input:
%  n      -  size of the image. Can be either a scalar n (in this case 
%            the size is n x n) or a vector [nrow, ncol] (in this case the
%            size is (nrow x ncol).
%            Default: n = 256.
%  options - Structure containing the following optional fields:
%    trueImage  : test image of size n, of type numeric, 2-D only,
%                 or character string indicating
%                 'pattern1'  : geometrical image
%                 'pattern2'  : geometrical image
%                 'sppattern' : sparse (edges of ) a geometrical image
%                 'ppower'    : random image with patterns of nonzero pixels
%                 'smooth'    : very smooth image
%                 'dot2'      : two small Gaussian shaped dots, e.g., a
%                               binary star
%                 'dotk'      : n/2 small Gaussian shaped dots, e.g., stars
%                               (placement is random, reset using rng(0))
%                 'satellite' : satellite test image 
%                 'hst'       : image of the Hubble space telescope
%                 Default: 'hst'.
%                 This image is then stored in the output vector x.
%    PSF        : point spread function. Can be either an array containing
%                 a PSF, of type double,
%                 or a character string to specify the type of PSF
%                 to be constructed. The following choices give spatially
%                 invariant blur:
%                 'gauss'    : Gaussian blur
%                 'defocus'  : out of focus blur
%                 'speckle'  : atmospheric speckle blur
%                 'shake'    : random, shaking motion
%                 'motion'   : linear motion with angle 45 degrees
%                 The following choice gives spatially variant blur:
%                 'rotation' : rotational blur round the image center.
%                 Default is 'gauss'.
%    BlurLevel  : If choosing one of the built-in PSFs, this sets the
%                 severity of the blur to one of the following:
%                 'mild'
%                 'medium'
%                 'severe'
%                 Default is 'medium'
%    BC         : Specify boundary condition:
%                 'zero'
%                 'periodic'
%                 'reflective' (or 'neumann' or 'reflexive')
%                 Default: 'reflective'
%                 Note that in this case an extended (or padded) test image
%                 is blurred using 'zero' boundary conditions, and then the 
%                 central subimage of size n is extracted from the exact and
%                 the blurred image. No inverse crime is committed, 
%                 i.e., A*x ~= b.
%    CommitCrime: To get an exact system Ax = b (i.e., commit the inverse
%                 crime), set this to:
%                 'on'
%                 Default is 'off' (do not commit the inverse crime).
%
% Output:   
%  A        - blurring matrix, which is a either a psfMatrix object in
%             the case of spatically invariant blur, or a sparse matrix
%             in the case of spatially variant blur
%  b        - blurred vector (i.e., blurred image with stacked columns)
%  x        - image vector, i.e., exact (unknown) image with stacked columns
%  ProbInfo - structure whose fields contain information about problem:
%               problemType : kind of test problem generated
%                             (in this case: 'deblurring')
%               xType       : solution type (in this case 'image2D')
%               bType       : data type (in this case 'image2D')
%               xSize       : size of image x
%               bSize       : size of image b
%               psf         : point spread function
%
% See also: PRblurdefocus, PRblurgauss, PRblurmotion, PRblurrotation,
% PRblurshake, PRblurspeckle, PRdiffusion, PRinvinterp2, PRnmr,
% PRseismic, PRspherical, PRtomo, PRnoise, PRshowb, PRshowx, fspecial

% Silvia Gazzola, University of Bath
% Per Christian Hansen, Technical University of Denmark
% James G. Nagy, Emory University
% April, 2018.

% This file is part of the IR Tools package and is distributed under the 
% 3-Clause BSD License. A separate license file should be provided as part 
% of the package.

% Set default values for options.
defaultopt = struct('trueImage', 'hst', 'PSF', 'gauss', ...
    'BlurLevel', 'medium', 'BC', 'reflective', 'CommitCrime', 'off');
  
% If input is 'defaults,' return the default options in X
if nargin == 1 && nargout <= 1 && strcmp(varargin,'defaults')
    A = defaultopt;
    return;
end

% Check for acceptable number of optional input arguments
switch length(varargin)
    case 0
        n = []; options = [];
    case 1
        if isa(varargin{1}, 'double')
            n = varargin{1}; options = [];
        else
            n = []; options = varargin{1};
        end
    case 2
        if isa(varargin{1}, 'double')
            n = varargin{1}; options = varargin{2};
        else
            n = varargin{2}; options = varargin{1};
        end
    otherwise
        error('Too many input parameters')
end

if isempty(options)
    options = defaultopt;
end

options = PRset(defaultopt, options);

image       = PRget(options, 'trueImage',      [], 'fast');
PSF         = PRget(options, 'PSF',         [], 'fast');
BlurLevel   = PRget(options, 'BlurLevel',   [], 'fast');
BC          = PRget(options, 'BC',          [], 'fast');
CommitCrime = PRget(options, 'CommitCrime', [], 'fast');

if strcmpi(BC, 'reflective') || strcmpi(BC, 'neumann')
    BC = 'reflexive';
end

% Initialization: set the default inputs
default_n = [256, 256];

if isnumeric(image)
    % Make sure user input image is square. If not, pad with zeros.
    if ~ismatrix(image)
        error('Expected user supplied true image to be a 2-D array of type numeric')
    else
        [nrow, ncol] = size(image);
        if ncol == 1
            error('Expected user supplied trueImage to be a 2D image array, not a vector')
        end
        if nrow ~= ncol
            padsize = [max(nrow,ncol)-nrow, max(nrow,ncol)-ncol];
            image = padarray(image, padsize, 'post');
        end
    end
end

[n, nbig, getidx] = GetDimensionInfo(n, PSF, image, default_n, CommitCrime);

%
% Defining the test image (depending on whether we commit the crime or not)
% Depending on whether we commit the crime or not path(path,'./PRblur_tools')
if strcmpi(image, 'pattern1')
    image = PatternTestImage1(n(1));
elseif strcmpi(image, 'pattern2')
    image = PatternTestImage2(n(1));
elseif strcmpi(image, 'sppattern')
    image = SpPatternTestImage1(n(1));
elseif strcmpi(image, 'ppower')
    image = ppower(nbig(1));
elseif strcmpi(image, 'smooth')
    image = smooth(nbig(1));
elseif strcmpi(image, 'dot2')
    image = StarImageRandom(nbig(1), 2);
elseif strcmpi(image, 'dotk')
    image = StarImageRandom(nbig(1), fix(nbig(1)/2));
elseif strcmpi(image, 'satellite')
    load('satellite','image')
% Avoid use of IPT     image = imresize(image,n(1)/size(image,1));
    [X,Y] = meshgrid(linspace(1,size(image,1),n(1)));
    image = interp2(image,X,Y,'linear');
    image = image/max(image(:));
elseif strcmpi(image, 'hst')
    image = double(imread('HSTgray.jpg'));
% Avoid use of IPT     image = imresize(image,n);
    [X,Y] = meshgrid(linspace(1,size(image,1),n(1)));
    image = interp2(image,X,Y,'linear');
    image = image/max(image(:));
else
    if ~isnumeric(image)
        error('incorrect image choice')
    end
    if strcmpi(CommitCrime, 'on') && any(n ~= size(image))
        error('When using CommitCrime ''on'', n should be same as size of input image')
    end
end    
image = padIm(image, nbig);

% Defining the PSF
if strncmpi(PSF,'rot',3)
    if n(1)~=n(2), error('For rotational blur, the image must be square'), end
    % In the case of rotaional blur, create the sparse blurring matrix.
    if strcmpi(BlurLevel,'mild')
        Rot = 15;
    elseif strcmpi(BlurLevel,'medium')
        Rot = 45;
    elseif strcmpi(BlurLevel,'severe')
        Rot = 90;
    else
        error('Invalid Blur Level')
    end
    RotStep = 1;
    
elseif isa(PSF, 'double') && ~isempty(PSF)
    % If PSF is an array then use it as it is, but may need to pad and
    % adjust center.
    np = size(PSF);
    idx1 = round(np/2); idx2 = round((np+1)/2);
    [ci, cj] = find(PSF == max(max(PSF(idx1(1):idx2(1),idx1(2):idx2(2)))));
    center = [ci, cj];
    [PSF, center] = padIm(PSF, n, center);
    % warning('We will do this, but note that you are committing an
    % inverse crime with the boundary condition.')
    %CommitCrime = 'on';
else
    % Otherwise create the PSF according to specification.
    [PSF, center] = PSFget(PSF,n,BlurLevel);
end

%
if strcmpi(CommitCrime, 'on')
    % If we commit crime, then we define the blurring matrix considering
    % the boundary conditions given by the user; in this case: Ax = b
    x = image(:);
    imsize = size(image);
    % Defining the blurring matrix.
    if strncmpi(PSF,'rot',3)
        A = rotBlur(Rot,RotStep,size(image,1),BC);
    else
        A = psfMatrix(PSF, BC, center, imsize);
    end
    b = A*x;    
else
    % If we don't commit crime, then we define a blurring matrix considering
    % zero boundary conditions; this matrix multiplies the enlarged or padded
    % image; in this case: Ax ~= b
    if strncmpi(PSF,'rot',3)
        A = rotBlur(Rot,RotStep,size(image,1),'zero');
        b = A*image(:);
        nb = sqrt(length(b));
        b = reshape(b,nb,nb);
    else
        A = psfMatrix(PSF, 'zero', center, size(image));
        b = A*image;
    end
    % Cutting the exact test image, and the blurred image.
    image = image(getidx(1):getidx(1)+n-1,getidx(2):getidx(2)+n-1);
    b = b(getidx(1):getidx(1)+n-1,getidx(2):getidx(2)+n-1);
    x = image(:);
    % Rescale x and b, in case the maximum was outside the border.
    b = b/max(x(:));
    x = x/max(x(:));
    % Define the blurring matrix with the BC specified by the user.
    if strncmpi(PSF,'rot',3)
        A = rotBlur(Rot,RotStep,size(image,1),BC);
        b = A*image(:);
    else
        A = psfMatrix(PSF, BC, center);
        b = b(:);
    end
end

% Providing information about the test problem.
ProbInfo.problemType = 'deblurring';
ProbInfo.xType = 'image2D';
ProbInfo.xSize = n;
ProbInfo.bType = 'image2D';
ProbInfo.bSize = n;
ProbInfo.psf = PSF;
if isa(A, 'psfMatrix')
    ProbInfo.psfCenter = center;
end

% Subfunctions ===========================================================

function [PSF, center] = PSFget(psfKind,n,BlurLevel)
%
% Generate the PSF with a user-defined level of blurring.
if strcmpi(psfKind,'gauss')
    % "qualitative" level of blurring
    if strcmpi(BlurLevel,'mild')
        y = 2;
    elseif strcmpi(BlurLevel,'medium')
        y = 4;
    elseif strcmpi(BlurLevel,'severe')
        y = 6;
    else
        error('Invalid Blur Level')
    end
    [PSF, center] = psfGauss(n, y);
elseif strcmpi(psfKind,'defocus')
    if strcmpi(BlurLevel,'mild')
        R = min(fix((n+1)/32) - 1);
    elseif strcmpi(BlurLevel,'medium')
        R = min(fix((n+1)/16) - 1);
    elseif strcmpi(BlurLevel,'severe')
        R = min(fix((n+1)/8) - 1);
    else
        error('Invalid Blur Level')
    end
    [PSF, center] = psfDefocus(n, R);
elseif strcmpi(psfKind,'speckle')
    % "qualitative" level of blurring
    if strcmpi(BlurLevel,'mild')
        Cn2 = 3.4765e-18;
    elseif strcmpi(BlurLevel,'medium')
        Cn2 = 2.16941e-17;
    elseif strcmpi(BlurLevel,'severe')
        Cn2 = 8.90505e-17;
    else
        error('Invalid Blur Level')
    end
    [PSF, center] = psfSpeckle(n, Cn2);
elseif strcmpi(psfKind,'motion')
    if strcmpi(BlurLevel,'mild')
        L = 5;
    elseif strcmpi(BlurLevel,'medium')
        L = 10;
    elseif strcmpi(BlurLevel,'severe')
        %L = 20;
        L = 50;
    else
        error('Invalid Blur Level')
    end
    [PSF, center] = psfMotion(n, L);
elseif strcmpi(psfKind,'shake')
    [PSF, center] = psfShake(n, BlurLevel);
else
    error('Invalid input type for psf')
end

function [PSF, center] = psfGauss(imsize, y, center)
%
%  Create a Gaussian PSF, with specified center, from parameters
%  in the vector y, and having dimension imsize.  The Gaussian
%  has the form:
%      P = exp(-x'*inv(C)*x)
%  where C is a 2-by-2 covariance matrix, with entries
%      C = [y(1)^2 y(3)^2; y(3)^2 y(2)^2]
%
%  If center is not specified, assume it is 
%      fix(m/2, n/2)
%
%  If size(y) == 1, then y(1) = y(2), and y(3) = 0
%  If size(y) == 2, then y(3) = 0.
%

m = imsize(1); n = imsize(2); 
if nargin < 3
  center = fix([m,n]/2);
end
if length(y) == 1
  y = [y; y; 0];
end
if length(y) == 2
  y = y(:);
  y = [y; 0];
end
d = (y(1)*y(2))^2 - y(3)^4;
if d <= 0
  error('Need (y(1)*y(2))^2-y(3)^4 > 0')
end

a = y(2)^2/d;
b = y(1)^2/d;
c = -y(3)^2/d;

[J,I] = meshgrid(1:n,1:m);

I = I - center(1);
J = J - center(2);
  
PSF = exp(-(a*(I.^2) + 2*c*(I.*J) + b*(J.^2))/2);
PSF = PSF/(2*pi*sqrt(d));
%
function [PSF, center] = psfDefocus(dim, R)
%PSFDEFOCUS Array with point spread function for out-of-focus blur.
%
%function [PSF, center] = psfDefocus(dim, R)
%
%            PSF = psfDefocus(dim);
%            PSF = pdfDefocus(dim, R);
%
%  Construct a defocus blur point spread function,  which is
%  1/(pi*R*R) inside a circle of radius R, and zero otherwise.
%
%  Input:
%      dim  Desired dimension of the PSF array.  For example,
%             PSF = psfDefocus(60) or
%             PSF = psfDefocus([60,60]) creates a 60-by-60 array,
%           while 
%             PSF = psfDefocus([40,50]) creates a 40-by-50 array.
%        R  Radius of defocus
%             Default is min(fix((dim+1)/2) - 1)
%
%  Output:
%      PSF  Array containing the point spread function.
%   center  [row, col] gives index of center of PSF

% Reference: See Chapter 3,
%            "Deblurring Images - Matrices, Spectra, and Filtering"
%            by P. C. Hansen, J. G. Nagy, and D. P. O'Leary,
%            SIAM, Philadelphia, 2006.
% Last revised April 25, 2007.

%
% Check inputs and set default parameters.
%
if (nargin < 1)
   error('dim  must be given.')
end
l = length(dim);
if l == 1
  m = dim;
  n = dim;
else
  m = dim(1);
  n = dim(2);
end
center = fix(([m,n]+1)/2);
if (nargin < 2)
   R = min(center - 1);
end

if R == 0
  % If R=0, then the PSF is a delta function, so the blurring matrix is 
  % the identity.
  PSF = zeros(m,n);
  PSF(center(1),center(2)) = 1;
elseif R <= min(center - 1)
  PSF = ones(m,n)/(pi*R*R);
  k = (1:max(m,n))';
  idx1 = meshgrid((k-center(1)).^2)' + meshgrid((k-center(2)).^2) > R^2;
  idx = idx1(1:m,1:n);
  PSF(idx) = 0;
else
    error('Blurring parameter R too big. It should be R <= min(center - 1)')
end
PSF = PSF / sum(PSF(:));

%
function [PSF, center] = psfMotion(imsize, L)
%
% Create a PSF for linear motion blur of length L and dimension imsize.

m = imsize(1); n = imsize(2); 
center = fix([m,n]/2);
% Modify L if the image is small.
L = min([L, min( m-center(1)+1 , n-center(2)+1 )]);
d = linspace(1,0,L);
P = diag(d) + 0.5*diag(d(1:end-1),1);
PSF = zeros(m,n);
PSF(center(1)+(0:L-1),center(2)+(0:L-1)) = P;

function I = PatternTestImage1(n)
%
%  Create a gray scale test image. The input value, n, is the 
%  dimension of the image, and should be a power of 2 (e.g., n = 256).

n2 = floor(n/2); n4 = floor(n/4); n8 = floor(n/8);

% First create a circle with radius r, center = (h,k).
C = ones(n2);
r = n8; h = n4; k = n4;
i = (1:n2)'; j = (1:n2)';
idx = meshgrid((i-h).^2) + meshgrid((j-k).^2)' > r^2;
C(idx) = 0;

% Now create a triangle, with vertices given by xv and yv.
xv = [n4, n8, n2-n8];
yv = [n8, n2-n8, n2-n8];
X = meshgrid(1:n2); Y = meshgrid(1:n2)';
T = inpolygon(X, Y, xv, yv);

%  Now create a rectangle.
xv = [n8, n2, n2, n8];
yv = [n8, n8, n2-n8, n2-n8];
X = meshgrid(1:n2); Y = meshgrid(1:n2)';
R1 = inpolygon(X, Y, xv, yv);
R = [R1, fliplr(R1)];

%  Finally, piece these together to form the image.
I = [C, T; R];
% Bring the image back to the original size
I = [I, zeros(2*n2,n-2*n2); zeros(n-2*n2,n)];

function I = PatternTestImage2(n)

I = zeros(n);
n2 = round(n/2); n3= round(n/3); n6 = round(n/6); n12 = round(n/12);
% Add a large ellipse.
T = zeros(n6,n3);
for i=1:n6
    for j=1:n3
      if ( (i/n6)^2 + (j/n3)^2 < 1 ), T(i,j) = 1; end
    end
end
T = [fliplr(T),T];  
T = [flipud(T);T];
I(2+(1:2*n6),n3-1+(1:2*n3)) = T;
% Add a smaller ellipse.
T = zeros(n6,n3);
for i=1:n6
    for j=1:n3
      if ( (i/n6)^2 + (j/n3)^2 < 0.6 ), T(i,j) = 1; end
    end
end
T = [fliplr(T),T];
T = [flipud(T);T];
I(n6+(1:2*n6),n3-1+(1:2*n3)) = I(n6+(1:2*n6),n3-1+(1:2*n3)) + 2*T;
% Correct for overlap.
f = find(I==3);
I(f) = 2*ones(size(f));
% Add a triangle.
T = triu(ones(n3,n3));
[mT,nT] = size(T);
I(n3+n12+(1:nT),1+(1:mT)) = 3*T;
% Add a cross.
T = zeros(2*n6+1,2*n6+1);
[mT,nT] = size(T);
T(n6+1,1:nT) = ones(1,nT);
T(1:mT,n6+1) = ones(mT,1);
I(n2+n12+(1:mT),n2+(1:nT)) = 4*T;
I = I/4;

function I = SpPatternTestImage1(n)
%
%  Create a gray scale test image. The input value, n, is the 
%  dimension of the image, and should be a power of 2 (e.g., n = 256).

n2 = floor(n/2); n4 = floor(n/4); n8 = floor(n/8);

% First create a circle with radius r, center = (h,k).
C = ones(n2);
r = n8; h = n4; k = n4;
i = (1:n2)'; j = (1:n2)';
idx = meshgrid((i-h).^2) + meshgrid((j-k).^2)' > r^2;
C(idx) = 0;

% Now create a triangle, with vertices given by xv and yv.
xv = [n4, n8, n2-n8];
yv = [n8, n2-n8, n2-n8];
X = meshgrid(1:n2); Y = meshgrid(1:n2)';
T = inpolygon(X, Y, xv, yv);

%  Now create a rectangle.
xv = [n8, n2, n2, n8];
yv = [n8, n8, n2-n8, n2-n8];
X = meshgrid(1:n2); Y = meshgrid(1:n2)';
R1 = inpolygon(X, Y, xv, yv);
R = [R1, fliplr(R1)];

%  Finally, piece these together to form the image.
I = [C, T; R];
% Bring the image back to the original size
I = [I, zeros(2*n2,n-2*n2); zeros(n-2*n2,n)];

D = spdiags([ones(n,1),-ones(n,1)], 0:1, n, n);
ID1 = abs(D*I); 
ID2 = abs(I*D');
I = ID1 + ID2;
I = I/max(max(I));

function I = smooth(n,p)
%SMOOTH Creates a 2D test image of a smooth function

% Per Christian Hansen, May 8, 2012, DTU Compute.

if nargin==1, p = 4; end

% Generate the image.
[L,M] = meshgrid(1:n);
sigma = 0.25*n;
c = [0.6*n 0.6*n; 0.5*n 0.3*n; 0.2*n 0.7*n; 0.8*n 0.2*n];
a = [1 0.5 0.7 0.9];
I = zeros(n);
for i=1:p
    I = I + a(i)*exp( - (L-c(i,1)).^2/(1.2*sigma)^2 - (M-c(i,2)).^2/sigma^2);
end
I = I/max(I(:));

function I = ppower(n,relnz,p,seed)
%PPOWER Creates a 2D test image with patterns of nonzero pixels

% Per Christian Hansen and Jakob Sauer Jrgensen, Oct. 1, 2014.

% Reference: J. S. Jrgensen, E. Y. Sidky, P. C. Hansen, and X. Pan,
% Empirical average-case relation between undersampling and sparsity in
% X-ray CT, submitted.

if nargin<2 || isempty(relnz), relnz = 0.3; end
if nargin<3 || isempty(p), p = 2; end
if nargin==4, rng(seed), end

P = randn(n,n);
[L,M] = meshgrid(1:n);
U = ( ( (2*L-1)/n - 1).^2 + ( (2*M-1)/n - 1).^2 ).^(-p/2);
I = U.*exp(2*pi*sqrt(-1)*P);
I = abs(ifft2(I));
f = sort(I(:),'descend');
k = round(relnz*n^2);
I( I < f(k) ) = 0;
I = I/max(I(:));

function A = rotBlur(Rot,RotStep,n,BC)
% Create blurring matrix for arbitrary rotational motion blur.
%   A = CreateBlurMatrix(Rot,RotStep,n,Boundaries)
% Input:
%   RotVec   Rotation in degrees.
%   RotStep  Rotation increment when computing rotational blur.
%   n        Size of the n-by-n image the matrix should be applied to.
%   BC       Boundary condition setting. The options are specified
%            in a string which may contain the following:
%            'zero'       : Zero boundary conditions.
%            'periodic'   :  Periodic boundary conditions.
%            'reflective' : Reflexive boundary conditions.
% Output:
%   A        The blurring matrix.

RotMatrix = zeros(3);
RotMatrix(1,1) = cos((pi/180)*RotStep);
RotMatrix(2,1) = sin((pi/180)*RotStep);
RotMatrix(1,2) = -RotMatrix(2,1);
RotMatrix(2,2) = RotMatrix(1,1);
RotMatrix(3,3) = 1;
A = speye(n^2);

for k = 1:Rot/RotStep
    
    % Setup transformation matrix
    T = [ (n-1)/2 0 (n-1)/2; 0 (n-1)/2 (n-1)/2; 0 0 1 ]*RotMatrix^k*...
        [ 2/(n-1) 0 -1; 0 2/(n-1) -1; 0 0 1 ];
    
    [Y,X] = ndgrid(0:n-1,0:n-1);
    TformdCoord = T\[ X(:) Y(:) ones(numel(X),1) ]';
    XInterp = TformdCoord(1,:)' + 1;
    YInterp = TformdCoord(2,:)' + 1;
        
    switch lower(BC)
       
        case 'zero' % Zero boundary conditions.
            RowIdx = find( XInterp >= 1 & XInterp < n & ...
                YInterp >= 1 & YInterp < n );
            XInterp = XInterp(RowIdx);
            YInterp = YInterp(RowIdx);
                
        case 'periodic' % Periodic boundary conditions.
            RowIdx = (1:n^2)';
            XInterp( XInterp >= n ) = mod( XInterp( XInterp >= n ), n );
            XLeft = n-1 + mod( XInterp( XInterp < 1 ), n );
            XLeft( XLeft >= n+1 ) = XLeft( XLeft >= n+1 ) - n;
            XLeft( XLeft >= n & XLeft < n+1 ) = ...
                XLeft( XLeft >= n & XLeft < n+1 ) - 1;
            XInterp( XInterp < 1 ) = XLeft;
            YInterp( YInterp >= n ) = mod( YInterp( YInterp >= n ), n );
            YUp = n-1 + mod( YInterp( YInterp < 1 ), n );
            YUp( YUp >= n+1 ) = YUp( YUp >= n+1 ) - n;
            YUp( YUp >= n & YUp < n+1 ) = YUp( YUp >= n & YUp < n+1 ) - 1;
            YInterp( YInterp < 1 ) = YUp;
            
        case 'reflexive' % Reflexive boundary conditions.
            RowIdx = (1:n^2)';
            XLeft = XInterp( XInterp < 1 );
            XLeft( XLeft < 1 & XLeft >= 0 ) = ...
                2 - mod( XLeft( XLeft < 1 & XLeft >= 0 ), n );
            XLeft( XLeft < 0 ) = n + 2 - mod( XLeft( XLeft < 0 ), n );
            XLeft( XLeft >= n ) = mod( XLeft( XLeft >= n ), n-1 );
            XInterp( XInterp < 1 ) = XLeft;
            XRight = XInterp( XInterp >= n );
            XRight( XRight >= n ) = n - mod( XRight( XRight >= n ), n ); 
            XRight( XRight == n ) = XRight( XRight == n ) - 1e-10;
            XRight( XRight >= 0 & XRight < 1 ) = ...
                XRight( XRight >= 0 & XRight < 1 ) + n-1;
            XInterp( XInterp >= n ) = XRight;
            YUp = YInterp( YInterp < 1 );
            YUp( YUp < 1 & YUp >= 0 ) = ...
                2 - mod( YUp( YUp < 1 & YUp >= 0 ), n );
            YUp( YUp < 0 ) = n + 2 - mod( YUp( YUp < 0 ), n );
            YUp( YUp >= n ) = mod( YUp( YUp >= n ), n-1 );
            YInterp( YInterp < 1 ) = YUp;
            YDown = YInterp( YInterp >= n );
            YDown( YDown >= n ) = n - mod( YDown( YDown >= n ), n ); 
            YDown( YDown == n ) = YDown( YDown == n ) - 1e-10;
            YDown( YDown >= 0 & YDown < 1 ) = ...
                YDown( YDown >= 0 & YDown < 1 ) + n-1;
            YInterp( YInterp >= n ) = YDown;
           
        otherwise
            error('Boundary condition not supported.')
    end
    
    XFloor = floor(XInterp);
    YFloor = floor(YInterp);
    I = [ RowIdx; RowIdx; RowIdx; RowIdx ];
    J = [ (XFloor-1)*n + YFloor;
          XFloor*n + YFloor;
          (XFloor-1)*n + YFloor + 1;
          XFloor*n + YFloor + 1      ];
    InterpWeights = ...
        [ (1-(XInterp-XFloor)).*(1-(YInterp-YFloor));
          (XInterp-XFloor).*(1-(YInterp-YFloor));
          (1-(XInterp-XFloor)).*(YInterp-YFloor);
          (XInterp-XFloor).*(YInterp-YFloor)          ];
    A = A + sparse(I,J,InterpWeights,n^2,n^2);    
end

A = A/(k+1);

%
function [PSF, center, phase] = psfSpeckle(imsize, Cn2)
%
%  Create a speckle PSF

if length(imsize) > 1
    m = imsize(1); n = imsize(2);
    if m ~= n
        warning('Sorry, but currently speckle blur is restricted square images. Resetting image size.')
        n = min(m, n);
    end
else 
    n = imsize;
end

% We'll assume the aperture diameter (in number of pixels) is half the
% size of the image size.  So if we have a 256-by-256 image, we'll use
% an aperture size of 128-by-128.
n_ap = n/2;
if n_ap ~= fix(n_ap)
    warning('Use even integers for input n. Resetting n = n + 1')
    n = n + 1;
    n_ap = n/2;
end
pupil_mask0 = PupilMask(n_ap,1);
pupil_mask = padarray(pupil_mask0, ceil([n_ap/2, n_ap/2]), 'pre');
pupil_mask = padarray(pupil_mask, floor([n_ap/2, n_ap/2]), 'post');

% pupil_mask0 is an n_ap -by- n_ap circle mask, which comes to the
% boundaries of this array.  If you want to use this to build an
% n-by-ny PSF, you need to pad this to size n-by-n.
%
% To get the normalized PSF (so that the pixel values sum to one),
% normalize the pupil_mask.
scaled_pupil_mask = n*pupil_mask/sqrt(sum(pupil_mask(:)));

[phase_layers, ~] = GenerateWavefront(n, [], Cn2);
phase = phase_layers{1};

PSF_shifted = abs( ifft2(scaled_pupil_mask.*exp(sqrt(-1)*phase)) ).^2;
PSF = fftshift(PSF_shifted);
center = [fix(n/2), fix(n/2)];

%=======================================

function [phi_symm, D_over_r0] = GenerateWavefront(N, D2, Cn2, nlayers)
%  function phi = GenerateWavefront(N, D, r0)
%
%  Inputs:
%  N          --- number of grid points in x-direction.
%  D2          --- diameter of the observation aperture [m]
%  Cn2       ---  structure parameter
%  nlayers   ---  the number of turbulent layers
%
%  Outputs:
%  phi_symm        ---  cell array, generated wavefront of "nlayers" layers
%  D2/r0pw   --- D_over_r0

%  last updated Sept 25, 2011

if nargin == 1, D2 = []; Cn2 = []; nlayers = []; end
if nargin == 2, Cn2 = []; nlayers = []; end
if nargin == 3, nlayers = []; end
if isempty(D2)
    D2 = 3.7;  %  diameter of the observation aperture [m]
               % JN: Changed this to 3.7, which is about the
               %     size of the Advanced Electro Optical System (AEOS)
               %     Telescope
end
if isempty(Cn2)
    %Cn2 = 1e-16;  %  structure parameter [m^(-2/3)], constant
    Cn2 = 5.5590e-17; % D_over_r0 = 5;
    %Cn2 = 8.1918e-015; %  in this case r0pw = 0.005, and D2/r0pw = 100
    %Cn2 = 5.6031e-016; % r0pw = 0.025, D2/r0pw = 20
end
if isempty(nlayers)
    %nlayers = 2;
    nlayers = 1;
end


%  determine geometry
wavelength = 0.744e-6;  %  optical wavelength [m]
                    % JN:  should be 0.744e-6
k = 2*pi / wavelength;  %  optical wavenumber [rad/m]
Dz = 50e3;  %  propagation distance [m]
            %  JN: This is the distance of the atmosphere through which 
            %      the light passes.  That is, above this distance the
            %      turbulence is too small to be considered.
            %      Could go possibly down to 25 kilometers.
            

%  use sinc to model pt source
%DROI = 4 * D2; % diam of obs-plane region of interest [m]
%D1 = wavelength *Dz /DROI;  %  width of central lobe [m]
%R = Dz;  %  wavefront radius of curvature [m]

%  PW coherence diameters [m]
p = linspace(0, Dz , 1e3);
r0pw = (0.423 * k^2 * Cn2  * Dz)^(-3/5);

l0 = 0;  %  inner scale [m]
L0= inf;  %outer scale [m]

%  log-amplitude variance
rytov_pw = 0.563 * k^(7/6)  * Dz^(5/6) * sum(Cn2 * (1-p/Dz).^(5/6) ...
    * (p(2) - p(1)));

if nlayers == 1
%    l0 = 0;  %  inner scale [m]
%    L0= inf;  %outer scale [m]
    
    delta = 10e-3;
   
    %  initialize array for phase screens
    %phi_asymm= zeros(N,N,n);
    phi_symm= cell(nlayers,1);
    
    [phi_symm{1}] = ft_phase_screen(r0pw,N, delta, L0, l0);
    
elseif nlayers > 1
    %  screen properties
    A = zeros(2, nlayers); %  matrix
    alpha = (0:nlayers-1) / (nlayers - 1);
    A(1,:) = ones(1, nlayers);
    A(2,:) = (1-alpha).^(5/6); %.* alpha.^(5/6);
    b = [r0pw.^(-5/3); rytov_pw/1.33*(k/Dz)^(5/6)];
    
    %  two methods solving Fried parameters for sublayers
    %  method I: normal equation
    if(rank(A) ~= 0)
        AAT = A*A';
        y = AAT\b;
        X_normal = A' * y;
    end
    
    %  method II: underdetermined LS problem with constrains
    %  initial guess
    %x0 = (nlayers/3*r0pw * ones(nlayers, 1)).^(-5/3);
    %x0 = zeros(3,1);
    %  objective function
    %fun = @(X)sum((A*X(:) - b).^2);
    %  constraints
    %x1 = zeros(nlayers, 1);
    %rmax = 0.1;  %  maximum Rytov number per partial prop
    %x2 = rmax/1.33*(k/Dz)^(5/6) ./A(2,:);
    %x2(A(2,:) ==0) = 50^(-5/3);
    %[X_LS, ~, ~, ~] = fmincon(fun, x0, [],[],[],[],x1, x2);
    %  check screen r0s
    r0scrn_normal = X_normal.^(-3/5);
    r0scrn_normal(isinf(r0scrn_normal)) = 1e6;
    
    %r0scrn_LS = X_LS.^(-3/5);
    %r0scrn_LS(isinf(r0scrn_LS)) = 1e6;
    % % %  check resulting r0pw & rytov_pw
    % bp_normal = A*X_normal(:);
    % [bp_normal(1)^(-3/5) bp_normal(2)*1.33*(Dz/k)^(5/6)];
    % [r0pw rytov_pw];
    %
    % bp_LS = A*X_LS(:);
    % [bp_LS(1)^(-3/5) bp_LS(2)*1.33*(Dz/k)^(5/6)];
    % [r0pw rytov_pw];
    
    n = nlayers;  %  number of planes
    z = (1:n-1) * Dz / (n-1);  % partial prop planes
    zt = [0 z];  % propagation plane locations
    %Delta_z = zt(2:n) - zt(1:n-1);  % propagation distances
    
    %  grid spacings
    alpha = zt / zt(n);
    %c =2;
    % D1p = D1 + c*wavelength*Dz/r0pw;
    % D2p = D2 + c*wavelength*Dz/r0pw;
    % deltan = linspace(0, 1.1*wavelength*Dz/D1p, 100);
    % delta1 = linspace(0, 1.1*wavelength*Dz/D2p, 100);
    delta1 = 10e-3;
    deltan = 10e-3;
    delta = (1-alpha) * delta1 + alpha * deltan;
    
    %  initialize array for phase screens
    %phi_asymm= zeros(N,N,n);
    phi_symm= cell(n,1);
    
    
    %  Choose Fried parameters for sublayers from either normal equation
    %  or minimization problem
    for idxscr = 1:n
        %phi_symm(:,:,idxscr) = ft_phase_screen(r0scrn_LS(idxscr), N, delta(idxscr), L0, l0);
        [phi_symm{idxscr}] = ft_phase_screen(r0scrn_normal(idxscr), N, delta(idxscr), L0, l0);
    end
    
end
D_over_r0 = D2/r0pw;

%=======================================

function [ phz] = ft_phase_screen(r0, N, delta, L0, l0)
%  function phz = ft_phase_screen(r0, N, delta, L0, l0)
%  Input
%   r0: Fried paramter;
%   N:  number of pixels along one dimension;
%   delta: side length of pixels;
%   L0: the outer scale;
%   l0: the inner scale.
%
%  Output
%   phz: the generated wavefront phase.

%  By Numerical Simulation of Optical Wave Propagation. Adjusted so that
%  the Fourier coefficients are Hermitian symmetric - when applying the
%  inverse Fourier transform, the results (wavefront phase) are directly
%  real.

% last updated Sept 25, 2011.

del_f = 1/(N*delta);
fx = (-N/2:N/2-1) * del_f;

[fx, fy] = meshgrid(fx);
[~, f] = cart2pol(fx, fy);
fm = 5.92/l0/(2*pi);
f0 = 1/L0;

PSD_phi = 0.023*r0^(-5/3) * exp((-f/fm).^2) ...
    ./ (f.^2 + f0^2).^(11/6);
PSD_phi(N/2+1, N/2+1) = 0;

% generated Hermitian symmetric Fourier coefficients .
cn_old = (randn(N) + 1i *randn(N)) .* sqrt(PSD_phi)*del_f;
cn = randn(N) + 1i*randn(N);
temp = diag((cn_old + cn_old')/2);
for k = 1:N
    cn(k,k) = temp(k);
end

if mod(N,2) == 1
    for k = 1:ceil(N/2)
        for j = 1:N
            cn(k,j) = conj(cn(mod(N-k+1,N)+1, mod(N-j+1,N)+1));
        end
    end
else
    cn(1,N/2+1) = cn(2,2);
    cn(N/2+1,1) = cn(3,3);
    for k = 1:N/2+1
        for j = 1:N
            cn(k,j) = conj(cn(mod(N-k+1,N)+1,mod(N-j+1,N)+1));
        end
    end
end
cn = cn.* sqrt(PSD_phi)*del_f;
phz = ifftshift(ifft2(ifftshift(cn)))*N^2;

%=======================================

function P = PupilMask(n, r1, r0)
%
%  P = PupilMask(n, r1, r0)
%  P = PupilMask(n, r1)
%
%  Constructs an n by n pupil mask with outer radius r1.  If only two input
%  arguments are given, the mask is circular.  Otherwise, it is annular
%  with inner radius r0.  The largest circle contained in the n by n square
%  has radius r1 = 1.
%
%  We assume n is an even integer. The central pixel has index
%  (n+1)/2.
%
%  Inputs:
%    n - size of mask to be constructed
%    r1 - outer radius
%    r0 - inner radius, if annular
%
%  Outputs:
%    mask - array of 1s and 0s representing the mask
%
% Author: John Bardsley, Sarah Knepper
% Date Created: 27 September 2009
% Date Last Modified: 27 September 2009
%

h = 2/n;
x = (-1:h:1-h)';
onevec = ones(n,1);
r = sqrt((x*onevec').^2 + (onevec*x').^2);
if nargin == 2
  P = (r <= r1);
else
  P = (r0 <= r) & (r <= r1);
end

function [PSF, center] = psfShake(nvec, BlurLevel)
% We assume square images, but the code below should work for rectangular
% images if we ever want to change.
m = nvec(1); n = nvec(2);
center = round([m, n]/2);
PointSource = zeros(m, n);
PointSource(center(1), center(2)) = 1;
if isa(BlurLevel,'numeric') && BlurLevel == 0
    PSF = PointSource;
    return
end

xdata = 0:0.25:1;
ydata = [100 50 25 12.5 6.25];
sfun = polyfit(xdata, ydata, 4);

if ~isa(BlurLevel, 'numeric')
    switch BlurLevel
        case 'mild'
            BlurLevelVal = 0.25; % mild blur
        case 'medium'
            BlurLevelVal = 0.5;  % medium blur
        case 'severe'
            BlurLevelVal = 0.75; % severe blur
        otherwise
            error('incorrect BlurLevel')
    end
else
    if 0 <= BlurLevel && BlurLevel <= 1
        BlurLevelVal = BlurLevel;
    else
        error('Incorrect value for BlurLevel')
    end
end
shift_scale = polyval(sfun, BlurLevelVal);
n_shifts = 10;   % Generate random points for the camera path
n_frames = 100;  % Fit the path points with a spline, and evaluate at
                 % this many values.
                 
dx_min = -n/shift_scale; dx_max = n/shift_scale;
dy_min = -m/shift_scale; dy_max = m/shift_scale;

dx0 = [0; (dx_max - dx_min)*rand(n_shifts-1,1) + dx_min];
dy0 = [0; (dy_max - dy_min)*rand(n_shifts-1,1) + dy_min];


dx = spline(1:n_shifts, dx0, linspace(1,n_shifts,n_frames));
dy = spline(1:n_shifts, dy0, linspace(1,n_shifts,n_frames));

% The following can be used if we want to include some a random
% rotation of the camera.  However, if we do, then the blur is no
% longer space invariant, so we'll need to construct a sparse matrix
% representation of the blurring operator.
dtheta = zeros(size(dx));

%  This is prehaps a bit clunky, but I wanted to use the same code as
%  is used for the case where rotation is included.
T = MotionBlurTransform(m/2, n/2, n_frames+1, dx, dy, dtheta(:));
PSF = ApplyMotionTransform(T, m, n, PointSource);

%
% --------------------
%

function T = MotionBlurTransform(cx, cy, nframes, deltax, deltay, delta_theta)
%
% Input:
%    [cx, cy] = center of image
%     nframes = number of frames
%      deltax = motion in x-direction
%      deltay = motion in y-direction
% delta_theta = rotaional motion (clockwise is postive angle)
%
% Output:   T = affine transformation of motion for each frame
%
T = zeros(3,3,nframes);
T(:,:,1) = eye(3);
for k = 2:nframes
  dx = deltax(k-1);
  dy = deltay(k-1);
  theta = delta_theta(k-1);
  %
  % Get spatial coordinates of center points of pixels:
  %   Note: If we assume the image is defined on a standard Euclidean
  %         coordinate system:
  %
  %              y ^
  %                |
  %                |
  %                |-------->
  %                         x 
  %
  % Now shift the coordinate system:
  % Create affine transformation to do the shifting of coordinates:
  % Note: According to our coordinate system, dx > 0 means move right,
  %       and dy > 0 means moves up.
  %
  % The affine transformation for this is:
  %
  T_shift = [1 0 0; 0 1 0; -dx -dy 1];
  %
  %  Note that standard rotation is about (0,0).  So we first need
  %  to shift the center of the image to the coordinate (0,0), then
  %  rotate, then shift back.
  %
  %  Shfit to (0,0):
  %
  SL = [1 0 0;0 1 0;-cx -cy 1];
  %  Rotate by angle theta.
  TR = [cos(theta), sin(theta), 0;-sin(theta), cos(theta), 0; 0, 0, 1];
  %  Now shift back.
  SR = [1 0 0;0 1 0; cx, cy, 1];
  % The combined transformation is:
  T_rot_center = SL*TR*SR;
  %
  %  Finally, the combined tranformation that shifts and rotates about
  %  center is:
  %
  T(:,:,k) = T_shift * T_rot_center;
end

%
%-----------------------
%
function image_out = ApplyMotionTransform(T, m, n, image_in)
%
%  Use the motion information, as described by affine transformations,
%  to create a sparse matrix and apply as A*image_in.
%
%  Inputs: T      = affine transformations describing motion
%          [m, n] = image size
%
%  Output: image_out is the motion blurred image

nframes = size(T,3);

% Get pixel centers of composite image.
[X, Y] = GetPixelCenters(m, n);
image_out = zeros(m*n,1);
for t = 1:nframes

  %  Use affine transfromation to transfrom coordinates:
  [Xnew, Ynew] = TransformCoordinates(T(:,:,t), X, Y);

  %  Get MATLAB indices corresponding to these new coordinates:
  [I, J] = SpaceToMatlabIndex(Xnew, Ynew);

  % Now build the sparse matrix that does the geometric
  % transformation on the image, and multiply.
  S = BuildInterpMatrix(I, J, m, n);
  image_out = image_out + S*image_in(:);
end
image_out = reshape(image_out, m, n) / nframes;

%
%--------------------
%
function [X, Y] = GetPixelCenters(m,n)
%
%           [X, Y] = GetPixelCenters(m,n);
%
%  Find the (x,y) coordinates of the centers of pixels of an image.
%  NOTE:  Here it is assumed that the image boundaries are defined by the
%         standard Euclidean coordinate system:
%              y ^
%                |
%                |
%                |-------->
%                         x
%         and the centers of the pixels are given at (x,y), where
%         x = 0, 1, 2, ..., n-1,
%         y = 0, 1, 2, ..., m-1
%
%  Input:  m, n = dimension of the image
%  Output: X = x-coordinates of centers of the pixels
%          Y = y-coordinates of centers of the pixels

[X, Y] = meshgrid(0:n-1, m-1:-1:0);

%
%----------------------
%
function [Xnew, Ynew] = TransformCoordinates(T, X, Y)
%
%  Transform (x,y) coordinates given by X and Y using the
%  affine transformation given by T.  That is,
%     [xnew, ynew] = [x, y, 1]*T
%
W = [X(:), Y(:), ones(length(X(:)), 1)] * T;
Xnew = reshape(W(:,1), size(X));
Ynew = reshape(W(:,2), size(Y));

%
%------------------------
%
function [I, J] = SpaceToMatlabIndex(X, Y)
%
%  Convert the Euclidean spatial coordinates given by (X,Y) to
%  MATLAB indices.  

m = size(Y,1);
I = m - Y;
J = X + 1;

%
%-----------------------
%
function S = BuildInterpMatrix(I, J, m, n)
%
%  Given pixel coordinates, (I,J) = (row, col), which
%  are not necessarily integers, this function computes
%  an interpolation matrix S so that y = S*x interpolates
%  pixel values in the image x to give pixel values in
%  the image y.
%
%  Here we use linear interpolation.
%
%  Input:
%    I and J are arrays of size m-by-n (same as image dimensions).
%       These contain coordinates of an image transformation.
%    m, n is the size of the image.
%
%  Output:
%    S is a sparse matrix

% First find integer coordinates that surround each (I(i,j),J(i,j))
% These will be the bilinear interpolation coordinates.
i0 = floor(I(:));
j0 = floor(J(:));
i1 = i0+1;
j1 = j0+1;

% To avoid playing games with indices, we are very sloppy
% around the border.  Interpolation is only done if ALL of the
% points that surround (I(i,j),J(i,j)) are within the image
% boarders.  This makes it easier to compute the interpolation
% weights without having to use special conditions near
% the borders.  If we assume black areas near the border,
% then this should not cause any problems - in particular, if we want
% to only find a PSF, this is okay to do.

% The first step, then, is to find the rows that will contain weights.
row_idx = find(1<=i0 & i1<=m & 1<=j0 & j1<=n);
i = [row_idx; row_idx; row_idx; row_idx];

% Since we only consider interior pixel values, then each
% row will have exactly four weights.  So next we find the
% four column indices where these weights should be put.
% We are assuming column ordering of the image.
i0 = i0(row_idx); i1 = i1(row_idx); j0 = j0(row_idx); j1 = j1(row_idx);
col_idx1 = i0 + m*(j0-1);
col_idx2 = i1 + m*(j0-1);
col_idx3 = i0 + m*(j1-1);
col_idx4 = i1 + m*(j1-1);
j = [col_idx1; col_idx2; col_idx3; col_idx4];

% Now we compute the weights that go into the matrix.
deltai = I(row_idx) - i0;
deltaj = J(row_idx) - j0;
w1 = (1 - deltai).*(1 - deltaj);
w2 = (1 - deltaj).*deltai;
w3 = (1 - deltai).*deltaj;
w4 = deltai.*deltaj;
s = [w1; w2; w3; w4];

% Now let's put the weights in the sparse matrix.
S = sparse(i, j, s, m*n, m*n);


function [n, nbig, getidx] = GetDimensionInfo(n, PSF, image, default_n, CommitCrime)
%
% This function is used to get dimension information for output
% images and PSF based on the given input information.

if isnumeric(n) && length(n) == 1
    n = [n, n];
end

if isempty(n) && ~isnumeric(PSF) && ~isnumeric(image)
    n = default_n; 
    %np = [n, n]; 
    if strcmpi(CommitCrime, 'off')
        nbig = 2*n; 
    else
        nbig = n;
    end
elseif isempty(n) && ~isnumeric(PSF) && isnumeric(image)
    n = size(image); 
    %np = n; 
    nbig = n;
    warning('Since no ouput image size was specified, we are force to commit the inverse crime for boundary condtions')
elseif isempty(n) && isnumeric(PSF) && ~isnumeric(image)     
    np = size(PSF); n = max(np, default_n); 
    if strcmpi(CommitCrime, 'off')
        nbig = 2*n; 
    else
        nbig = n;
    end
elseif isempty(n) && isnumeric(PSF) && isnumeric(image)
    nx = size(image); np = size(PSF);
    if nx(1) < np(1) || nx(2) < np(2)
        error('PSF size should be smaller than image size')
    else
        n = np; 
        if strcmpi(CommitCrime, 'off')
            nbig = nx;
            if nbig < 2*n
                warning('Due to the input image and PSF sizes, a partial inverse crime may be committed at the boundaries')
            end
        else
            nbig = n;
        end
    end        
elseif isnumeric(n) && ~isnumeric(PSF) && ~isnumeric(image)
    if strcmpi(CommitCrime, 'off')
        nbig = 2*n; 
    else
        nbig = n;
    end
elseif isnumeric(n) && isnumeric(PSF) && ~isnumeric(image)
    np = size(PSF);
    if n(1) < np(1) || n(2) < np(2)
        error('PSF size should be smaller than image size')
    else
        if strcmpi(CommitCrime, 'off')
            nbig = 2*n; 
        else
            nbig = n;
        end
    end
elseif isnumeric(n) && ~isnumeric(PSF) && isnumeric(image)
    nx = size(image);
    if nx(1) < n(1) || nx(2) < n(2)
        error('PSF size should be smaller than image size')
    else
        if strcmpi(CommitCrime, 'off')
            nbig = nx;
            if nx(1) < 2*n(1) || nx(2) < n(2)
                warning('Due to the input image and PSF sizes, a partial inverse crime may be committed at the boundaries')
            end
        else
            nbig = n;
        end
    end
elseif isnumeric(n) && isnumeric(PSF) && isnumeric(image)
    nx = size(image); np = size(PSF);
    if n(1) < min(nx(1), np(1)) || n(2) < min(nx(2), np(2))
        error('Input PSF and image sizes should be smaller then n')
    elseif nx(1) < np(1) || nx(2) < np(2)
        error('Input PSF size should be smaller than input image size')
    else
        if strcmpi(CommitCrime, 'off')
            nbig = nx;
            warning('Due to the input image and PSF sizes, a partial inverse crime may be committed at the boundaries')
        else
            nbig = n;
        end
    end
else
    error('Something is wrong with inputs, or this code!')
end
getidx = fix((nbig-n)/2)+1;

function x = StarImageRandom(n, nstars)
%
%  n      - size of image
%  nstars - specified number of stars, default is 100
%           if nstars = 2, then create binary star image in center

% Stars are modeled as circular Gaussians.
% "centers" specify locations of stars, where the value "shift" is used
%           to make sure the stars are not too close to the boundary
%           of the image domain
% "sigmas" specify random variances for the Gaussians, 
%          in the range [sigma_min, sigma_max]
% "magnitudes" specify random star magnitudes, in the range [mag_min, mag_max]

if nargin == 0
    n = []; nstars = [];
elseif nargin == 1
    nstars = [];
elseif nargin > 2
    error('incorrect number of inputs')
end
if isempty(n), n = 256; end
if isempty(nstars), nstars = 100; end

shift = 10;
sigma_max = 2; sigma_min = 1;
mag_max = 1; mag_min = 0.1;
if nstars == 2
    centers = round([n/2, n/2]);
    centers = [centers; centers + [0, shift]];
else
    centers = shift + randi(n-2*shift, nstars, 2);
end
sigmas = (sigma_max - sigma_min)*rand(nstars, 1) + sigma_min;
magnitudes = (mag_max - mag_min)*rand(nstars, 1) + mag_min;
x = zeros(n, n);
[J,I] = meshgrid(1:n,1:n);
for k = 1:nstars
    f = 1/sigmas(k)^2;
    stark = f*exp(-((0.5*f)*((I-centers(k,1)).^2) + (0.5*f)*((J-centers(k,2)).^2)));
    x = x + magnitudes(k)*stark;
end
x = x/max(x(:));