function  [A,Y,tol] = syl_gen(n,d,err)

%  <Purpose>
%     Construct a Sylvester matrix of two polynomials with perturbation;
%
%  <Syntax>
%     [A,Y,tol] = syl_gen(n,d,err);
%
%  <Description>
%     Matrix A is a 2n-by-2n Sylvester matrix of two randomly generated
%     polynomials of degree n with perturbation err. d is the degree of
%     the numerical GCD within err.
%
%  <Input>
%       n -- the degree of polynomials;
%       d -- the degree of numerical GCD (default: d = 10);
%     err -- the perturbation of coefficients (default: err = 0.0);
%
%  <Output>
%       A -- the 2n-by-2n Sylvester matrix;
%       Y -- the orthogonal complement of the numerical kernel;
%     tol -- the suggested threshold;
%

if ( n <= 2 )
    error('n > 2 is required.')
end

if nargin <= 1
    d = min(10, round(n/2));
end

if nargin <= 2
    err = 0.0;
    tol = sqrt(10*n)*1e-12;
else
    if (err < 1e-12)
        tol = sqrt(10*n)*1e-10;
    else
        tol = sqrt(n)*err*10;
    end
end;

if ( d < 0 )
    error('d > 0 is required.')
end

if ( d >= n )
    error('d < n is required.')
end

if ( err < 0 )
    error('err > 0.0 is required.')
end

% generate random polynomials with a GCD
u = [1; randn(d,1)];
v = randn(n-d+1,1);
w = randn(n-d+1,1);
f = conv(u,v) + err*randn(n+1,1);
g = conv(u,w) + err*randn(n+1,1);

% construct the Sylvester matrix
A = zeros(2*n,2*n);
for k = 1:n
    A(k:n+k,k) = f;
    A(k:n+k,n+k) = g;
end;

% Construct the orthogonal complement of the numerical kernel
[~,~,Y] = svd(A);
Y = Y(:,1:n-d);
