function [ark,Basis,C] = NumericalRankUpdate(A,pth,a,C,RC)

% <Purpose>
%   Updating the numerical rank of an inserted matrix.
%
% <Syntax>
%   [r,Basis,C] = NumericalRankUpdate(A,pth,vec,C,RC);
%
% <Input Parameters>
%   1.   A -- the target matrix.
%   2. pth -- the index of row/column to be inserted.
%   3. vec -- the row/column vector to be inserted.
%   4.   C -- cell array contains data required by updating/downdating.
%   5.  RC -- string parameter
%             If RC = 'row', then the pth row is inserted.
%             If RC = 'column', then the pth column is inserted.
%
% <Output Parameters>
%   1. r     -- the numerical rank of the inserted matrix.
%   2. Basis -- (optional)
%       When running the high rank-revealing algorithm, its columns
%       form an orthonormal basis of the numerical kernel;
%       When running the low rank-revealing algorithm, its columns
%       form an orthonormal basis of the numerical range;
%   3.     C -- (optional)
%       Matlab cell array contains data required by updating/downdating.
%       When running the high rank-revealing algorithm,
%       C{1,1} = rank : the numerical rank of the new matrix;
%       C{2,1} = Basis : an orthonormal basis of the numerical kernel;
%       C{3,1} = Q : the Q in the kernel stacked QR factorization;
%       C{4,1} = R : the R in the kernel stacked QR factorization;
%       C{5,1} = tau : the scaling factor in the new kernel stacked matrix;
%       C{6,1} = tol : the rank decision threshold;
%       When running the low rank-revealing algorithm,
%       C{1,1} = rank : the numerical rank of the new matrix;
%       C{2,1} = U : U in the USV+E decomposition of the new matrix;
%       C{3,1} = V : V in the USV+E decomposition of the new matrix;
%       C{4,1} = S : S in the USV+E decomposition of the new matrix;
%       C{5,1} = tol : the rank decision threshold;
%

if ( ~strcmp(RC,'row') && ~strcmp(RC,'column') )
    error('The sixth input should be "row" or "column". ')
end

if (size(C,1) == 6)
    ark = C{1,1};
    NB = C{2,1};
    Q = C{3,1};
    R = C{4,1};
    tau = C{5,1};
    tol = C{6,1};
    if ( strcmp(RC,'row') )
        [ark,NB,R,Q] = hrowup(ark,Q,R,NB,a,pth,tol);
    else
        [ark,NB,R,Q] = hcolup(ark,Q,R,NB,tau,a,pth,tol);
    end
    Basis = NB;
    C{1,1} = ark;
    C{2,1} = NB;
    C{3,1} = Q;
    C{4,1} = R;
elseif (size(C,1) == 5)
    ark = C{1,1};
    U = C{2,1};
    V = C{3,1};
    S = C{4,1};
    tol = C{5,1};
    if ( strcmp(RC,'row') )
        [ark,U,S,V] = lrowup(ark,A,U,S,V,a,pth,tol);
    else
        [ark,U,S,V] = lcolup(ark,A,U,S,V,a,pth,tol);
    end
    Basis = U;
    C{1,1} = ark;
    C{2,1} = U;
    C{3,1} = V;
    C{4,1} = S;
end;
%-----------------------------------------------------------------------
% End of function
%-----------------------------------------------------------------------

function [ark,Y,R,Q] = hrowup(ark,Q,R,NB,a,pth,tol)

% <Purpose>
%   Updating the kernel stacked QR factorization for inserting a row.
%
% <Syntax>
%   [ark,Y,R,Q] = hrowup(ark,Q,R,NB,a,pth,tol);
%
% <Input Parameters>
%    1.   ark -- the numerical rank.
%    2.     Q -- the Q in the kernel stacked QR factorization.
%    3.     R -- the R in the kernel stacked QR factorization.
%    4.    NB -- an orthonormal basis for numerical kernel.
%    5.     a -- the row to be inserted.
%    6.   pth -- the index of row to be inserted.
%    7.   tol -- the rank-decision threshold.
%
% <Output Parameters>
%    1.   ark -- the numerical rank of the inserted matrix.
%    2.     Y -- an orthonormal basis for the numerical kernel.
%    3.     R -- the R in the kernel stacked QR factorization.
%    4.     Q -- the Q in the kernel stacked QR factorization.

if (nargin < 1)
    error('Not enough input arguments.')
end
[m,n] = size(R);
if (m <= n)
    error('The system needs to be overdetermined;')
end;

nul = n - ark;
y = (a*NB)';
if (norm(y,2) < tol)
    Y = NB;
else
    [u,~] = housevec(y);
    H = eye(nul) - (2*u)*u';
    Y = NB*H(:,2:nul);
    ark = ark + 1;
    if (nargout > 2)
        Q(1:nul,:) = H*Q(1:nul,:);
        [Q,R] = qrdown(Q,R,1);
        nul = nul - 1; m = m - 1;
    end;
end;

if (nargout > 2)
    R = [a;R];
    Q = [[1,zeros(1,m)];[zeros(m,1),Q]];
    nk = nul + pth;
    b = Q(1,:);
    Q(1:nk-1,:) = Q(2:nk,:);
    Q(nk,:) = b;
    for j = 1:n
        [T,~] = givensmt(R(j:j+1,j));
        R(j:j+1,j:n) = T*R(j:j+1,j:n); R(j+1,j) = 0;
        Q(:,j:j+1) = Q(:,j:j+1)*T';
    end;
end;
%-----------------------------------------------------------------------
% End of function hrowup
%-----------------------------------------------------------------------

function [ark,Y,R,Q] = hcolup(ark,Q,R,NB,tau,b,pth,tol)

% <Purpose>
%   Updating the kernel stacked QR factorization for inserting a column.
%
% <Syntax>
%   [ark,Y,R,Q] = hcolup(ark,Q,R,NB,tau,b,pth,tol);
%
% <Input Parameters>
%    1.   ark -- the numerical rank.
%    2.     Q -- the Q in the kernel stacked QR factorization.
%    3.     R -- the R in the kernel stacked QR factorization.
%    4.    NB -- an orthonormal basis for numerical kernel.
%    5.   tau -- scaling factor.
%    6.     b -- the column to be inserted.
%    7.   pth -- the index of column to be inserted.
%    8.   tol -- the rank decision threshold.
%
% <Output Parameters>
%    1.   ark -- the numerical rank of the inserted matrix.
%    2.     Y -- an orthonormal basis for the numerical kernel.
%    3.     R -- the R in the kernel stacked QR factorization.
%    4.     Q -- the Q in the kernel stacked QR factorization.

if (nargin < 1)
    error('Not enough input arguments.')
end
[m,n] = size(R);
if (m <= n)
    error('The system needs to be overdetermined;')
end;

nul = n - ark;    %  nullity;
n1 = n + 1;
% update Q, R
h = ( [zeros(1,nul) b']*Q )';
[u,t] = housevec(h(n1:m));
h(n1) = t; h(n1+1:m) = 0;
Q(:,n1:m) = Q(:,n1:m) - 2*(Q(:,n1:m)*u)*u';
R = [R h];
n = n1;

% find the possible additional null vector
[v,s] = singular_n(R,tau,tol);
Y = [NB;zeros(1,nul)];
if (s < tol)
    Y = [v Y];
    if (nargout > 2)
        R = [tau*v'; R];
        [R,trans] = hesqr(R);
        Q = [ [1,zeros(1,m)]; [zeros(m,1),Q] ];
        for j = 1:size(trans,2)
            T = [trans(:,j),[-conj(trans(2,j));conj(trans(1,j))]];
            Q(:,j:j+1) = Q(:,j:j+1)*T';
        end
    end;
else
    ark = ark + 1;
end;

% additional computation if the column is inserted in the middle
if (pth < n)
    Y = Y([1:pth-1,n,pth:n-1],:);
    if (nargout > 2)
        R = R(:,[1:pth-1,n,pth:n-1]);
        for j = n-1:-1:pth
            [T,d] = givensmt(R(j:j+1,pth));
            R(j:j+1,pth) = [d;0];
            R(j:j+1,j+1:n) = T*R(j:j+1,j+1:n);
            Q(:,j:j+1) = Q(:,j:j+1)*T';
        end;
    end;
end;
%-----------------------------------------------------------------------
% End of function hcolup
%-----------------------------------------------------------------------

function [v,s] = singular_n(R,scale,tol)

% <Purpose>
%   Computing the smallest singular value of an upper triangular matrix.
%
% <Syntax>
%   [v,s] = sigular_n(R,scale,tol);
%
% <Input parameters>
%   1.     R  -- an upper triangular matrix.
%   2. scale  -- the norm of R.
%   3.   tol  -- the rank decision threshold.
%
% <Output Parameters>
%   1.     v  -- the smallest singular vector.
%   2.     s  -- the smallest singular value.

n = size(R,2);
t1 = tol/sqrt(n);
t2 = max(1.0,scale)*eps*sqrt(n);
ztol = max(t1,t2);

v = randn(n,1);
tmp = 1/norm(v);
v = tmp*v;
u = zeros(1,n);
max_iter = 8;
for k = 1:n
    if (abs(R(k,k)) < eps)
        R(k,k) = eps;
    end
end;

for k = 1:max_iter
    %--------------------------- forward elimination
    u(1) = v(1)/R(1,1);
    for kk = 2:n
        u(kk) = (v(kk) - u(1:kk-1)*R(1:kk-1,kk))/R(kk,kk);
    end;
    tmp = 1/norm(u);
    u = tmp*u;
    %--------------------------- backward substitution
    v(n) = u(n)/R(n,n);
    for kk = (n-1):-1:1
        v(kk) = (u(kk) - R(kk,kk+1:n)*v(kk+1:n))/R(kk,kk);
    end;
    s = 1/norm(v);
    v = s*v;
    %---------------------------
    if (k > 1)
        if (abs(s - s_old)/s < ztol)
            break;
        end
        if (s < tol)
            break;
        end
    end;
    s_old = s;
end;
%-----------------------------------------------------------------------
% End of function
%-----------------------------------------------------------------------

function [A,trans] = hesqr(A)

% <Syntax>
%   [A,trans] = hesqr(A);
%
% <Input parameters>
%          A -- an upper Hessenberg matrix. (m >= n)
%
% <Output parameters>
%   1.     A -- an upper triangular matrix.
%   2. trans -- rotation used.

[~,n] = size(A);
trans = zeros(2,n);
for j = 1:n
    [G,d] = givensmt(A(j:j+1,j));
    if (abs(d) < eps)
        A(j:j+1,j) = [eps;0];
        trans(1:2,j) = [1;0];
    else
        if (j < n)
            A(j:j+1,j:n) = G*A(j:j+1,j:n);
        else
            A(n:n+1,n) = [d;0];
        end
        trans(:,j) = G(:,1);
    end
end
%-----------------------------------------------------------------------
% End of function
%-----------------------------------------------------------------------

function [G,d] = givensmt(v)

% <Purpose>
%   Computing the Givens matrix G such that G*v = [d;0].
%
% <Syntax>
%   [G,d] = givensmt(v);
%
% <Input Parameters>
%      v -- a target vector. ( 2-by-1 )
%
% <Output Parameters>
%   1. G -- the 2x2 Givens rotation matrix.
%   2. d -- G*v = [d;0].

d = norm(v,2);
if (abs(d) < eps)
    G = eye(2);
else
    c = v(1)/d; s = v(2)/d;
    G = [conj(c),conj(s);-s,c];
end;
%-----------------------------------------------------------------------
% End of function
%-----------------------------------------------------------------------

function [Q,R] = qrdown(Q,R,pth)

% <Description>
%   QR-downdating the p-th row.
%
% <Syntax>
%   [Q,R] = qrdown(Q,R,pth);

[m,n] = size(R);
if (pth < 1)
    error('the 3rd input should be a positive integer')
elseif (pth > m)
    error('the 3rd input argument is too large')
else
    Q = Q([pth,1:pth-1,pth+1:m],:);
end;

q = Q(1,:)';
for j = m-1:-1:1
    [T,d] = givensmt(q(j:j+1));
    q(j:j+1) = [d;0];
    Q(:,j:j+1) = Q(:,j:j+1)*T';
    if (j <= n)
        R(j:j+1,j:n) = T*R(j:j+1,j:n);
    end;
end;
Q = Q(2:m,2:m);  R = R(2:m,:);
%-----------------------------------------------------------------------
% End of function
%-----------------------------------------------------------------------

function [u,t] = housevec(v)

% <Purpose>
%   Computing the Householder vector u in H = I - 2*u*u'
%   that makes Hv = t*e_1
%
% <Syntax>
%   [u,t] = housevec(v);
%
% <Input Parameters>
%      v -- a target vector.
%
% <Output Parameters>
%   1. u -- the vector in H = I - 2*u*u'.
%   2. t -- Hv = t*e_1

if (abs(v(1)) < eps)
    s = norm(v);
else
    s = norm(v)*v(1)/abs(v(1));
end;
u = v;
u(1) = u(1) + s;
nrm = norm(u);
if nrm > eps
    u = u/nrm;
end;
t = -s;
%-----------------------------------------------------------------------
% End of function
%-----------------------------------------------------------------------

function [ark,U,S,V] = lrowup(k,A,U,S,V,a,pth,tol,num_ref)

% <Purpose>
%   Updating the USV+E decomposition for inserting a row
%
% <Syntax>
%   [ark,U,S,V] = lrowup(k,A,U,S,V,a,pth,tol,num_ref);
%
% <Input Parameters>
%   1.      k  -- the numerical rank of A.
%   2.      A  -- the target matrix A.
%   3.  U,S,V  -- the USV+E decomposition.
%   4.      a  -- the row to be inserted.
%   5.    pth  -- the index of row to be inserted.
%   6.    tol  -- the rank decision tolerance.
%   7. num_ref -- the number of refinement.
%
% <Output Parameters>
%   1.     ark -- the numerical rank of the new matrix.
%   2.   U,S,V -- the USV+E decomposition of the new matrix.

m = size(U,1); n = size(V,1);
if (nargin < 9)
    num_ref = 0;
end
%-----------------------------------
a1 = a*V;
nrma1 = norm(a1);
if (nrma1 < sqrt(n)*eps)
    ark = k;
    return
end
vt = a-a1*V';
beta = norm(vt);

An = [A(1:pth-1,:);a;A(pth:m,:)];
if (beta > tol)
    if (beta < 50*tol)
        [ark,U,S,V] = Refine_sv(An,V,tol);
        return
    end
    ark = k + 1;
    tmp = 1/beta;
    x = tmp*vt';
    if (beta < 0.1*nrma1)
        tmpv = V*(V'*x); x = x-tmpv;
        tmp = 1/norm(x);
        x = tmp*x;
    end
    V = [V x];
else
    if (beta > 0.5*tol)
        [ark,U,S,V] = Refine_sv(An,V,tol);
        return
    end
    ark = k;
end
%----------------
U = An*V;
[U,~] = qr(U,0);
V = (U'*An)';
[V,~] = qr(V,0);
for j = 1: num_ref
    U = An*V;
    [U,~] = qr(U,0);
    V = (U'*An)';
    [V,~] = qr(V,0);
end
U = An*V;
[U,S] = qr(U,0);
%-----------------------------------------------------------------------
% End of function
%-----------------------------------------------------------------------

function [k,U,S,V] = Refine_sv(A,V,tol)

% <Purpose>
%   Refine the SV in the USV+E decomposition.
%
% <Syntax>
%   [k,U,S,V] = Refine_sv(A,V,tol);

k = size(V,2);
scale = norm(A,inf);

U = A*V;
[U,~] = qr(U,0);
V = (U'*A)';
[V,~] = qr(V,0);
[~,n] = size(A);
t2 = max(1.0,scale)*eps*sqrt(n);
x = randn(n,1);
tmp = 1/norm(x);
x = tmp*x;
max_iter = 8;
for j = 1:max_iter
    %----------------------------------------
    tv = (x'*V)';
    tu = V*tv;
    h = x - tu;
    y = A*h;
    %----------------------------------------
    nrmy = norm(y);
    y = (1/nrmy)*y;
    if (j > 1)
        check = abs(nrmy-os)/nrmy;
        if (check < t2), break; end
    end
    os = nrmy;
    %-----------------------------
    tv = (y'*A)';
    tu = (tv'*V)';
    h = V*tu;
    x = tv - h;
    %-----------------------------
    nrmx = norm(x);
    x = (1/nrmx)*x;
    eta = (tol/nrmx)^(2*j);
    if (eta < t2), break; end
end

if (nrmx > tol)
    k = k + 1;
    V = [V x];
end
U = A*V;
[U,S] = qr(U,0);
%-----------------------------------------------------------------------
% End of function
%-----------------------------------------------------------------------

function [ark,U,S,V] = lcolup(k,A,U,S,V,b,pth,tol,num_ref)

% <Purpose>
%   Updating the USV+E decomposition for inserting a column
%
% <Syntax>
%   [ark,U,S,V] = lcolup(k,A,U,S,V,b,pth,tol,num_ref);
%
% <Input Parameters>
%   1.      k  -- the numerical rank of A.
%   2.      A  -- the target matrix A.
%   3.  U,S,V  -- the USV+E decomposition.
%   4.      b  -- the column to be inserted.
%   5.    pth  -- the index of column to be inserted.
%   6.    tol  -- the rank decision tolerance.
%   7. num_ref -- the number of refinement.
%
% <Output Parameters>
%   1.     ark -- the numerical rank of the new matrix.
%   2.   U,S,V -- the USV+E decomposition of the new matrix.

m = size(U,1); n = size(V,1);
if (nargin < 9)
    num_ref = 0;
end
%-----------------------------------
b1 = (b'*U)';
nrmb1 = norm(b1);
if (nrmb1 < sqrt(m)*eps)
    ark = k;
    return
end
ut = b-U*b1;
beta = norm(ut);

An = [A(:,1:pth-1) b A(:,pth:n)];
if (beta > tol)
    if (beta < 50*tol)
        [ark,U,S,V] = Refine_su(An,U,tol);
        return
    end
    ark = k + 1;
    tmp = 1/beta;
    y = tmp*ut;
    if (beta < 0.1*nrmb1)
        tmpu = U*(y'*U)'; y = y-tmpu;
        tmp = 1/norm(y);
        y = tmp*y;
    end
    U = [U y];
else
    if (beta > 0.5*tol)
        [ark,U,S,V] = Refine_su(An,U,tol);
        return
    end
    ark = k;
end
%----------------
V = (U'*An)';
[V,~] = qr(V,0);
U = An*V;
[U,~] = qr(U,0);
for j = 1: num_ref
    V = (U'*An)';
    [V,~] = qr(V,0);
    U = An*V;
    [U,~] = qr(U,0);
end
V = (U'*An)';
[V,S] = qr(V,0);
S = S';
%-----------------------------------------------------------------------
% End of function
%-----------------------------------------------------------------------

function [k,U,S,V] = Refine_su(A,U,tol)

% <Purpose>
%   Refine the SU in the USV+E decomposition.
%
% <Syntax>
%   [k,U,S,V] = Refine_su(A,U,tol);

k = size(U,2);
scale = norm(A,inf);

V = (U'*A)';
[V,~] = qr(V,0);
U = A*V;
[U,~] = qr(U,0);
[m,~] = size(A);
t2 = max(1.0,scale)*eps*sqrt(m);
y = randn(m,1);
tmp = 1/norm(y);
y = tmp*y;
max_iter = 8;
for j = 1:max_iter
    %-----------------------------
    tv = (y'*U)';
    tu = U*tv;
    h = y - tu;
    x = (h'*A)';
    %-----------------------------
    nrmx = norm(x);
    tmp = 1/nrmx;
    x = tmp*x;
    if (j > 1)
        check = abs(nrmx-os)/nrmx;
        if (check < t2), break; end
    end
    os = nrmx;
    %----------------------------------------
    tv = A*x;
    tu = (tv'*U)';
    h = U*tu;
    y = tv - h;
    %----------------------------------------
    nrmy = norm(y);
    tmp = 1/nrmy;
    y = tmp*y;
    eta = (tol/nrmy)^(2*j);
    if (eta < t2), break; end
end

if (nrmy > tol)
    k = k + 1;
    U = [U y];
end
V = (U'*A)';
[V,S] = qr(V,0);
S = S';
%-----------------------------------------------------------------------
% End of function
%-----------------------------------------------------------------------
