function [ark,Basis,C] = NumericalRankDowndate(A,pth,C,RC)

% <Purpose>
%   Downdating the numerical rank of a deleted matrix.
%
% <Syntax>
%   [r,Basis,C] = NumericalRankDowndate(A,pth,C,RC);
%
% <Input Parameters>
%   1.   A -- the target matrix.
%   2. pth -- the index of row/column to be deleted.
%   3.   C -- cell array contains data required by updating/downdating.
%   4.  RC -- string parameter
%             If RC = 'row', then the pth row is deleted.
%             If RC = 'column', then the pth column is deleted.
%
% <Output Parameters>
%   1. r     -- the numerical rank of the deleted matrix.
%   2. Basis -- (optional)
%       When running the high rank-revealing algorithm, its columns
%       form an orthonormal basis of the numerical kernel;
%       When running the low rank-revealing algorithm, its columns
%       form an orthonormal basis of the numerical range;
%   3.     C -- (optional)
%       Matlab cell array contains data required by updating/downdating.
%       When running the high rank-revealing algorithm,
%       C{1,1} = rank : the numerical rank of the new matrix;
%       C{2,1} = Basis : an orthonormal basis of the numerical kernel;
%       C{3,1} = Q : the Q in the kernel stacked QR factorization;
%       C{4,1} = R : the R in the kernel stacked QR factorization;
%       C{5,1} = tau : the scaling factor in the new kernel stacked matrix;
%       C{6,1} = tol : the rank decision threshold;
%       When running the low rank-revealing algorithm,
%       C{1,1} = rank : the numerical rank of the new matrix;
%       C{2,1} = U : U in the USV+E decomposition of the new matrix;
%       C{3,1} = V : V in the USV+E decomposition of the new matrix;
%       C{4,1} = S : S in the USV+E decomposition of the new matrix;
%       C{5,1} = tol : the rank decision threshold;
%

if ( ~strcmp(RC,'row') && ~strcmp(RC,'column') )
    error('The sixth input should be "row" or "column". ')
end

if (size(C,1) == 6)
    ark = C{1,1};
    NB = C{2,1};
    Q = C{3,1};
    R = C{4,1};
    tau = C{5,1};
    tol = C{6,1};
    if ( strcmp(RC,'row') )
        [ark,NB,R,Q] = hrowdown(ark,Q,R,NB,tau,pth,tol);
    else
        [ark,NB,R,Q] = hcoldown(ark,Q,R,NB,pth,tol);
    end
    Basis = NB;
    C{1,1} = ark;
    C{2,1} = NB;
    C{3,1} = Q;
    C{4,1} = R;
elseif (size(C,1) == 5)
    ark = C{1,1};
    U = C{2,1};
    V = C{3,1};
    S = C{4,1};
    tol = C{5,1};
    if ( strcmp(RC,'row') )
        [ark,U,S,V] = lrowdown(ark,A,U,S,V,pth,tol);
    else
        [ark,U,S,V] = lcoldown(ark,A,U,S,V,pth,tol);
    end
    Basis = U;
    C{1,1} = ark;
    C{2,1} = U;
    C{3,1} = V;
    C{4,1} = S;
end;
%-----------------------------------------------------------------------
% End of function
%-----------------------------------------------------------------------

function [ark,Y,R,Q] = hrowdown(ark,Q,R,NB,tau,pth,tol)

% <Purpose>
%   Downdating the kernel stacked QR factorization for deleting a row.
%
% <Syntax>
%   [ark,Y,R,Q] = hrowdown(ark,Q,R,NB,tau,pth,tol);
%
% <Input Parameters>
%   1. ark -- the numerical rank;
%   2.   Q -- the Q in the kernel stacked QR factorization;
%   3.   R -- the R in the kernel stacked QR factorization;
%   4.  NB -- an orthonormal basis for numerical kernel;
%   5. tau -- scaling factor;
%   6. pth -- the index of row to be deleted;
%   7. tol -- the rank decision threshold;
%
% <Output Parameters>
%   1. ark -- the numerical rank of the deleted matrix;
%   2.   Y -- an orthonormal basis for the numerical kernel;
%   3.   R -- the R in the kernel stacked QR factorization;
%   4.   Q -- the Q in the kernel stacked QR factorization;

if (nargin < 1)
    error('Not enough input arguments.')
end
[m,n] = size(R);
if (m <= n)
    error('The system needs to be overdetermined;')
end;

nul = n - ark;    %  nullity;
[Q,R] = qrdown(Q,R,pth+nul);     % downdate the QR decomposition
m = m - 1;

% find the additional kernel vector if it exists
[v,s] = singular_n(R,tau,tol);
if (s < tol)
    Y = [v,NB];           % append the new kernel vector
    ark = ark - 1;
    if (nargout > 2)
        R = [tau*v'; R];  % update the QR decomposition
        [R,trans] = hesqr(R);
        Q = [ [1,zeros(1,m)]; [zeros(m,1),Q] ];
        for j = 1:size(trans,2)
            T = [trans(:,j),[-conj(trans(2,j));conj(trans(1,j))]];
            Q(:,j:j+1) = Q(:,j:j+1)*T';
        end
    end
else
    Y = NB;
end;
%-----------------------------------------------------------------------
% End of function hrowdown
%-----------------------------------------------------------------------

function [ark,Y,R,Q] = hcoldown(ark,Q,R,NB,pth,tol)

% <Purpose>
%   Downdating the kernel stacked QR factorization for deleting a column.
%
% <Syntax>
%   [ark,Y,R,Q] = hcoldown(ark,Q,R,NB,pth,tol);
%
% <Input Parameters>
%   1. ark -- the numerical rank;
%   2.   Q -- the Q in the kernel stacked QR factorization;
%   3.   R -- the R in the kernel stacked QR factorization;
%   4.  NB -- an orthonormal basis for numerical kernel;
%   5. pth -- the index of column to be deleted;
%   6. tol -- the rank decision threshold;
%
% <Output Parameters>
%   1. ark -- the numerical rank of the deleted matrix;
%   2.   Y -- an orthonormal basis for the numerical kernel;
%   3.   R -- the R in the kernel stacked QR factorization;
%   4.   Q -- the Q in the kernel stacked QR factorization;

if (nargin < 1)
    error('Not enough input arguments.')
end
[m,n] = size(R);
if (m <= n)
    error('The system needs to be overdetermined;')
end;

nul = n - ark;    %  nullity;
if (norm(NB(pth,:)) < tol)
    ark = ark - 1;
    Y = NB([1:pth-1,pth+1:n],:);
    if (nargout > 2)
        R = R(:,[1:pth-1,pth+1:n]);
    end
else
    u = housevec(NB(pth,:)');
    Y = NB - (2*(NB*u))*u';
    Y = Y([1:pth-1,pth+1:n],2:nul);
    if (nargout > 2)
        Q(1:nul,:) = Q(1:nul,:) - (2*u)*(u'*Q(1:nul,:));
        [Q,R] = qrdown(Q,R,1);
        R = R(:,[1:pth-1,pth+1:n]);
    end;
end;

if (pth < n && nargout > 2)
    for j = pth:n-1
        [T,d] = givensmt(R(j:j+1,j));
        R(j:j+1,j) = [d;0];
        R(j:j+1,j+1:n-1) = T*R(j:j+1,j+1:n-1);
        Q(:,j:j+1) = Q(:,j:j+1)*T';
    end
end;
%-----------------------------------------------------------------------
% End of function hcoldown
%-----------------------------------------------------------------------

function [v,s] = singular_n(R,scale,tol)

% <Purpose>
%   Computing the smallest singular value of an upper triangular matrix.
%
% <Syntax>
%   [v,s] = sigular_n(R,scale,tol);
%
% <Input parameters>
%   1.     R  -- an upper triangular matrix.
%   2. scale  -- the norm of R.
%   3.   tol  -- the rank decision threshold.
%
% <Output Parameters>
%   1.     v  -- the smallest singular vector.
%   2.     s  -- the smallest singular value.

n = size(R,2);
t1 = tol/sqrt(n);
t2 = max(1.0,scale)*eps*sqrt(n);
ztol = max(t1,t2);

v = randn(n,1);
tmp = 1/norm(v);
v = tmp*v;
u = zeros(1,n);
max_iter = 8;
for k = 1:n
    if (abs(R(k,k)) < eps)
        R(k,k) = eps;
    end
end;

for k = 1:max_iter
    %--------------------------- forward elimination
    u(1) = v(1)/R(1,1);
    for kk = 2:n
        u(kk) = (v(kk) - u(1:kk-1)*R(1:kk-1,kk))/R(kk,kk);
    end;
    tmp = 1/norm(u);
    u = tmp*u;
    %--------------------------- backward substitution
    v(n) = u(n)/R(n,n);
    for kk = (n-1):-1:1
        v(kk) = (u(kk) - R(kk,kk+1:n)*v(kk+1:n))/R(kk,kk);
    end;
    s = 1/norm(v);
    v = s*v;
    %---------------------------
    if (k > 1)
        if (abs(s - s_old)/s < ztol)
            break;
        end
        if (s < tol)
            break;
        end
    end;
    s_old = s;
end;
%-----------------------------------------------------------------------
% End of function
%-----------------------------------------------------------------------

function [A,trans] = hesqr(A)

% <Syntax>
%   [A,trans] = hesqr(A);
%
% <Input parameters>
%          A -- an upper Hessenberg matrix. (m >= n)
%
% <Output parameters>
%   1.     A -- an upper triangular matrix.
%   2. trans -- rotation used.

[~,n] = size(A);
trans = zeros(2,n);
for j = 1:n
    [G,d] = givensmt(A(j:j+1,j));
    if (abs(d) < eps)
        A(j:j+1,j) = [eps;0];
        trans(1:2,j) = [1;0];
    else
        if j < n
            A(j:j+1,j:n) = G*A(j:j+1,j:n);
        else
            A(n:n+1,n) = [d;0];
        end
        trans(:,j) = G(:,1);
    end
end
%-----------------------------------------------------------------------
% End of function
%-----------------------------------------------------------------------

function [G,d] = givensmt(v)

% <Purpose>
%   Computing the Givens matrix G such that G*v = [d;0].
%
% <Syntax>
%   [G,d] = givensmt(v);
%
% <Input Parameters>
%      v -- a target vector. ( 2-by-1 )
%
% <Output Parameters>
%   1. G -- the 2x2 Givens rotation matrix.
%   2. d -- G*v = [d;0].

d = norm(v,2);
if (abs(d) < eps)
    G = eye(2);
else
    c = v(1)/d; s = v(2)/d;
    G = [conj(c),conj(s);-s,c];
end;
%-----------------------------------------------------------------------
% End of function
%-----------------------------------------------------------------------

function [Q,R] = qrdown(Q,R,pth)

% <Description>
%   QR-downdating the p-th row.
%
% <Syntax>
%   [Q,R] = qrdown(Q,R,pth);

[m,n] = size(R);
if (pth < 1)
    error('the 3rd input should be a positive integer')
elseif (pth > m)
    error('the 3rd input argument is too large')
else
    Q = Q([pth,1:pth-1,pth+1:m],:);
end;

q = Q(1,:)';
for j = m-1:-1:1
    [T,d] = givensmt(q(j:j+1));
    q(j:j+1) = [d;0];
    Q(:,j:j+1) = Q(:,j:j+1)*T';
    if (j <= n)
        R(j:j+1,j:n) = T*R(j:j+1,j:n);
    end;
end;
Q = Q(2:m,2:m);  R = R(2:m,:);
%-----------------------------------------------------------------------
% End of function
%-----------------------------------------------------------------------

function [u,t] = housevec(v)

% <Purpose>
%   Computing the Householder vector u in H = I - 2*u*u'
%   that makes Hv = t*e_1
%
% <Syntax>
%   [u,t] = housevec(v);
%
% <Input Parameters>
%      v -- a target vector.
%
% <Output Parameters>
%   1. u -- the vector in H = I - 2*u*u'.
%   2. t -- Hv = t*e_1

if (abs(v(1)) < eps)
    s = norm(v);
else
    s = norm(v)*v(1)/abs(v(1));
end;
u = v;
u(1) = u(1) + s;
nrm = norm(u);
if (nrm > eps)
    u = u/nrm;
end;
t = -s;
%-----------------------------------------------------------------------
% End of function
%-----------------------------------------------------------------------

function [ark,U,S,V] = lrowdown(k,A,U,S,V,pth,tol,num_ref)

% <Purpose>
%   Downdating the USV+E decomposition for deleting a row.
%
% <Syntax>
%   [ark,U,S,V] = lrowdown(ark,A,U,S,V,pth,tol,num_ref);
%
% <Input Parameters>
%   1.     ark -- the numerical rank of A.
%   2.      A  -- the target matrix A.
%   3.   U,S,V -- the USV+E decomposition.
%   4.     pth -- the index of row to be deleted.
%   5.     tol -- the rank decision tolerance.
%   6. num_ref -- the number of refinement.
%
% <Output Parameters>
%   1.     ark -- the numerical rank of the new matrix.
%   2.   U,S,V -- the USV+E decomposition of the new matrix.

[m,n] = size(A);
if (nargin < 8)
    num_ref = 0;
end
%----------------------------
a = norm(A(pth,:));
if (a < sqrt(n)*eps)
    U = [U(1:pth-1,:);U(pth+1:m,:)];
    ark = k;
    return
end

An = [A(1:pth-1,:);A(pth+1:m,:)];
U = An*V;
[U,~] = qr(U,0);
V = (U'*An)';
[V,S] = qr(V,0);
for j = 1: num_ref
    U = An*V;
    [U,~] = qr(U,0);
    V = (U'*An)';
    [V,S] = qr(V,0);
end
S = S';
scale = norm(S,inf);
[~,s] = singular_n(S,scale,tol);
if (s > tol)
    ark = k;
else
    ark = k - 1;
    [X,S,Y] = svd(S);
    U = U*X; U = U(:,1:ark);
    V = V*Y; V = V(:,1:ark);
    S = S(1:ark,1:ark);
end
%-----------------------------------------------------------------------
% End of function lrowdown
%-----------------------------------------------------------------------

function [ark,U,S,V] = lcoldown(k,A,U,S,V,pth,tol,num_ref)

% <Purpose>
%   Downdating the USV+E decomposition for deleting a column.
%
% <Syntax>
%   [ark,U,S,V] = lcoldown(ark,A,U,S,V,pth,tol,num_ref);
%
% <Input Parameters>
%   1.     ark -- the numerical rank of A.
%   2.       A -- the target matrix A.
%   3.   U,S,V -- the USV+E decomposition.
%   4.     pth -- the index of column to be deleted.
%   5.     tol -- the rank decision tolerance.
%   6. num_ref -- the number of refinement.
%
% <Output Parameters>
%   1.     ark -- the numerical rank of the new matrix.
%   2.   U,S,V -- the USV+E decomposition of the new matrix.

[m,n] = size(A);
if (nargin < 8)
    num_ref = 0;
end
%----------------------------
b = norm(A(:,pth));
if (b < sqrt(m)*eps)
    V = [V(:,1:pth-1) V(:,pth+1:n)];
    ark = k;
    return
end

An = [A(:,1:pth-1) A(:,pth+1:n)];
V = (U'*An)';
[V,~] = qr(V,0);
U = An*V;
[U,S] = qr(U,0);
for j = 1: num_ref
    V = (U'*An)';
    [V,~] = qr(V,0);
    U = An*V;
    [U,S] = qr(U,0);
end
scale = norm(S,inf);
[~,s] = singular_n(S,scale,tol);
if (s > tol)
    ark = k;
else
    ark = k - 1;
    [X,S,Y] = svd(S);
    U = U*X; U = U(:,1:ark);
    V = V*Y; V = V(:,1:ark);
    S = S(1:ark,1:ark);
end
%-----------------------------------------------------------------------
% End of function lcoldown
%-----------------------------------------------------------------------
